/* USER.C

   Module providing functions for keeping track of system user information,
   such as user id's, group id's, and which users are in which groups.

   $Header: user.c,v 1.7 91/11/08 12:26:13 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


/* HEADER FILES =========================================================== */

#include <sys/types.h>

#include <my-types.h>
#include <my-defs.h>

#include "top.h"
#include "gen.h"
#include "user.h"

/* LOCAL MACRO DEFINITIONS ================================================ */

/* Radix of the ASCII character set */
#define CHAR_SET_RADIX 256

/* Hash table size -- should be prime and not divide CHAR_SET_RADIX */
#define ID_HASH_TABLE_SIZE 103

/* LOCAL TYPES ============================================================ */

/* Elements of ID hash table */
typedef struct id_hash {
    struct id_hash *next;	/* pointer to next element in bucket */
    uid_t id;			/* user or group system id */
    String name;		/* user or group login-id */
    UserNameList id_list;	/* linked list of users in a group */
} IDHashStruct;

/* LOCAL VARIABLES ======================================================== */

/* hash table */
static IDHashStruct *IDHashTable[ID_HASH_TABLE_SIZE];

/* LOCAL FUNCTIONS ======================================================== */

#ifdef MACRO_FUNCTION
static int user_HashID(name,kind)
  String name;
  IDType kind;
/* Hash 'name' with 'kind' to produce an index in the range
   [0,ID_HASH_TABLE_SIZE) such that two like id's of different kinds are
   guaranteed to return different values.
*/
#endif MACRO_FUNCTION

#define user_HashID(_name,_kind)\
    (((_name)+((int)_kind)) % ID_HASH_TABLE_SIZE)

static IDHashStruct *user_GetID(id,kind)
  uid_t id;
  IDType kind;
/* RETURNS a pointer to the IDHashStruct containing 'id' or type 'kind'; NULL
   if no such id is found.
*/
{
    IDHashStruct *curr_id;

    StepLinkedList(curr_id,IDHashTable[user_HashID(id,kind)]) {
	if (curr_id->id == id) { /* exact match */
	    return(curr_id);
	}
    }
    return(NULL);
}

int user_CompareNames(un1,un2)
  UserName **un1,**un2;
/* Comparison routine passed to qsort(3) to sort an array of pointers to
   UserName's.
*/
{
    return(strcmp((*un1)->name,(*un2)->name));
}

/* GLOBAL FUNCTIONS ======================================================= */

String User_RegisterName(id,kind,name)
  uid_t id;
  IDType kind;
  String name;
/* IMPLEMENTATION: We make a new copy of 'name' when we register it.
*/
{
    int hash_val;
    IDHashStruct *curr_id;

    /* search for the box; return NULL if it is found */
    if (user_GetID(id,kind)) return(NULL);

    /* build a new BoxHashStruct */
    curr_id = AllocOne(IDHashStruct);
    curr_id->id = id;
    CopyString(curr_id->name,name);
    curr_id->id_list = NULL;

    /* add the IDHashStruct to the correct bucket */
    hash_val = user_HashID(id,kind);
    NextOf(curr_id) = IDHashTable[hash_val];
    IDHashTable[hash_val] = curr_id;

    return(name);
}

String User_GetName(id,kind)
  uid_t id;
  IDType kind;
{
    IDHashStruct *curr_id;

    if ((curr_id=user_GetID(id,kind))) {
	return(curr_id->name);
    }
    return(NULL);
}

String User_AddUserToGroup(group_id,user_name)
  gid_t group_id;
  String user_name;
/* IMPLEMENTATION: We do not make a new copy of 'user_name', but instead
   point to the name stored in the gen.c hash table.
*/
{
    IDHashStruct *curr_id;
    UserName *u_name;

    if ((curr_id=user_GetID(group_id,GroupID))) {
	/* make a new UserName */
	u_name = User_NewUserName();
	u_name->name = Gen_GetBoxName(user_name,UserBox);

	/* add it to the front of the group */
	SpliceIntoList(curr_id->id_list,u_name);

	/* return success */
	return(u_name->name);
    }
    return(NULL);
}

UserNameList User_GetGroupMembers(group_id)
  gid_t group_id;
{
    IDHashStruct *curr_id;

    if ((curr_id=user_GetID(group_id,GroupID))) {
	return(curr_id->id_list);
    }
    return(NULL);
}

Boolean User_IsInList(list,user_name)
  UserNameList list;
  String user_name;
{
    UserName *curr_user;

    StepLinkedList(curr_user,list) {
	if (!strcmp(user_name,curr_user->name)) {
	    return(True);
	}
    }
    return(False);
}

UserNameList User_NewList(name)
  String name;
{
    UserName *result;

    result = User_NewUserName();
    NextOf(result) = NULL;
    result->name = name;
    return(result);
}

UserNameList User_NewCopyList(list)
  UserNameList list;
{
    UserNameList result,*next_ptr;

    next_ptr = &result;
    StepInitializedLinkedList(list) {
	*next_ptr = User_NewUserName();
	(*next_ptr)->name = list->name;
	next_ptr = &(NextOf(*next_ptr));
    }
    *next_ptr = (UserName *)NULL;
    return(result);
}

void User_DestroyList(list)
  UserNameList list;
{
    UserName *list_name;

    while(list) {
	list_name = list;
	Next(list);
	User_DestroyUserName(list_name);
    }
}

Boolean User_IsSameList(list1,list2)
  UserNameList list1,list2;
{
    int len1,len2,i;
    UserName *curr;
    ULint hash = 0L;
    UserName **array1,**array2;

    /* check that lengths and hash value for both is the same */
    for ((len1=0,curr=list1); curr; (len1++,Next(curr))) {
	hash ^= (ULint)(curr->name);
    }
    for ((len2=0,curr=list2); curr; (len2++,Next(curr))) {
	hash ^= (ULint)(curr->name);
    }
    if ((len1 != len2) || hash) { return(False); }

    /* copy both to arrays */
    array1 = AllocPtrArray(UserName,len1);
    array2 = AllocPtrArray(UserName,len2);
    for((i=0,curr=list1); curr; (i++,Next(curr))) { array1[i]=curr; }
    for((i=0,curr=list2); curr; (i++,Next(curr))) { array2[i]=curr; }

    /* sort both arrays */
    (void)qsort((char *)array1,len1,sizeof(UserName *),user_CompareNames);
    (void)qsort((char *)array2,len2,sizeof(UserName *),user_CompareNames);

    /* step down both to make sure they are identical */
    StepIndex(i,0,len1) {
	if ((array1[i])->name != (array2[i])->name) { break; }
    }
    Dealloc(array1); Dealloc(array2);
    return((i < len1) ? False : True);
}

void User_MinusListsD(list1,list2)
  UserNameList list1;
  INOUT UserNameList *list2;
{
    String curr2_name;
    UserName *curr1,*curr2;

    if (!list1) { return; }	/* return if 'list1' is empty */
    while (*list2) {
	curr2_name = (*list2)->name;
	StepLinkedList(curr1,list1) {
	    if (curr1->name == curr2_name) { break; }
	}
	if (curr1) {		/* item found that's in both lists */
	    curr2 = *list2;
	    *list2 = NextOf(curr2);
	    User_DestroyUserName(curr2);
	} else {
	    list2 = &(NextOf(*list2));
	}
    }
}

UserNameList User_NewOrLists(list1,list2)
  UserNameList list1,list2;
/* IMPLENTATION: This routine is slow, requiring O(n^2) time. It uses a naive
   traversal of both lists to decide which names are in both.
*/
{
    String name2;		/* name pointed to by current 'list2' */
    UserNameList result;
    UserName *curr1,*new_ptr;

    /* copy list1 into result */
    result = User_NewCopyList(list1);

    /* splice in copies of all items in list2 not appearing in list1 */
    StepInitializedLinkedList(list2) {
	name2 = list2->name;
	StepLinkedList(curr1,list1) {
	    if (curr1->name == name2) { break; }
	}
	if (curr1 == NULL) {	/* name2 *not* in list1 */
	    new_ptr = User_NewList(name2);
	    SpliceIntoList(result,new_ptr);
	}
    }
    return(result);
}

UserNameList User_NewAndLists(list1,list2)
  UserNameList list1,list2;
/* IMPLENTATION: This routine is slow, requiring O(n^2) time. It uses a naive
   traversal of both lists to decide which names are in both.
*/
{
    String name2;		/* name pointed to by current 'list2' */
    UserNameList result = NULL;
    UserName *curr1,*new_ptr;

    /* splice in copies of all items in both lists */
    StepInitializedLinkedList(list2) {
	name2 = list2->name;
	StepLinkedList(curr1,list1) {
	    if (curr1->name == name2) { break; }
	}
	if (curr1 != NULL) { /* name2 *is* in list1 */
	    new_ptr = User_NewList(name2);
	    SpliceIntoList(result,new_ptr);
	}
    }
    return(result);
}
