/* TOP.C

   Top-level control file for fs2iff, the program to probe a filesystem and
   produce an equivalent IFF file.

   $Header: top.c,v 1.9 91/11/15 17:55:47 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


/* HEADER FILES =========================================================== */

/* Used for MAXPATHLEN definition */
#include <sys/param.h>

/* Used to get time of day this program was run */
#include <sys/types.h>
#include <sys/time.h>

/* Misc system header files */
#include <stdio.h>
#include <strings.h>
#include <my-types.h>
#include <my-defs.h>

/* Local header files */
#include "top.h"
#include "gen.h"
#include "perm.h"
#include "search.h"

/* GLOBAL VARIABLES ======================================================= */

String Argv0;
FILE *fp;
Boolean Quiet = False;
Boolean DummyUsers = False;
Boolean DummyFiles = False;
Boolean DirPerms = False;
int MaxNumPerms;

/* LOCAL FUNCTIONS ======================================================== */

#define DATE_LEN 28
#define DATE_STR "%hour:%min [%noon|%am], %3Wday, %3Month %day %year"

static String top_DateString()
/* RETURNS a String containing the current date and time in the form:
   "12:52 pm, Mon, Sep 17 1990" or "12:00 noon, Tue, Oct 3 1990".
*/
{
    char *buff,*fdate();
    struct tm *tm;
    struct timeval tp;
    struct timezone tzp;

    if (gettimeofday(&tp,&tzp)) {
	perror(Argv0);
	exit(-1);
    }
    tm = localtime((time_t *)&tp.tv_sec);
    buff = AllocArray(char,DATE_LEN+1);
    return(fdate(buff,DATE_STR,tm));
}

/* GLOBAL FUNCTIONS ======================================================= */

int main(argc,argv)
  int argc;
  char *argv[];
/* See top.h for a description of the command-line syntax.
*/
{
    Path absolute_path;
    String search_root,output_name;
    int option;
    Boolean err_flag = False;
    char *getwd();

    /* globals needed to use getopt(3) */
    extern int getopt();
    extern char *optarg;
    extern int optind;

    /* assign global Argv0 */
    if ((Argv0=rindex(argv[0],'/'))) { Argv0++; }
    else { Argv0 = argv[0]; }

    /* process command-line arguments */
    output_name = (String)NULL;
    while ((option=getopt(argc,argv,"qufdo:")) != EOF) {
	switch (option) {
	  case 'q': Quiet = True; break;
	  case 'u': DummyUsers = True; break;
	  case 'f': DummyFiles = True; break;
	  case 'd': DirPerms = True; break;
	  case 'o': output_name = optarg; break;
	  default:  err_flag = True; break;
	}
    }
    if (DirPerms) { DummyFiles = True; }
    MaxNumPerms = 1 + (DirPerms ? (int)InDel : (int)Read);
    Assert(optind <= argc);
    if (err_flag || optind < (argc-1)) {
	fprintf(stderr,"SYNTAX: %s",SYNTAX);
	exit(-1);
    }
    Assert((optind == argc) || (optind == (argc-1)));
    if (optind == (argc-1) && strlen(argv[optind]) > MAXPATHLEN) {
	fprintf(stderr,"%s: Argument pathname '%s' too long.\n",
		Argv0,argv[1]);
	exit(-1);
    }

    /* open output file if specified */
    if (output_name != (String)NULL) {
	if ((fp=fopen(output_name,"w")) == NULL) {
	    fprintf(stderr,"%s: Output file '%s' could not be opened.\n",
		    output_name);
	    exit(-1);
	}
    } else {
	fp = stdout;
    }

    /* expand arugment pathnames to absolute names */
    if ((optind == argc) || (*(argv[optind]) != '/')) {
	if (getwd(absolute_path) == NULL) {
	    fprintf(stderr,"%s: getwd(3) call failed", Argv0);
	    exit(-1);
	}
	if (optind != argc) {
	    (void)strcat(absolute_path,"/");
	    (void)strcat(absolute_path,argv[optind]);
	}
    } else {
	(void)strcpy(absolute_path,argv[optind]);
    }

    /* add tailing '/' to absolute_path if necessary */
    if (absolute_path[strlen(absolute_path)-1] != '/')
	(void)strcat(absolute_path,"/");

    /* print file header */
    fprintf(fp,"# fs2iff(1)-created IFF file ");
    if (DummyUsers || DummyFiles || !DirPerms) {
	fprintf(fp,"(with");
	if (DummyUsers) { fprintf(fp," -u"); }
	if (DummyFiles) { fprintf(fp," -f"); }
	if (DirPerms)   { fprintf(fp," -d"); }
	fprintf(fp,")");
    } else {
	fprintf(fp,"(no command-line options)");
    }
    fprintf(fp,"\n#\n");
    fprintf(fp,"# Date:      %s\n",top_DateString());
    fprintf(fp,"# Directory: %s\n",absolute_path);

    /* generate user boxes from /etc/passwd and /etc/group */
    fprintf(fp,"\n# === ALL SUBJECT BOXES ===\n");
    Gen_UserRoleBoxes();

    /* search pathname 'path' */
    fprintf(fp,"\n# === ALL OBJECT BOXES ===\n\n");
    CopyString(search_root,absolute_path);
    (void)strcpy(absolute_path,"/");
    Search_TopLevelPath(absolute_path,search_root);
    return(0);
}
