/* GEN.C

   $Header: gen.c,v 1.20 91/11/18 19:42:47 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


/* HEADER FILES =========================================================== */

#include <stdio.h>
#include <pwd.h>
#include <grp.h>

#include <sys/types.h>
#include <sys/stat.h>		/* for struct stat type */

#include <my-types.h>
#include <my-defs.h>

#include "top.h"
#include "gen.h"
#include "user.h"

/* LOCAL MACRO DEFINITIONS ================================================ */

/* Radix of the ASCII character set */
#define CHAR_SET_RADIX 256

/* Hash table size -- should be prime and not divide CHAR_SET_RADIX */
#define BOX_HASH_TABLE_SIZE 103

/* LOCAL TYPES ============================================================ */

/* BoxHashStruct -- elements of box hash table */
typedef struct box_hash {
    struct box_hash *next;	/* pointer to next element in bucket*/
    String name;		/* box name */
    BoxSysname sysname;		/* corresponding sysname */
} BoxHashStruct;

/* UserSysname -- structure for linked list of User box sysnames */
typedef struct user_sysname {
    struct user_sysname *next;	/* pointer to next element in list */
    BoxSysname sysname;		/* User box sysname */
} UserSysname;

/* LOCAL VARIABLES ======================================================== */

/* head of linked list of User box sysnames */
static UserSysname *gen_UserListHead = NULL;

/* strings corresponding to each BoxType for the macro BoxTypeToString() */
static String gen_BoxTypeStrings[] = {
    "file", "device", "socket", "dir", "dir-dummy", "home", "root", "",
    "user", "group", "world"
};

/* hash table */
/* NOTE: last bucket will *always* be a NULL pointer, since get_HashName()
   returns an index to this bucket for the empty string only.
*/
static BoxHashStruct *BoxHashTable[BOX_HASH_TABLE_SIZE+1];

/* Used by Gen_BeginInside(), GenAddInsideChild(), and Gen_EndInside() */
static BoxSysname parent_num;	/* sysname of parent */
static int child_num;		/* # of children printed on current line */

/* LOCAL MACRO FUNCTIONS ================================================== */

/* BoxTypeToString(BoxType) - evaluates to string corresponding to the type */
#define BoxTypeToString(_kind) gen_BoxTypeStrings[(int)(_kind)]

/* maximum # of children to include in a single INSIDE entry */
#define MAX_CHILDREN 8

/* GLOBAL VARIABLES ======================================================= */

BoxSysname world_sysname;

/* LOCAL FUNCTIONS ======================================================== */

static void gen_WriteName(name)
  String name;
/* Write the name 'name' inside double-quotes to the global 'fp', quoting any
   double-quote characters inside the string as necessary.
*/
{
    String temp;

    (void)fputc('"',fp);
    while ((temp=index(name,'"')) != (String)NULL) {
	*temp = '\0';
	fprintf(fp,"%s\\\"",name);
	name = temp+1;
    }
    fprintf(fp,"%s",name);
    (void)fputc('"',fp);
}

#define MAX_VAL   0x4000L
#define MAX_MASK  0x3fffL
#define START_VAL 0x2001L
#define A_VAL 101		/* must be congruent to 1 mod 4 */
#define B_VAL  25		/* must be congruent to 1 mod 2 (i.e., odd) */

static BoxSysname gen_Box(name,kind)
  String name;
  BoxType kind;
/* IMPLEMENTATION NOTE: This routine generates the next box sysname using a
   linear congruential random number generator, as described on pg 273 of
   "Number Theory in Science and Communication" by Schroeder.
*/
{
    static BoxSysname next_box_name = START_VAL;
    static BoxSysname offset = 0L;

    BoxSysname result = next_box_name+1;
    fprintf(fp,">BOX sysname=%lu; type=%s; name=",
	   result+offset,BoxTypeToString(kind));
    gen_WriteName(name);
    (void)fputc(';',fp);

    /* compute new value */
    next_box_name *= A_VAL;
    next_box_name += B_VAL;
    next_box_name &= MAX_MASK;
    if (next_box_name == START_VAL) { offset += MAX_VAL; }
    return(result);
}

static int gen_HashName(name,hash_size)
  String name;
  int hash_size;
/* RETURNS a hash value in the range [0,hash_size), or hash_size if
   name==NULL.

   IMPLEMENTATION: The hash value computed is simply the string 'name', when
   written out in binary, taken as an integer mod 'hash_size'.
*/
{
    register int hash_val;

    if (name == (String)NULL) {
	/* return index of empty bucket */
	hash_val = hash_size;
    } else {
	for (hash_val=0; *name != '\0'; name++) {
	    hash_val *= CHAR_SET_RADIX;
	    hash_val += (int)(*name);
	    hash_val %= hash_size;
	}
    }
    return(hash_val);
}

#ifdef MACRO_FUNCTION
static int gen_HashBox(name,kind)
  String name;
  BoxType kind;
/* Hash 'name' with 'kind' to produce an index in the range
   [0,BOX_HASH_TABLE_SIZE) such that two boxes with the same name and
   different kinds are guaranteed to return different values.

   IMPLEMENTATION: The RETURN value is the value produced by gen_HashName()
   plus 'kind' mod BOX_HASH_TABLE_SIZE.
*/
#endif MACRO_FUNCTION

#define gen_HashBox(_name,_kind)\
    ((gen_HashName((_name),BOX_HASH_TABLE_SIZE)+((int)_kind))\
     % BOX_HASH_TABLE_SIZE)

static BoxSysname gen_RegisterBox(name,kind,sysname)
  String name;
  BoxType kind;
  BoxSysname sysname;
/* Associates the BoxSysname 'sysname' with the box having name 'name' and
   BoxType 'kind'.

   RETURNS 'sysname' if no such box already exists in the table; 0L otherwise.
*/
{
    int hash_val;
    BoxHashStruct *curr_box;

    /* search for the box; return 0L if it is found */
    StepLinkedList(curr_box,BoxHashTable[(hash_val=gen_HashBox(name,kind))]) {
	if (strcmp(curr_box->name,name) == 0) {	/* exact match */
	    return(0L);
	}
    }

    /* build a new BoxHashStruct */
    curr_box = AllocOne(BoxHashStruct);
    CopyString(curr_box->name,name);
    curr_box->sysname = sysname;

    /* add the BoxHashStruct to the correct bucket */
    NextOf(curr_box) = BoxHashTable[hash_val];
    BoxHashTable[hash_val] = curr_box;

    return(sysname);
}

static BoxHashStruct *gen_GetBox(name,kind)
  String name;
  BoxType kind;
/* RETURNS a pointer to the BoxHashStruct for the box having name 'name' and
   type 'kind', or NULL if no such box exists.
*/
{
    BoxHashStruct *curr_box;

    StepLinkedList(curr_box,BoxHashTable[gen_HashBox(name,kind)]) {
	if (strcmp(curr_box->name,name) == 0) {	return(curr_box); }
    }
    return(NULL);
}

static void gen_AddToUserList(sysname)
  BoxSysname sysname;
/* Adds a structure for the BoxSysname 'sysname' to the linked list headed by
   the local variable 'gen_UserListHead'.
*/
{
    UserSysname *curr;
    curr = AllocOne(UserSysname);
    curr->sysname = sysname;
    SpliceIntoList(gen_UserListHead,curr);
}

static void gen_RemoveFromUserList(sysname)
  BoxSysname sysname;
/* Searches the linked list headed by the local variable 'gen_UserListHead'
   for an item having sysname 'sysname', and if one such is found, removes it
   from the list and deallocates its space.
*/
{
    UserSysname **curr,*match;

    for (curr = &gen_UserListHead; *curr; curr = &((*curr)->next)) {
	if ((*curr)->sysname == sysname) {
	    /* remove the matched structure from the list and free space */
	    match = *curr;
	    *curr = match->next;
	    Dealloc(match);
	    break;
	}
    }
}

static void gen_UserBoxes()
/* Reads the /etc/passwd file and generates a box for each user on the system;
   the sysname corresponding to each generated box is registered. Also, the
   sysname of each user box is added to the linked list headed by the local
   variable gen_UserListHead.

   A notice of duplicate user id's is sent to stderr; only the first such id
   is processed; the rest are skipped.
*/
{
    struct passwd *u_ent;

    fprintf(fp,"\n# User Boxes\n");
    (void)setpwent();
    while ((u_ent = getpwent()) != NULL) {
	if (User_RegisterName((uid_t)u_ent->pw_uid,UserID,u_ent->pw_name)
	    == NULL) {
	    if (!Quiet) {
		fprintf(stderr,"%s: duplicate user id %d (%s); skipping...\n",
			Argv0,u_ent->pw_uid,u_ent->pw_name);
	    }
	} else {
	    gen_AddToUserList(Gen_BoxSaveSysname(u_ent->pw_name,UserBox));
	}
    }
    (void)endpwent();
}

static void gen_GroupBoxes()
/* Reads the /etc/group file and generates a group box for each group in the
   system, along with containment information for all users who are members of
   that group. The sysname corresponding to each generated group box is
   registered. The users should have already been processed (and their
   sysnames registered) before this routine is called.

   A notice of duplicate group id's is sent to stderr; only the first such id
   is processed; the rest are skipped. When the members of a group are
   processed, those users skipped by gen_UserBoxes() because they were
   duplicate ID's, or those users not appearing in the /etc/passwd file are
   skipped. In both of these latter cases, a message is sent to stderr.

   If the global 'DummyUsers' is True, then a dummy user box is created
   inside each group box with the name "GROUP-USER-<groupname>".
*/
{
    String *g_member,dummy_name;
    BoxSysname curr_group,curr_member,dummy_group;
    struct group *g_ent;

    fprintf(fp,"\n# Group Boxes and All User Box Containment Info\n");
    (void)setgrent();
    while ((g_ent = getgrent()) != NULL) {
	if (User_RegisterName((gid_t)g_ent->gr_gid,GroupID,g_ent->gr_name)
	    == NULL) {
	    if (!Quiet) {
		fprintf(stderr,"%s: duplicate group id %d (%s); skipping...\n",
			Argv0,g_ent->gr_gid,g_ent->gr_name);
	    }
	} else {
	    /* create boxes for this group and a dummy user inside it */
	    curr_group = Gen_BoxSaveSysname(g_ent->gr_name,GroupBox);
	    if (DummyUsers) {
		dummy_name = malloc((unsigned)(strlen(DUMMY_GROUP_NAME)+
					       strlen(g_ent->gr_name)+1));
		(void)strcat(strcpy(dummy_name,DUMMY_GROUP_NAME),
			     g_ent->gr_name);
		dummy_group = Gen_Box(dummy_name,UserBox);
		Dealloc(dummy_name);
	    }

	    /* generate INSIDE entries for this group */
	    Gen_BeginInside(curr_group);
	    StepPtrArray(g_member,g_ent->gr_mem) {
		if ((curr_member = Gen_GetBoxSysname(*g_member,UserBox))) {
		    gen_RemoveFromUserList(curr_member);
		    Gen_AddInsideChild(curr_member);
		    (void)User_AddUserToGroup((gid_t)g_ent->gr_gid,*g_member);
		} else if (!Quiet) {
		    fprintf(stderr,
			 "%s: unregistered group member '%s'; skipping...\n",
			    Argv0,*g_member);
		}
	    }
	    if (DummyUsers) { Gen_AddInsideChild(dummy_group); }
	    Gen_EndInside();
	}
    }
    (void)endgrent();
}

static void gen_WorldBoxes()
/* Generates a world box and containment information for all groups read from
   the /etc/groups file.

   If the global 'DummyUsers' is True, then a dummy user box is created
   inside the world box with the name "WORLD-USER"
*/
{
    BoxSysname dummy_world,curr_group;
    UserSysname *curr,*next;
    struct group *g_ent;

    fprintf(fp,"\n# World Box and Group Containment Info\n");
    world_sysname = Gen_BoxSaveSysname("world",WorldBox);
    if (DummyUsers) { dummy_world = Gen_Box(DUMMY_WORLD_NAME,UserBox); }
    Gen_BeginInside(world_sysname);
    (void)setgrent();
    while ((g_ent=getgrent()) != NULL) {
	if (curr_group = Gen_GetBoxSysname(g_ent->gr_name,GroupBox)) {
	    Gen_AddInsideChild(curr_group);
	}
    }
    (void)endgrent();
    for (curr=gen_UserListHead; curr; curr=next) {
	Gen_AddInsideChild(curr->sysname);
	next = NextOf(curr);
	Dealloc(curr);
	if (!Quiet) {
	    fprintf(stderr,
		    "%s: user id %lu not in any group; adding to world...\n",
		    Argv0,curr->sysname);
	}
    }
    if (DummyUsers) { Gen_AddInsideChild(dummy_world); }
    Gen_EndInside();
}

/* GLOBAL FUNCTIONS ======================================================= */

void Gen_Arrow(from,to,perm,parity)
  BoxSysname from,to;
  String perm;
  Parity parity;
{
    static ArrowSysname last_arrow_name = 0L;
    fprintf(fp,">ARROW sysname=%lu; from=%lu; to=%lu; permissions={%s}; ",
	    ++last_arrow_name,from,to,perm);
    fprintf(fp,"parity=%s;\n",((parity == Pos) ? "pos" : "neg"));
}

BoxSysname Gen_Box(name,kind)
  String name;
  BoxType kind;
{
    BoxSysname result = gen_Box(name,kind);
    (void)fputc('\n',fp);
    return(result);
}

BoxSysname Gen_ObjBox(name,kind,st)
  String name;
  BoxType kind;
  struct stat *st;
{
    /* generate the box */
    BoxSysname result = gen_Box(name,kind);

    /* add extra attributes */
    if (st != NULL) {
	if ((st->st_mode & S_IFMT) != S_IFLNK) { /* not a symbolic link */
	    String subj_name;
	    if ((subj_name=User_GetName(st->st_uid,UserID)) != NULL) {
		(void)fprintf(fp," owner=%s;",subj_name);
	    }
	    if ((subj_name=User_GetName(st->st_gid,GroupID)) != NULL) {
		(void)fprintf(fp," group=%s;",subj_name);
	    }
	}
	if (((st->st_mode & S_IFMT) == S_IFDIR) && (st->st_mode & S_ISVTX)) {
	    /* sticky directory */
	    (void)fputs(" sticky=true;",fp);
	}
	if ((st->st_mode & S_IFMT) == S_IFREG) { /* regular file */
	    if (st->st_mode & S_ISUID) { (void)fputs(" setuid=true;",fp); }
	    if (st->st_mode & S_ISGID) { (void)fputs(" setgid=true;",fp); }
	}
    }
    (void)fputc('\n',fp);
    return(result);
}

BoxSysname Gen_BoxSaveSysname(name,kind)
  String name;
  BoxType kind;
{
    BoxSysname sysname;

    /* make a new box */
    sysname = Gen_Box(name,kind);

    /* register the new box and its sysname */
    (void)gen_RegisterBox(name,kind,sysname);

    /* return the new sysname */
    return(sysname);
}

String Gen_GetBoxName(name,kind)
  String name;
  BoxType kind;
{
    BoxHashStruct *curr_box;

    if (curr_box=gen_GetBox(name,kind)) { return(curr_box->name); }
    return(NULL);
}

BoxSysname Gen_GetBoxSysname(name,kind)
  String name;
  BoxType kind;
{
    BoxHashStruct *curr_box;

    if (curr_box=gen_GetBox(name,kind)) { return(curr_box->sysname); }
    return(0L);
}

void Gen_BeginInside(parent)
  BoxSysname parent;
{
    parent_num = parent;
    child_num = 0;
}

void Gen_AddInsideChild(child)
  BoxSysname child;
{
    if (child_num++ == 0) {
	(void)fprintf(fp,">INSIDE parent=%lu; children={",parent_num);
	(void)fprintf(fp,"%lu",child);
    } else {
	(void)fprintf(fp,",%lu",child);
    }
    if (child_num >= MAX_CHILDREN) {
	(void)fputs("};\n",fp);
	child_num = 0;
    }
}

void Gen_EndInside()
{
    if (child_num > 0) (void)fputs("};\n",fp);
}

void Gen_UserRoleBoxes()
/* Generates the boxes and containment information for all users, groups, and
   world on the filesystem.
*/
{
    gen_UserBoxes();
    gen_GroupBoxes();
    gen_WorldBoxes();
}
