;;;             -*- Mode: Lisp; Package: MIRO; -*-
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

;;; 
;;; MIRO EDITOR TOOLKIT
;;;
;;; This file contains the definitions of various prototype objects.
;;; These include:
;;;  - fonts
;;;  - line styles
;;;  - the Miro cursor bitmap
;;;  - button templates
;;;  - scroll bars
;;;

#|
============================================================
Change log:
    11/04/91 ky ; Added lots of new objects to shield the rest of
		; rest of the system from some Garnet internal stuff.
		; Added a few new functions to support new interfaces.
    05/10/91 ky ; Added miro-aggregadget.  Added :use-default-values
		; menu to objects.
    04/18/91 ky ; Moved get-arg-list, read-string to this file.
    04/17/91 ky ; Added skip-comments-and-whitespace.
    04/16/91 ky ; Added run-miro-tool.
    04/10/91 ky ; Added completion-list.
    04/01/91 ky ; Added dbox-labeled-text-input, call-schema.
    03/22/91 ky ; Added function get-output to read output from
		; multiple streams.
    12/7/90  ky ; Lower the dialog window when restarting from
		; disksave.
    11/27/90 ky ; Added RGB values for colors to *options*.  Added
		; function :use-default-values to *options*.
    11/26/90 ky ; Added *options*.  Changed :really-light-gray to
		; :light-gray-for-dbox-emphasis and made it depend on
		; the :change-background-for-dbox-emphasis option.
    11/20/90 ky ; Use gray-scale when possible.  Added
		; miro::reconnect-garnet, miro:change-garnet-display
		; to call the garnet functions and update our grays.
		; Change the background color of menu-window,
		; sb-window, and work-window when we de-activate the
		; buttons.
    10/31/90 ky ; Fixed alignment of text strings in
		; menu-button-panel.
    9/21/90 ky  ; Flush output after printing a prompt in
		; get-string-from-user.
    9/14/90 ky  ; Deleted function fix-garnet-display.
		; Use gray-fill for "out-of-service" buttons instead
		; of dark-gray-fill.
    9/13/90 ky  ; Moved export declaration to miro-defs.lisp.  Added
		; some declarations to get rid of compiler warnings.
    9/12/90 ky  ; Changed font for stars to use standard Garnet fonts
		; to avoid further problems with font changes (let
		; Garnet worry about it!)
		;
    8/23/90 ky  ; Changed font-path for stars to
		; /afs/cs/project/miro/editor/fonts/ since
		; /usr/misc/.X11tra/lib/x11fonts/ is not guaranteed to
		; exist.  Don't call fix-garnet-display any more;
		; the "test" Garnet handles disksaves properly.
    8/10/90 ky  ; Added function min-max to guarantee a number within
		; the specified range.
		;
    7/23/90 ky  ; Added scroll-agg, which consists of a trill device,
		; a label, and a reset button.  scroll-agg is used for
		; the horizontal and vertical offsets, and for scaling.
    7/3/90  ky	; Added *started-from-disksave* and set it in
		; disksave's init function so that exit-editor will
		; know whether or not to exit lisp as well.
    7/2/90  ky	; disksave works now.
    6/25/90 ky  ; Added functions "block-interference" and
		; "allow-interference" to set/reset *dont-interfere*,
		; change button colors, etc.
		;
		; Added function "disksave".  This function will
		; eventually save a copy of the current lisp and
		; arrange for whatever is necessary to set up the
		; editor when the lisp is restarted.
		;
		; Added function "fix-garnet-display", which attempts
		; to re-initialize the display from the DISPLAY
		; environment variable.  This function is apparently
		; still missing some variable that needs to be
		; re-initialized.
    6/8/90  ky  ; For compatibility with Allegro Common Lisp, call the
		; function "find-on-path", defined in miro-load.lisp,
		; to find the cursor bitmaps on the miro: path.
    6/1/90  ky  ; Deleted *dont-even-exit*, since exit is now treated
		; like any other command.
    5/8/90  ky  Added a variable, *dont-even-exit*, which may be set
                to prevent the exit command from working.  This is
                useful for commands that use *y-n-buttons*.
    4/25/90 ky	Added a variable, *dont-interfere*, which may be set by
		commands (such as display) that don't want other commands to
		interfere with them.  Also added a macro, dont-interfere, which
		may be used to call a function only if *dont-interfere* is nil.
    4/17/90 ky	Added command-button-panel (a panel of text buttons)
	        and dbox-tbutton-entry (the text-button equivalent
		of dbox-button-entry.)
    2/21/90 amz added dialog box line style, dbox-button-entry
    2/15/90 amz added box asterisk fonts
    1/17/90 amz cleaned up font stuff a little, 
                added arrow asterisk fonts 
    9/5/89   amz Switched to o-formulas
    8/7/89   amz Added round-buttons
    08/03/89 amz Moved line styles and cursor bitmaps here.
                 Added new line styles for constraint pictures.
                 Changed to new version of aggrestuff.
    Nov 1988 Philippe Marchal Created.
============================================================
|#

(in-package "MIRO" :use `("LISP" "KR"))

(proclaim '(function push-help-string))	; defined in miro-main.lisp
(proclaim '(function pop-help-string))	; defined in miro-main.lisp
(proclaim '(function print-startup-msg)) ; defined in miro-main.lisp

; debugging flag
(defparameter *test-debug* NIL)

;; indicates whether the editor was started from a saved core image.
(defvar *started-from-disksave* nil)

; indicates whether a function that should not be interrupted is in use
(defvar *dont-interfere* nil)
(defmacro dont-interfere (fn)
  `#'(lambda (&rest args)
	     (unless *dont-interfere* (apply #',fn args))))


;;;============================================================
;;; call-schema  -- call a method, giving the schema as the first
;;; argument.
;;;============================================================
(defmacro call-schema (schema method &rest args)
  `(kr-send ,schema ,method ,schema ,@args))

;;;------------------------------------------------------------
;;; miro-aggregadget -- an aggregadget that sets default values
;;;------------------------------------------------------------
(create-instance 'miro-aggregadget opal:aggregadget
		 ;; do we need to reset the parent too?
		 (:responsible-for-parent nil)
		 )
(define-method :use-default-values miro-aggregadget
  (obj &optional original-callers)
  (unless (position obj original-callers)
	  (opal:do-components
	   obj #'(lambda (c)
		   (call-schema c :use-default-values original-callers))
	   :type opal:aggregadget)
	  (when (g-value obj :responsible-for-parent)
		(call-schema (g-value obj :parent) :use-default-values
			     (cons obj original-callers)))
	  )
  )

(define-method :initialize miro-aggregadget (obj &optional top-agg)
  (if top-agg (call-prototype-method obj top-agg)
    (call-prototype-method obj))
  (call-schema obj :use-default-values))

;;;------------------------------------------------------------
;;; miro-frame
;;;------------------------------------------------------------
(create-instance 'miro-frame opal:aggregadget
		 (:component-to-frame nil)
		 (:border-width
		  (o-formula (gv *fonts-and-styles*
				 :db-frame-width)))
		 (:line-style (o-formula (gv *fonts-and-styles*
					     :db-frame-style)))
		 (:filling-style (o-formula (gv *colors* :white)))
		 (:top (o-formula (- (gvl :component-to-frame :top)
				     (gvl :border-width))))
		 (:left (o-formula (- (gvl :component-to-frame :left)
				      (gvl :border-width))))
		 (:width (o-formula (+ (gvl :component-to-frame
					    :width)
				       (* 2 (gvl :border-width)))))
		 (:height (o-formula (+ (gvl :component-to-frame
					     :height)
					(* 2 (gvl :border-width)))))
		 (:parts
		  `((:frame ,opal:rectangle
			    (:border-width
			     ,(o-formula (gvl :parent :border-width)))
			    (:line-style
			     ,(o-formula (gvl :parent :line-style)))
			    (:filling-style
			     ,(o-formula (gvl :parent :filling-style)))
			    (:top ,(o-formula (gvl :parent :top)))
			    (:left ,(o-formula (gvl :parent :left)))
			    (:width ,(o-formula (gvl :parent :width)))
			    (:height ,(o-formula (gvl :parent :height)))
			    )
		    ))
		 )

;;;============================================================
;;; BUTTON PANELS
;;;============================================================

;;;------------------------------------------------------------
;;; Menu-Button-Panel: define my own radio-button to make size, etc
;;; consistent. Used in all button menus
;;;------------------------------------------------------------
(create-instance 'menu-button-panel opal:aggregadget
		 (:top 0)
		 (:left 0)
		 (:default-value nil)
		 (:value-list nil)
		 (:items '("button-1" "button-2"))
		 (:toggle nil)
		 (:value (o-formula
			  (let ((i (position (gvl :buttons :value-obj)
					     (get-values
					      (gvl :buttons
						   :radio-button-list)
					      :components))))
			    (when (and i (gvl :buttons :value-obj
					      :selected))
				  (nth i (or (gvl :value-list)
					     (gvl :items)))))))
		 (:button-diameter 20)
		 (:shadow-offset 3)
		 (:gray-width 2)
		 (:text-on-left-p nil)
		 (:h-align :left)
		 (:font (o-formula (gv *fonts-and-styles* :button-label-font)))
		 (:direction :vertical)
		 (:fixed-width-p T)
		 (:selection-function nil)
		 (:parts
		  `((:buttons
		     ,Garnet-gadgets:Radio-Button-Panel
		     (:top ,(o-formula (gvl :parent :top)))
		     (:left ,(o-formula (gvl :parent :left)))
		     (:items ,(o-formula (gvl :parent :items)))
		     (:button-diameter
		      ,(o-formula (gvl :parent :button-diameter)))
		     (:shadow-offset ,(o-formula (gvl :parent
						      :shadow-offset)))
		     (:gray-width ,(o-formula (gvl :parent
						   :gray-width)))
		     (:text-on-left-p
		      ,(o-formula (gvl :parent :text-on-left-p)))
		     (:h-align ,(o-formula (gvl :parent :h-align)))
		     (:font ,(o-formula (gvl :parent :font)))
		     (:direction ,(o-formula (gvl :parent :direction)))
		     (:fixed-width-p ,(o-formula (gvl :parent :fixed-width-p)))
		     (:selection-function
		      ,#'(lambda (obj nv)
			   (call-schema (g-value obj :parent)
					:selection-function nv)
			   (call-prototype-method obj nv)
			   ))
		     )))
		 )

(define-method :set-value menu-button-panel (obj new-value)
  (let* ((button-obj (g-value obj :buttons))
	 (i (position new-value (or (g-value obj :value-list)
				   (g-value obj :items))
		     :test #'equal))
	 (button (when i (nth i (get-values
				 (g-value button-obj :radio-button-list)
				 :components))))
	 )
    (when button (s-value button :selected T))
    (s-value button-obj :value-obj button)
    )
  )

(define-method :use-default-values menu-button-panel
  (obj &optional original-callers)
  (declare (ignore original-callers))
  (call-schema obj :set-value (g-value obj :default-value)))

(define-method :initialize menu-button-panel (obj &optional top-agg)
  (if top-agg (call-prototype-method obj top-agg)
    (call-prototype-method obj))
  (s-value (g-value obj :buttons :radio-button-press) :how-set
	   (if (g-value obj :toggle) :toggle :set))
  (call-schema obj :use-default-values)
  )

;;;------------------------------------------------------------
;;; Command-Button-Panel: define a text-button panel to keep things
;;; consistent.  Used in "commands-menu" and the exit commands in
;;; "*box-db*".  Expects the following slots: :left, :top, :direction,
;;; :items.  The user may also wish to override :v-spacing, :h-spacing,
;;; and :rank-margin.
;;;------------------------------------------------------------
(create-instance 'command-button-panel opal:aggregate
		 (:top 0)
		 (:left 0)
		 (:font (o-formula (gv *fonts-and-styles*
				       :button-label-font)))
		 (:gray-width 2)
		 (:shadow-offset 2)
		 (:text-offset 4)
		 (:rank-margin 6)
		 (:v-spacing 5)
		 (:h-spacing 35)
		 (:final-feedback-p nil)
		 (:direction :vertical)
		 (:fixed-width-p nil)
		 (:fixed-width-size nil)
		 (:fixed-height-p nil)
		 (:fixed-height-size nil)

		 (:active-fill opal:white-fill)
		 (:inactive-fill opal:black-fill)
		 (:inactive-list nil)
		 (:panel-active T)
		 (:selection-function nil)
		 (:value (o-formula (gvl :button-panel :value)))

		 (:button-panel nil)
		 (:actions nil)
		 )

(defun *command-button-panel-default-action* (obj nv)
  (let* ((rank (position nv (g-value obj :items)))
	 (active (and (g-value obj :parent :panel-active)
		      (not
		       (position nv
				 (g-value obj :parent
					  :inactive-list)
				 :test #'equal))))
	 (action (nth rank (g-value obj :parent :actions)))
	 (action-fn (if (functionp action) (when active action)
		      (if active (first action) (second action))))
	 (selection (g-value obj :parent :selection-function))
	 (selection-fn (if (functionp selection)
			   (when active selection)
			 (if active (first selection)
			   (second selection))))
	 (parent (g-value obj :parent))
	)
    (when active (call-prototype-method obj nv))
    (when action-fn (funcall action-fn parent nv))
    (when selection-fn (funcall selection-fn parent nv))
    ))

(define-method :initialize command-button-panel (obj)
  (call-prototype-method obj)

  ;; make sure we don't carry over and try to delete the button panel
  ;; of an ancestor
  (s-value obj :button-panel nil)

  (call-schema obj :notice-new-items))

(define-method :notice-new-items command-button-panel (obj)
  ;; get rid of old button panel
  (let ((panel (g-value obj :button-panel)))
    (when panel
	  (s-value obj :button-panel nil)
	  (when (schema-p panel)
		(opal:remove-component obj panel)
		(opal:destroy panel)
		)
	  ))

  ;; split the item list
  (let (items actions)
    (dolist (item (g-value obj :items))
	    (cond
	     ((and item (listp item))
	      (push (first item) items)
	      (push (second item) actions))
	     (T (push item items)
		(push nil actions))
	     ))
    (setf items (reverse items))
    (s-value obj :actions (reverse actions))

    ;; create the button panel
    (let ((panel (create-instance nil garnet-gadgets:text-button-panel
				  (:items items)
				  (:top (o-formula (gvl :parent :top)))
				  (:left (o-formula (gvl :parent :left)))
				  (:font (o-formula (gvl :parent :font)))
				  (:gray-width
				   (o-formula (gvl :parent :gray-width)))
				  (:shadow-offset
				   (o-formula (gvl :parent :shadow-offset)))
				  (:text-offset
				   (o-formula (gvl :parent :text-offset)))
				  (:rank-margin
				   (o-formula (gvl :parent :rank-margin)))
				  (:v-spacing
				   (o-formula (gvl :parent :v-spacing)))
				  (:h-spacing
				   (o-formula (gvl :parent :h-spacing)))
				  (:final-feedback-p
				   (o-formula (gvl :parent :final-feedback-p)))
				  (:selection-function
				   #'*command-button-panel-default-action*)
				  (:fixed-width-p
				   (o-formula (gvl :parent :fixed-width-p)))
				  (:fixed-height-p
				   (o-formula (gvl :parent :fixed-height-p)))
				  )))
      (opal:add-component obj panel)
      (s-value obj :button-panel panel)
      ;; have to set direction after :parent exists
      (s-value panel :direction (o-formula (gvl :parent :direction)))
      (s-value panel :original-fixed-width-size
	       (copy-formula (get-value panel :fixed-width-size)))
      (s-value panel :original-fixed-height-size
	       (copy-formula (get-value panel :fixed-height-size)))
      (s-value panel :fixed-width-size
	       (o-formula (or (gvl :parent :fixed-width-size)
			      (gvl :original-fixed-width-size))))
      (s-value panel :fixed-height-size
	       (o-formula (or (gvl :parent :fixed-height-size)
			      (gvl :original-fixed-height-size))))

      ;; set filling styles for the buttons
      (dolist (button (get-values (g-value panel :text-button-list)
				  :components))
	      (s-value (g-value button :white-field) :filling-style
		       (o-formula
			(if (and (gvl :parent :parent :parent :parent
				      :panel-active)
				 (not (position (gvl :parent :string)
						(gvl :parent :parent
						     :parent :parent
						     :inactive-list)
						:test #'equal)))
			    (gvl :parent :parent :parent :parent
				 :active-fill)
			  (gvl :parent :parent :parent :parent
			       :inactive-fill)
			  )))
	      )
      )))

(define-method :activate-panel command-button-panel (obj)
  (unless (g-value obj :panel-active)
	  (s-value obj :panel-active T)))

(define-method :inactivate-panel command-button-panel (obj)
  (when (g-value obj :panel-active)
	(s-value obj :panel-active nil)))

(define-method :activate-item command-button-panel (obj item)
  (let* ((oitems (g-value obj :inactive-list))
	 (items (remove item oitems :test #'equal))
	 )
    (unless (equal items oitems)
	    (s-value obj :inactive-list items))
    ))

(define-method :inactivate-item command-button-panel (obj item)
  (pushnew item (g-value obj :inactive-list) :test #'equal)
  )

;;;------------------------------------------------------------
;;; Dbox-Button-Entry: 
;;; A dbox-button-entry consists of a Label and a radio button panel.
;;; parts = label + button-list
;;; expects the following slots:
;;; - :label-string : string for label of entry (eg, "PARITY:")
;;; - :buttons : a list of the form accepted by the :items slot of the
;;; garnet button gadgets.
;;;------------------------------------------------------------
(create-instance 'dbox-button-entry miro-aggregadget
		 (:label-string "")
		 (:default-value nil)
		 (:value-list nil)
		 (:items '("button 1" "button 2"))
		 (:value (o-formula (gvl :button-list :value)))
		 (:toggle nil)
		 (:selection-function nil)
		 (:parts
		  `(
		    (:label ,opal:text
			    (:top ,(o-formula (+ (gvl :parent :top) 4)))
			    (:left ,(o-formula (gvl :parent :left)))
			    (:font ,(o-formula (gv *fonts-and-styles*
						   :button-label-font)))
			    (:string ,(o-formula (gvl :parent
	        				      :label-string))))
		    (:button-list ,menu-button-panel
			    (:top ,(o-formula (gvl :parent :top)))
			    (:left ,(o-formula (+ (gvl :parent :left)
						  (gvl :parent :label
						       :width) 5)))
			    (:toggle ,(o-formula (gvl :parent :toggle)))
			    (:selection-function
			     ,(o-formula (gvl :parent :selection-function)))
			    (:default-value ,(o-formula
					      (gvl :parent
						   :default-value)))
			    (:direction ,:horizontal)
			    ;; add this so that buttons aren't as
			    ;; spread out 
			    (:fixed-width-p NIL)
			    (:waiting-priority inter:high-priority-level)
			    (:running-priority inter:high-priority-level)
			    (:value-list ,(o-formula (gvl :parent
							  :value-list)))
			    (:items ,(o-formula (gvl :parent :items)))
			    )
		    )))

(define-method :set-value dbox-button-entry (obj new-value)
  (call-schema (g-value obj :button-list) :set-value new-value))

;;;------------------------------------------------------------
;;; Dbox-TButton-Entry: 
;;; A dbox-tbutton-entry consists of a Label and a command button panel.
;;; parts = label + button-list
;;; expects the following slots:
;;; - :label-string : string for label of entry (eg, "PARITY:")
;;; - :items : a list of the form accepted by the :items slot of the
;;; garnet button gadgets.
;;;------------------------------------------------------------
(create-instance 'dbox-tbutton-entry opal:aggregadget
		 (:label-string "")
		 (:items nil)
		 (:parts
		  `(
		    (:label ,opal:text
			    (:top ,(o-formula (+ (gvl :parent :top) 4)))
			    (:left ,(o-formula (gvl :parent :left)))
			    (:font ,(o-formula (gv *fonts-and-styles*
						   :button-label-font)))
			    (:string ,(o-formula (gvl :parent
						      :label-string))))
		    (:button-list ,command-button-panel
			    (:top ,(o-formula (gvl :parent :top)))
			    (:left ,(o-formula (+ (gvl :parent :left)
						  (gvl :parent :label
						       :width) 5)))
			    (:direction ,:horizontal)
			    (:h-spacing 5)
			    (:waiting-priority inter:high-priority-level)
			    (:running-priority inter:high-priority-level)
			    (:items ,(o-formula (gvl :parent :items))))
		    )))

;;;============================================================
;;; scroll-agg: includes a trill-device, label, and reset button.
;;;============================================================
(create-instance 'scroll-agg opal:aggregadget
		 (:left 0) (:top 0)
		 (:label-string "")
		 (:val-1 0)
		 (:val-2 0)
		 (:value-feedback-p t)
		 (:selection-function nil)
		 (:scr-incr 1)
		 (:page-incr 5)
		 (:max-string-width 60)
		 (:overlapping nil)
		 (:value (o-formula (gvl :scroll :value)))
		 (:parts
		  `(
		    (:label ,opal:text
			    (:top ,(o-formula (gvl :parent :top)))
			    (:left ,(o-formula (gvl :parent :left)))
			    (:font ,(o-formula (gv *fonts-and-styles*
						   :button-label-font)))
			    (:string ,(o-formula (gvl :parent :label-string)))
			    )
		    (:reset-button ,menu-button-panel
				   (:left ,(o-formula (gvl :parent :left)))
				   (:top
				    ,(o-formula (+ (gvl :parent :top) 14)))
				   (:direction ,:horizontal)
				   (:button-diameter 12)
				   (:shadow-offset 1)
				   (:gray-width 1)
				   (:selection-function
				    ,#'(lambda (gadget-object value)
					 (declare (ignore value))
					 (s-value (g-value
						   gadget-object
						   :parent :scroll)
						  :value 0)
					 (s-value gadget-object :value
						  nil)
					 (let ((act (g-value
						     gadget-object
						     :parent
						     :reset-actions)))
					   (when act (funcall act)))
					 ))
				   (:text-on-left-p t)
				   (:items ("Reset"))
				   )
		    (:scroll ,garnet-gadgets:trill-device
			     (:left ,(o-formula (+ (gvl :parent :left)
						   (gvl :parent
							:max-string-width))))
			     (:top ,(o-formula (+ (gvl :parent :top)
						  5)))
			     (:scr-incr ,(o-formula (gvl :parent :scr-incr)))
			     (:page-incr ,(o-formula (gvl :parent :page-incr)))
			     (:val-1 ,(o-formula (gvl :parent :val-1)))
			     (:val-2 ,(o-formula (gvl :parent :val-2)))
			     (:value-feedback-p ,(o-formula (gvl
							     :parent
							     :value-feedback-p)))
			     (:value 0)
			     (:selection-function ,(o-formula (gvl
							       :parent
							       :selection-function)))
			     )
		    ))
		 )

;;;============================================================
;;; SCROLL BARS
;;;============================================================
;;; took these out -- use gadget scroll bars now 3/5/90


;;;============================================================
;;; Helper functions
;;;============================================================

;;; Asks the user for a string with prompt
(defun get-string-from-user (prompt)
  (format t "~%~A " prompt)
  (force-output)
  (read-line))

(defun min-max (n min max)
  (cond
   ((< n min) min)
   ((> n max) max)
   (T n)
   ))
;;;------------------------------------------------------------
;;; turn off "interfering" commands
;;;------------------------------------------------------------
(defun block-interference (&key help-msg leave-inters-alone
				leave-work-window-alone menu
				leave-priorities-alone)
  (unless menu
	  (setf *dont-interfere* T)
	  (call-schema popup-command-menu :inactivate-menu)
	  )
  (call-schema help-buttons :inactivate-panel)
  (s-value menu-background :filling-style
	   (o-formula (gv *colors* :light-gray-for-dbox-emphasis)))
  (s-value sb-background :filling-style
	   (o-formula (gv *colors* :light-gray-for-dbox-emphasis)))
  (s-value box-creation1-background :filling-style
	   (o-formula (gv *colors* :light-gray-for-dbox-emphasis)))
  (s-value box-creation2-background :filling-style
	   (o-formula (gv *colors* :light-gray-for-dbox-emphasis)))
  (unless leave-work-window-alone
	  (s-value work-background :filling-style
		   (o-formula (gv *colors* :light-gray-for-dbox-emphasis))))
  (unless leave-inters-alone
	  (inter:change-active low-priority-level nil)
	  (inter:change-active medium-priority-level nil))
  (when help-msg (push-help-string help-msg))
  (opal:update-all)
  )

;;;------------------------------------------------------------
;;; turn "interfering" commands back on
;;;------------------------------------------------------------
(defun allow-interference (&key leave-help-alone leave-inters-alone)
  (call-schema popup-command-menu :activate-menu)
  (call-schema help-buttons :activate-panel)
  (s-value menu-background :filling-style (o-formula (gv *colors* :white)))
  (s-value sb-background :filling-style (o-formula (gv *colors* :white)))
  (s-value box-creation1-background :filling-style (o-formula (gv *colors* :white)))
  (s-value box-creation2-background :filling-style (o-formula (gv *colors* :white)))
  (s-value work-background :filling-style (o-formula (gv *colors* :white)))
  (unless leave-inters-alone
	  (inter:change-active low-priority-level T)
	  (inter:change-active medium-priority-level T))
  (setf *dont-interfere* nil)
  (unless leave-help-alone (pop-help-string))
  (opal:update-all)
  )


;;;============================================================
;;; dbox-labeled-button-input -- the buttons with a title used in
;;; dialog boxes.  User may set :left, :top, :title-string, :items,
;;; :direction. The value may be explicitly set by calling :set-value
;;; with the new value, expressed in terms of :value-list.
;;; :value contains the current value, obtained by taking the value in
;;; :value-list corresponding to the button pressed, or nil if nothing
;;; is currently selected.
;;;============================================================
(create-instance 'dbox-labeled-button-input miro-aggregadget
		 (:left 0) (:top 0)
		 (:default-value nil)
		 (:title-string "Select a button:")
		 (:items '("button-1" "button-2"))
		 (:value-list nil)
		 (:direction :horizontal)
		 (:toggle nil)
		 (:selection-function nil)

		 (:value (o-formula (gvl :buttons :value)))

		 (:parts
		  `((:title ,opal:text
			    (:left ,(o-formula (gvl :parent :left)))
			    (:top ,(o-formula (gvl :parent :top)))
			    (:string ,(o-formula (gvl :parent
						      :title-string)))
			    (:font ,(o-formula (gv *fonts-and-styles*
						   :button-label-font))))
		    (:buttons ,menu-button-panel
			      (:default-value ,(o-formula
						(gvl :parent
						     :default-value)))
			      (:selection-function
			       ,(o-formula (gvl :parent :selection-function)))
			      (:left ,(o-formula (+ 5 (gvl :parent
							   :left))))
			      (:top ,(o-formula (+ 5 (gvl :parent
							  :title :top)
						   (gvl :parent :title
							:height))))
			      (:direction  ,(o-formula (gvl :parent
							    :direction)))
			      (:toggle ,(o-formula (gvl :parent :toggle)))
			      (:items ,(o-formula (gvl :parent :items)))
			      (:value-list ,(o-formula (gvl :parent
							    :value-list)))
			      )
		    ))
		 )

(define-method :set-value dbox-labeled-button-input (obj new-value)
  (call-schema (g-value obj :buttons) :set-value new-value))

;;;============================================================
;;; dbox-reset-button -- the reset button used by dialog boxes.  The
;;; user may set :left, :top.
;;;============================================================
(create-instance 'dbox-reset-button command-button-panel
		 (:left 0) (:top 0)
		 (:items `(("Use Default Values"
			    ,#'(lambda (obj &rest args)
				 (declare (ignore args))
				 (call-schema (g-value obj :parent)
					      :use-default-values)
				 (opal:update dialog-window)
				 ))
			   ))
		 )

;;;============================================================
;;; dbox-labeled-multi-text -- similar to dbox-labeled-text-input (see
;;; below) but uses multi-text instead of a scrolling box
;;;============================================================
(create-instance 'dbox-labeled-multi-text miro-aggregadget
		 (:left 0) (:top 0)
		 (:default-value "")
		 (:width (o-formula
			  (if (gvl :visible)
			      (+ (gvl :label :width)
				 (gvl :frame :width)
				 5)
			    0)))
		 (:height (o-formula
			   (if (gvl :visible)
			       (max (gvl :label :height)
				    (gvl :frame :height))
			     0)))
		 (:label-string "")
		 (:label-font (o-formula (gv *fonts-and-styles*
					     :button-label-font)))
		 (:text-font (o-formula (gv *fonts-and-styles*
					    :label-font)))
		 (:multi-text T)
		 (:value (o-formula (gvl :text :string)))
		 (:parts
		  `((:label ,opal:text
			    (:left ,(o-formula (gvl :parent :left)))
			    (:top ,(o-formula (gvl :parent :top)))
			    (:string ,(o-formula (gvl :parent
						      :label-string)))
			    (:font ,(o-formula (gvl :parent
						    :label-font)))
			    )
		    (:frame ,opal:rectangle
			  (:left ,(o-formula (+ (gvl :parent :label :left)
						(gvl :parent :label :width)
						5)))
			  (:top ,(o-formula (+ (gvl :parent :top))))
			  (:width ,(o-formula (+ (gvl :parent :text :width)
						 10)))
			  (:height ,(o-formula (+ (gvl :parent :text :height)
						  6)))
			  )
		    (:text ,opal:aggregadget
			   (:left ,(o-formula (+ (gvl :parent :frame
						      :left)
						 5)))
			   (:top ,(o-formula (+ (gvl :parent :frame :top)
						3)))
			   (:string ,(o-formula (if (gvl :parent :multi-text)
						    (gvl :multi-text :string)
						  (gvl :single-text :string))))
			   (:parts
			    ((:multi-text ,opal:cursor-multi-text
					  (:left ,(o-formula (gvl :parent :left)))
					  (:top ,(o-formula (gvl :parent :top)))
					  (:visible ,(o-formula
						      (and (gvl
							    :parent
							    :parent
							    :visible)
							   (gvl :parent
								:parent
								:multi-text))))
					  (:font ,(o-formula (gvl
							      :parent
							      :parent
							      :text-font)))
					  (:cursor-index nil)
					  (:string "")
					  )
			     (:single-text ,opal:cursor-text
					   (:left ,(o-formula (gvl
							       :parent
							       :left)))
					   (:top ,(o-formula (gvl
							      :parent
							      :top)))
					   (:visible ,(o-formula
						       (and
							(gvl :parent
							     :parent
							     :visible)
							(not
							 (gvl :parent
							      :parent
							      :multi-text)))))
					   (:font ,(o-formula
						    (gvl :parent
							 :parent
							 :text-font)))
					   (:cursor-index nil)
					   (:string "")
					   )
			     )))
		    ))
		 (:interactors
		  `((:text-inter ,inter:text-interactor
				 (:feedback-obj nil)
				 (:window ,(o-formula (gvl
						       :operates-on
						       :window)))
				 (:start-where
				  ,(o-formula (list :in-box
						    (gvl :operates-on :frame))))
				 (:obj-to-change
				  ,(o-formula (if (gvl :operates-on :multi-text)
						  (gvl :operates-on
						       :text :multi-text)
						(gvl :operates-on
						     :text :single-text))))
				 (:stop-event (:any-mousedown #\RETURN))
				 )
		    ))
		 )

(define-method :start-text-inter dbox-labeled-multi-text (obj)
  (inter:start-interactor (g-value obj :text-inter)))

(define-method :set-value dbox-labeled-multi-text (obj new-value)
  (let ((text (if (g-value obj :multi-text)
		  (g-value obj :text :multi-text)
		(g-value obj :text :single-text))))
    (s-value text :string (or new-value ""))
    (s-value text :cursor-index nil)))

(define-method :use-default-values dbox-labeled-multi-text
  (obj &optional original-callers)
  (declare (ignore original-callers))
  (call-schema obj :set-value (g-value obj :default-value)))

;;;============================================================
;;; dbox-labeled-text-input -- the labeled scrolling text input used
;;; in dialog boxes.  User may set :left, :top, :scrolling-box-width,
;;; :title-string.  The input value is in :value.  The value may be
;;; explicitly set by calling :set-value with the new string.
;;;============================================================
(create-instance 'dbox-labeled-text-input miro-aggregadget
		 (:left 0) (:top 0)
		 (:default-value "")
		 (:width (o-formula
			  (if (gvl :visible)
			      (max (gvl :title :width)
				   (+ 5 (gvl :scrolling-box-width)))
			    0)))
		 (:height (o-formula
			   (if (gvl :visible)
			       (+ (g-value (gvl :name) :height)
				  (g-value (gvl :title) :height)
				  5)
			     0)))
		 (:scrolling-box-width 300)
		 (:title-string "Enter/Edit a string:")
		 (:value (o-formula (gvl :name :value)))
		 (:parts
		  `((:title ,opal:text
			    (:left ,(o-formula (gvl :parent :left)))
			    (:top ,(o-formula (gvl :parent :top)))
			    (:string ,(o-formula (gvl :parent
						      :title-string)))
			    (:font ,(o-formula (gv *fonts-and-styles*
						   :button-label-font)))
			    )
		    (:name ,garnet-gadgets:scrolling-labeled-box
			   (:top ,(o-formula (+ 5 (gvl :parent :top)
						(gvl :parent :title
						     :height))))
			   (:left ,(o-formula (+ (gvl :parent :left)
						 5)))
			   (:width ,(o-formula
				     (gvl :parent
					  :scrolling-box-width)))
			   (:field-font ,(o-formula (gv *fonts-and-styles*
						  :fixed-label-font)))
			   (:value "")
			   (:label-string "")
			   (:label-offset 0)
			   ))))

(define-method :initialize dbox-labeled-text-input
  (obj &optional top-agg)
  (if top-agg (call-prototype-method obj top-agg)
    (call-prototype-method obj))
  (s-value (g-value obj :name :field-text :text-edit) :stop-event
	   '(#\RETURN :leftdown))
  )

(define-method :start-text-inter dbox-labeled-text-input (obj)
  (inter:start-interactor (g-value obj :name :field-text :text-edit)))

(define-method :set-value dbox-labeled-text-input (obj new-value)
  (s-value (g-value obj :name) :value (if new-value new-value "")))

(define-method :use-default-values dbox-labeled-text-input
  (obj &optional original-callers)
  (declare (ignore original-callers))
  (call-schema obj :set-value (g-value obj :default-value)))

;;;============================================================
;;; dbox-filename-selector --  Allows the user to request directory
;;; listings while entering a filename.  User may set :left, :top,
;;; :title-string, :scrolling-box-width.
;;;============================================================
(create-instance
 'dbox-filename-selector miro-aggregadget
 (:left 0) (:top 0)
 (:default-value "")
 (:title-string "Filename:")
 (:scrolling-box-width 300)
 (:value (o-formula (gvl :text :value)))
 (:showing-completions nil)
 (:parts
  `((:text ,dbox-labeled-text-input
	   (:default-value ,(o-formula (gvl :parent :default-value)))
	   (:top ,(o-formula (gvl :parent :top)))
	   (:left ,(o-formula (gvl :parent :left)))
	   (:scrolling-box-width
	    ,(o-formula (gvl :parent :scrolling-box-width)))
	   (:title-string ,(o-formula (gvl :parent
					   :title-string)))
	   )
    (:show-completion-button
     ,command-button-panel
     (:visible ,(o-formula (and (gvl :parent :visible)
				(not (gvl :parent
					  :showing-completions))
				)))
     (:top ,(o-formula (+ (gvl :parent :top)
			  (gvl :parent :text :height)
			  5)))
     (:left ,(o-formula (+ 5 (gvl :parent :left))))
     (:items (("Show Completions"
	       ,#'(lambda (obj &rest args)
		    (declare (ignore args))
		    (s-value (g-value obj :parent)
			     :showing-completions T)
		    ))))
     )
    (:hide-completion-button
     ,command-button-panel
     (:visible ,(o-formula (and (gvl :parent :visible)
				(gvl :parent
				     :showing-completions)
				)))
     (:top ,(o-formula (+ (gvl :parent :top)
			  (gvl :parent :text :height)
			  5)))
     (:left ,(o-formula (+ 5 (gvl :parent :left))))
     (:items (("Hide Completions"
	       ,#'(lambda (obj &rest args)
		    (declare (ignore args))
		    (s-value (g-value obj :parent)
			     :showing-completions nil)
		    ))))
     )
    (:completion-menu
     ,garnet-gadgets:scrolling-menu
     (:left ,(o-formula (+ 5 (gvl :parent :hide-completion-button
				  :left)
			   (gvl :parent :hide-completion-button
				:width))))
     (:top ,(o-formula (gvl :parent :hide-completion-button :top)))
     (:can-be-visible ,(o-formula (and (gvl :parent :visible)
				       (gvl :parent
					    :showing-completions))))
     (:visible ,(o-formula (and (gvl :can-be-visible)
				(progn
				  (opal:notice-items-changed
				   (gvl :menu-item-list))
				  T)
				)))
     (:item-font ,(o-formula (gv *fonts-and-styles*
				 :button-label-font)))
     (:final-feedback-p nil)
     (:multiple-p nil)
     (:title nil)
     (:title-font ,(o-formula (gv *fonts-and-styles* :microscopic-font)))
     (:items ,(o-formula (when (gvl :can-be-visible)
			       (completion-list (gvl :parent :text
						     :value)))))
     (:menu-selection-function
      ,#'(lambda (obj value)
	   (call-schema (g-value obj :parent :text) :set-value
			(g-value value :item))
	   (s-value (g-value obj :parent) :showing-completions nil)
	   ))
     )
    ))
 )

(define-method :start-text-inter dbox-filename-selector (obj)
  (call-schema (g-value obj :text) :start-text-inter))

(define-method :set-value dbox-filename-selector (obj new-value)
  (call-schema (g-value obj :text) :set-value (if new-value new-value
						"")))

;;;============================================================
;;; reconnect-garnet, change-garnet-display: call the appropriate
;;; garnet function and update our colors.
;;;============================================================

(defun reconnect-garnet (&rest args)
  (apply 'opal:reconnect-garnet args)
  (s-value *colors* :color-display (g-value opal:color :color-p))
  (opal:update-all)
  )

(defun change-garnet-display (&rest args)
  (apply 'opal:change-garnet-display args)
  (s-value *colors* :color-display (g-value opal:color :color-p))
  (opal:update-all)
  )


;;;============================================================
;;; make-temporary-file -- obtains a unique temporary file name by
;;; taking "basename" and adding version numbers to it if necessary
;;; until an unused name is found.  Touches the new file and returns a
;;; string containing the filename.
;;;============================================================
(defun make-temporary-file (basename)
  (do* ((fname basename (format nil "~A_~S" basename version))
	(version 1 (+ version 1))
	)
       ((with-open-file (file fname :direction :output :if-exists nil
			      :if-does-not-exist :create) file)
	fname)))
				

;;;============================================================
;;; skip-comments-and-whitespace -- skip over line comments and
;;; whitespace an an input file
;;;============================================================
(defun skip-comments-and-whitespace (file comment-char)
  (let ((*eof* :eof)
	c)
    (loop
     (setq c (peek-char T file nil *eof*))
     ;; skip comments -- peek-char will skip whitespace for us
     (if (equal c comment-char) (read-line file nil *eof*) (return))
     ))
  )

;;;============================================================
;;; get-output -- reads output from a list of streams and returns a
;;; list of strings in the same order.  Returns nil for nil streams.
;;;============================================================
(defun list-of-nil (n)
  (if (> n 0) (cons nil (list-of-nil (- n 1))) nil))

(defun get-output (process-id &rest streams)
  ;; loop until we reach eof on all the streams
  (do ((results (list-of-nil (length streams)))
       (wait-results nil)
       (wr nil)
       (*eof* :eof)
       )
      ((and streams
	    (every #'(lambda (s)
		       (if s
			   (let ((c (read-char-no-hang s nil *eof*)))
			     (cond ((equal c *eof*) T)
				   (c (unread-char c s) nil)
				   (T nil)))
			 T)
		       )
		   streams))
       (progn
	 (dolist (s streams) (when s (close s)))
#-cmu	 (do () ((null (and (pushnew (setq wr (reverse (multiple-value-list (sys:os-wait))))
				     wait-results :test #'equal)
			    (car wr)))))
         (append
	  (mapcar #'(lambda (r) (coerce (reverse r) 'string))
		  results)
	  (cdr (assoc process-id wait-results)))
	 ))
      ;; try to read from each of the streams
      (setf results
	    (mapcar
	     #'(lambda (s r)
		 (when s
		       (do* ((c (read-char-no-hang s nil *eof*)
				(read-char-no-hang s nil *eof*))
			     )
			    ((or (equal c *eof*) (null c)))
			    (push c r)))
		 r)
	     streams results))
      ))

;;;============================================================
;;; run-miro-tool -- runs a miro tool and returns a list containing
;;; stdout and stderr.
;;;============================================================
(defun run-miro-tool (tool arg-list &key (stdout nil) (stderr nil))
  (let* ((full-path (equal (elt tool 0) #\/))
	 (run-args '(:wait nil))
	 result-streams)

    ;; error output
    (if stderr
	(progn (push :supersede run-args)
	       (push :if-error-output-exists run-args)
	       (push stderr run-args)
	  )
      (push :stream run-args))
    (push #+cmu :error #-cmu :error-output run-args)

    ;; output
    (if stdout
	(progn (push :supersede run-args)
	       (push :if-output-exists run-args)
	       (push stdout run-args)
	  )
      (push :stream run-args))
    (push :output run-args)
    
    #+cmu (push arg-list run-args)
    (push #+cmu (format nil "~A~A" (if full-path "" "ambig:") tool)
	  #-cmu (format nil "~A~A~{ ~S~}"
			(if full-path "" Miro-AmbigPath) tool arg-list)
	  run-args)
    (setq result-streams
	  #+cmu (apply #'extensions:run-program run-args)
	  #-cmu (multiple-value-list (apply #'excl:run-shell-command
					    run-args))
	  )
    #+cmu (get-output nil
		      (extensions:process-output
		       result-streams)
		      (extensions:process-error
		       result-streams))
    #-cmu (get-output (third result-streams)
		      (first result-streams)
		      (second result-streams))
    ))

;;;============================================================
;;; disksave - saves the current state of lisp to a file
;;;============================================================

(defun disksave (file)
  (let ((init-fn #'(lambda ()
			(miro::reconnect-garnet)
			(setf *started-from-disksave* T)
			(opal:lower-window dialog-window)
			(print-startup-msg)
			)
		 ))
#+cmu
    (progn
      (format T "~%disksave: garbage collecting...~%")
      (extensions:gc)
      (format T "~%disksave: disconnecting garnet...~%")
      (opal:disconnect-garnet)
      (format T "~%disksave: saving this lisp...~%")
      (extensions:save-lisp file
       :init-function init-fn
       :print-herald T)
      )
#-cmu
    (progn
      (format T "~%disksave: garbage collecting...~%")
      (excl:gc)
      (format T "~%disksave: disconnecting garnet...~%")
      (opal:disconnect-garnet)
      (format T "~%disksave: saving this lisp...~%")
      (excl:dumplisp
       :name file
       :restart-function init-fn
       :read-init-file nil)
      )
    ))

;;;============================================================
;;; completion-list -- returns a list of the possible completions of
;;; path (relative to the current directory if path is not absolute).
;;;============================================================
(defun completion-list (path)
  (let* ((ls-streams			; run ls
	  #+cmu
	  (extensions:run-program
	   "ls" (list "-F1d" (concatenate 'string path "*"))
	   :output :stream :error :stream :wait nil)
	  #-cmu
	  (multiple-value-list
	   (excl:run-shell-command
	    (concatenate 'string "ls -F1d " path "*") :output :stream
	    :error-output :stream :wait nil))
	  )
	 (whitespace '(#\space #\tab #\page #\return))
	 (results			; get the results
	  #+cmu (get-output nil
			    (extensions:process-output ls-streams)
			    (extensions:process-error ls-streams))
	  #-cmu (get-output nil (first ls-streams) (second ls-streams))
	  )
	 (paths (first results))
	 (errors (second results))

	 ;; strip off socket, executable markers.  Don't strip off "@"
	 ;; (symbolic link) since "@" should serve as a reminder to
	 ;; add or remove "L" from the ls argument list (the meaning
	 ;; of -L is reversed at CMU).
	 (unwanted-trailers "=*")
	 )
    (when (null-string errors)
	  (remove nil
		  (mapcar #'(lambda (p)
			      (let* ((path (string-trim whitespace p))
				     (last (- (length path) 1))
				     )
				(cond
				 ((= last -1) nil)
				 ((position (elt path last)
					    unwanted-trailers)
				  (subseq path 0 last))
				 (T path))))
			  (get-line-list paths)))
	  )
    )
  )

;;;------------------------------------------------------------
;;; Read an unquoted string.  If we find out that we aren't looking at
;;; an unquoted string, read whatever is there.
;;;------------------------------------------------------------
(defun read-string (;; the file to read from
		    file

		    ;; the end-of-file character
		    eof
		    
		    &key
		    ;; characters that signal that the next object
		    ;; should be read normally.
		    (non-string-chars
		     '(#\" #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8
		       #\9))

		    ;; characters that signal the end of the string.
		    (terminating-chars
		     '(#\space #\newline #\tab #\page #\return #\;))

		    ;; whitespace characters - these will be skipped
		    ;; over if they occur at the beginning of the
		    ;; string.
		    (white-space
		     '(#\space #\newline #\tab #\page #\return))

		    ;; the comment character
		    (comment-char #\#)

		    ;; parens
		    (open-paren #\{)
		    (close-paren #\})
		    (elt-separator #\,)
		    )

  (push open-paren non-string-chars)
  (push close-paren terminating-chars)

  (let ((first-char
	 ;; Skip over white space and comments
	 (do ((c (read-char file nil eof) (read-char file nil eof)))
	     ((or (and (not (member c white-space))
		       (not (when (equal c comment-char)
				  (read-line file nil eof)
				  T)))
		  (equal c eof))
	      c))))

    (cond
     ;; empty stream
     ((equal first-char eof) eof)

     ;; If we aren't looking at an unquoted string, just read
     ;; whatever is there.
     ((member first-char non-string-chars)
      (unread-char first-char file)
      (if (equal first-char open-paren) (read-list file eof
						   :non-string-chars
						   non-string-chars
						   :open-paren
						   open-paren
						   :close-paren
						   close-paren
						   :elt-separator
						   elt-separator)
	(read file nil eof))
      )

     ;; Read the string, character-by-character.  Accumulate the
     ;; characters in reverse order.  When we encounter a terminating
     ;; character, we remove that character from our list, reverse the
     ;; list, and coerce it to a string.  We also put the terminating
     ;; character back onto the input stream in case anyone else wants to
     ;; look at it.
     (T
      (do* ((c first-char (read-char file nil eof))
	    (c-list (list c) (cons c c-list)))
	   ((or (equal c eof) (member c terminating-chars))
	    (progn
	      (unless (equal c eof) (unread-char c file))
	      (coerce (reverse (cdr c-list)) 'string)
	      )))
      ))))

;;;------------------------------------------------------------
;;; Convert a string into a list of lines.
;;;------------------------------------------------------------
(defun get-line-list (str)
  (unless (null-string str)
	  (with-input-from-string
	   (s str)
	   (let ((*eof* :eof))
	     (do* ((line nil (read-line s nil *eof*))
		   (line-list nil (cons line line-list)))
		  ((equal line *eof*)
		   (reverse (cdr line-list)))
		  ))
	   )))
					  

;;;------------------------------------------------------------
;;; Convert a string containing an argument list into a list of
;;; arguments (strings)
;;;------------------------------------------------------------
(defun get-arg-list (arg-string)
  (unless (null-string arg-string)
	  (with-input-from-string
	   (str arg-string)
	   (let ((*eof* :eof))
	     (do* ((arg nil (read-string str *eof*
					 :non-string-chars
					 '(#\")))
		   (arg-list nil (cons arg arg-list)))
		  ((equal arg *eof*)
		   (reverse (cdr arg-list))))
	     ))))

;;;------------------------------------------------------------
;;; print the date
;;;------------------------------------------------------------
(defun print-date (file)
  (let ((date (multiple-value-list (get-decoded-time))))
    (format file "# ~A ~A ~S, ~S  ~2,'0D:~2,'0D:~2,'0D~%~%"
	    ;; day of week
	    (nth (seventh date)
		 '("Monday" "Tuesday" "Wednesday" "Thursday"
		   "Friday" "Saturday" "Sunday"))
	    ;; month
	    (nth (- (fifth date)1)
		 '("January" "February" "March" "April" "May" "June"
		   "July" "August" "September" "October" "November"
		   "December"))
	    ;; day
	    (fourth date)
	    ;; year
	    (sixth date)
	    ;; hour
	    (third date)
	    ;; minute
	    (second date)
	    ;; second
	    (first date)
	    )
    ))

;;;------------------------------------------------------------
;;; execute a function in the specified directory with the specified
;;; directory as *default-pathname-defaults*
;;;------------------------------------------------------------
(defun with-directory (dir pathname-default f &rest args)
  (let ((current-dir (excl:current-directory))
	(result nil))
    (excl:chdir (pathname dir))
    (let ((*default-pathname-defaults*
	   (make-pathname :directory pathname-default :type "fasl"))
	  (*dot* (make-pathname :directory dir :type "fasl")))
      (push *dot* system:*load-search-list*)
      (push *dot* system:*require-search-list*)
      (setq result (apply f args))
      (pop system:*load-search-list*)
      (pop system:*require-search-list*)
      )
    (excl:chdir current-dir)
    result))

;;;------------------------------------------------------------
;;; add/remove a dialog box from the dialog window
;;;------------------------------------------------------------
(defun add-dbox (dbox)
  (unless (g-value dbox :parent)
	  (opal:add-component dialog-agg dbox)
	  (opal:update dialog-window)
	  (mark-as-changed dialog-window :aggregate)
	  (opal:update dialog-window T)))

(defun remove-dbox (dbox)
  (when (g-value dbox :parent)
	(opal:remove-component dialog-agg dbox)
	(opal:update dialog-window)
	(mark-as-changed dialog-window :aggregate)
	(opal:update dialog-window T)))

(defun add-menu (menu)
  (unless (g-value menu :parent)
	  (opal:add-component menu-agg menu)
	  (opal:update menu-window)
	  (mark-as-changed menu-window :aggregate)
	  (opal:update menu-window T)))

(defun remove-menu (menu)
  (when (g-value menu :parent)
	(opal:remove-component menu-agg menu)
	(opal:update menu-window)
	(mark-as-changed menu-window :aggregate)
	(opal:update menu-window T)))
