;;;             -*- Mode: Lisp; Package: MIRO; -*-
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

;;; 
;;; MIRO EDITOR - Parse
;;;
;;; This file contains the functions necessary for read-miro-file
;;; bug: doesn't work if there's a comment between an attr-value pair
;;;      and the next entry (problems with the lisp reader....)
;;;

#|
============================================================
Change log:
    11/19/91 ky ; Got rid of most of this file -- parsing is done
		; mostly in C.
    11/04/91 ky ; Support "permissions" attribute for arrows.
    04/22/91 ky ; Changed read-list-or-string to handle "lisp" lists.
    04/18/91 ky ; Moved get-arg-list, read-string to miro-toolkit.lisp.
    02/19/91 ky ; Added support for reading a lisp file instead of an
		; iff file.
    02/12/91 ky ; Re-did box size/position assignment.
    02/11/91 ky ; read-list returns :empty-list instead of nil for an
		; empty list.  This allows parse-alist to see that it
		; did in fact read something.
		;
		; Don't complain about inside entries that have no
		; children -- this is legal.
		;
		; Assume that (0,0) is the lower left corner when
		; assigning box positions.
		;
    02/08/91 ky ; Let the user know how many entries we have
		; parsed.
		;
		; Put boxes into a hash table as we parse them.
		;
    02/07/91 ky ; Fixed comment-between-entries and empty-list bugs in
		; the parser.
    02/05/91 ky ; Added function minimum-picture-size,
		; assign-size-and-loc.
    02/01/91 ky ; Added function process-inside-entry to hold onto
		; insideness information.  Added function
		; set-minimum-box-size to compute the minimum size a
		; box can have and still hold it's children and label.
    12/18/90 ky ; Change the readtable in read-list-or-string if
		; needed.
    12/13/90 ky ; Handles lists as arrow types.  Added function
		; read-list-or-string.
    12/12/90 ky ; Added function read-list, to be called by
		; read-string when a list is encountered.
    11/9/90  ky ; Check for eof when reading an unquoted string.
    11/5/90  ky ; Added new variable, *adjust-sysnames* to indicate
		; whether or not sysnames should be made unique.
    8/10/90 ky  ; Use :points instead of :x1, etc. to set the location
		; of an arrow.
		;
    7/5/90  ky  ; Added function get-arg-list to parse an argument
		; list.  Check for valid values when parsing an iff
		; file.
    6/25/90 ky  ; When reading from a file, assign sysnames that are
		; unique in the current picture.
		;
		; Set *package* to "MIRO" when reading so that the
		; user doesn't have to be in the "MIRO" package before
		; starting the editor.
		;
		; Added function "read-string" which reads symbols as
		; strings to preserve case-sensitivity.  Added calls
		; to "read-from-string" to convert these back to
		; symbols where necessary.
    9/6/89 amz  Added actual object creation.
    9/5/89 amz  Created parser.

============================================================
|#

(in-package "MIRO" :use `("LISP" "KR"))

(defvar *lisp-readtable* (copy-readtable nil))

;;;------------------------------------------------------------
;;; create new readtable for iff syntax, (returns new readtable)
;;;------------------------------------------------------------
(defun create-iff-table ()
  (let ((new-readtable (copy-readtable)))
    ;; # is comment character
    (set-syntax-from-char #\# #\; new-readtable)
    ;; /* ... */ is comment

    ;; read property value list (return list of property values)
    ;; set left bracket
    (set-macro-character #\{ #'property-list-reader nil new-readtable)
    ;; set right bracket so won't be included in last p-value
    (set-macro-character #\} (get-macro-character #\) *lisp-readtable*) nil new-readtable)
    ;; reset ;
    (set-macro-character #\; #'iff-semicolon-reader nil new-readtable)
    ;; reset ,
    (set-macro-character #\, #'iff-comma-reader nil new-readtable)
    ;; = is special
    (set-macro-character #\= #'iff-equal-reader nil new-readtable)
    ;; return new readtable
    new-readtable))

(defun property-list-reader (stream char)
  (declare (ignore char))
  (read-delimited-list #\} stream T))

(defun iff-semicolon-reader (stream char) 
  (declare (ignore stream char)) 
  :semi)

(defun iff-comma-reader (stream char)
  (declare (ignore stream char))
  :comma)

(defun iff-equal-reader (stream char)
  (declare (ignore stream char))
  :equal)

(defparameter *iff-readtable* (create-iff-table)) 

;;;------------------------------------------------------------
;;; Read a list, preserving case.  Returns :empty-list instead of nil
;;; if the list is empty.
;;;------------------------------------------------------------
(defun read-list (file eof &key (open-paren #\{) (close-paren #\})
		       (non-string-chars '(#\" #\0 #\1 #\2 #\3 #\4 #\5
					   #\6 #\7 #\8 #\9))
		       (elt-separator #\,))
  (let ((c (read-char file nil eof))
	)
    (cond
     ((equal c eof) eof)
     ((equal (peek-char T file nil eof) close-paren)
      (read-char file nil eof)
      :empty-list)
     (T
      (let ((result
	     (do* ((elt nil (read-string file eof
					 :non-string-chars
					 non-string-chars
					 :open-paren open-paren
					 :close-paren close-paren
					 :elt-separator elt-separator
					 :terminating-chars
					 (list elt-separator)))
		   (elt-list nil (if (equal elt "") elt-list
				   (cons elt elt-list)))
		   )
		  ((or (equal elt eof)
		       (equal (if elt-list (read-char file nil eof)
				(peek-char T file nil eof))
			      close-paren))
		   (reverse elt-list))
		  ))
	    )
	(if result result :empty-list)
	))
     )
    ))
;;;------------------------------------------------------------
;;; Try to read a list from a string.
;;;------------------------------------------------------------
(defun read-list-or-string (str &optional use-lisp-readtable)
  (let* ((change-readtable
	  (if (and (not use-lisp-readtable)
		   (equal *readtable* *lisp-readtable*))
	      (progn (setq *readtable* *iff-readtable*) T)))
	 (open-paren (if use-lisp-readtable #\( #\{))
	 (close-paren (if use-lisp-readtable #\) #\}))
	 (elt-separator (if use-lisp-readtable #\space #\,))
	 (result
	  (cond
	   ((null-string str) str)
	   (T (let* ((*eof* :eof)
		     (list-str (if (equal open-paren
					  (car (coerce str 'list)))
				   str (format nil "~A~A~A" open-paren
					       str close-paren)))
		     )
		(with-input-from-string (s list-str)
					(read-string s *eof*
						     :non-string-chars
						     '(#\")
						     :open-paren
						     open-paren
						     :close-paren
						     close-paren
						     :elt-separator
						     elt-separator))
		))
	   ))
	 )
    (when change-readtable (setq *readtable* *lisp-readtable*))
    result
    ))
