;;;             -*- Mode: Lisp; Package: MIRO; -*-
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

;;; 
;;; MIRO EDITOR OBJECTS
;;;
;;; This file contains the functions to create the basic graphical objects
;;; of the Miro Editor: miro-box and miro-arrow
;;;

#|
============================================================
Change log:
   11/04/91 ky  ; Added support for automatic box creation (verifier).
   02/21/91 ky  ; Added default :box/points value to miro-box/arrow.
   02/08/91 ky  ; Use a hash table in find-box.  Hopefully, this will
		; speed things up for "tiny" real-world applications
		; that have huge numbers of entries compared to what
		; we're used to...
		;
		; Deleted find-miro-obj-name since it is inefficient
		; and no one seems to call it.
		;
   02/05/91 ky  ; find-box can look for either a sysname or a string.
   11/5/90  ky  ; find-box looks in :boxes for boxes if :components is
		; empty.
   9/24/90 ky   ; Moved some information from ambig-menu to
		; ambig-status in an attempt to avoid display errors.
   9/21/90 ky   ; Warn the user that ambig results may be invalid when
		; the picture changes.
   9/18/90 ky   ; Delete undo-objecs if we created a new object.
   9/13/90 ky   ; A couple of changes to get rid of compile-time
		; warnings.
   9/12/90 ky   ; Use :draw-function :xor.
		;
   8/10/90 ky   ; Use :box (for boxes) or :points (for arrows) to
		; get/set size and position information, since slots
		; like :height and :width may contain bogus values if
		; the object is not visible for some reason.
		;
		; Added function find-attached-box to figure out which
		; box an arrow should be attached to.  This function
		; uses :box instead of :height, etc, which is what the
		; opal version uses.
		;
   7/31/90 ky   ; Use pic-sp instead of vert-sb, hor-sb.
		;
   7/23/90 ky   ; "miro-box"es scale themselves unless :fixed-size-p
		; is set.  :fixed-size-p is used for boxes in the
		; menu.  The scaling information is kept somewhat
		; seperate from horizontal and vertical offsets in an
		; attempt to speed things up; this doesn't seem to
		; help much.
		;
		; copy-miro-box now calls translate-points so that the
		; location of the new box will properly reflect the
		; current scale.
		;
   7/2/90  ky	; Don't use :xor as a :draw-function.  This causes
		; visibility problems in some cases.
   6/25/90 ky   ; Deleted ":label-string" since it was not always
		; updating/being updated by ":label :string".
   5/8/90  ky  Use push-error-msg for error messages.
   2/19/90 amz changed arrow selection to be two lines instead of polygon
   2/15/90 amz added stars to boxes
   2/9/90 amz added new copy-miro-{box,arrow} 
   2/5/90 amz changed arrow selection feedback to work (sort of) for
              containment arrows
   2/1/90 amz changed to single arrow type
   1/18/90 amz added :selected-feedback parts to box and arrow
   1/17/90 amz changed to asterisks for arrow stars
   1/5/90 amz changed arrow positioning along side of boxes
   9/5/89  amz Changed to o-formulas
   8/3/89  amz Changed to new version of aggrestuff.
   7/21/89 amz added labels to arrows
   7/18/89 amz change neg arrow to draw X on angle of arrow
   7/13/89 amz merged with my version 1, added miro-role and miro-type to boxes
   05/04 prm   Added text interactor for the boxes labels
   4/8/89   afm   added sysname to boxes and arrows
                  added function create-miro-box to make things consistent
   04/07 prm   Changed to match KR v2.0
   3/29/89  afm   moved X to middle of arrow
   Nov 1988 Philippe Marchal Created
============================================================
|#


(in-package "MIRO" :use `("LISP" "KR"))
(proclaim '(function destroy-undo-objects)) ; defined in miro-cmnds.lisp


;;;============================================================
;;; DEFAULT SIZES.
;;;============================================================

;;; Boxes
(defparameter *minimum-box-width* 60)
(defparameter *minimum-box-height* 30)

;;; Arrows
(defparameter *arrow-head-length* 12)
(defparameter *arrow-head-width* 12)
(defparameter *cross-x* 9)
(defparameter *cross-y* 9)
(defparameter *con-arrow-margin* 17)
(defparameter *arrow-star-offset* 8)

;;; distance of feedback box from box
(defparameter *box-selection-sep* 4)
(defparameter *box-selection-sep2* 8)

;;; width of arrow feedback
(defparameter *half-arrow-height* 10)

(defparameter *hit-threshold* 7) ; was 5



;;;============================================================
;;; PARTS
;;;  Aggregates used as parts of miro boxes or arrows:
;;;   - box-star-agg: a collection of "stars" used to make a starred
;;;   box 
;;;   - arrow-selection-feedback: two lines used to indicate arrow
;;;   selection. 
;;;============================================================

;;;------------------------------------------------------------
;;; create an aggregate with a collection of asterisks to put on
;;; starred boxes. 
;;;------------------------------------------------------------
(create-instance 'box-star-agg opal:aggregadget
		 (:visible (o-formula (gvl :parent :starred)))
		 ;; these slots can be used for all the star locations
		 (:left-1-3 (o-formula (+ (gvl :left)
					  (floor (gvl :width) 3))))
		 (:left-2-3 (o-formula (+ (gvl :left)
					  (* 2 (floor (gvl :width) 3)))))
		 (:offset-top (o-formula 
			       (- (gvl :top) 9)))
		 (:offset-bottom (o-formula
				  (- (+ (gvl :top)
					(gvl :height)) 10)))
		 (:font (o-formula 
			 (if (gvl :parent :thick)
			     (gv *fonts-and-styles*
				 :thick-box-asterisk-font)
			   (gv *fonts-and-styles*
			       :thin-box-asterisk-font))))
		 (:parts
		  `((:top-1 ,opal:text
			   (:visible ,(o-formula (gvl :parent :visible)))
			   (:top ,(o-formula 
				   (gvl :parent :offset-top)))
			   (:left ,(o-formula (gvl :parent
						   :left-1-3)))
			   (:font ,(o-formula (gvl :parent :font)))
			   (:string "*"))
		    
		    (:top-2 ,opal:text
			   (:visible ,(o-formula (gvl :parent :visible)))
			   (:top ,(o-formula 
				   (gvl :parent :offset-top)))
			   (:left ,(o-formula (gvl :parent :left-2-3)))
			   (:font ,(o-formula (gvl :parent :font)))
			   (:string "*"))
		    (:bottom-1 ,opal:text
			   (:visible ,(o-formula (gvl :parent :visible)))
			   (:top ,(o-formula 
				   (gvl :parent :offset-bottom)))
			   (:left ,(o-formula (gvl :parent :left-1-3)))
			   (:font ,(o-formula (gvl :parent :font)))
			   (:string "*"))
		    (:bottom-2 ,opal:text
			   (:visible ,(o-formula (gvl :parent :visible)))
			   (:top ,(o-formula 
				   (gvl :parent :offset-bottom)))
			   (:left ,(o-formula (gvl :parent :left-2-3)))
			   (:font ,(o-formula (gvl :parent :font)))
			   (:string "*"))
		    )))

;;;------------------------------------------------------------
;;; create  an aggregate with two lines to use as arrow feedback
;;; expects slots :{x,y}{1,2} to be set when used
;;; ends of feedback lines are flush with boxes
;;;------------------------------------------------------------
(create-instance 'arrow-selection-feedback opal:aggregadget
		 (:visible (o-formula (gvl :parent :selected)))
		 (:arrow-type (o-formula (gvl :parent :arrow-type)))
		 (:parts
		  `(
		    ;; line to left of con arrow or above other arrows
		    (:line-1 ,opal:line
		     (:line-style ,(o-formula (gv *fonts-and-styles*
						  :selected-dash)))
		     (:x1 ,(o-formula 
			    (cond
			     ((eq (gvl :parent :arrow-type)
				  :con)
			      (- (gvl :parent :x1)
				 *half-arrow-height*) )
			     (T
			      (gvl :parent :x1)))))
		     (:y1 ,(o-formula 
			    (cond
			     ((eq (gvl :parent :arrow-type)
				  :con)
			      (gvl :parent :y1))
			     (T
			      (- (gvl :parent :y1)
				 *half-arrow-height*)))))
		     (:x2 ,(o-formula 
			    (cond
			     ((eq (gvl :parent :arrow-type)
				  :con)
			      (- (gvl :parent :x2)
				 *half-arrow-height*) )
			     (T
			      (gvl :parent :x2)))))
		     (:y2 ,(o-formula 
			    (cond
			     ((eq (gvl :parent :arrow-type)
				  :con)
			      (gvl :parent :y2))
			     (T
			      (- (gvl :parent :y2)
				 *half-arrow-height*))))))
		    (:line-2 ,opal:line
		     (:line-style ,(o-formula (gv *fonts-and-styles*
						  :selected-dash)))
		     (:x1 ,(o-formula 
			    (cond
			     ((eq (gvl :parent :arrow-type)
				  :con)
			      (+ (gvl :parent :x1)
				 *half-arrow-height*) )
			     (T
			      (gvl :parent :x1)))))
		     (:y1 ,(o-formula 
			    (cond
			     ((eq (gvl :parent :arrow-type)
				  :con)
			      (gvl :parent :y1))
			     (T
			      (+ (gvl :parent :y1)
				 *half-arrow-height*)))))
		     (:x2 ,(o-formula 
			    (cond
			     ((eq (gvl :parent :arrow-type)
				  :con)
			      (+ (gvl :parent :x2)
				 *half-arrow-height*) )
			     (T
			      (gvl :parent :x2)))))
		     (:y2 ,(o-formula 
			    (cond
			     ((eq (gvl :parent :arrow-type)
				  :con)
			      (gvl :parent :y2))
			     (T
			      (+ (gvl :parent :y2)
				 *half-arrow-height*))))))
		    ))) ; end arrow-selection-feedback

;;;============================================================
;;; MIRO BOXES
;;;============================================================
;;;------------------------------------------------------------
;;; Definition of a miro-box,
;;; The position and size are in the :box slot (l t w h),
;;; the label in :label-string 
;;; also has slots :box-type, :box-role, :thick and :starred
;;;
;;; :parts =  frame (+ star) + label + selected-feedback
;;;------------------------------------------------------------
(create-instance 'miro-box opal:aggregadget
	   (:object-type :miro-box)
	   (:sysname 0)
	   (:box '(0 0 0 0))
	   (:fixed-size-p nil)
	   (:v-offset (o-formula (second (gv pic-sp :real-value))))
	   (:h-offset (o-formula (first (gv pic-sp :real-value))))
	   (:scaled-left
	    (o-formula
	     (let ((left (first (gvl :box))))
	       (if (gvl :fixed-size-p) left
		 (round (* left (gv zoom-agg :scale)))))))
	   (:left (o-formula (- (gvl :scaled-left) (gvl :h-offset))))
	   (:scaled-top
	    (o-formula
	     (let ((top (second (gvl :box))))
	       (if (gvl :fixed-size-p) top
		 (round (* top (gv zoom-agg :scale)))))))
	   (:top (o-formula (- (gvl :scaled-top) (gvl :v-offset))))
	   (:box-type "Unknown")
	   (:box-role "Unknown")
	   (:from-arrows NIL)
	   (:to-arrows NIL)
	   (:from-lenP1 (o-formula (1+ (length (gvl :from-arrows)))))
	   (:to-lenP1 (o-formula (1+ (length (gvl :to-arrows)))))
	   ;; sort a list of the objects on the other end of the arrows by their
	   ;; tops.  The FROM objects are at the other end of arrows TO this obj.
	   (:sorted-from-arrows-with-boxes (o-formula
			  (sort (mapcar #'(lambda (ar) 
					    (list ar (g-value ar :to)))
					(gvl :from-arrows))
				#'< :key 
				#'(lambda (pair) (g-value (cadr pair) :top)))))
	   (:sorted-to-arrows-with-boxes (o-formula
			  (sort (mapcar #'(lambda (ar) 
					    (list ar (g-value ar :from)))
					  (gvl :to-arrows))
				 #'< :key 
				 #'(lambda (pair) (g-value (cadr pair) :top)))))
	   (:selected nil)
	   ;; used in constraint boxes only
	   (:thick NIL)
	   (:starred NIL)
	   (:hit-threshold *hit-threshold*)
	   (:select-outline-only T)
	   (:parts
	    `((:frame
	      ,opal:roundtangle
	      (:select-outline-only
	       ,(o-formula (gvl :parent :select-outline-only)))
	      (:radius 15)
	      (:hit-threshold ,*hit-threshold*)
	      (:line-style ,(o-formula (if (gvl :parent :thick)
					   (gv *fonts-and-styles*
					       :thick-solid)
					 (gv *fonts-and-styles*
					     :thin-solid))))
	      (:left ,(o-formula (gvl :parent :left)))
	      (:top ,(o-formula (gvl :parent :top)))
	      (:width ,(o-formula (let ((w (third (gvl :parent :box))))
				    (if (gvl :parent :fixed-size-p) w
				      (round
				       (* w (gv zoom-agg :scale)))))))
	      (:height ,(o-formula (let ((h (fourth (gvl :parent :box))))
				     (if (gvl :parent :fixed-size-p) h
				       (round
					(* h (gv zoom-agg :scale)))))))
	      )
	    (:star 
	     ,box-star-agg
	     (:left ,(o-formula (gvl :parent :left)))
	     (:top ,(o-formula (gvl :parent :top)))
	     (:width ,(o-formula (let ((w (third (gvl :parent :box))))
				   (if (gvl :parent :fixed-size-p) w
				     (round
				      (* w (gv zoom-agg :scale)))))))
	     (:height ,(o-formula (let ((h (fourth (gvl :parent :box))))
				    (if (gvl :parent :fixed-size-p) h
				      (round
				       (* h (gv zoom-agg :scale)))))))
	     )
	    (:label
	      ,opal:cursor-multi-text
	      (:font ,(o-formula (gv *fonts-and-styles* :label-font)))
	      (:string "")
	      (:cursor-index NIL)
	      (:left ,(o-formula (+ (gvl :parent :frame :left) 10)))
	      (:top ,(o-formula (+ (gvl :parent :frame :top) 5))))
	    (:selected-feedback
	      ,opal:rectangle
	      ;; need radius if change to roundtangle
	      ;	       (:radius 15)
	      (:select-outline-only
	       ,(o-formula (gvl :parent :select-outline-only)))
	      (:hit-threshold ,*hit-threshold*)
	      (:line-style ,(o-formula (gv *fonts-and-styles*
					   :selected-dash)))
	      ;; visible iff selected
	      (:visible ,(o-formula (and (gvl :parent :selected)
					 (gvl :select-outline-only))))
	      (:left ,(o-formula (- (gvl :parent :left)
				    *box-selection-sep*)))
	      (:top ,(o-formula (- (gvl :parent :top)
				   *box-selection-sep*)))

	      (:width ,(o-formula (+
				   (let ((w (third (gvl :parent :box))))
				     (if (gvl :parent :fixed-size-p) w
				       (round
					(* w (gv zoom-agg :scale)))))
				   *box-selection-sep2*)))
	      (:height ,(o-formula (+
				    (let ((h (fourth (gvl :parent :box))))
				      (if (gvl :parent :fixed-size-p) h
					(round
					 (* h (gv zoom-agg :scale)))))
				    *box-selection-sep2*)))
	      ))))

;;;------------------------------------------------------------
;;; MENU-BOX
;;; create a slightly different miro box for use in the menu.
;;; Specifically, the location is not a function of the scroll bars,
;;; and there is no label.
;;;------------------------------------------------------------
(create-instance 'menu-box miro-box
		 (:fixed-size-p T)
		 (:select-outline-only nil)
		 (:left (o-formula (first (gvl :box))))
		 (:top (o-formula (second (gvl :box)))))



;;;------------------------------------------------------------
;;; Create-Miro-Box creates a new box with the given parameters,
;;; (prompts user for type and role), and starts text interactor to
;;; input name.
;;;------------------------------------------------------------
(setq *file-boxes-to-add* nil)
(setq *user-boxes-to-add* nil)

;; possible values :file :user nil
(setq *box-creation-list* nil)

(defun box-list-eq (expected actual)
  (not
   (if (stringp expected)
      (position nil
		(mapcar
		 #'(lambda (b)
		     (string-equal (g-value b :box-role) expected))
		 actual))
     (set-exclusive-or expected actual)
     )))
    
(defun insideness-ok (obj agg expected-parents expected-children)
  (push-help-string "Checking Insideness...")
  (build-containment-lists agg obj)
  (let* ((parent-specified (listp expected-parents))
	 (parents (when (and parent-specified expected-parents)
			(mapcar #'(lambda (p)
				    (find-box (cdr p)))
				expected-parents)))
	 (parent-role (unless parent-specified
			      (if (eq expected-parents :user)
				  "user" "file")))
	 (actual-parents
	  (if parent-specified
	      (remove
	       nil
	       (mapcar
		#'(lambda (b)
		    (when (position obj (g-value b :direct-contains))
			  b))
		(g-value obj :contained-by)))
	    (g-value obj :contained-by)))
	 (parent-differs
	  (not (box-list-eq (if parent-specified parents parent-role)
			    actual-parents)))
	 (child-specified (listp expected-children))
	 (children (when (and child-specified expected-children)
			 (mapcar #'(lambda (p)
				     (find-box (cdr p)))
				 expected-children)))
	 (child-role (unless child-specified
			     (if (eq expected-children :user)
				 "user" "file")))
	 (actual-children (g-value obj :direct-contains))
	 (child-differs
	  (not (box-list-eq (if child-specified children child-role)
			    actual-children)))
	 (child-msg nil)
	 (parent-msg nil)
	 )
    (when child-differs
	  (setq
	   child-msg
	   (if child-specified
	       (format
		nil
		(if children
		    "Box doesn't contain all the~%specified children."
		  "Box shouldn't contain~%anything."))
	     (format nil "Box may only contain ~S~%boxes."
		     child-role))))

     (when parent-differs
	   (setq
	    parent-msg
	    (if parent-specified
		(format nil
			(if parents
			    "Box not contained by the~%specified parents."
			  "Box shouldn't be inside anything."))
	      (format nil "Box may only be inside ~S~%boxes."
		      parent-role))))

    (pop-help-string)

    ;; return
    (if child-msg
	(if parent-msg
	    (format nil "~A~%~A" child-msg parent-msg)
	  child-msg)
      parent-msg)
    ))


(defun create-miro-box (point-list thick starred agg text-inter window)
  (declare (ignore text-inter window))
  (let ((new-object (create-instance NIL miro-box
				     (:box point-list)
				     (:thick thick)
				     (:starred starred)
				     (:sysname (get-new-number))
				     ;; change to a pop-up menu? (form)
				     (:box-type "Unknown")
				     ;;	(:box-role (get-role-from-user))
				     ))
	(fix-perm-args nil)
	)
    (case *box-creation-list*
	  (:file
	   (let* ((file (pop *file-boxes-to-add*))
		  (path (first file))
		  (name (second file))
		  (type (third file))
		  (inside (fourth file))
		  (sysname (g-value new-object :sysname))
		  (inside-error (insideness-ok new-object obj-agg
					       inside nil))
		  )

	     ;; abort if there is an insideness problem
	     (when inside-error
		   (push-error-msg inside-error)
		   (push file *file-boxes-to-add*)
		   (opal:destroy-me new-object)
		   (prepare-next-box-creation)
		   (return-from create-miro-box)
		   )

	     ;; set string, role, type
	     (s-value (g-value new-object :label) :string name)
	     (s-value new-object :box-role "file")
	     (s-value new-object :box-type type)

	     ;; delete box from update list; set sysname; add to
	     ;; appropriate hash tables
	     (let* ((add-list (g-value verify-update-menu
				       :file-box-addition-list))
		    (wb (g-value verify-update-menu :workbench))
		    (utable (fourth wb))
		    (ftable (sixth wb))
		    (btable (seventh wb))
		    (itable (ninth wb))
		    (path-no-trailer (string-right-trim '(#\/) path))
		    )
	       ;; delete box from update list
	       (setq add-list
		     (remove path add-list :test #'string= :key
			     #'car))

	       ;; set box's sysname in box addition lists
	       (dolist (b add-list)
		       (when (string= (car (second b)) path)
			     (setf (second b) (cons path sysname))
			     ))
	       (dolist (b *file-boxes-to-add*)
		       (dolist (box (fourth b))
			       (when (string= (car box) path)
				     (setf (cdr box) sysname)
				     )))
	       (s-value verify-update-menu :file-box-addition-list
			add-list)

	       ;; add box to appropriate hash tables
	       (when (and inside (listp inside))
		     (dolist (i inside)
			     (pushnew sysname (gethash (cdr i)
						       itable))))
	       (setf (gethash sysname btable) (list "file" type name))
	       (setf (gethash path-no-trailer ftable) sysname)

	       ;; remember to fix permissions
	       (maphash #'(lambda (k v)
			    (declare (ignore v))
			    (push (list k path-no-trailer) fix-perm-args))
			utable)
	       )
	     ))
	  (:user
	   (let* ((user (pop *user-boxes-to-add*))
		  (name (first user))
		  (type (second user))
		  (inside (third user))
		  (members (fourth user)) ; could be a group
		  (inside-error (insideness-ok new-object obj-agg
					       inside members))
		  (sysname (g-value new-object :sysname))
		  )
	     ;; abort if there is an insideness problem
	     (when inside-error
		   (push-error-msg inside-error)
		   (push user *user-boxes-to-add*)
		   (opal:destroy-me new-object)
		   (prepare-next-box-creation)
		   (return-from create-miro-box)
		   )

	     ;; set string, role, type
	     (s-value (g-value new-object :label) :string name)
	     (s-value new-object :box-role "user")
	     (s-value new-object :box-type type)

	     ;; delete box from update list
	     (s-value verify-update-menu :user-box-addition-list
		      (remove name (g-value verify-update-menu
					    :user-box-addition-list)
			      :test #'string=))

	     ;; add box to tables
	     (let* ((wb (g-value verify-update-menu :workbench))
		    (utable (fourth wb))
		    (ftable (sixth wb))
		    (btable (seventh wb))
		    (gtable (fifth wb))
		    (itable (ninth wb))
		    )
	       (setf (gethash name (if (string-equal type "user") utable
				     gtable)) sysname)
	       (setf (gethash sysname btable) (list "user" type name))
	       (when (and inside (listp inside))
		     (dolist (i inside)
			     (pushnew sysname (gethash (cdr i)
						       itable))))

	       ;; remember to fix permissions
	       (maphash #'(lambda (k v)
			    (declare (ignore v))
			    (push (list name k) fix-perm-args))
			ftable)
	       )
	     ))
	  )
    ;; put the new box in agg's hashtable
    (when (g-value agg :hashtbl)
	  (setf (gethash (g-value new-object :sysname)
			 (g-value agg :hashtbl))
		new-object))

    (opal:add-component agg new-object)

    ;; fix permissions if necessary
    (when fix-perm-args (fix-perms fix-perm-args))

    ;; we may have invalidated undo-objects
    (destroy-undo-objects)
    ;; we may have invalidated ambig results
    (when (g-value ambig-status :guaranteed-valid)
	  (s-value ambig-status :guaranteed-valid nil))
					;start interactor to enter box label
    
    (if (null-string (g-value new-object :label :string))
	(progn (setq *objects-to-display* (list (list new-object)))
	       (display-next-object))
      (prepare-next-box-creation)
      )
    ))

;;;------------------------------------------------------------
;;; Copy-Miro-Box creates a new miro box just like the box1 at an
;;; offset of delta-x and delta-y with different sysname, and adds the
;;; new box to the obj-agg. 
;;;------------------------------------------------------------
(defun copy-miro-box (box1 delta-x delta-y agg)
  (let* ((box (g-value box1 :box))
	 (new-object
	  (create-instance NIL miro-box
			   (:box (list (+ (first box) delta-x)
				       (+ (second box) delta-y)
				       (third box)
				       (fourth box)))
			   (:sysname (get-new-number))
			   (:box-type (g-value box1 :box-type))
			   (:box-role (g-value box1 :box-role))
			   (:thick (g-value box1 :thick))
			   (:starred (g-value box1 :starred))
			   (:from-arrows NIL)
			   (:to-arrows NIL)
			   ))
	 )
    (opal:add-component agg new-object)

    ;; put the new box in agg's hashtable
    (when (g-value agg :hashtbl)
	  (setf (gethash (g-value new-object :sysname)
			 (g-value agg :hashtbl))
		new-object))
    ;; we may have invalidated undo-objects
    (destroy-undo-objects)
    ;; we may have invalidated ambig results
    (when (g-value ambig-status :guaranteed-valid)
	  (s-value ambig-status :guaranteed-valid nil))
    new-object))

;;;============================================================
;;; MIRO ARROWS
;;;============================================================

;;;-----------------------------------------------------------------------
;;; Definition of a miro-arrow.
;;; The locations of the ends are in :x1,:y1,:x2 and :y2
;;; the label in :label-string 
;;; also has slots :arrow-type, :thick, :neg, and :starred
;;; internally maintained slot :alpha used to figure angle of cross.
;;;
;;; :parts =  shaft + head + label + {down-cross-bar + up-cross-bar} +
;;; star + selected-feedback 
;;;-----------------------------------------------------------------------

(create-instance 'miro-arrow opal:aggregadget
		 (:object-type :miro-arrow)
		 (:sysname 0)
		 (:points '(0 0 0 0))
		 (:fixed-size-p nil)
		 (:select-outline-only T)
		 (:h-offset (o-formula (first (gv pic-sp :real-value))))
		 (:v-offset (o-formula (second (gv pic-sp :real-value))))
		 (:scaled-x1
		  (o-formula
		   (let ((x1 (first (gvl :points))))
		     (if (gvl :fixed-size-p) x1
		       (round (* x1 (gv zoom-agg :scale)))))))
		 (:x1 (o-formula (- (gvl :scaled-x1) (gvl :h-offset))))
		 (:scaled-y1
		  (o-formula
		   (let ((y1 (second (gvl :points))))
		     (if (gvl :fixed-size-p) y1
		       (round (* y1 (gv zoom-agg :scale)))))))
		 (:y1 (o-formula (- (gvl :scaled-y1) (gvl :v-offset))))
		 (:scaled-x2
		  (o-formula
		   (let ((x2 (third (gvl :points))))
		     (if (gvl :fixed-size-p) x2
		       (round (* x2 (gv zoom-agg :scale)))))))
		 (:x2 (o-formula (- (gvl :scaled-x2) (gvl :h-offset))))
		 (:scaled-y2
		  (o-formula
		   (let ((y2 (fourth (gvl :points))))
		     (if (gvl :fixed-size-p) y2
		       (round (* y2 (gv zoom-agg :scale)))))))
		 (:y2 (o-formula (- (gvl :scaled-y2) (gvl :v-offset))))
		 (:arrow-type :syn)
		 (:thick NIL)
		 (:neg NIL)
		 (:starred NIL)
		 (:label-left (o-formula (gvl :label :left)))
		 (:label-top (o-formula (gvl :label :top)))
		 ;; the following slot (the angle of the arrow)
		 ;; is used to compute the arrow head
		 (:alpha (o-formula (atan (- (gvl :shaft :y1) (gvl :shaft :y2))
					  (- (gvl :shaft :x2) (gvl
							       :shaft
							       :x1)))))
		 ;; the next 2 slots are used for calculating the
		 ;; position of the star
		 (:star-center-x (o-formula (+ (gvl :x2)
					       (round (* *arrow-star-offset*
							 (cos (gvl :alpha)))))))
		 (:star-center-y (o-formula (- (gvl :y2)
					       (round (* *arrow-star-offset*
							 (sin (gvl :alpha)))))))
		 (:parts
		  `((:shaft ,opal:line
			    (:hit-threshold ,*hit-threshold*)
			    (:line-style ,(o-formula (cond 
					; :thick & sem
						      ((and (gvl :parent :thick) 
							    (eq (gvl :parent :arrow-type)
								:sem))
						       (gv *fonts-and-styles*
							   :thick-dash))
					; :thick & not sem
						      ((gvl :parent :thick) 
						       (gv *fonts-and-styles*
							   :thick-solid))
					; not :thick & sem
						      ((eq (gvl :parent :arrow-type)
							   :sem)
						       (gv *fonts-and-styles*
							   :thin-dash))
					; not :thick and not sem
						      (T (gv *fonts-and-styles*
							     :thin-solid)))))
			    (:x1 ,(o-formula (gvl :parent :x1)))
			    (:y1 ,(o-formula (gvl :parent :y1)))
			    (:x2 ,(o-formula (gvl :parent :x2)))
			    (:y2 ,(o-formula (gvl :parent :y2))))
		    (:head ,opal:arrowhead
			   (:head-x ,(o-formula (gvl :parent :shaft :x2)))
			   (:head-y ,(o-formula (gvl :parent :shaft :y2)))
			   (:from-x ,(o-formula (gvl :parent :shaft :x1)))
			   (:from-y ,(o-formula (gvl :parent :shaft :y1)))
			   (:length ,*arrow-head-length*)
			   (:diameter ,*arrow-head-width*)
			   (:line-style ,(o-formula (if (gvl :parent :thick)
							(gv *fonts-and-styles*
							    :thick-solid)
						      (gv *fonts-and-styles*
							  :thin-solid)))))
		    (:label ,opal:cursor-text
			    (:font ,(o-formula (gv *fonts-and-styles*
						   :label-font)))
			    (:string "")
			    (:cursor-index NIL)

					; can't use :parent :left or :parent :width in these
					; formulas -- they're NIL 
			    (:left ,(o-formula
				     (cond 
					; line is almost straight, or sloping down
				      ( (< (- (gvl :parent :y1)
					      (gvl :parent :y2)) 10)
					(+ (gvl :parent :shaft :left)
					   (round (/ (gvl :parent
							  :shaft
							  :width) 2))))  
				  
				      ;; width of string doesn't work right now, so put below line instead 
				      ;; (this may be better, regardless...)
					;line is sloping up
				      ( T
					(+ (+ (gvl :parent :shaft :left)
					      (round (/ (gvl :parent
							     :shaft
							     :width) 2))) 
					   3)))))
			    (:top ,(o-formula
				    (-
				     (cond 
					; line is almost straight, or sloping down
				      ( (< (- (gvl :parent :y1)
					      (gvl :parent :y2)) 10)
					(- (+ (gvl :parent :shaft :top) 
					      (round (/ (gvl :parent :shaft
							     :height) 2)))
					   15
					   ;; subtract more if neg
					   (if (gvl :parent :neg) 8 0)))
					;line is sloping up
				      ( T
					(+ (+ (gvl :parent :shaft :top) 
					      (round (/ (gvl :parent :shaft
							     :height) 2))) 
					   5
					   (if (gvl :parent :neg) 4 0))))
				     (if (gv *options* :large) 10 0)
				     ))))
					; change to this	'(:cross cross-agg ....)
					; these are angled now.
					; general form : x1 = x0 - (cross-x*cos(alpha)) - (cross-y*sin(alpha))
		    (:down-cross-bar ,opal:line
				     (:visible ,(o-formula (gvl :parent :neg))) 
				     (:line-style ,(o-formula (if (gvl :parent
								       :thick)
								  (gv *fonts-and-styles*
								      :thick-solid)
								(gv *fonts-and-styles*
								    :thin-solid))))
				     (:x1 ,(o-formula (round (- (-
								 (/ (+ (gvl :parent :x1) 
								       (gvl :parent :x2)) 2)
								 (* *cross-x* 
								    (cos (gvl :parent :alpha))))
								(* *cross-y* 
								   (sin (gvl :parent :alpha)))))))
				     (:y1 ,(o-formula (round (+ (-
								 (/ (+ (gvl :parent :y1) 
								       (gvl :parent :y2)) 2)
								 (* *cross-y* 
								    (cos (gvl :parent :alpha))))
								(* *cross-x* 
								   (sin (gvl :parent
									     :alpha)))))))
				     (:x2 ,(o-formula (round (+ (+
								 (/ (+ (gvl :parent :x1) 
								       (gvl :parent :x2)) 2)
								 (* *cross-x* 
								    (cos (gvl :parent :alpha))))
								(* *cross-y* 
								   (sin (gvl :parent
									     :alpha))))))) 
				     (:y2 ,(o-formula (round (- (+
								 (/ (+ (gvl :parent :y1) 
								       (gvl :parent :y2)) 2)
								 (* *cross-y* 
								    (cos (gvl :parent :alpha))))
								(* *cross-x* 
								   (sin (gvl :parent
									     :alpha))))))))
		    (:up-cross-bar ,opal:line
				   (:visible ,(o-formula (gvl :parent :neg))) 
				   (:line-style ,(o-formula (if (gvl :parent :thick)
								(gv *fonts-and-styles*
								    :thick-solid)
							      (gv *fonts-and-styles*
								  :thin-solid))))
				   (:x1 ,(o-formula (round (+ (-
							       (/ (+ (gvl :parent :x1) 
								     (gvl :parent :x2)) 2)
							       (* *cross-x* 
								  (cos (gvl :parent :alpha))))
							      (* *cross-y* 
								 (sin (gvl :parent :alpha)))))))
				   (:y1 ,(o-formula (round (+ (+
							       (/ (+ (gvl :parent :y1) 
								     (gvl :parent :y2)) 2)
							       (* *cross-y* 
								  (cos (gvl :parent :alpha))))
							      (* *cross-x* 
								 (sin (gvl :parent :alpha)))))))
				   (:x2 ,(o-formula (round (- (+
							       (/ (+ (gvl :parent :x1) 
								     (gvl :parent :x2)) 2)
							       (* *cross-x* 
								  (cos (gvl :parent :alpha))))
							      (* *cross-y* 
								 (sin (gvl :parent :alpha)))))))
				   (:y2 ,(o-formula (round (- (-
							       (/ (+ (gvl :parent :y1) 
								     (gvl :parent :y2)) 2)
							       (* *cross-y* 
								  (cos (gvl :parent :alpha))))
							      (* *cross-x* 
								 (sin (gvl :parent
									   :alpha))))))))

		    (:star ,opal:text
			   (:visible ,(o-formula (gvl :parent :starred)))
			   ;; will this fix visibility problems?
			   ;; this causes visibility problems in some cases...
			   (:draw-function :xor)
			   (:top ,(o-formula (- (gvl :parent
						     :star-center-y)
						9)))
			   (:left ,(o-formula (- (gvl :parent
						      :star-center-x) 4)))
			   (:font ,(o-formula
				    (if (gvl :parent :thick)
					(gv *fonts-and-styles*
					    :thick-arrow-asterisk-font)
				      (gv *fonts-and-styles*
					  :thin-arrow-asterisk-font))))
			   (:string "*"))
		    (:selected-feedback 
		     ,arrow-selection-feedback
		     ;; visible iff selected
		     (:visible ,(o-formula
				 (and (gvl :parent :selected)
				      (gvl :parent :select-outline-only))))
		     (:x1 ,(o-formula (gvl :parent :x1)))
		     (:y1 ,(o-formula (gvl :parent :y1)))
		     (:x2 ,(o-formula (gvl :parent :x2)))
		     (:y2 ,(o-formula (gvl :parent :y2))) 
		     )			; end selected-feedback
		    )))
		 ; end miro-arrow


;;;------------------------------------------------------------
;;; Create-Miro-Arrow creates a miro-arrow from x1,y1 to x2,y2
;;; (between the boxes "from" and "to"), of the specied type and with
;;; the appropriate attributes
;;;------------------------------------------------------------

(defun create-miro-arrow (x1 y1 x2 y2 from to neg thick type starred)
;  (when *test-debug* (format T "In create-miro-arrow~%"))
  (create-instance NIL miro-arrow
		   (:points (list x1 y1 x2 y2))
		   (:arrow-type type)
		   (:starred starred)
		   (:thick thick)
		   (:neg neg)
		   (:from from)
		   (:to to)
		   (:sysname (get-new-number))))


;;; Find the box, if any, at position x,y.  This is intended to be
;;; used to figure out which box an arrow should be attached to.
;;; Constraint-to should be T if the box is the to-box of a constraint
;;; arrow.
(defun find-attached-box (agg x y &optional constraint-to)
  (declare (ignore constraint-to))	; remove this when constraint
					; arrows are implemented properly
  (dovalues (obj agg :components)
    (let ((box (g-value obj :box)))
      (when box
	(let* ((threshold (or (g-value obj :hit-threshold) 0))
	       (left (first box))
	       (top (second box))
	       (right (+ left (third box)))
	       (bottom (+ top (fourth box)))
	       (inside (and (<= (- left threshold) x (+ right threshold))
			    (<= (- top threshold) y (+ bottom threshold))
			    ))
	       (on-left (<= (- left threshold) x (+ left threshold)))
	       (on-right (<= (- right threshold) x (+ right threshold)))
	       (on-top (<= (- top threshold) y (+ top threshold)))
	       (on-bottom (<= (- bottom threshold) y (+ bottom threshold)))
	       )
	  (when (and inside (or on-left on-right on-top on-bottom))
	    (return-from find-attached-box obj))
	  )
	)
      )
    )
  nil
  )
;;;------------------------------------------------------------
;;; Copy-Miro-Arrow creates a new
;;; modelled after draw-arrow-between-boxes
;;;------------------------------------------------------------
(defun copy-miro-arrow (arrow-to-copy delta-x delta-y agg)
  ;; to recognize inner boxes, have to use point-to-leaf to find
  ;; frame, and then look at its parent
  (let* ((points (g-value arrow-to-copy :points))
	 (new-x1 (+ (first points) delta-x))
	 (new-y1 (+ (second points) delta-y))
	 (new-x2 (+ (third points) delta-x))
	 (new-y2 (+ (fourth points) delta-y))
	 (from-box (find-attached-box obj-agg new-x1 new-y1))
	 (to-box (find-attached-box obj-agg new-x2 new-y2))
	 )
    ;; If one end of the arrow is not inside a box, or is from and to
    ;; the same box, beep.
    (if (or (null from-box) (null to-box)
	    (not (eq (g-value from-box :object-type) :miro-box))
	    (not (eq (g-value to-box :object-type) :miro-box))
	    (eq from-box to-box))
	  (push-error-msg "Couldn't copy arrow.")
      ;;  else draw the arrow.
      (let (new-arrow)
	(setq new-arrow 
	      (create-instance NIL miro-arrow 
		;; coordinates will be set later
		(:sysname (get-new-number))
		(:from from-box)
		(:to to-box)
		(:neg (g-value arrow-to-copy :neg))
		(:thick (g-value arrow-to-copy :thick))
		(:arrow-type (g-value arrow-to-copy :arrow-type))
		(:starred (g-value arrow-to-copy :starred))))
	(set-arrow-formulas new-arrow)
	(push new-arrow (g-value from-box :from-arrows))
	(push new-arrow (g-value to-box :to-arrows))
	;; add to aggregate
	(opal:add-component agg new-arrow)
	;; we may have invalidated undo-objects
	(destroy-undo-objects)
	;; we may have invalidated ambig results
	(when (g-value ambig-status :guaranteed-valid)
	      (s-value ambig-status :guaranteed-valid nil))
	new-arrow))))


;;;------------------------------------------------------------
;;; Draw-Arrow-Between-Boxes draws an arrow between the two boxes
;;; which are under the points in point-list. The arrow is drawn
;;; according to the relatives positions of the boxes.
;;;------------------------------------------------------------

(defun draw-arrow-between-boxes (point-list neg thick type starred agg text-inter window)
  (declare (ignore text-inter window))
  ;; to recognize inner boxes, have to use point-to-leaf to find
  ;; frame, and then look at its parent
  (let* ((from-box (find-attached-box agg (first point-list) (second point-list)))
	 (to-box (find-attached-box agg (third point-list) (fourth point-list)))
	 )
    ;; If one end of the arrow is not inside a box, or is from and to
    ;; the same box, beep.
;    (when *test-debug* (format T "In draw-arrow-between-boxes~%"))
    (if (or (null from-box) (null to-box)
	    (not (eq (g-value from-box :object-type) :miro-box))
	    (not (eq (g-value to-box :object-type) :miro-box))
	    (eq from-box to-box))
	(push-error-msg "Couldn't draw arrow.")
    ;;  else draw the arrow.
	(let ((new-arrow
	       (create-miro-arrow 
		;; coordinates will be set later
		0 0 0 0
		from-box to-box neg thick type starred)))
	  (set-arrow-formulas new-arrow)
	  (opal:add-component agg new-arrow)
	  (push new-arrow (g-value from-box :from-arrows))
	  (push new-arrow (g-value to-box :to-arrows))
	  ;; we may have invalidated undo-objects
	  (destroy-undo-objects)
	  ;; we may have invalidated ambig results
	  (when (g-value ambig-status :guaranteed-valid)
		(s-value ambig-status :guaranteed-valid nil))
	  (setq *objects-to-display* (list (list new-arrow)))
	  (display-next-object)
	  ;; take s-value ... out when interactor is fixed 
	  ;; doesn't work for starred arrows!
	  ;;   (s-value (g-value new-arrow :label) :string (get-label-from-user))
	  ))))

;;;============================================================
;;; ARROW END FORMULAS
;;;============================================================

;;;------------------------------------------------------------
;;; set formulas for ends of arrows based on arrow type
;;;------------------------------------------------------------
(defun set-arrow-formulas (arrow)
  (let ((kind (g-value arrow :arrow-type)))
    (cond ((or (equal kind :syn) (equal kind :sem))
	   (attach-arrow-to-boxes arrow))
	  ((equal kind :con) 
	   (attach-arrow-inside arrow))
	  ;; this shouldn't be necessary
	  (t
	   (attach-arrow-to-boxes arrow))
	  )))

;;;------------------------------------------------------------
;;; set the formulas of the ends of arrow so that the arrow goes from
;;; the right hand side of the :from box to the left hand side of the
;;; :to box, unless the to box is completely to the left of the from
;;; box, in which case, reverse.
;;; Position the y coords of the arrows to spread out along the side of
;;; the boxes.  
;;;------------------------------------------------------------
(defun attach-arrow-to-boxes (arrow)
  (s-value
   arrow :points
   (o-formula
    (let* ((to-box (gvl :to :box))
	   (from-box (gvl :from :box))
	   (to-left (first to-box))
	   (to-top (second to-box))
	   (to-width (third to-box))
	   (to-height (fourth to-box))
	   (from-left (first from-box))
	   (from-top (second from-box))
	   (from-width (third from-box))
	   (from-height (fourth from-box))
	   (test-fn #'(lambda (key pair) (eq key (car pair))))
	   )
      (declare (inline test-fn))
      (list
       ;; formula for x1 -- attach to :from box
       (cond
	;; if :to left of :from, attach to left
	((< (+ to-left to-width) from-left) from-left)
	;; otherwise, attach to right
	(t (+ from-left from-width))
	)

       ;; formula for y1 (order based on to boxes)
       (+ from-top (* (floor from-height (gvl :from :from-lenP1))
		      (1+ (or (position
			       arrow
			       (gvl :from :sorted-from-arrows-with-boxes)
			       :test test-fn)
			      0))))

       ;; formula for x2 -- attach to :to box
       (cond
	;; if :to left of :from, attach to right
	((< (+ to-left to-width) from-left) (+ to-left to-width))
	;; otherwise, attach to left
	(t to-left))

       ;; formula for y2 (order based on from boxes)
       (+ to-top (* (floor to-height (gvl :to :to-lenP1))
		    (1+ (or (position
			     arrow
			     (gvl :to :sorted-to-arrows-with-boxes)
			     :test test-fn)
			    0))))
       )))))

;;;------------------------------------------------------------
;;; set the formulas of the ends of arrow so that the arrow goes from
;;; the top/bottom of the :from box to just inside the bottom/top of the
;;; :to box. 
;;;------------------------------------------------------------
(defun attach-arrow-inside (arrow)
  (s-value
   arrow :points
   (o-formula
    (let* ((to-box (gvl :to :box))
	   (from-box (gvl :from :box))
	   (to-left (first to-box))
	   (to-top (second to-box))
	   (to-width (third to-box))
	   (to-height (fourth to-box))
	   (from-left (first from-box))
	   (from-top (second from-box))
	   (from-width (third from-box))
	   (from-height (fourth from-box))
	   )
      (list
       ;; formula for x1 -- attach to middle of :from box
       (+ from-left (floor from-width 2))

       ;; formula for y1 -- attach to top or bottom of :from box
       (cond ((> to-top (+ from-top from-height))
	      (+ from-top from-height))
	     (T from-top))

       ;; formula for x2 -- attach to middle of :to box
       (+ to-left (floor to-width 2))

       ;; formula for y2 -- put inside :to box
       (cond
	;; :from above :to -> attach inside top
	((> to-top (+ from-top from-height))
	 (+ to-top *con-arrow-margin*))
	;; :from below :to -> attach inside bottom
	(T (- (+ to-top to-height) *con-arrow-margin*)))
       )))))


;;;============================================================
;;; HELPER FUNCTIONS
;;;============================================================

;;; ------------------------------------------------------------
;;; unique ID's for objects
;;; ------------------------------------------------------------

(defvar *unique-num* 0)

(defun get-new-number ()
  (setq *unique-num* (+ *unique-num* 1)))

;;;------------------------------------------------------------
;;; Functions to get and change the object string -- do this
;;; to avoid problems when string changed by interactor
;;;------------------------------------------------------------
(defun get-object-string (obj)
  (g-value obj :label :string))

(defun set-object-string (obj string)
  (s-value (g-value obj :label) :string string))

;;; ----------------------------------------
;;;
;;; Prompts the string input
;;;
;;; ----------------------------------------
(defun get-type-from-user ()
  (get-string-from-user "Please enter the box type: "))

(defun get-role-from-user ()
  (get-string-from-user "Please enter the box role (user or file): "))

(defun get-label-from-user ()
  (get-string-from-user "Please enter the box label: "))


;;;------------------------------------------------------------
;;; returns T iff box1 is completely inside area
;;;------------------------------------------------------------
(defun box-inside-area-p (box area)
  (let ((box-left (g-value box :left))
	(box-top (g-value box :top))
	(area-left (first area))
	(area-top (second area)))
    ;; box starts and ends inside
    (and (and (< area-left box-left)
	      (< area-top box-top))
	 (and (> (+ area-left (third area))
		 (+ box-left (g-value box :width)))
	      (> (+ area-top (fourth area))
		 (+ box-top (g-value box :height)))))))

;;;------------------------------------------------------------
;;; returns T iff arrow is completely inside area
;;;------------------------------------------------------------
(defun arrow-inside-area-p (arrow area)
    (and 
     ;; (x1,y1) inside
     (point-inside-area-p (g-value arrow :x1) (g-value arrow :y1)
			  (first area) (second area) (third area)
			  (fourth area))
     ;; (x2,y2) inside
     (point-inside-area-p (g-value arrow :x2) (g-value arrow :y2)
			  (first area) (second area) (third area)
			  (fourth area))))

;;;------------------------------------------------------------
;;; returns T iff (x,y) is inside (top,left,width,height)
;;;------------------------------------------------------------
(defun point-inside-area-p (x y left top width height)
  (and
   ;; > top,left
   (and (> x left) (> y top))
   ;; < bottom,right
   (and (< x (+ left width)) (< y (+ top height)))))

;;;------------------------------------------------------------
;;; returns T iff obj is completely inside point-list
;;;------------------------------------------------------------
(defun obj-inside-area-p (obj point-list)
  (if (eq (g-value obj :object-type) :miro-box)
      (box-inside-area-p obj point-list)
    (arrow-inside-area-p obj point-list)))

;;;------------------------------------------------------------
;;; Find-box searches through the components of agg looking for
;;; a miro box with :sysname name or label name. Returns the miro box,
;;; or nil if not found
;;; used in read-miro-box routines to set box info for arrows.
;;; Uses a hash table to help with subsequent lookups.
;;;------------------------------------------------------------
(defvar *hash-stats* nil)
(defun find-box (name &optional (agg obj-agg))
  (let ((box nil)
	(hashtbl (g-value agg :hashtbl))
	(components nil)
	(equal-fn (if (numberp name)
		      #'(lambda (b n) (eq (g-value b :sysname) n))
		    #'(lambda (b n) (string= (get-object-string b) n))
		    ))
	)

    ;; create a new hash table if this agg doesn't have one yet.
    (unless hashtbl
	    (setq hashtbl
		  (s-value agg :hashtbl
			   (make-hash-table :test #'equal))))

    ;; try to look up name in the hash table
    (setq box (gethash name hashtbl))

    ;; did we find a valid entry?
    (cond
     ((null box)			; nope
      (when *hash-stats*
	    (format T "didn't find box in hashtbl~%"))
      )
     ((or (not (schema-p box))
	  (not (g-value box :is-a)))	; box deleted
      (remhash name hashtbl)
      (when *hash-stats*
	    (format T "found destroyed box in hashtbl~%"))
      )
     ((not (funcall equal-fn box name))	; wrong box
      (remhash name hashtbl)
      (when *hash-stats*
	    (format T "found wrong box in hashtbl~%"))
      )
     (T					; found it!!!
      (when *hash-stats*
	    (format T "found box in hashtbl~%"))
      (return-from find-box box))
     )

    ;; if we got this far, we didn't find the box in the hash table
    (unless (setq components (get-values agg :components))
	    (setq components (g-value agg :boxes)))

    ;; do a linear search through the components looking for the box
    ;; -- yuck
    (setq box nil)
    (dolist (obj components)
	    (when (and (eq (g-value obj :object-type) ':miro-box)
		       (funcall equal-fn obj name))
		  (setq box obj)))

    ;; if we found it, put it into the hash table for future
    ;; reference.
    (when box
	  (let ((sysname (g-value box :sysname))
		(strname (g-value box :label :string))
		)
	    (setf (gethash sysname hashtbl) box)
	    (when strname (setf (gethash strname hashtbl) box))
	    ))

    ;; return it
    box))
