;;;             -*- Mode: Lisp; Package: MIRO; -*-
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

;;; 
;;; MIRO EDITOR - MAIN FILE
;;;
;;; This is the main file of the Miro Editor. It contains:
;;; - default sizes (variables)
;;; - the creation of the windows of the editor 
;;; - the top-level function, run-editor
;;; The editor is started by (run-editor) and stopped with the exit-editor
;;; command button.
;;;
;;; The editor window is made of three main parts:
;;; - the tools and commands zone on the left with a set of tools to create
;;;   Miro objects, and a set of commands to edit these objects,
;;; - the workbench on the right, where the Miro objects are created.
;;; a help window, where information and error messages are displayed.
;;;

#|
============================================================
Change log:
    11/04/91 ky ; Rearranged/added windows.  Fixed handling of error
		; messages in help string.
    04/08/91 ky ; Redraw everything before entering main event loop.
    02/12/90 ky ; Added *drawing-area-width*, *drawing-area-height*.
    12/11/90 ky ; Took out gc notification for now -- causes
		; problems with multiple windows
    12/4/90  ky ; Added *dialog-left*, *dialog-top*.
    12/3/90  ky ; Added dialog-window, dialog-agg, dialog-label.
    11/21/90 ky ; Added help-background.  Hopefully this will fix the
		; microvax's display problem (help-window was black).
    11/20/90 ky ; Use filling-styles from *colors* in the scrollpad.
		; Put a background in menu-window, sb-window,
		; work-window.
    11/6/90  ky ; Arrange for the user to be notified when garbage
		; collection takes place.
    9/19/90 ky  ; Changed help-window default string to "Press the
		; help button for help."
    9/18/90 ky  ; Moved startup help message into *help-string*.
    9/13/90 ky  ; Moved export declaration to miro-defs.lisp.
		; Added some declarations to avoid compile-time
		; warnings.
    9/12/90 ky  ; Changed the title from "GARNET MIRO" to "Miro Editor".
		;
		; Tell the scrollpad to show the bounding box of the
		; picture.
		;
    8/24/90 ky  ; Use double-buffered windows except for miro-window.
		; Don't need to worry about border widths any more.
		; Moved startup message printing to function
		; "print-startup-msg" to allow it to be reprinted
		; easily after a disksave.
		;
    7/31/90 ky  ; Adjusted a few sizes/positions to make room for the
		; scrollpad.
		;
		; Swapped help-window and sb-window.
		;
		; Create pic-sp (a scrollpad) to do the job of hor-sb
		; and vert-sb.
		;
		; Changed help messages to conform to the size of the
		; new help window.
		;
    7/23/90 ky  ; Scroll bars are now trill devices.  sb-agg and
		; sb-window were added to contain the scroll bars and
		; the scaling trill device.
		;
		; Made help-window a little bigger.
		;
    6/25/90 ky  ; We don't need to leave the user in the miro package
		; any more.
    6/13/90 ky  ; Leave the user in the miro package at the end of
		; "run-editor".  We can't do this in miro-load.lisp
		; because the value of *package* gets reset at the end
		; of a load.
    6/8/90  ky  ; Added call to "force-output" to the end of "run-editor".
    6/8/90  ky  ; Changes for compatibility with Allegro Common Lisp:
                ;   - Don't attempt to tell the user about garbage
		;     collections in the help window in ACL, since
		;     there doesn't seem to be any way to find out
		;     when a gc starts, only when it ends.
                ;   - In ACL, call inter:main-event-loop and tell the
		;     user how to break out of and restart the loop.
    5/23/90 ky  ; Check border widths and set them by hand if they
		; haven't been set already.  Print a warning message
		; when this is done; this code should never need to be
		; executed, but it is needed to work around a bug.
    5/10/90 ky  ; Take border widths into account in :height, :width
		; formulas.
    5/1/90  ky  Added functions push-error-msg, pop-error-msg for
                temporarily displaying error messages.  Added
                :error-msg to help-string.
    4/27/90 ky  run-editor arranges for a message to be displayed in
                the help window when a gc is in progress.  Added variables
                *default-gc-before*, *default-gc-after*, *gc-use-help-window*.
    4/25/90 ky	Added functions push-help-string and pop-help-string to make
		restoring previous help messages easier.  These functions use
		help-string's :saved-strings to keep track of the strings saved
		by push-help-string.
    4/10/90 amz changed to multi-text in help window. 
                added function set-help-string
    2/21/90 amz added dialog box creation
    1/16/90 amz added obj-agg and feedback-agg
    1/5/90 amz moved interactor stuff to miro-work
    9/5/89 amz  Changed to o-formulas
    8/9/89 amz  Changed menu to include constraint options.
    8/3/89 amz  Changed to new version of aggrestuff.
    5/4/89 prm   Added text interactor for the boxes labels
    3/30/89 afm  added function save-workbench
                 first pass at producing output in interm file format
    Nov. 1988 Brad Myers, Philippe Marchal -- Created.

============================================================
|#


(in-package "MIRO" :use `("LISP" "KR"))

(proclaim '(function create-menus)) ; defined in miro-menus.lisp
(proclaim '(function create-dialog-boxes)) ; defined in miro-menus.lisp
(proclaim '(function create-workbench-inters)) ; defined in miro-inters.lisp

;; remember the previous gc notify functions so that we can restore them later
#+cmu (defvar *default-gc-before*)
(defvar *default-gc-after*)
(defvar *gc-use-help-window* nil)

;;;============================================================
;;; DEFAULT SIZES
;;;============================================================

; window parameters
(defparameter *help-top* 355)
(defparameter *sb-top* 520)
(defparameter *menu-width* 200)
; these may be changed by window manager
(defparameter *window-left* 1)
(defparameter *window-top* 1)
(defparameter *window-width* 1000)
(defparameter *window-height* 720)
(defparameter *dialog-left* 220)
(defparameter *dialog-top* 40)
(defparameter *drawing-area-width* 2000)
(defparameter *drawing-area-height* 2000)


;;;============================================================
;;; WINDOW SET UP
;;;============================================================

;;;------------------------------------------------------------
;;; Create-Windows creates all the windows and aggregates needed by
;;; the editor, and the scrollbars. 
;;;
;;; There are three parts to the editor: the tools on the left, the
;;; help window below, and the workbench on the right separated by a
;;; vertical line. 
;;; 
;;; Structure:
;;;    Window (aggregate) {:components}
;;;
;;; miro-window (miro-agg) {vert-sb, hor-sb}
;;;   menu-window (menu-agg) 
;;;   help-window (help-agg)
;;;   work-window (work-agg) {feedback-agg, obj-agg}
;;;------------------------------------------------------------
(defun create-windows ()
  (setq *all-windows* nil)

  ;; The main window and aggregate
  (create-instance 'miro-window inter:interactor-window
		       (:left *window-left*) (:top *window-top*)
		       (:width *window-width*) (:height *window-height*)
		       (:min-width *window-width*)
		       (:min-height *window-height*)
		       (:title "Miro Editor") (:icon-title "Miro Editor")
		       (:cursor (cons miro-cursor miro-mask))
		       (:double-buffered-p nil)
		       )
  (create-instance 'miro-agg opal:aggregadget
		   (:left 0) (:top 0)
		   (:width (o-formula (gvl :window :width)))
		   (:height (o-formula (gvl :window :height)))
		   (:overlapping NIL))
  (s-value miro-window :aggregate miro-agg)
  (opal:update miro-window)
  (push miro-window *all-windows*)

  ;; the dialog window
  (create-instance 'dialog-window inter:interactor-window
		   (:left *dialog-left*) (:top *dialog-top*)
		   (:width (o-formula (max 100 (gvl :aggregate :width))))
		   (:height (o-formula (max 20 (gvl :aggregate :height))))
		   (:title "Miro Dialog") (:icon-title "Miro Dialog")
		   (:cursor (cons miro-cursor miro-mask))
		   (:double-buffered-p T)
		   (:aggregate
		    (create-instance 'dialog-agg opal:aggregadget)))
  (opal:update dialog-window)
  (create-instance 'dialog-label opal:aggregadget
		   (:left 0) (:top 0)
		   (:width (o-formula (gvl :frame :width)))
		   (:height (o-formula (gvl :frame :height)))
		   (:parts
		    `((:frame ,miro-frame
			      (:component-to-frame
			       ,(o-formula (gvl :parent :label)))
			      (:where :back)
			      )
		      (:label ,opal:multi-text
			      (:top ,(o-formula (gv *fonts-and-styles*
						    :db-frame-width)))
			      (:left ,(o-formula (gv *fonts-and-styles*
						     :db-frame-width)))
			      (:font ,(o-formula (gv *fonts-and-styles*
						     :label-font)))
			      (:string "Dialog Window")
			      )
		      )))
  (opal:add-component dialog-agg dialog-label :where :back)
  (opal:update dialog-window)
  (opal:lower-window dialog-window)
  (push dialog-window *all-windows*)
			  
  (create-instance 'sb-window inter:interactor-window
		   (:left 1) (:top *sb-top*)
		   (:width *menu-width*) 
		   (:height (o-formula (- (gvl :parent :height) *sb-top* 3)))
		   (:border-width 1)
		   (:parent miro-window)
		   (:cursor (cons miro-cursor miro-mask))
		   (:aggregate
		    (create-instance 'sb-agg opal:aggregadget
				     (:left 1) (:top 1)
				     (:width (o-formula (gvl :window :width)))
				     (:height (o-formula (gvl :window :height)))
				     (:overlapping nil)))
		   (:double-buffered-p t)
		   )

  (create-instance 'pic-sp scrollpad
		   (:real-value '(0 0))
		   (:left (o-formula (+ (gvl :parent :left) 5)))
		   (:top (o-formula (+ (gvl :parent :top) 35)))
		   (:white-fill (o-formula (gv *colors* :white)))
		   (:light-gray-fill (o-formula (gv *colors* :light-gray)))
		   (:gray-fill (o-formula (gv *colors* :medium-gray)))
		   (:black-fill (o-formula (gv *colors* :black)))
		   ;; the bounding box of the picture
		   (:bb-box nil)
		   (:bb-size
		    (o-formula
		     (let* ((box (gvl :bb-box))
			    (scale (gv zoom-agg :scale))
			    (pad-height (- (gvl :pad :height) 2))
			    (pad-width (- (gvl :pad :width) 2))
			    (work-height (+ (second (gvl :val-2))
					    (gv work-window :height)))
			    (work-width (+ (first (gvl :val-2))
					   (gv work-window :width)))
			    (left (round (* pad-width scale (first box))
					 work-width))
			    (top (round (* pad-height scale (second box))
					work-height))
			    (width (min (round (* pad-width scale (third box))
					       work-width)
					(- pad-width left)))
			    (height (min (round (* pad-height scale (fourth box))
						work-height)
					 (- pad-height top)))
			    )
		       (list left top width height)
		       )))
		   (:show-bb (o-formula (and (gvl :scroll-p) (gvl :bb-box) T)))
		   (:scr-incr '(10 10))
		   (:page-incr '(100 100))
		   (:indicator-text-p '(nil nil))
		   (:cursor-size
		    (o-formula
		     (let* ((ww-height (gv work-window :height))
			    (ww-width (gv work-window :width))
			    (pad-height (- (gvl :pad :height) 2))
			    (pad-width (- (gvl :pad :width) 2))
			    (x (/ ww-width
				 (+ (first (gvl :val-2)) ww-width)))
			    (y (/ ww-height
				  (+ (second (gvl :val-2)) ww-height)))
			    (newx (round (* x pad-width)))
			    (newy (round (* y pad-height)))
			   )
		       (list
			(cond
			 ((< newx 10) 10)
			 ((> newx pad-width) pad-width)
			 (T newx))
			(cond
			 ((< newy 10) 10)
			 ((> newy pad-height) pad-height)
			 (T newy))
			)
		       )))
		   (:val-1 '(0 0))
		   (:val-2 (o-formula
			    (let ((scale (gv zoom-agg :scale))
				  (width (gv work-window :width))
				  (height (gv work-window :height))
				  )
			      (list (max 1
					 (round
					  (+ (* 1000 scale)
					     (- (* width scale)
						width))))
				    (max 1
					 (round
					  (+ (* 1000 scale)
					     (- (* height scale)
						height))))
				    ))))
		   (:scroll-p (o-formula (and (> (first (gvl :val-2)) 1)
					      (> (second (gvl :val-2))
						 1))))
		   (:selection-function
		    #'(lambda (obj nv)
			(push-help-string "Redrawing...")
			(s-value obj :real-value nv)
			(opal:update work-window)
			(pop-help-string)
			))
		   )
  (opal:add-component sb-agg pic-sp)
  (create-instance 'sb-background opal:rectangle
		   (:top 0) (:left 0)
		   (:width (o-formula (gvl :parent :width)))
		   (:height (o-formula (gvl :parent :height)))
		   (:line-style opal:no-line)
		   (:filling-style (o-formula (gv *colors* :white)))
		   )
  (opal:add-component sb-agg sb-background :where :back)
  (push sb-window *all-windows*)

  ;; the menu window and aggregate
  (create-instance 'menu-window inter:interactor-window
		   (:left 1) (:top 1)
		   (:width *menu-width*)
		   (:height (- *help-top* 4))
		   (:border-width 1)
		   (:cursor (cons miro-cursor miro-mask))
		   (:parent miro-window)
		   (:double-buffered-p t)
		   )
  (create-instance 'menu-agg opal:aggregadget
		   (:left 0) (:top 0)
		   (:width (o-formula (gvl :window :width)))
		   (:height (o-formula (gvl :window :height)))
		   (:overlapping NIL))
  (s-value menu-window :aggregate menu-agg)
  (create-instance 'menu-background opal:rectangle
		   (:top 0) (:left 0)
		   (:width (o-formula (gvl :parent :width)))
		   (:height (o-formula (gvl :parent :height)))
		   (:line-style opal:no-line)
		   (:filling-style (o-formula (gv *colors* :white)))
		   )
  (opal:add-component menu-agg menu-background :where :back)
  (push menu-window *all-windows*)

  ;; the help window and aggregate
  (create-instance 'help-window inter:interactor-window
		   (:left 1)
		   (:top *help-top*)
		   (:width *menu-width*)
		   (:height (- *sb-top* *help-top* 3))
		   (:border-width 1)
		   (:cursor (cons miro-cursor miro-mask))
		   (:parent miro-window)
		   (:double-buffered-p t)
		   )
  (create-instance 'help-agg opal:aggregadget
		   (:left 0) (:top 0)
		   (:width (o-formula (gvl :window :width)))
		   (:height (o-formula (gvl :window :height)))
		   (:overlapping NIL))
  (s-value help-window :aggregate help-agg)
  (create-instance 'help-background opal:rectangle
		   (:top 0) (:left 0)
		   (:width (o-formula (gvl :parent :width)))
		   (:height (o-formula (gvl :parent :height)))
		   (:line-style opal:no-line)
		   (:filling-style (o-formula (gv *colors* :white)))
		   )
  (opal:add-component help-agg help-background :where :back)
  (push help-window *all-windows*)

  ;; box creation windows
  (create-instance 'box-creation1-window inter:interactor-window
		   (:left (+ *menu-width* 4))
		   (:top (o-formula (- (gvl :parent :height) 67)))
		   (:width (o-formula
			    (- (round (- (gvl :parent :width)
					 *menu-width* 3) 2) 3)))
		   (:height 64)
		   (:border-width 1)
		   (:cursor (cons miro-cursor miro-mask))
		   (:parent miro-window)
		   (:double-buffered-p T)
		   )
  (create-instance 'box-creation1-agg opal:aggregate
		   (:left 0) (:top 0)
		   (:height (o-formula (gvl :window :height)))
		   (:width (o-formula (gvl :window :width)))
		   (:overlapping NIL)
		   )
  (s-value box-creation1-window :aggregate box-creation1-agg)
  (create-instance 'box-creation1-background opal:rectangle
		   (:top 0) (:left 0)
		   (:width (o-formula (gvl :parent :width)))
		   (:height (o-formula (gvl :parent :height)))
		   (:line-style opal:no-line)
		   (:filling-style (o-formula (gv *colors* :white)))
		   )
  (opal:add-component box-creation1-agg box-creation1-background :where :back)
  (push box-creation1-window *all-windows*)

  (create-instance 'box-creation2-window inter:interactor-window
		   (:left (o-formula (+ (gv box-creation1-window :left)
					(gv box-creation1-window :width)
					3)))
		   (:top (o-formula (gv box-creation1-window :top)))
		   (:width (o-formula (- (gvl :parent :width)
					 (gvl :left) 3)))
		   (:height 64)
		   (:border-width 1)
		   (:cursor (cons miro-cursor miro-mask))
		   (:parent miro-window)
		   (:double-buffered-p T)
		   )
  (create-instance 'box-creation2-agg opal:aggregate
		   (:left 0) (:top 0)
		   (:height (o-formula (gvl :window :height)))
		   (:width (o-formula (gvl :window :width)))
		   (:overlapping NIL)
		   )
  (s-value box-creation2-window :aggregate box-creation2-agg)
  (create-instance 'box-creation2-background opal:rectangle
		   (:top 0) (:left 0)
		   (:width (o-formula (gvl :parent :width)))
		   (:height (o-formula (gvl :parent :height)))
		   (:line-style opal:no-line)
		   (:filling-style (o-formula (gv *colors* :white)))
		   )
  (opal:add-component box-creation2-agg box-creation2-background :where :back)
  (push box-creation2-window *all-windows*)

  ;; the workbench window and aggregates
  (create-instance 'work-window inter:interactor-window
		   (:left (+ *menu-width* 4))
		   (:top 1)
		   (:width (o-formula (- (gvl :parent :width)
					 (gvl :left) 3)))
		   (:height (o-formula (- (gv box-creation1-window :top) 4)))
		   (:border-width 1)
		   (:cursor (cons miro-cursor miro-mask))
		   (:parent miro-window)
		   (:double-buffered-p t)
		   )
  (create-instance 'work-agg opal:aggregate
		   (:left 0) (:top 0)
		   (:width (o-formula (gvl :window :width)))
		   (:height (o-formula (gvl :window :height)))
		   (:overlapping T))
  (s-value work-window :aggregate work-agg)
  (create-instance 'work-background opal:rectangle
		   (:top 0) (:left 0)
		   (:width (o-formula (gvl :parent :width)))
		   (:height (o-formula (gvl :parent :height)))
		   (:line-style opal:no-line)
		   (:filling-style (o-formula (gv *colors* :white)))
		   )
  (opal:add-component work-agg work-background :where :back)
  ;; create aggregate for objects in workbench
  (create-instance 'obj-agg opal:aggregate
		   (:left 0) (:top 0)
		   (:width (o-formula (gvl :window :width)))
		   (:height (o-formula (gvl :window :height)))
		   (:overlapping T))
  (opal:add-component work-agg obj-agg)
  ;; create aggregate for feedback objects in workbench
  (create-instance 'feedback-agg opal:aggregate
		   (:left 0) (:top 0)
		   (:width (o-formula (gvl :window :width)))
		   (:height (o-formula (gvl :window :height)))
		   (:overlapping T))
  (opal:add-component work-agg feedback-agg)
  (push work-window *all-windows*)
  )


(defun set-up-box-creation-windows ()
  (create-instance 'box-creation1-string opal:multi-text
		   (:left 5) (:top 2)
		   (:font (o-formula (gv *fonts-and-styles* :label-font)))
		   (:cursor-index nil)
		   (:string "No boxes waiting to be created.")
		   )
  (opal:add-component box-creation1-agg box-creation1-string)

  (create-instance 'box-creation2-string opal:multi-text
		   (:left 70) (:top 20)
		   (:font (o-formula (gv *fonts-and-styles* :label-font)))
		   (:cursor-index nil)
		   (:string "File:")
		   )
  (opal:add-component box-creation2-agg box-creation2-string)
  (setq *box-creation-list* nil)
  (setq *file-boxes-to-add* nil)
  (setq *user-boxes-to-add* nil)
  (prepare-next-box-creation)
  )

(defun set-box-creation1-string (str)
  (s-value box-creation1-string :string str))

(defun set-box-creation2-string (str)
  (s-value box-creation2-string :string str))

(defun set-up-help ()
  (create-instance 'help-string opal:multi-text
		   (:left 5)
		   (:top 5)
		   (:error-msg nil)
		   (:saved-strings nil)
		   (:string (format nil "Press the ~S button~%for help."
				    "Help"))
		   (:font (o-formula (gv *fonts-and-styles* :label-font)))
		   (:cursor-index NIL))
  (opal:add-component help-agg help-string)
  )

;; set-help-string changes the text in the help window
(defun set-help-string (str)
  (if (g-value help-string :error-msg)
      (s-value help-string :saved-strings
	       (cons str (cdr (g-value help-string :saved-strings))))
    (s-value help-string :string str)))

;; push-help-string saves the text in the help window and displays the new
;; text
(defun push-help-string (str)
  (push (cons (g-value help-string :string)
	      (g-value help-string :error-msg))
	(g-value help-string :saved-strings))
  (s-value help-string :string str)
  (when (g-value help-string :error-msg)
	(s-value help-string :error-msg nil)
	(inter:change-active error-inter nil))
  (opal:update help-window))

;; push-error-msg pushes a string onto the help window and sets
;; :error-msg.
(defun push-error-msg (str)
  (push-help-string str)
  (s-value help-string :error-msg T)
  (inter:change-active error-inter T)
  (inter:beep))

;; pop-help-string restores the previous contents of the help window
(defun pop-help-string (&optional (dont-touch-errors T))
  (let ((saved-strings (g-value help-string :saved-strings)))

    (cond
     ;; current help string is an error message and we want to leave
     ;; it alone
     ((and dont-touch-errors (g-value help-string :error-msg))
      (s-value help-string :saved-strings
	       (remove nil saved-strings :key #'cdr :count 1)))

     ;; help string isn't an error message -- we want to delete it
     (T
      (s-value help-string :saved-strings (cdr saved-strings))
      (let* ((ss (car saved-strings))
	     (str (car ss))
	     (emsg (cdr ss))
	     )
	(s-value help-string :string (or str ""))
	(s-value help-string :error-msg emsg)
	(inter:change-active error-inter emsg)
	(when emsg (inter:beep))
	)))

    (opal:update help-window)
    ))

;; pop-error-msg calls pop-help-string if help-string's :error-msg
;; flag is T.
(defun pop-error-msg ()
  (when (g-value help-string :error-msg) (pop-help-string nil)))


;;;============================================================
;;; RUN EDITOR
;;;============================================================

;;;------------------------------------------------------------
;;; Run-Editor is the main function. It creates the windows, menus, and
;;; interactors needed to run the editor.
;;;------------------------------------------------------------
(defun run-editor (&optional no-main-event-loop)
  (s-value *colors* :color-display (g-value opal:color :color-p))
  (create-priorities)
  (create-windows)
  (create-menus menu-agg)
  (create-workbench-inters)
  (create-dialog-boxes) 
  (set-up-help)
  (set-up-box-creation-windows)
  (dolist (w *all-windows*) (opal:update w))

  ;; display a message on the help window when a gc is in progress
  #+cmu
  (progn
    (setf *default-gc-before* extensions:*gc-notify-before*)
    (setf *default-gc-after* extensions:*gc-notify-after*)
    (setf *gc-use-help-window* nil)
    (setf extensions:*gc-notify-before*
	  #'(lambda (bytes-in-use)
	      (funcall *default-gc-before* bytes-in-use)
	      (when *gc-use-help-window*
		    (push-help-string "Garbage collection in
progress.  Please wait...")
		    (inter:beep))))
    (setf extensions:*gc-notify-after*
	  #'(lambda (bytes-retained bytes-freed new-trigger)
	      (funcall *default-gc-after* bytes-retained bytes-freed new-trigger)
	      (when *gc-use-help-window*
		    (pop-help-string)
		    (inter:beep))))
    (setf *gc-use-help-window* T))
#|
  #-cmu
  (progn
    (setf *gc-use-help-window* nil)
    (setf excl:*global-gc-behavior* nil)
    (setf *default-gc-after* excl:*gc-after-hook*)
    (setf (sys:gsgc-switch :hook-after-gc) T)
    (setf excl:*gc-after-hook*
	  #'(lambda (global-p bytes-to-newspace bytes-to-oldspace utl-time)
	      (funcall *default-gc-after* global-p bytes-to-newspace
		       bytes-to-oldspace utl-time)
	      ;; this is a hack, but I couldn't find a "proper" way to
	      ;; get at this information...
	      (unless (or global-p (< excl::.gc-copied-to-oldspace.
				      excl:*tenured-bytes-limit*))
		      (if *gc-use-help-window*
			  (progn
			    (push-help-string
			     "Global garbage collection in
progress.  Please wait...")
			    (opal:update help-window)
			    (inter:beep))
			(format
			 T
			 "~%Global garbage collection in progress.  Please wait...~%")
			)
		      (excl:gc t)
		      (if *gc-use-help-window*
			  (progn
			    (pop-help-string)
			    (opal:update help-window)
			    (inter:beep))
			(format T "~%Global garbage collection completed.~%")
		      ))))
    (setf *gc-use-help-window* T)
    )
|#
  (unless no-main-event-loop (print-startup-msg))
  )

(setq *help-string*
      (format nil "Welcome to the Miro editor~%
The menu to the left of the screen allows you to select the type of
picture you wish to draw (instance or constraint), the type of object
(box or arrow), and the attributes of the object.

Mouse operations in the main window:
  - Right button: create new object. For boxes, press down where you
       want the top left corner of the box, and move to where you want
       the bottom right corner, then let go of the mouse button. For
       arrows, press down near an edge of the box at the tail of the
       arrow, and let up near an edge of the box at the head of the
       arrow. Once the object is created, enter its label.  
  - Left button: select/deselect object.
  - Middle button: sweep out an area. All objects in the area will be
       selected. 
  - Shift left: move selected objects.
  - Shift right: resize selected object (must have exactly one object
       selected).
  - Shift middle: display a menu with additional commands."))

(defun print-startup-msg ()
  (format T "~%~A~%" *help-string*)
  (force-output)

  #-cmu (progn (funcall (g-value redraw-inter :stop-action))
	       (inter:main-event-loop))
  )
