/* TOP.C

   Top-level control file for Miro constraint picture compiler cpc(1).

   $Header: top.c,v 1.5 91/10/25 18:21:27 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   Note: If the macro DEBUG is defined, the -d command-line switch for
   debugging is available in the resultant executable.
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "id-hash.h"
#include "parser.h"
#include "parser.g"

#include "parse-pred.h"
#include "hints.h"
#include "id-table.h"
#include "objs.h"
#include "extract-objs.h"
#include "compile.h"
#include "top.h"

/* Exit codes */
#define OKAY_EXIT     (0)	/* no errors */
#define CMD_LINE_EXIT (1)	/* problem with command-line */
#define OPEN_EXIT     (2)	/* file could not be opened */
#define PARSE_EXIT    (3)	/* problem reading or parsing a file */
#define ERROR_EXIT    (4)	/* constraint contained a non-parse error */

/* GLOBAL VARIABLES ======================================================== */

String argv0;
Boolean AtomicOnly = False;

/* LOCAL FUNCTIONS ========================================================= */

static void InitModules()
{
    InitParser();
    ConfigureHashTable();
}

ShutDownModules()
{
    ShutDownParser();
}

static Boolean ProcessIffFile(iff_fp,filename,pict,build_tree)
  FILE *iff_fp;
  String filename;
  INOUT Pict *pict;
  Boolean build_tree;
/* Parse and process the file 'iff_fp' named 'filename'. Store the resulting
   picture objects in 'pict'. If 'filename' is NULL, "stdin" is used as the
   name of the input file when errors are reported. If 'build_tree' is True,
   call the routine BuildTypeTree() to construct the BoxType tree.

   Returns True iff errors encountered parsing the IFF file.
*/
{
    Boolean result = False;
    Entry *p_tree;		/* parse tree returned from parse */

    /* process the input file; parse and extract from the parse tree */
    if (((p_tree=ParseFile(iff_fp))==NULL           /* build parse tree */
	 || Extract(p_tree,pict)                    /* extract data */
	 || (build_tree && BuildTypeTree(pict)))) { /* build type tree */
	fprintf(stderr,"%s: %d error(s) encountered parsing file '%s'.\n",
		argv0,parse_error_cnt,(filename ? filename : "stdin"));
	result = True;
    }

    /* wrap up this parsing */
    FreeParseTree(p_tree);
    return(result);
}

static int ProcessInput(argc,argv)
  int argc;
  char *argv[];
/* Returns one of the exit codes defined at the beginning of this file.
*/
{
    int option;			/* command-line flag */
    Boolean err_flag = False;	/* error encountered processing command-line */
    Boolean parse_err = False;	/* error encountered parsing IFF file */
    FILE *input_fp;		/* file to read IFF file from */
    FILE *type_fp;		/* file to read IFF type info from */
    FILE *hint_fp;		/* file to read (textual) hint info from */
    FILE *output_fp;		/* file to send output code to */
    String type_name = TYPE_FILE_NAME; /* name of input type file */
    String hint_name = HINT_FILE_NAME; /* name of input hint file */
    String output_name;		/* name of output file */
    Pict *pict;

    /* globals needed to use getopt(3) */
    int getopt();
    extern char *optarg;
    extern int optind;

    /* initialized command-line processing */
    optind = 1;
    output_name = (String)NULL;

    /* process command-line arguments */
    while ((option=getopt(argc,argv,"ah:o:t:")) != EOF) {
	switch(option) {
	  case 'a': AtomicOnly = True; break;
	  case 'h': hint_name = optarg; break;
	  case 'o': output_name = optarg; break;
	  case 't': type_name = optarg; break;
	  default:  err_flag = True; break;
	}
    }

    /* check for command-line errors */
    if (err_flag == True || (argc-optind) > 1) {
	fprintf(stderr,SYNTAX_STRING,argv0);
	return(CMD_LINE_EXIT);
    }

    /* create new picture objects stucture */
    pict = NewPict();

    /* open and process a 'type_fp' file */
    if ((type_fp=fopen(type_name,"r")) == NULL) {
	/* type file not found in current directory */
	if (*type_name != '/') {
	    /* if relative path, try looking in TYPE_FILE_DIR */
	    String full_name;
	    full_name =
		AllocArray(char,strlen(TYPE_FILE_DIR)+strlen(type_name)+1);
	    strcpy(full_name,TYPE_FILE_DIR);
	    strcat(full_name,type_name);
	    type_fp = fopen(full_name,"r");
	    Dealloc(full_name);
	}
    }
    if (type_fp == NULL) {
	fprintf(stderr,"%s: Type file '%s' could not be opened for reading.\n",
		argv0,type_name);
	return(OPEN_EXIT);
    }
    if (ProcessIffFile(type_fp,type_name,pict,True)) parse_err = True;
    fclose(type_fp);
#ifdef OLD_DEBUG
    DisplayIdHashTable(pict->table);
#endif OLD_DEBUG

    /* open & process an 'input_fp' file */
    if (argc > optind) {
	pict->input = argv[optind];
	if ((input_fp=fopen(pict->input,"r")) == NULL) {
	    fprintf(stderr,"%s: File '%s' could not be opened for reading.\n",
		    argv0,argv[optind]);
	    return(OPEN_EXIT);
	}
    } else {
	input_fp = stdin;
	pict->input = (String)NULL;
    }
    if (ProcessIffFile(input_fp,pict->input,pict,False)) parse_err = True;
    if (input_fp != stdin) fclose(input_fp);

    /* open and process a 'hint_fp' file */
    if ((hint_fp=fopen(hint_name,"r")) == NULL) {
	/* hint file not found in current directory */
	if (*hint_name != '/') {
	    /* if relative path, try looking in HINT_FILE_DIR */
	    String full_name;
	    full_name =
		AllocArray(char,strlen(HINT_FILE_DIR)+strlen(hint_name)+1);
	    strcpy(full_name,HINT_FILE_DIR);
	    strcat(full_name,hint_name);
	    hint_fp = fopen(full_name,"r");
	    Dealloc(full_name);
	}
    }
    if (hint_fp == NULL) {
	fprintf(stderr,"%s: Hint file '%s' could not be opened for reading.\n",
		argv0,hint_name);
	return(OPEN_EXIT);
    }
    if (ProcessHintFile(hint_fp,hint_name,pict)) parse_err = True;
    fclose(hint_fp);

    /* return immediately if parse errors encountered */
    if (parse_err) return(PARSE_EXIT);

    /* open output_fp file if necessary */
    if (output_name != (String)NULL) {
	if ((output_fp=fopen(output_name,"w")) == NULL) {
	    fprintf(stderr,"%s: File '%s' could not be opened for writing.\n",
		    argv0,output_name);
	    return(OPEN_EXIT);
	}
    } else {
	output_fp = stdout;
    }

    /* compile the picture 'pict', sending output to 'output_fp' FILE */
    if (CompilePict(output_fp,pict) == True) { /* ill-formed constraint */
	return(ERROR_EXIT);
    }

    /* close output_fp if necessary */
    if (output_fp != stdout) fclose(output_fp);
    return(OKAY_EXIT);
}

/* GLOBAL FUNCTIONS ======================================================== */

int main(argc,argv)
  int argc;
  char *argv[];
{
    int exit_code;
    argv0 = argv[0];
    InitModules();
    if ((exit_code=ProcessInput(argc,argv)) != OKAY_EXIT) {
	fprintf(stderr,"%s: no object code generated.\n",argv0);
    }
    ShutDownModules();
    return(exit_code);
}
