/* PARSE-PRED.C

   Module for parsing box predicates.

   $Header: parse-pred.c,v 1.2 91/10/21 01:29:46 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon.
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "parser.g"

#include "parse-pred.h"
#include "parse-pred.g"
#include "pred.y.h"

/* LOCAL VARIABLES ========================================================= */

#ifdef DEBUG
/* NOTE: '#' is currently unsupported */
static char *RelStrings[] = { "=", "#", "<", "<=", ">", ">=" };
#endif DEBUG

static int curr_line_no;

/* GLOBAL VARIABLES ======================================================== */

String yyin2_string;

/* LOCAL DEBUGGING FUNCTIONS =============================================== */

#ifdef DEBUG

static void Indent(level)
  int level;
{
    register int i = 2 * level;
    while (i--) fputc(' ',stderr);
}

static void DisplayVarExpr(v_exp)
  VarExpr *v_exp;
{
    if (v_exp->prefix) fprintf(stderr,"\"%s\" ",v_exp->prefix);
    fprintf(stderr,"$%s",v_exp->name);
    if (v_exp->suffix) fprintf(stderr," \"%s\"",v_exp->suffix);
}

static void DisplaySimplePred(pred,level)
  SimplePred *pred;
  int level;
{
    Indent(level);
    fprintf(stderr,"%s %s ",pred->attr_name,RelStrings[(int)(pred->rel)]);
    switch (pred->kind) {
      case VarPredKind:
	DisplayVarExpr(pred->u.var_expr);
	break;
      case StringPredKind:
	fprintf(stderr,"\"%s\"",pred->u.string);
	break;
      case IntPredKind:
	fprintf(stderr,"%d",pred->u.int_val);
	break;
    }
    fputc('\n',stderr);
}

static void DisplayTopPred(pred,level)
  Predicate *pred;
  int level;
{
    switch (pred->kind) {
      case Conjunction:
	Indent(level++);
	fputs("&&\n",stderr);
	break;
    }
    if (pred->pred) DisplayTopPred(pred->pred,level);
    DisplaySimplePred(pred->simple_pred,level);
}
#endif DEBUG

/* LOCAL FUNCTIONS ========================================================= */

void FreeVarExpr(v)
  VarExpr *v;
{
    Dealloc(v->prefix);
    Dealloc(v->suffix);
    Dealloc(v->name);
    Dealloc(v);
}

void FreeSimplePred(p)
  SimplePred *p;
{
    Dealloc(p->attr_name);
    switch(p->kind) {
      case VarPredKind:
	FreeVarExpr(p->u.var_expr);
	break;
      case StringPredKind:
	Dealloc(p->u.string);
	break;
    }
    Dealloc(p);
}

/* DEBUGGING FUNCTIONS ===================================================== */

#ifdef DEBUG

void DisplayPred(p)
  Predicate *p;
{
    if (p != NULL) DisplayTopPred(p,1);
}
#endif DEBUG

/* GLOBAL FUNCTIONS ======================================================== */

Predicate *ParsePred(p,line_no)
  String p;
  int line_no;
{
    extern YYSTYPE yyval2;	/* yacc's last formed value */

    curr_line_no = line_no;
    yyin2_string = p;
    return(yyparse2() ? (Predicate *)NULL : yyval2.pred);
}

void FreePred(p)
  Predicate *p;
{
    if (p != (Predicate *)NULL) {
	FreePred(p->pred);
	FreeSimplePred(p->simple_pred);
	Dealloc(p);
    }
}

void yyerror2(s)
  String s;
{
    ParseErrorS(curr_line_no,"%s in box predicate",s);
}
