/* OBJS.C

   Module for creating and storing picture objects (boxes, arrows, etc.) and
   the relations between them.

   $Header: objs.c,v 1.7 91/11/13 03:13:02 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "box-type.h"
#include "interval.h"
#include "var.h"
#include "id-table.h"
#include "objs.h"

/* LOCAL VARIABLES ========================================================= */

/* ":SUBJ" and ":OBJ" intervals for 'side' attribute */
static Intrvl *SubjIntrvl,*ObjIntrvl;

/* "pos" and "neg" intervals for 'parity' attribute */
static Intrvl *PosIntrvl,*NegIntrvl;

/* "T" and "NIL" intervals for 'direct' attribute */
static Intrvl *TIntrvl,*NilIntrvl;

/* LOCAL DEBUGGING FUNCTIONS =============================================== */

#ifdef DEBUG

static void DisplaySysnameList(elt_list)
  EltList *elt_list;
{
    StepLinkedList(elt_list,elt_list) {
	fprintf(stderr,"%d%s",elt_list->elt->sysname,
		(elt_list->next == (EltList *)NULL) ? "\n" : ", ");
    }
}

static void DisplayAdjacencies(elt_list)
  EltList *elt_list;
{
    StepLinkedList(elt_list,elt_list) {
	fprintf(stderr,"  %3d: ",elt_list->elt->sysname);
	DisplaySysnameList(elt_list->elt->adj_elts);
    }
}

#endif DEBUG

/* LOCAL FUNCTIONS ========================================================= */

static Elt *NewElt()
{
    Elt *result;

    result = AllocOne(Elt);
    result->sysname = NO_SYSNAME;
    result->thickness = UnknownThickness;
    result->adj_elts = (EltList *)NULL;
    result->frontier = result->marked = False;
    result->order = NO_ORDER;;
    result->rank = NO_RANK;
    return(result);
}

static void FreeBox(b)
  Box *b;
{
    switch (b->kind) {
      case PredKind:
	FreePred(b->u1.pred);
	break;
      case IntvlKind:
	fputs("ERROR: Can't free IntvlKind of box.\n",stderr);
	exit(-1);
    }
    Dealloc(b);
}

static void FreePermList(l)
  PermList *l;
{
    PermList *temp;
    while (l != (PermList *)NULL) {
	temp = l;
	l = l->next;
	Dealloc(temp);
    }
}

static void FreeArrow(a)
  Arrow *a;
{
    if (a->kind != Containment) { FreePermList(a->u.perm_list); }
    Dealloc(a);
}

static Intrvl *InitEqIntrvl(attr_name,kind)
  String attr_name;
  IntrvlKind kind;
{
    Intrvl *result = AllocOne(Intrvl);
    IntrvlRange *rng = AllocOne(IntrvlRange);
    Value *val = AllocOne(Value);

    result->name = attr_name;
    result->kind = kind;
    result->range = rng;
    rng->low = rng->high = val;
    val->next = (Value *)NULL;
    val->kind = EqComp;
    return(result);
}

static Intrvl *InitEqStringIntrvl(attr_name,key_name)
  String attr_name,key_name;
/* Create and return a new Intrvl representing an equality interval (in fact,
   the 'low' and 'high' values are identical pointers) on the attribute named
   'attr_name' where the lower and upper values are the same string 'key_name'
   (hence, the resulting Intrvl has a 'kind' of StringKind).
*/
{
    Intrvl *result = InitEqIntrvl(attr_name,StringKind);
    CopyString(result->range->low->u.string,key_name);
    return(result);
}

static IntrvlList *FillOneItem(i,one_item)
  Intrvl *i;
  IntrvlList *one_item;
{
    if (i != (Intrvl *)NULL) {
	one_item->next = (IntrvlList *)NULL;
	one_item->i = i;
	return(one_item);
    }
    return((IntrvlList *)NULL);
}

static Boolean InRangeList(rng,attr)
  IntrvlRange *rng;
  String attr;			/* (not necessarily in table) */
/* Return True iff the attribute named 'attr' is in either Value list
   'rng->low' or 'rng->high' associated with a variable.
*/
{
    register Value *curr;
    StepLinkedList(curr,rng->low) {
	if (SameString(attr,curr->u.v_bnd->attr_name)) { return(True); }
    }
    StepLinkedList(curr,rng->high) {
	if (SameString(attr,curr->u.v_bnd->attr_name)) { return(True); }
    }
    return(False);
}

static int EqCnt(b,var)
  Box *b;
  Var *var;
/* Returns the number of attributes compared to the variable 'var' for
   equality in the box 'b'.
*/
{
    IntrvlList *i_list;

    StepLinkedList(i_list,b->u1.intvls) {
	Intrvl *i = i_list->i;
	if (i->kind == VarKind && i->name == var->name) {
	    VarCnt cnt;
	    (void)CountVarRange(i->range,&cnt,(IdHashTable)NULL,False);
	    return(cnt.eq_cnt);
	}
    }
    return(0);
}

static Intrvl *IntrvlInBox(b,attr)
  Box *b;
  String attr;			/* (not necessarily in table) */
/* Return a pointer to the Intrvl in 'b' with an attribute named 'attr'. If
   there is no such interval, check to see if there is a variable that
   interval associated with 'b' that contains 'attr' in its high or low list.
   If not, we return NULL. If so, we have to decide if the variable is unbound 
   or bound when this box is processed. In the former case, we return NULL; in
   the latter case, we return the special interval NoIntrvl.

   We now describe what it means for a variable to "bound" or "unbound" when a
   box is processed. If the variable is not in the 'first_vars' list of the
   box, then the variable is bound (since it is first bound at some other
   box), and we return NoIntrvl. Otherwise, it is unbound, and we return NULL.
*/
{
    IntrvlList *curr;

    /* look for a match on the attribute name directly */
    StepLinkedList(curr,b->u1.intvls) {
	Intrvl *i = curr->i;
	if (i->kind != VarKind && SameString(i->name,attr)) {
	    return(curr->i);
	}
    }

    /* look for a match on the attribute inside a variable interval */
    StepLinkedList(curr,b->u1.intvls) {
	Intrvl *i = curr->i;
	if (i->kind == VarKind && InRangeList(i->range,attr)) {
	    Var *var;
	    if ((var=SearchVarList(b->first_vars,i->name)) == NULL) {
		return(NoIntrvl);
	    } else {
		/* require that no var compared to > 1 attr in same box */
		Assert(EqCnt(b,var) == 1);
	    }
	}
    }
    return((Intrvl *)NULL);
}

static IntrvlList *NewBoxTypeIntrvlList(bt)
  BoxType *bt;
{
    IntrvlList *result = AllocOne(IntrvlList);
    result->next = (IntrvlList *)NULL;
    result->i = InitEqIntrvl(TYPE_ATTR,BoxTypeKind);
    result->i->range->low->u.box_type = bt;
    return(result);
}

static void IntrvlsFromSubtree(attr_name,bt,list_ptr_ptr)
  String attr_name;
  BoxType *bt;
  INOUT IntrvlList **list_ptr_ptr;
{
    BoxTypeList *child;
    IntrvlList *cell;

    /* scan children recursively */
    StepLinkedList(child,bt->children) {
	IntrvlsFromSubtree(attr_name,child->bt,list_ptr_ptr);
    }

    /* prepend interval for this BoxType */
    cell = NewBoxTypeIntrvlList(bt);
    SpliceIntoList(*list_ptr_ptr,cell);
}

static IntrvlList *TypeRangeToList(i)
  Intrvl *i;
/* Given an Intrvl 'i' containing a range of box-types, this routine creates
   and returns a *list* of intervals, each of which is an EqIntrvl(), such
   the list contains one interval for each type in the range specified by 'i'.
   The space for each new interval and the list cells themselves is allocated
   separately by this routine. It will be subsequently freed by
   FreeIntrvlList() below.

   Pre-Condition: Assumes 'i' is in canonical form.
*/
{
    IntrvlList *result = NULL;
    IntrvlList *cell;
    BoxType *curr,*high_type;
    Value *high = i->range->high;

    if (i->range->low != NULL) {
	curr = i->range->low->u.box_type;
	high_type = (high != NULL) ? high->u.box_type : (BoxType *)NULL;
	while (curr != high_type) {
	    cell = NewBoxTypeIntrvlList(curr);
	    SpliceIntoList(result,cell);
	}
    } else {
	BoxTypeList *child;
	Assert(high != NULL);
	/*
	 * check if this is a trivial upper bound by SUBJ_NAME or OBJ_NAME; if
	 * so, then return NULL, since the "side" will already have taken this
	 * "type" interval into account */
	if (high->kind == StrictComp &&
	    (SameString(high->u.box_type->name,SUBJ_NAME)
	     || SameString(high->u.box_type->name,OBJ_NAME))) {
	    return((IntrvlList *)NULL);
	}
	/*
	 * otherwise, generate IntrvlList values for children of this type */
	StepLinkedList(child,high->u.box_type->children) {
	    IntrvlsFromSubtree(i->name,child->bt,&result);
	}
    }
    if (high && high->kind == EqComp) {
	cell = NewBoxTypeIntrvlList(high->u.box_type);
	SpliceIntoList(result,cell);
    }
    return(result);
}

static Intrvl *RelevantBoxSideIntrvl(b)
  Box *b;
{
    Intrvl *type_i;
    Value *val;

    if ((type_i=IntrvlInBox(b,TYPE_ATTR)) != NULL
	&& type_i != NoIntrvl) { /* in case 'type' compared to variable */
	val = type_i->range->high ? type_i->range->high : type_i->range->low;
	switch (val->u.box_type->class) {
	  case Subject: return(SubjIntrvl);
	  case Object:  return(ObjIntrvl);
	}
    }
    return((Intrvl *)NULL);
}

static Intrvl *RelevantBoxSysnameIntrvl(b_elt)
  BoxElt *b_elt;
/* There is an indirect "sysname" interval relevant to 'b_elt' if there is
   some arrow incident on 'b_elt' whose order number is less than that of
   'b_elt'. If there is such an arrow, we return the special interval
   NoIntrvl; otherwise NULL.
*/
{
    EltList *curr;

    StepLinkedList(curr,b_elt->adj_elts) {
	if (curr->elt->order < b_elt->order) {
	    return(NoIntrvl);
	}
    }
    return((Intrvl *)NULL);
}

static IntrvlList *RelevantBoxIntrvl(b_elt,attr,rng,one_item)
  BoxElt *b_elt;
  String attr;
  OUT Boolean *rng;
  IntrvlList *one_item;
/* Look for an interval in 'b_elt' involving an attribute named 'attr'. If no
   interval is found, there are two special cases to check: the cases when
   'attr' is either "side" or "sysname". We must do special computations in
   these cases to determine which interval to return. The 'rng' value is set
   by taking the complement of the EqIntrvl() predicate.
*/
{
    Box *b = b_elt->u.b;
    Intrvl *result;

    *rng = False;
    if ((result=IntrvlInBox(b,attr)) != NULL) {
	if (SameString(attr,TYPE_ATTR)) {
	    return(TypeRangeToList(result));
	}
	*rng = NotOf(result==NoIntrvl || EqIntrvl(result));
    } else {
	if (SameString(attr,SIDE_ATTR)) {
	    result = RelevantBoxSideIntrvl(b);
	} else if (SameString(attr,SYSNAME_ATTR)) {
	    result = RelevantBoxSysnameIntrvl(b_elt);
	}
    }
    return(FillOneItem(result,one_item));
}

Intrvl *ArrowBoxIntrvl(a_elt,b_elt)
  Elt *a_elt,*b_elt;
{
    Elt *min_elt = MinBoxAdjOrder(b_elt);
    return((min_elt->order < a_elt->order) ? NoIntrvl : NULL);
}

static IntrvlList *NewArrowPermIntrvlList(perm_name)
  String perm_name;
{
    IntrvlList *result = AllocOne(IntrvlList);
    result->next = (IntrvlList *)NULL;
    result->i = InitEqStringIntrvl(PERM_ATTR,perm_name);
    return(result);
}

static IntrvlList *RelevantArrowIntrvl(a_elt,name,rng,one_item)
  ArrowElt *a_elt;
  String name;
  OUT Boolean *rng;
  IntrvlList *one_item;
{
    Intrvl *result;
    Arrow *a = a_elt->u.a;

    *rng = False;		/* no arrow attribute can become a range */

    /* handle attributes common to all arrows */
    if (SameString(name,FROM_ATTR)) {
	result = ArrowBoxIntrvl(a_elt,a->from);
    } else if (SameString(name,TO_ATTR)) {
	result = ArrowBoxIntrvl(a_elt,a->to);
    } else if (SameString(name,SYSNAME_ATTR)) {
	result = ArrowBoxIntrvl(a_elt,a->to);
    } else if (SameString(name,PARITY_ATTR)) {
	result = (a->parity == Pos) ? PosIntrvl : NegIntrvl;
    } else if (SameString(name,PERM_ATTR)) {
	IntrvlList *cell;
	IntrvlList *head = NULL;
	PermList *curr;
	StepLinkedList(curr,a->u.perm_list) {
	    cell = NewArrowPermIntrvlList(curr->perm);
	    SpliceIntoList(head,cell);
	}
	return(head);
    } else if (SameString(name,DIRECT_ATTR)) {
	result = (a->u.starred) ? (Intrvl *)NULL : TIntrvl;
    }
    return(FillOneItem(result,one_item));
}

static Boolean OppSides(b1,b2)
  Box *b1,*b2;
/* Returns True iff boxes 'b1' and 'b2' must be on opposite sides.
*/
{
    return(MakeBoolean(((int)(b1->side)|(int)(b2->side))==(int)BothSides));
}

static Boolean Contains(b1,b2,pict)
  Box *b1,*b2;
  Pict *pict;
/* Returns True iff box 'b2' is contained in 'b1' (by some chain of
   containment arrows in 'pict'), or if the two boxes are the same.
*/
{
    if (b1 == b2) {
	return(True);
    } else {
	ArrowList *curr;
	StepLinkedList(curr,pict->arrows) {
	    Arrow *a = curr->elt->u.a;
	    if (a->kind == Containment && a->from->u.b == b2
		&& Contains(b1,a->to->u.b,pict)) {
		return(True);
	    }
	}
	return(False);
    }
}

static Boolean ContainmentRel(b1,b2,pict)
  Box *b1,*b2;
  Pict *pict;
/* Returns True iff box 'b1' is contained in 'b2' (by some chain of
   containment arrows in 'pict') or vice-versa.
*/
{
    return(MakeBoolean(Contains(b1,b2,pict) || Contains(b2,b1,pict)));
}

static Boolean DisjointBoxes(b1,b2)
  Box *b1,*b2;
/* Returns True iff the intervals for 'b1' and 'b2' are such that no instance
   box can match both. This occurs precisely when both boxes contain intervals
   for the same attribute that are disjoint. For example, boxes having the box
   predicates `name < "cat"' and `name > "dog"' have disjoint intervals for
   the "name" attribute.
*/
{
    IntrvlList *i1,*i2;
    StepLinkedList(i1,b1->u1.intvls) {
	if (i1->i->kind != VarKind) {
	    StepLinkedList(i2,b2->u1.intvls) {
		if (i1->i->kind == i2->i->kind
		    && i1->i->name == i2->i->name
		    && DisjointIntrvls(i1->i,i2->i)) {
		    return(True);
		}
	    }
	}
    }
    return(False);
}

static int ValueListLen(v)
  Value *v;
/* Returns the length of the linked list starting at 'v'.
*/
{
    int result = 0;
    StepInitializedLinkedList(v) { result++; }
    return(result);
}

/* GLOBAL DEBUGGING FUNCTIONS ============================================== */

#ifdef DEBUG

void DisplayAllAdjacencies(pict)
  Pict *pict;
{
    fprintf(stderr,"\nBox Adjacencies:\n");
    DisplayAdjacencies(pict->boxes);
    fprintf(stderr,"\nArrow Adjacencies:\n");
    DisplayAdjacencies(pict->arrows);
}

#endif DEBUG

/* GLOBAL FUNCTIONS ======================================================== */

BoxElt *NewBox()
{
    BoxElt *result;
    Box *b;

    result = (BoxElt *)NewElt();
    result->kind = BoxEltKind;
    b = result->u.b = AllocOne(Box);
    b->kind = PredKind;
    b->u1.pred = (Predicate *)NULL;
    b->starred = False;
    b->side = NeitherSide;
    b->u2.contained = False;
    b->first_vars = (VarList *)NULL;
    return(result);
}

ArrowElt *NewArrow()
{
    ArrowElt *result;
    Arrow *a;

    result = (ArrowElt *)NewElt();
    result->kind = ArrowEltKind;
    a = result->u.a = AllocOne(Arrow);
    a->from = a->to = (BoxElt *)NULL;
    return(result);
}

void FreeElt(elt)
  Elt *elt;
{
    switch (elt->kind) {
      case BoxEltKind: FreeBox(elt->u.b); break;
      case ArrowEltKind: FreeArrow(elt->u.a); break;
    }
    Dealloc(elt);
}

void FreeEltList(l)
  EltList *l;
{
    EltList *temp;

    while (l) {
	temp = l;
	l = l->next;
	FreeElt(temp->elt);
	Dealloc(temp);
    }
}

Pict *NewPict()
{
    int i;
    Pict *result;
    result = AllocOne(Pict);
    result->elt_cnt = 0;
    result->boxes = (BoxList *)NULL;
    result->arrows = (ArrowList *)NULL;
    result->elt = (Elt **)NULL;
    result->range.low = 1;
    result->range.high = INFINITY;
    result->box_types = (BoxTypeList *)NULL;
    result->var_head = (VarList *)NULL;
    result->table = NewIdHashTable();
    StepIndex(i,0,NUM_ELT_KINDS) {
	result->elts[i] = (EltList *)NULL;
	result->attrs[i] = (HdsHintList *)NULL;
	result->hds[i] = (Hds *)NULL;
    }
    return(result);
}

Elt *MinBoxAdjOrder(b_elt)
  BoxElt *b_elt;
{
    EltList *adj;
    Elt *result = b_elt;
    int min = b_elt->order;

    StepLinkedList(adj,b_elt->adj_elts) {
	if (adj->elt->order < min) {
	    result = adj->elt;
	    min = result->order;
	}
    }
    return(result);
}

void InitRelevantIntrvl(pict)
  Pict *pict;
{
    TableEntry *tbl;

    /* initialize "side" intervals */
    tbl = FindTableId(pict->table,AttrNameId,SIDE_ATTR);
    Assert(tbl != NULL);
    SubjIntrvl = InitEqStringIntrvl(tbl->name,SUBJ_KEY);
    ObjIntrvl = InitEqStringIntrvl(tbl->name,OBJ_KEY);

    /* initialize "parity" intervals */
    tbl = FindTableId(pict->table,AttrNameId,PARITY_ATTR);
    Assert(tbl != NULL);
    PosIntrvl = InitEqStringIntrvl(tbl->name,TRUE_KEY);
    NegIntrvl = InitEqStringIntrvl(tbl->name,FALSE_KEY);

    /* initialize "direct" intervals */
    tbl = FindTableId(pict->table,AttrNameId,DIRECT_ATTR);
    Assert(tbl != NULL);
    TIntrvl = InitEqStringIntrvl(tbl->name,TRUE_KEY);
    NilIntrvl = InitEqStringIntrvl(tbl->name,FALSE_KEY);
}

IntrvlList *RelevantIntrvl(elt,name,rng,one_item)
  Elt *elt;
  String name;
  OUT Boolean *rng;
  IntrvlList *one_item;
{
    IntrvlList *result;
    switch (elt->kind) {
      case BoxEltKind:
	result = RelevantBoxIntrvl(elt,name,rng,one_item);
	break;
      case ArrowEltKind:
	result = RelevantArrowIntrvl(elt,name,rng,one_item);
	break;
    }
    return(result);
}

void FreeIntrvlList(i_list,one_item)
  IntrvlList *i_list,*one_item;
{
    IntrvlList *temp;
    if (i_list != (IntrvlList *)NULL && i_list != one_item) {
	while (i_list != NULL) {
	    temp = i_list;
	    i_list = i_list->next;
	    FreeIntrvl(temp->i,True);
	    Dealloc(temp);
	}
    }
}

Var *SearchVarList(v_list,var_name)
  VarList *v_list;
  String var_name;		/* (in table) */
{
    StepInitializedLinkedList(v_list) {
	if (v_list->var->name == var_name) { return(v_list->var); }
    }
    return((Var *)NULL);
}

BoxList *PotentialSamePrecursors(b_elt,pict)
  BoxElt *b_elt;
  Pict *pict;
/* IMPLEMENTATION NOTE: There are ways to make this test more sophisticated.
   Currently, we declare a box element 'b_elt_2' as being potentially the same
   as 'b_elt' if it is NOT:
     1) on the opposite side of 'b_elt',
     2) it is contained in 'b_elt' or vice-versa, OR
     3) the two box elements cannot be matched by the same instance box due to
        the fact that they have intervals for the same attribute name that are
        disjoint. 
   We could also rule out two boxes being potentially the same if there is
   some box they are box contained in (or that they both contain) by direct
   containment arrows only, yet the number of such direct containment arrows
   from each to the commonly containing (contained) box is not equal.
*/
{
    Box *b1 = b_elt->u.b;
    BoxList *result = (BoxList *)NULL,*temp;
    register int i;

    StepIndex(i,0,b_elt->order) {
	Elt *elt = pict->elt[i];
	if (elt->kind == BoxEltKind) {
	    Box *b2 = elt->u.b;
	    if (!(OppSides(b1,b2)
		  || ContainmentRel(b1,b2,pict)
		  || DisjointBoxes(b1,b2))) {
		temp = AllocOne(BoxList);
		temp->elt = elt;
		SpliceIntoList(result,temp);
	    }
	}
    }
    return(result);
}

void FreeBoxListSkeleton(bl)
  BoxList *bl;
{
    register BoxList *temp;
    while (bl) { temp = bl; bl = bl->next; Dealloc(temp); }
}

double CountVarRange(r,cnt,table,eql_yet)
  IntrvlRange *r;
  OUT VarCnt *cnt;
  IdHashTable table;
  Boolean eql_yet;		/* is there an equality yet? */
{
    Value *v_low,*v_high;
    double result = 1.0;

    /* count number of low and high values */
    cnt->low_cnt = ValueListLen(r->low);
    cnt->high_cnt = ValueListLen(r->high);

    /* count # of equalities, increment cnt->eq_cnt as each is found */
    cnt->eq_cnt = 0;
    StepLinkedList(v_low,r->low) {
	StepLinkedList(v_high,r->high) {
	    String low_name = v_low->u.v_bnd->attr_name;
	    if (low_name == v_high->u.v_bnd->attr_name) {
		if (table != NULL) {
		    if (eql_yet) {
			TableEntry *tbl;
			tbl = FindTableId(table,AttrNameId,low_name);
			Assert(tbl != NULL && tbl->kind == AttrNameId);
			result *= 1.0/(double)(tbl->u.attr->range_size);
		    } else {
			eql_yet = True;
		    }
		}
		cnt->low_cnt--; cnt->high_cnt--;
		cnt->eq_cnt++;
	    }
	}
    }
    return(result);
}
