
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* INTERVAL.H

   Type definitions and function declarations for the interval.c module, which
   provides facilities for creating and manipulating value intervals.

   $Header: interval.h,v 1.2 91/11/13 03:10:14 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   There is an interval associated with each:
     1) attribute in a single box predicate
     2) variable in a constraint picture

   The canonical forms for each IntrvlKind are:

   IntKind:
     Both upper and lower bounds are EqComp's (add or subtract 1 as
     necessary).
   BoxTypeKind:
     The lower bound is an EqComp (take parent type as necessary). If both the
     lower and upper types are non-NULL, then the upper bound is an EqComp as
     well (take child along path from upper to lower value).
   BoolKind:
     Both upper and lower bounds are EqComp's, and 'low' and 'high' are always
     the same Value.
*/

#ifndef INTERVAL_H
#define INTERVAL_H

#include <my-types.h>

/* BoxType in Value type */
#include "box-type.h"

/* MACRO DEFINITIONS ======================================================= */

/* value-less interval */
#define NoIntrvl ((Intrvl *)0x4)

/* TYPE DEFINITIONS ======================================================== */

typedef enum {
    StrictComp, EqComp		/* strict vs non-strict comparison */
} CompKind;

typedef struct var_bound {
    String attr_name;		/* (in table) */
    String prefix,suffix;
} VarBound;

typedef union val {
    int int_val;		/* kind == IntKind */
    String string;		/* kind == StringKind (*not* in table) */
    Boolean bool;		/* kind == BoolKind */
    BoxType *box_type;		/* kind == BoxTypeKind (in table)*/
    VarBound *v_bnd;		/* kind == VarKind */
} Val;

typedef struct value {
    struct value *next;		/* next value if kind==VarKind */
    CompKind kind;		/* strict inequality comparision? */
    Val u;			/* actual value */
} Value;

typedef enum {
    IntKind, StringKind, BoolKind, BoxTypeKind, VarKind
} IntrvlKind;

typedef struct intrvl_range {
    Value *low,*high;		/* interval [low,high] */
} IntrvlRange;

typedef struct intrvl {
    String name;		/* name of variable or attribute (in table) */
    IntrvlKind kind;		/* kind of the 'low' and 'high' values */
    IntrvlRange *range;		/* interval for this variable or attribute */
} Intrvl;

typedef struct intrvl_list {
    struct intrvl_list *next;
    Intrvl *i;
} IntrvlList;

/* FUNCTION DECLARATIONS =================================================== */

void FreeIntrvl( /* Intrvl *i, Boolean free_values */ );
/* Deallocate the space allocated to 'i'. If 'free_values' is True, then free
   the 'low' and 'high' Values of 'i'.
*/

Boolean FormAllIntrvls( /* Pict *pict */ );
/* Set the 'u1.intvls' field of each Box. This field is computed from the
   'u1.pred' field, which is freed. The 'kind' field is set to IntvlKind to
   indicate the change. If there are any errors found due to empty intervals
   within a Box (e.g., specifying "sysname < 2 && sysname > 5"), then an error
   is reported to stderr.

   Returns True iff any errors are found.
*/

void DisplayIntrvl( /* Intrvl *i */ );
/* Write a description of the interval 'i' to stderr. No spaces and/or
   newlines either preceed or follow the interval output.
*/

Boolean EqIntrvl( /* Intrvl *intrvl */ );
/* Returns True iff 'intrvl' is an Intrvl containing equal low and high
   values, thereby representing a *single* value.
*/

Boolean SameVal( /* IntrvlKind kind, Val v1, Val v2 */ );
/* Returns True iff 'v1' and 'v2', when consider to be Val's of kind 'kind',
   are equal.
*/

void CopyVal( /* IntrvlKind kind, Val *v1, Val v2 */ );
/* Copies the value 'v2' into '*v1'. Value 'v2' should be of kind 'kind'.
*/

String ValName( /* IntrvlKind kind, PrimValueType type, Val val */ );
/* Returns the "name" of the single value contained in 'val' of kind 'kind'.
   If 'kind' is StringKind, then 'type' is used to determine if the underlying
   string should be quoted or not. In the case of integer, string, and boolean
   values, a short-lived string representation is returned.
*/

Boolean InsideIntrvl( /* Slot *slot, Intrvl *intrvl */ );
/* Returns True if 'slot' and 'intrvl' have the same 'kind' *and* the value
   'slot->key' is in the interval 'intrvl'. Also returns True if 'intrvl' is
   NULL OR 'slot->key' is the special value 'Others' and 'intrvl' is the
   special value NoIntrvl.
*/

Boolean DisjointIntrvls( /* Intrvl *i1, Intrvl *i2 */ );
/* Returns True iff the intervals 'i1' and 'i2' are disjoint.

   PRE-CONDITION: 'i1' and 'i2' must both have the same 'kind', and neither of
   them should be of kind VarKind.
*/

#endif INTERVAL_H
