/* HDS.C

   Module for constructing representations of the IPQL's HDS structures.

   $Header: hds.c,v 1.12 91/11/14 22:08:20 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <strings.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "attr.h"
#include "hints.h"
#include "hds.h"
#include "id-table.h"
#include "objs.h"

/* MACRO DEFINITIONS ======================================================= */

/* maximum length of an iterator path */
#define MAX_PATH_LEN 100

/* amount to tab on each element */
#define TAB 3

/* extra amount to tab for each construct */
#define LET_TAB       6
#define LET_STAR_TAB  7
#define WHEN_AND_TAB 11

/* HDS/Discriminator Tree variable suffix */
#define DT_SUFFIX "-dt"

/* LOCAL VARIABLES ========================================================= */

/* names of each kind of graphical element */
static String EltName[] = { "box", "syn", "con", "obj", "subj" };

/* names of each kind of HDS structure (indexed by HdsKind) */
static String HdsName[] = { "bst", "xht", "ht" };

/* names of each thickness */
static String ThickName[] = { "thick", "thin" };

/* character buffer for paths */
static char path_buff[MAX_PATH_LEN];

/* LOCAL FUNCTIONS ========================================================= */

Hds *NewHds(hint)
  HdsHint *hint;
/* Create and return a pointer to a new Hds. All fields of the new HDS are
   initialized appropriately. The 'hint' and 'kind' fields are those
   associated with the Hint structure 'hint'.
*/
{
    Hds *result;

    result = AllocOne(Hds);
    result->hint = hint;
    result->kind = hint->kind;
    result->slots = (Slot *)NULL;
    return(result);
}

static int EltIndex(elt)
  Elt *elt;
/* Return BOX, SYN, SEM, or CON according to 'elt'.
*/
{
    int result;
    switch (elt->kind) {
      case BoxEltKind: result = BOX; break;
      case ArrowEltKind:
	switch (elt->u.a->kind) {
	  case Syntax: result = SYN; break;
	  case Semantics: result = SEM; break;
	  case Containment: result = CON; break;
	}
	break;
    }
    return(result);
}

static void InitElts(pict)
  INOUT Pict *pict;
/* Initialize the 'pict->elts[]' arrays to point to BOX, SYN, SEM, and CON
   elements in their rank order (i.e., the order in which they appear in the
   array 'pict->elt[]').
*/
{
    int num;			/* number (index) of current element */
    EltList *curr;
    int kind;

    StepIndexBackwards(num,0,pict->elt_cnt) {
	curr = AllocOne(EltList);
	curr->elt = pict->elt[num];
	kind = EltIndex(curr->elt);
	SpliceIntoList(pict->elts[kind],curr);
    }
}

#ifdef MACRO_FUNCTION

static int HdsToElt( /* hds_index */ );
/* Given the 'hds_index' BOX, SYN, CON, SUBJ, or OBJ this function returns the
   index of the element kind for that HDS type, namely, BOX, SYN, CON, or SEM.
*/
#endif MACRO_FUNCTION

#define HdsToElt(_hds_index)\
  (((_hds_index)==SUBJ || (_hds_index)==OBJ) ? SEM : (_hds_index))

/* ----------------------------------------------------------------------------
 *
 * NOTE: Given an IntrvlList 'intrvls' and a corresponding range indicator
 * 'is_rng', the rules for matching that IntrvlList with a slot are as
 * follows:
 *
 * 1) (intrvls==NULL || intrvls->i==NoIntrvl) => match *any* slot
 * 2) Otherwise, match the slot if at least *one* of the Intrvl values in
 *    'intrvls' contains the slot value.
 * 3) (is_rng) => additionally match ":OTHERS" slot
 *
 * The function IsOthersRelevant() checks if conditions (1) or (3) are
 * satisfied to decide if an ":OTHERS" slot should be added. In this case,
 * there are two kinds of ":OTHERS" slots to add: either a true ":OTHERS"
 * slot, or a ":NEW-SLOT" slot. The rules for determining which slot to add
 * are:
 *
 * a) (intrvls==NULL) => add ":OTHERS" slot
 * b) (intrvls->i==NoIntrvl || is_rng) => add ":NEW-SLOT" slot
 *
 * In case (a), there is no reason to create a separate slot for each added
 * key value, since we will never be supplying a key to distinguish in this
 * case. In case (b), however, we will either be selecting on a particular key
 * (in the intrvls->i==NoIntrvl case; e.g., to select a particular sysname),
 * or we will be selecting on a range of keys (in the is_rng==True case). Once
 * a slot has been created, future uses of that slot should check to see if
 * the slot should be "upgraded" from a case-(a) slot to a case-(b) slot.
 * These upgraded work because case-(b) slots can be used even for elements
 * falling to case-(a), but not the other way around. The iteration is
 * slightly less efficient when a case-(b) iterator is used in case-(a),
 * however.
 *
 * The function MatchesSlot() returns a Boolean as conditions (1), (2), or (3)
 * is satisfied.
 *
 * ------------------------------------------------------------------------- */

static Boolean IsOthersRelevant(hds,intrvls,is_rng)
  Hds *hds;
  IntrvlList *intrvls;
  Boolean is_rng;
/* Returns True iff 'intrvls' and 'is_rng' indicate that an ":OTHERS" slot is
   necessary. If so, it also has the SIDE-EFFECT of adding an ":OTHERS" slot
   to 'hds->slots' if none exists.
*/
{
    /* check for condition (1), (2), or (3) */
    if (intrvls==NULL || intrvls->i==NoIntrvl || is_rng) {
	/*
	 * find :OTHERS slot */
	Slot *slot;
	StepLinkedList(slot,hds->slots) {
	    if (slot->key == Others) { break; }
	}
	/*
	 * install new :OTHERS slot if none exists */
	if (slot == (Slot *)NULL) {
	    slot = AllocOne(Slot);
	    slot->key = Others;	/* NOTE: no 'kind' value is installed */
	    slot->hds = (Hds *)NULL;
	    slot->new_slot = NotOf(intrvls==NULL);
	    SpliceIntoList(hds->slots,slot);
	}
	/*
	 * promote 'hds' to a BST-HDS if necessary */
	if (is_rng) { hds->kind = BstHds; }
	return(True);
    } else {
	return(False);
    }
}

static void ExtendLeafHds(hds,intrvls,is_rng)
  Hds *hds;
  IntrvlList *intrvls;
  Boolean is_rng;
/* Extends the HDS 'hds', which is an HDS keying off the attribute applicable
   to 'intrvls'. If 'intrvls' and 'is_rng' so indicate, an Others slot is
   added. Moreover, if 'is_rng' is True, then 'hds' is "upgraded" to a BST-HDS.
*/
{
    /* test if this is a specific value; if not, this test may still have the
       side-effect of installing an Others slot */
    if (!IsOthersRelevant(hds,intrvls,is_rng)) {
	/*
	 * install slot according to (single value in) 'intrvl' */
	IntrvlList *curr;
	StepLinkedList(curr,intrvls) {
	    Intrvl *intrvl = curr->i;
	    IntrvlKind kind = intrvl->kind;
	    Val key = intrvl->range->low->u;
	    Slot *slot;
	    /*
	     * see if a key for this intrvl already exists */
	    StepLinkedList(slot,hds->slots) {
		/* skip ":OTHERS" slot */
		if (slot->key != Others && SameVal(kind,*slot->key,key)) {
		    break;
		}
	    }
	    /*
	     * if not, install new slot for 'key' */
	    if (slot == (Slot *)NULL) {
		slot = AllocOne(Slot);
		slot->kind = kind;
		slot->key = AllocOne(Val);
		CopyVal(kind,slot->key,key);
		slot->hds = (Hds *)NULL;
		SpliceIntoList(hds->slots,slot);
	    }
	}
    }
}

static Boolean MatchesSlot(slot,intrvls,is_rng)
  Slot *slot;
  IntrvlList *intrvls;
  Boolean is_rng;
/* Returns True iff the Slot 'slot' is in at least one of the interrvals of
   the list 'intrvls' having the corresponding range indicator 'is_rng' (see
   rules for matching above).
*/
{
    if (intrvls==NULL || intrvls->i==NoIntrvl
	|| (is_rng && slot->key==Others)) {
	/*
	 * upgrade OTHERS slot to NEW-KEY if necessary */
	if (slot->key==Others && (intrvls!=NULL || is_rng)) {
	    slot->new_slot = True;
	}
	return(True);
    } else {
	/* this slot is relevant if it is inside at least one Intrvl */
	IntrvlList *curr;
	StepLinkedList(curr,intrvls) {
	    if (InsideIntrvl(slot,curr->i)) { break; }
	}
	return(MakeBoolean(curr != NULL));
    }
}

static Boolean MatchesSlot2(slot,intrvls,is_rng)
  Slot *slot;
  IntrvlList *intrvls;
  Boolean is_rng;
/* Like MatchesSlot(), only this predicate is used when deciding what to
   iterate over. The only difference is that we do *not* iterate over items in
   'slot' if 'is_rng' is True *unless* the 'slot' is Others (even in the case
   where the 'slot' value is inside one of the intervals in 'intrlvs'
*/
{
    if (is_rng && slot->key != Others) {
	return(False);
    } else {
	return(MatchesSlot(slot,intrvls,is_rng));
    }
}

static void ExtendThroughHds(elt,hint,intrvls,is_rng,hds_ptr)
  Elt *elt;
  HdsHint *hint;
  IntrvlList *intrvls;
  Boolean is_rng;
  INOUT Hds **hds_ptr;
{
    Hds *hds = *hds_ptr;
    Boolean is_rng2;
    IntrvlList *intrvls2,one_item2; /* list of intervals for *this* HDS */
    Slot *slot;
    void ExtendHdsRecursively();

    /* determine relevant intervals of 'elt' in *this* HDS */
    intrvls2 = RelevantIntrvl(elt,hds->hint->name,&is_rng2,&one_item2);

    /* install "OTHERS" slot if necessary and none exists */
    (void)IsOthersRelevant(hds,intrvls2,is_rng2);

    /* recursively search relevant slots */
    StepLinkedList(slot,hds->slots) {
	if (MatchesSlot(slot,intrvls2,is_rng2)) {
	    ExtendHdsRecursively(elt,hint,intrvls,is_rng,&(slot->hds));
	}
    }

    /* free space used by 'intrvls2' (if necessary) */
    FreeIntrvlList(intrvls2,&one_item2);
}

static void ExtendHdsRecursively(elt,hint,intrvls,is_rng,hds_ptr)
  Elt *elt;
  HdsHint *hint;
  IntrvlList *intrvls;
  Boolean is_rng;
  INOUT Hds **hds_ptr;
{
    /* don't progress with empty list unless an HDS already exists */
    if (*hds_ptr == (Hds *)NULL && intrvls == (IntrvlList *)NULL) { return; }

    /* determine if we are at the end of a branch or at an internal node */
    if (*hds_ptr == (Hds *)NULL	       /* at a leaf... */
	|| (*hds_ptr)->hint == hint) { /* ...or we are at a matching node */
	/* create a new HDS if necessary */
	if  (*hds_ptr == (Hds *)NULL) { *hds_ptr = NewHds(hint); }
	ExtendLeafHds(*hds_ptr,intrvls,is_rng);
    } else {
	ExtendThroughHds(elt,hint,intrvls,is_rng,hds_ptr);
    }
}

static void ExtendHds(elt,hint,hds_ptr)
  Elt *elt;
  HdsHint *hint;
  INOUT Hds **hds_ptr;
/* Determine if the intervals of 'elt' have any bearing on the attribute named
   'hint->name'. If so, then attempt to "extend" *all* pertinent branches of
   the HDS rooted at '*hds_ptr'.

   The leaves of this HDS may already contain HDS structures for this
   attribute. If so, simply add relevant slots; if not, create the necessary
   HDS and add relevant slots. In either case, if the interval of 'elt'
   requires iteration over a *range* of values, promote the corresponding Hds
   structures to be BST-HDS's, and make sure that the Hds contains an
   ":OTHERS" slot.
*/
{
    IntrvlList *intrvls,one_item;
    Boolean is_rng;		/* is this interval over a range of values? */

    /* determine relevant intervals for this attribute */
    intrvls = RelevantIntrvl(elt,hint->name,&is_rng,&one_item);

    /* recursively extend the hds with the relevant interval 'intrvls' */
    ExtendHdsRecursively(elt,hint,intrvls,is_rng,hds_ptr);

    /* if 'intrvls' is a list of intervals with size > 1, the space for that
       list was allocated dynamically, so free it */
    FreeIntrvlList(intrvls,&one_item);
}

static Boolean UnnecessarySide(hds,sub_hds)
  Hds *hds;
  OUT Hds **sub_hds;
/* Returns True iff 'hds' keys on SIDE_ATTR with a single slot,
   containing an hds that keys on TYPE_ATTR with *no* OTHERS slot. In this
   case, '*sub_hds' is made to point to the "TYPE_ATTR" hds.
*/
{
    if (SameString(hds->hint->name,SIDE_ATTR)) {
	Slot *single,*slot;
	/*
	 * check this hds has exactly one slot */
	single = hds->slots;
	if (single == NULL || single->next != NULL) { return(False); }
	/*
	 * check that the sub-hds keys off "type" */
	*sub_hds = single->hds;
	if (*sub_hds && SameString((*sub_hds)->hint->name,TYPE_ATTR)) {
	    /*
	     * make sure it has no ":OTHERS" slot */
	    StepLinkedList(slot,(*sub_hds)->slots) {
		if (slot->key == Others) { break; }
	    }
	    if (slot == NULL) {	/* no :OTHERS slot */
		return(True);
	    }
	}
    }
    return(False);
}

#ifdef OBSOLETE
static Boolean ArrowCross(hds,sub_hds)
  Hds *hds;
  OUT Hds **sub_hds;
/* Returns True iff 'hds' keys on FROM_ATTR with a single OTHERS slot,
   containing an hds that keys on TO_ATTR with a single OTHERS slot. In this
   case, '*sub_hds' is made to point to the "TO_ATTR" hds, and that HDS's
   'hint' field to the special HdsHint value 'FROM_TO_CROSS'.
*/
{
    if (SameString(hds->hint->name,FROM_ATTR)) {
	Slot *single1 = hds->slots;
	/*
	 * check this hds has exactly one OTHERS slot */
	if (single1==NULL || single1->next!=NULL || single1->key!=Others) {
	    return(False);
	}
	/*
	 * check that the sub-hds keys off "to" */
	*sub_hds = single1->hds;
	if (*sub_hds && SameString((*sub_hds)->hint->name,TO_ATTR)) {
	    Slot *single2 = (*sub_hds)->slots;  
	    /*
	     * check this hds has exactly one OTHERS slot */
	    if (single2==NULL || single2->next!=NULL || single2->key!=Others) {
		return(False);
	    } else {
		(*sub_hds)->hint = FROM_TO_CROSS;
		return(True);
	    }
	}
    }
    return(False);
}
#endif OBSOLETE

static void OptimizeHds(hds_ptr)
  INOUT Hds **hds_ptr;
/* For now, the only optimization this makes is to eliminate any Hds keying
   off the "side" attribute that contains only one slot pointing to a Hds
   keying off the "type" attribute that does not contain an ":OTHERS" slot.

   A future optimization would be to merge a "from" Hds with an ":OTHERS" slot
   only followed by a "to" Hds with an ":OTHERS" slot only.
*/
{
    Hds *hds = *hds_ptr;
    Hds *sub_hds;
    Slot *slot;

    if (hds == NULL) {		/* base case */
	return;
    } else if (UnnecessarySide(hds,&sub_hds)) {
	/*
	 * change orig hds to skip "side" hds */
	*hds_ptr = sub_hds;
	/*
	 * Free space used by the hds, its slot, and its slot's key */
	if (hds->slots->key != Others) { Dealloc(hds->slots->key); }
	Dealloc(hds->slots);
	Dealloc(hds);
    } else {
	/*
	 * optimize recursively for each slot */
	StepLinkedList(slot,hds->slots) {
	    OptimizeHds(&(slot->hds));
	}
    }
}

static void GenerateHds(fp,hds_num,hds,path)
  FILE *fp;
  int hds_num;
  Hds *hds;
  String path;
/* Write IPQL commands to 'fp' to construct the Hds structure 'hds' for the
   element kind 'hds_num' (e.g., BOX, SYN, etc.). 'path' contains the current
   path of key values in the recursion.
*/
{
    String arg;			/* argument to use in creation of HDS */
    String end;			/* points to current end of string */
    Slot *curr;
    HdsHint *hint;

    /* generate constructors for this level */
    fprintf(fp,"(extend *%s%s* '(%s) ",EltName[hds_num],DT_SUFFIX,path);
    if (hds == NULL) {
	/*
	 * base case: generate a bag-hds */
	fputs("(make-bag))\n",fp);
    } else {
	/*
	 * valid non-leaf-hds: write it and recurse */
	hint = hds->hint;
	fprintf(fp,"(make-%s ",HdsName[(int)hds->kind]);
	fprintf(fp,"#'%s-%s",EltName[hds_num],hint->name);
	arg = (hds->kind == hint->kind) ? hint->arg : hint->bst_arg;
	if (*arg != '\0') { fprintf(fp," %s",arg); } /* non-empty arg */
	fputs("))\n",fp);
	/*
	 * recursively generate constructors for the slots */
	end = path + strlen(path);
	Assert(hds->slots != NULL);
	StepLinkedList(curr,hds->slots) {
	    /*
	     * append separating space if not first key name */
	    strcpy(end,((end==path) ? "" : " "));
	    /*
	     * append key name to path */
	    if (curr->key == Others) {
		/* append ":OTHERS" or ":NEW-SLOT" to 'path' */
		strcat(end,curr->new_slot ? NEW_SLOT_KEY : OTHERS_KEY);
	    } else {
		/*
		 * append key value (as a string), possibly in quotes */
		strcat(end,ValName(curr->kind,hint->type,*curr->key));
	    }
	    GenerateHds(fp,hds_num,curr->hds,path);
	}
    }
}

static void Indent(tab,fp)
  int tab;
  FILE *fp;
{
    register int i;
    StepIndex(i,0,tab) { fputc(' ',fp); }
}

static void IndentTab(tab,fp)
  int tab;
  FILE *fp;
{
    fputc('\n',fp);
    Indent(tab,fp);
}

static Boolean AllSubtreesRelevant(elt,hds)
  Elt *elt;
  Hds *hds;
/* Returns True iff 'elt' could be classified in *every* subtree of 'hds'.
   This is trivially true if 'hds' == NULL (i.e., we are at a slot with a NULL
   Hds pointer, indicating a LIST-HDS). The purpose of this routine is to
   decide if we know for certain that all elements below the hds-sub-tree
   rooted at 'hds' will be accessed so we can cut the iterator path short.
*/
{
    IntrvlList *intrvls,one_item;
    Boolean is_rng;
    Slot *slot;
    Boolean result;

    /* base case */
    if (hds == NULL) { return(True); }

    /* recursively check all slots of this hds */
    intrvls = RelevantIntrvl(elt,hds->hint->name,&is_rng,&one_item);
    if (intrvls != NULL && intrvls->i == NoIntrvl) {
	/* in this case, we need to select a particular value */
	result = False;
    } else {
	Assert(hds->slots != NULL);
	StepLinkedList(slot,hds->slots) {
	    if (is_rng || !MatchesSlot(slot,intrvls,is_rng)
		|| !AllSubtreesRelevant(elt,slot->hds)) {
		break;
	    }
	}
	result = MakeBoolean(slot == NULL);
    }
    FreeIntrvlList(intrvls,&one_item);
    return(result);
}

static void GenValue(v,kind,type,fp)
  Value *v;
  IntrvlKind kind;
  PrimValueType type;
  FILE *fp;
{
    if (v == NULL) {
	fputs(FALSE_KEY,fp);
    } else {
	fputc('(',fp);
	fputs(ValName(kind,type,v->u),fp);
	fputs(" . ",fp);
	fputs((v->kind==EqComp ? TRUE_KEY : FALSE_KEY),fp);
	fputc(')',fp);
    }
}

static void GenRange(rng,kind,type,fp)
  IntrvlRange *rng;
  IntrvlKind kind;
  PrimValueType type;
  FILE *fp;
{
    GenValue(rng->low,kind,type,fp);
    fputs(" . ",fp);
    GenValue(rng->high,kind,type,fp);
}

static void AppendAdjBox(elt,from_flag,force_all,end)
  Elt *elt;
  int from_flag;
  Boolean force_all;		/* force an :ALL keyword? */
  INOUT String end;
{
    Elt *min_elt;		/* adj elt of minimum order */
    BoxElt *b_elt;		/* adj box of interest */

    /* check if arrow end restricted by anchored box */
    Assert(elt->kind == ArrowEltKind);
    b_elt = from_flag ? elt->u.a->from : elt->u.a->to;
    min_elt = MinBoxAdjOrder(b_elt);
    if (min_elt->order < elt->order) {
	if (min_elt->kind == BoxEltKind || b_elt->order < elt->order) {
	    sprintf(end,",e%d-%s",b_elt->order,NAME);
	} else {
	    sprintf(end,",e%d-%s",min_elt->order,
		    (min_elt->u.a->from == b_elt ? TAIL : HEAD));
	}
    } else if (force_all) {
	/* if not, iterate over all arrow end values */
	strcpy(end,ALL_KEY);
    }
}

static Boolean InValList(attr,v)
  String attr;			/* name of attribute (in table) */
  Value *v;
{
    StepInitializedLinkedList(v) {
	if (v->u.v_bnd->attr_name == attr) { return(True); }
    }
    return(False);
}

static String EqNonFirstVarName(elt,attr)
  Elt *elt;
  String attr;			/* name of attribute (in table) */
/* If 'elt' is has kind BoxElt kind, return the name of the variable compared
   to the attribute 'attr' for equality in the box 'b' so long as the variable
   is not "first" with this box. Otherwise, return NULL.
*/
{
    IntrvlList *curr;

    if (elt->kind == BoxEltKind) {
	Box *b = elt->u.b;
	StepLinkedList(curr,b->u1.intvls) {
	    Intrvl *i = curr->i;
	    if (i->kind == VarKind
		&& InValList(attr,i->range->low)
		&& InValList(attr,i->range->high)
		&& SearchVarList(b->first_vars,i->name) == NULL) {
		return(i->name);
	    }
	}
    }
    return((String)NULL);
}

static void AppendSlotKey(elt,slot,hint,is_rng,rng_cnt,path)
  Elt *elt;
  Slot *slot;
  HdsHint *hint;
  Boolean is_rng;		/* is this elt a range for this slot? */
  int rng_cnt;			/* index of last range (if necessary) */
  INOUT String path;
/* Append the "name" of the next key to the iterator construction path 'path'.
   This name may be a keyword, such as ':OTHERS'; a value, such as ':SUBJ',
   'NIL', or 'pos'; or a variable name substitution, such as ',e2-name',
   ',e1-head', or ',rng1' (notice the commas before these names so they will
   be evaluated).
*/
{
    EltList *adj;		      /* adjacent element */
    int from_flag;		      /* are we on the keyword "from"? */
    String end = path + strlen(path); /* end of current path (write here) */

    if (SameString(hint->name,SYSNAME_ATTR)) {
	Assert(elt->kind==BoxEltKind);
	/*
	 * check if box sysname is restricted by anchored arrow */
	StepLinkedList(adj,elt->adj_elts) {
	    if (adj->elt->order < elt->order) break;
	}
	if (adj != NULL) {
	    if (adj->elt->u.a->from == elt) {
		sprintf(end,",e%d-%s",adj->elt->order,TAIL);
	    } else {
		sprintf(end,",e%d-%s",adj->elt->order,HEAD);
	    }
	} else {
	    /* if not, iterate over all boxes */
	    strcpy(end,ALL_KEY);
	}
    } else if ((from_flag=SameString(hint->name,FROM_ATTR))
	       || SameString(hint->name,TO_ATTR)) {
	AppendAdjBox(elt,from_flag,True,end);
    } else if (slot->key == Others) {
	String var_name;	/* name of variable for attribute */
	/*
	 * :OTHERS slot -- determine if it is due to a variable, a range, or
	   simply to the fact that this box has no relevant intervals for the
	   attribute at this level */
	if ((var_name=EqNonFirstVarName(elt,hint->name)) != NULL) {
	    /* if a variable, append the name of the variable */
	    sprintf(end,",var-%s",var_name);
	} else if (is_rng) {
	    /* if a range, append the name of the range variable */
	    sprintf(end,",rng%d",rng_cnt);
	} else {
	    /* otherwise, not relevant to this attr; iterate over OTHERS */
	    strcpy(end,OTHERS_KEY);
	}
	return;
    } else {
	/*
	 * otherwise, simply append the value of the slot */
	strcpy(end,ValName(slot->kind,hint->type,*slot->key));
    }
}

static void GenHdsIterators(elt,tab,fp,hds,it_cnt,rng_cnt,path,sem)
  Elt *elt;
  int tab;
  FILE *fp;
  Hds *hds;
  int *it_cnt;			/* next iterator index to use */
  int *rng_cnt;			/* next range index to use */
  String path;
  String sem;
/* Generates "let*" bindings of iterator constructs to iterator variables. The
   iterator constructs take the form: (MAKE-ITER-ON *<elt>DT_SUFFIX* <path>),
   where <elt> is the element kind (i.e., "box", "syn", "sem", or "con"), and
   <path> is *one* of the search paths for the element 'elt'.

   SIDE-EFFECT: '*it_cnt' is set to the next iterator to use.
*/
{
    /* see if slots of *all* subtrees are relevant */
    if (AllSubtreesRelevant(elt,hds)) {
	String prefix = (*sem) ? "-sem" : "";
	if ((*it_cnt + *rng_cnt) > 0) { IndentTab(tab+LET_STAR_TAB,fp); }
	fprintf(fp,"(itr%d (make%s-iter-on%s *%s%s* `(%s)))",(*it_cnt)++,
		prefix,sem,EltName[EltIndex(elt)],DT_SUFFIX,path);
    } else {
	/*
	 * otherwise, generate iterators for relevant slots */
	IntrvlList *intrvls,one_item;
	Boolean is_rng;
	Slot *slot;		/* current slot */
	String end = path + strlen(path);
	/*
	 * determine relevant intervals for this attribute */
	intrvls = RelevantIntrvl(elt,hds->hint->name,&is_rng,&one_item);
	/*
	 * bind to a range value if necessary */
	if (is_rng) {
	    if ((*it_cnt + *rng_cnt) > 0) { IndentTab(tab+LET_STAR_TAB,fp); }
	    fprintf(fp,"(rng%d (make-range (",(*rng_cnt)++);
	    GenRange(intrvls->i->range,intrvls->i->kind,hds->hint->type,fp);
	    fputs(")))",fp);
	}
	/*
	 * extend path along all relevant slots */
	StepLinkedList(slot,hds->slots) {
	    if (MatchesSlot2(slot,intrvls,is_rng)) {
		/*
		 * append separating space if not first key name */
		strcpy(end,((end==path) ? "" : " "));
		/*
		 * append proper value to 'path' */
		AppendSlotKey(elt,slot,hds->hint,is_rng,*rng_cnt-1,end);
		/*
		 * recursive call */
		GenHdsIterators(elt,tab,fp,slot->hds,it_cnt,rng_cnt,path,sem);
	    }
	}
	FreeIntrvlList(intrvls,&one_item);
    }
}

static int GenIterators(elt,tab,fp,hds)
  Elt *elt;
  int tab;
  FILE *fp;
  Hds *hds;
/* Returns the index of the *last* iterator.
*/
{
    int i;
    int it_cnt=0;		/* iterator index */
    int rng_cnt=0;		/* range index */
    String path = path_buff;
    String sem_end;
    static char sem[20];

    *path = *sem = '\0';
    if (EltIndex(elt) == SEM) {
	sprintf(sem," %c *%s%s* `(",(elt->u.a->parity==Pos) ? '1' : '0',
		EltName[SUBJ],DT_SUFFIX);
	sem_end = sem + strlen(sem);
	AppendAdjBox(elt,1,False,sem_end);
	strcat(sem_end,")");
    }
    GenHdsIterators(elt,tab,fp,hds,&it_cnt,&rng_cnt,path,sem);
    if (it_cnt > 1) {
	IndentTab(tab+LET_STAR_TAB,fp);
	fprintf(fp,"(itr%d (make-union-it",it_cnt++);
	StepIndex(i,0,it_cnt-1) {
	    fprintf(fp," itr%d",i);
	}
	fputs("))",fp);
    }
    return(it_cnt-1);
}

static Boolean GenUniquenessTest(elt,fp,pict,tab)
  Elt *elt;			/* current element */
  FILE *fp;
  Pict *pict;
  INOUT int *tab;
/* Return True iff 'elt' is a BoxEltKind and a conditional test is required to
   guarantee that it is different from some of the other boxes ordered before
   it. If True is returned, this function also has the side effect of
   generating the conditional code to 'fp' and incrementing the 'tab' value.
*/
{
    BoxList *bl,*curr;

    if (elt->kind == BoxEltKind
	&& (bl=PotentialSamePrecursors((BoxElt *)elt,pict)) != NULL) {
	Boolean multiple = MakeBoolean(bl->next != NULL);
	/*
	 * Generate "when" test */
	Indent(++(*tab),fp);
	fputs("(when ",fp);
	if (multiple) { fputs("(and ",fp); }
	StepLinkedList(curr,bl) {
	    if (curr != bl) { IndentTab((*tab)+WHEN_AND_TAB,fp); }
	    fprintf(fp,"(not (eq e%d e%d))",elt->order,curr->elt->order);
	}
	if (multiple) { fputc(')',fp); } /* close "and" */
	fputc('\n',fp);
	/*
	 * Deallocate structures forming 'bl' list skeleton */
	FreeBoxListSkeleton(bl);
	return(True);
    }
    return(False);
}

static Boolean GenSysnameBindings(num,fp,elt,elt_ix,elt_name,tab)
  int num;
  FILE *fp;
  Elt *elt;
  int elt_ix;
  String elt_name;
  INOUT int *tab;
{
    Boolean result;
    EltList *adj;		/* current adjacent element */
    int after_cnt;		/* number of adj elts ranked after this one */

    /* count adjacent elements ranked before and after this element */
    after_cnt = 0;
    StepLinkedList(adj,elt->adj_elts) {
	if (adj->elt->order > elt->order) { after_cnt++; }
    }

    /* introduce sysname variables? */
    if (result=MakeBoolean(after_cnt > 0)) {
	Indent(++(*tab),fp);
	fputs("(let (",fp);
	switch (elt->kind) {
	  case BoxEltKind:
	    fprintf(fp,"(e%d-%s (box-sysname e%d))",num,NAME,num);
	    break;
	  case ArrowEltKind:
	    if (elt_ix == SEM) { elt_name = "sem"; }
	    if (elt->u.a->from->order > elt->order) {
		fprintf(fp,"(e%d-%s (%s-from e%d))",num,TAIL,elt_name,num);
	    }
	    if (after_cnt >= 2) { IndentTab((*tab)+LET_TAB,fp); }
	    if (elt->u.a->to->order > elt->order) {
		fprintf(fp,"(e%d-%s (%s-to e%d))",num,HEAD,elt_name,num);
	    }
	    break;
	}
    }
    return(result);
}

static String EqAttrName(b,var)
  Box *b;
  Var *var;
/* Return the name of an attribute compared for equality to the variable 'var'
   in the box 'b'. It is assumed that the variable 'var' indeed appears in the
   box predicate for 'b' and that there is at least one attribute compared for
   equality to this variable in that box predicate.
*/
{
    IntrvlList *i_list;
    IntrvlRange *rng;
    Value *low,*high;

    StepLinkedList(i_list,b->u1.intvls) {
	if (i_list->i->name == var->name) { break; }
    }
    Assert(i_list != NULL);
    rng = i_list->i->range;
    StepLinkedList(low,rng->low) {
	StepLinkedList(high,rng->high) {
	    if (low->u.v_bnd->attr_name == high->u.v_bnd->attr_name) {
		return(low->u.v_bnd->attr_name);
	    }
	}
    }
    Assert(False);		/* we should return inside previous loop */
    return(NULL);		/* make lint happy */
}

static void GenVarBindings(num,elt,fp,tab,let_started)
  int num;
  Elt *elt;
  FILE *fp;
  INOUT int *tab;
  INOUT Boolean *let_started;
{
    VarList *curr;

    if (elt->kind == BoxEltKind) {
	Box *b = elt->u.b;
	StepLinkedList(curr,b->first_vars) {
	    if (!(*let_started)) {
		(*let_started) = True;
		Indent(++(*tab),fp);
		fputs("(let (",fp);
	    } else {
		IndentTab((*tab)+LET_TAB,fp);
	    }
	    fprintf(fp,"(var-%s (box-%s e%d))",curr->var->name,
		    EqAttrName(b,curr->var),num);
	}
    }
}
			   
static void GenQuery(num,tab,fp,pict)
  int num,tab;
  FILE *fp;
  Pict *pict;
{
    void GenQueryTop();
    int it_cnt;			/* number of iterators constructed */
    Boolean when_test;		/* when test generated after query? */
    Boolean let_after;		/* sysname/variable bound after query? */
    Elt *elt = pict->elt[num];	/* current element */
    int elt_ix;			/* index of this element */
    String t_name,elt_name;	/* names of thickness and elt kind */

    /* initialize */
    t_name = ThickName[(int)(elt->thickness)];
    elt_name = EltName[(elt_ix=EltIndex(elt))];

    /* introduce element and opening LET* */
    Indent(tab+=TAB,fp); fputs(";;\n",fp);
    Indent(tab,fp); fprintf(fp,";; Element e%d\n",num);
    Indent(tab,fp);
    fputs("(let* (",fp);

    /* generate iterator values and close variable list of "let*" */
    it_cnt = GenIterators(pict->elt[num],tab,fp,pict->hds[elt_ix]);
    fputs(")\n",fp);		/* close variable list of "let*" */

    /* query */
    Indent(++tab,fp);
    fprintf(fp,"(%s e%d itr%d\n",t_name,num,it_cnt);

    /* generate when test to guarantee unique box matchings if necessary */
    when_test = GenUniquenessTest(elt,fp,pict,&tab);

    /* generate sysname bindings if necessary */
    let_after = GenSysnameBindings(num,fp,elt,elt_ix,elt_name,&tab);

    /* generate bindings for first-use variables in boxes */
    GenVarBindings(num,elt,fp,&tab,&let_after);

    /* close variable list of "let" */
    if (let_after) { fputs(")\n",fp); }

    /* recursive call */
    GenQueryTop(num+1,tab,fp,pict);
    if (let_after) { fputc(')',fp); }	/* close "let" for sysnames/vars */
    if (when_test) { fputc(')',fp); }	/* close "when" for uniqueness test */
    fputc(')',fp);		        /* close thick/thin query */
    fputc(')',fp);		        /* close "let*" for iterators */
}

static void GenThinRange(num,range,fp)
  int num;
  Range *range;
  FILE *fp;
{
    int i;
    fprintf(fp,"(thin-range (%d ",range->low);
    if (range->high == INFINITY) { fputs("NIL",fp); }
    else { fprintf(fp,"%d",range->high); }
    fputs(") `(",fp);		/* close range, open var list */
    StepIndex(i,0,num) {
	if (i > 0) { fputc(' ',fp); }
	fprintf(fp,",e%d",i);
    }
    fputs(")\n",fp);		/* close var list */
}

static void GenQueryTop(num,tab,fp,pict)
  int num,tab;
  FILE *fp;
  Pict *pict;
{
    if (num >= pict->elt_cnt) { /* end of query */
	Indent(tab+=TAB,fp); fputs(";;\n",fp);
	Indent(tab,fp); fputs("(constraint-end ",fp);
	if (pict->range.high == INFINITY) { fputs("NIL",fp); }
	else { fprintf(fp,"%d",pict->range.high); }
	fputc(')',fp);
    } else {
	Elt *elt = pict->elt[num];
	if (elt->thickness==Thin
	    && (num==0 || pict->elt[num-1]->thickness==Thick)) {
	    /* thick->thin transition */
	    Indent(tab+=TAB,fp); fputs(";;\n",fp);
	    Indent(tab,fp); fputs(";; Thick -> Thin\n",fp);
	    Indent(tab,fp);
	    GenThinRange(num,&(pict->range),fp);
	    GenQuery(num,tab,fp,pict);
	    fputc(')',fp);	/* close "thin-range" */
	} else {
	    GenQuery(num,tab,fp,pict);
	}
    }
}

/* GLOBAL FUNCTIONS ======================================================== */

void ConstructHdsStructures(pict)
  INOUT Pict *pict;
{
    int elt_index,hds_index;
    Hds **hds_ptr;
    HdsHint *hint;
    HdsHintList *hds_hint;
    EltList *curr;

    /* initialize 'pict->elts[]' arrays */
    InitElts(pict);

    /* init objs.c module for calls to RelevantIntrvl() */
    InitRelevantIntrvl(pict);

    /* build the HDS structures for each kind of element */
    StepIndex(hds_index,0,NUM_HDS_KINDS) {
	elt_index = HdsToElt(hds_index);
	hds_ptr = &(pict->hds[hds_index]);
	/*
	 * consider each attribute for this kind of element*/
	StepLinkedList(hds_hint,pict->attrs[hds_index]) {
	    hint = hds_hint->hds_hint;
	    /*
	     * consider each element of this kind */
	    StepLinkedList(curr,pict->elts[elt_index]) {
		ExtendHds(curr->elt,hint,hds_ptr);
	    }
	}
	/*
	 * optimize this hds */
	OptimizeHds(hds_ptr);
    }
}

void GenerateHdsConstructors(fp,pict)
  FILE *fp;
  Pict *pict;
/* NOTE: If the "type" BOX attribute is a range, then we have to add the slot
   values according to the DFS in-order traversal of the type tree. If there
   is no ":OTHERS" slot, then we only add the slots installed explicitly;
   otherwise, we must install all of them. See "ipql/Test/boxtype-rng.cl".
*/
{
    int i;
    String path = path_buff;

    fputs("\n;;; construct HDS structures\n",fp);
    StepIndex(i,0,NUM_HDS_KINDS) {
	if (pict->hds[i] != NULL) {
	    *path = '\0';
	    GenerateHds(fp,i,pict->hds[i],path);
	}
    }
}

void GenerateQuery(fp,pict)
  FILE *fp;
  Pict *pict;
/* NOTE: When we generate a query for a box, we check if any arrow incident on
   the box has a higher number; if so, we generate a (let...) to bind a
   variable to the box's sysname. Similarly, when we generate a query for an
   arrow, we check if each of the boxes incident on the arrow has a higher
   number; if so, we generate a (let...) to bind a variable to the head and
   tail numbers of the arrow. Moreover, we also check for each such higher
   numbered box if there is some arrow with a *lower* number than this arrow
   incident on the same box; if so, we must generate a (when...) check that
   both (all) arrows are pointing at the same box.
*/
{
    fputs("\n;;; query instance picture\n",fp);
    fputs("(constraint-begin\n",fp);
    GenQueryTop(0,0,fp,pict);
    fputs(")\n",fp);		/* close off "constraint-begin" */
}
