
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* SYSNAME.H

   Declarations for types/functions implementing a hash table for sysname's.

   $Header: sysname.h,v 1.2 91/08/30 16:27:30 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   OVERVIEW OF THIS MODULE

   This module provides facilities for storing Box and Arrow elements by their
   sysnames. The objects are stored in individual hash tables. There is a
   different type of hash table defined for each kind of element: one for
   boxes and one for arrows. It also provides a function to compute the
   sysname of an arrow given the two sysnames of the boxes it connects.

   WHAT THIS MODULE PROVIDES

   This module provides the types "BSysnameTable" and "ASysnameTable" for
   storing boxes and arrows, respectively, in hash tables by sysname. It
   provides functions for inserting and retrieving Box and Arrow elements
   into/from these tables. Finally, it provides the function CantorPair() for
   pairing two box sysnames (BSysname's) to form a unique arrow sysname
   (ASysname).
*/

#ifndef SYSNAME_H
#define SYSNAME_H

#include <my-types.h>

#include "sysname-types.h"
#include "elts.h"

/* TYPES ================================================================== */

typedef struct b_sysname_entry {
    struct b_sysname_entry *next;
    BSysname sysname;
    Box *b;
} BSysnameEntry;

typedef struct sysname_table {
    ULint size;			/* max hash table size */
    ULint cnt;			/* total current number of elements in table */
    ULint threshold;		/* max # of elements before rehash */
    union {
	BSysnameEntry **b_tbl;	/* table (array) of boxes */
	Arrow **a_tbl;		/* table (array) of arrows */
    } u;
} SysnameTable,*BSysnameTable,*ASysnameTable;

/* GLOBAL DEBUGGING FUNCTION DECLARATIONS ================================= */

#ifdef DEBUG

void DisplayBSysnameTable( /* BSysnameTable tbl */ );
/* Display the current contents of the BSysnameTable 'tbl'.
*/

void DisplayASysnameTable( /* ASysnameTable tbl */ );
/* Display the current contents of the ASysnameTable 'tbl'.
*/

#endif DEBUG

/* GLOBAL FUNCTION DECLARATIONS =========================================== */

BSysnameTable NewBSysnameTable( /* void */ );
/* Create and return a new, initialized (empty) BSysnameTable.
*/

ASysnameTable NewASysnameTable( /* void */ );
/* Create and return a new, initialized (empty) BSysnameTable.
*/

ASysname CantorPair( /* BSysname x,y */ );
/* This routine implements an injection: Z+ X Z+ --> Z+, where Z+ is the set
   of positive integers. The unique pairing is formed using a slight
   modification (for efficiency) of Cantor's classical dovetail pattern over
   the grid of natural numbers.
*/

void AddBoxSysname( /* BSysnameTable tbl, BSysname sysname, Box *b */ );
/* Adds the box 'b' to 'tbl' under the sysname 'sysname'.
*/

Arrow *AddArrowSysname( /* ASysnameTable tbl,ASysname sysname,FullArrow *a */);
/* Adds the arrow 'a' to 'tbl' under the sysname 'sysname'. Returns a pointer
   to the Arrow in the hash table.
*/

Box *FindBoxSysname( /* BSysnameTable tbl, BSysname sysname */ );
/* Returns the Box with sysname 'sysname' in 'tbl', or NULL if none exists.
*/

Arrow *FindArrowSysname(/* ASysnameTable tbl,ASysname sysname,PermSet perm */);
/* Returns the Arrow with sysname 'sysname' in 'tbl' having one of the
   permissions represented by 'perm', or NULL if no such arrow exists.
*/

#endif SYSNAME_H
