/* RELS.H

   Type/function definitions for the rels.c module, which manages relations
   between subject and object boxes in the form of witness sets.

   $Header: rels.h,v 1.5 91/10/28 11:44:37 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#ifndef RELS_H
#define RELS_H

#include <my-types.h>

#include "sysname-types.h"

/* MACRO DEFINITIONS ======================================================= */

/* WBlock value for a visited/unvisited object box */
#define UnvisitedBlk ((WBlock *)NULL)
#define DeletedBlk   ((WBlock *)0x04)

/* TYPE DEFINITIONS ======================================================== */

typedef ULint FingerPrint;

typedef struct w_set {
    struct w_set *next;
    FingerPrint hash_val;	       /* hash value for this WSet */
    Uint ref_cnt;		       /* reference cnt of pointers to this */
#ifdef MEASURE
    Uint tot_ref_cnt;		       /* total diff. refs to this WSet */
#endif MEASURE
    struct arrow_list *al[PARITY_CNT]; /* lists of Pos/Neg arrows */
} WSet;

typedef WSet **WSetGrp;		/* "group" of WSet pointers */

typedef struct w_block {
    Uint ref_cnt;		/* reference count */
#ifdef MEASURE
    Uint tot_ref_cnt;		/* total diff refs to this WBlock */
#endif MEASURE
    WSetGrp w_set_grp;		/* "group" (array) of Witness sets */
} WBlock;

typedef struct w_set_tbl {
    ULint size;			/* number of buckets in table */
    ULint threshold;		/* max number of elements in table */
    ULint cnt;			/* number of elements in table */
    WSet **tbl;			/* hash table of WSet's */
} *WSetTable,WSetTableStruct;

typedef Boolean *BoolVector;
typedef ULint *BitVector;

typedef struct access_block {
    struct access_block *next;	/* pointer to next block in bucket */
    ULint index;		/* index of this block */
    FingerPrint hash_val;	/* hash value for this BitVector */
    BitVector bits;		/* block of bits */
} AccessBlock;

typedef struct access_table {
    ULint size;			/* size of table */
    ULint threshold;		/* max # of elements in table */
    ULint cnt;			/* number of elements in table */
    AccessBlock **tbl;		/* array of AccessBlock's */
} AccessTable;

/* FUNCTION DECLARATIONS =================================================== */

WSet *NewWSet( /* void */ );
/* Returns a new, empty WSet.
*/

#ifdef MACRO_FUNCTION

void CopyWSet( /* WSet *w_set */ );
/* "Copy" and return witness set 'w_set'. The set is not actually copied; a
   pointer to the set is returned after updating its reference count.
*/

void FreeWSet( /* WSet *w_set */ );
/* Deallocate the witness set 'w_set' stored in 'WSetTbl'. This frees the
   memory used by the witness set iff the reference count becomes 0.
*/
#endif MACRO_FUNCTION

#ifdef MEASURE
#define CopyWSet(_w_set)\
  (((_w_set)->ref_cnt)++,((_w_set)->tot_ref_cnt)++,(_w_set))
#else
#define CopyWSet(_w_set)\
  (((_w_set)->ref_cnt)++,(_w_set))
#endif MEASURE

#define FreeWSet(_w_set)\
  if (--((_w_set)->ref_cnt) == 0) { DeallocWSet((_w_set)); }

void DeallocWSet( /* WSet *w_set */ );
/* Really deallocate the space occupied by 'w_set'. This also removes 'w_set'
   from the WSetTable in the global variable 'WSetTbl'. A core dump will occur
   if 'w_set' is not contained in this table. That is, the 'w_set' should
   already have been inserted in the table via a call to FindOrAddWSet() below.

   Users should not call this function directly, but should instead use
   FreeWSet() above.
*/

WSet *NewWSetFromArrow( /* Arrow *a, PermSet perm */ );
/* Create and return a new WitnessSet corresponding to the single Arrow 'a'
   restricted to permission 'perm'.
*/

#ifdef MACRO_FUNCTION

Boolean PosWSet( /* WSet *w_set */ );
/* Returns True iff 'w_set' contains no negative arrows.
*/

Boolean NegWSet( /* WSet *w_set */ );
/* Returns True iff 'w_set' contains no positive arrows.
*/

Boolean AmbigWSet( /* WSet *w_set */ );
/* Returns True iff 'w_set' contains positive *and* negative arrows.
*/
#endif MACRO_FUNCTION

#define PosWSet(_w_set) ((_w_set)->al[(int)Neg] == (ArrowList *)NULL)
#define NegWSet(_w_set) ((_w_set)->al[(int)Pos] == (ArrowList *)NULL)

#define AmbigWSet(_w)\
  ((_w)->al[(int)Pos] != (ArrowList *)NULL && \
   (_w)->al[(int)Neg] != (ArrowList *)NULL)

WBlock *NewWBlock( /* int size, WSet *w_set */ );
/* Return a new WBlock having size 'size'. Each pointer in the block
   is initialized to 'w_set' (which is usually passed as NULL).
*/

#ifdef MACRO_FUNCTION

WBlock CoypWBlock( /* WBlock w_blk */ );
/* "Copy" and return WBlock 'w_blk'. The block is not actually copied; a
   pointer to the block is return after updating its reference count.
*/

void FreeWBlock( /* WBlock w_blk, int size */ );
/* Deallocate the WBlock'w_blk' having size 'size' as created by NewWBlock()
   above. This frees the memory and those of the WSet's it contains iff the
   reference count becomes 0. Every pointer of the WBlock array is assumed to
   be non-NULL. The WSet's in the WBlock must reside in the WSetTable named by
   the global variable 'WSetTbl'.
*/
#endif MACRO_FUNCTION

#ifdef MEASURE
#define CopyWBlock(_w_blk)\
  (((_w_blk)->ref_cnt)++,((_w_blk)->tot_ref_cnt)++,(_w_blk))
#else
#define CopyWBlock(_w_blk)\
  (((_w_blk)->ref_cnt)++,(_w_blk))
#endif MEASURE

#define FreeWBlock(_w_blk,_size)\
  if (--((_w_blk)->ref_cnt) == 0) { DeallocWBlock((_w_blk),(_size)); }

void DeallocWBlock( /* WBlock w_blk, int size */ );
/* Really deallocate the space occupied by 'w_blk' and the WSet's it
   contains. This function has the same conditions as FreeWBlock() above.

   Users should not call this function directly, but should instead use
   FreeWBlock() above.
*/

Boolean DisplayParity(/* FILE *fp, WSet *w_set, BSysname subj,obj */);
/* Print an indication of the parity of 'w_set' (which should be the witness
   set between the box 'subj' and 'obj') to the stream 'fp'. If 'PrintLisp' is
   true, then the description is written in a lisp-like format.

   Returns True iff all arrows in 'w_set' are positive.
*/

WSetTable NewWSetTable( /* void */ );
/* Create and return a new, empty table of WSet's.
*/

#ifdef OBSOLETE
void ResetWSetTable( /* INOUT WSetTable tbl */ );
/* Deallocate the space used by any WSet's in 'tbl'. All WSet's are removed
   from the table, with the exception of the WSet named by the global variable
   'DefaultSet'.
*/
#endif OBSOLETE

WSet *FindOrAddWSet( /* INOUT WSet *w_set */ );
/* Searches for a WSet identical to 'w_set' in the WSetTable in the global
   variable 'WSetTbl'. If one is not found, 'w_set' is added to the table and
   returned. If one *is* found, then the space occupied by 'w_set' is freed,
   and the matching WSet in the table is returned. The 'hash_val' field of
   'w_set' is assumed to be garbage on entry; it is set in the returned
   result.
*/

WSet *MinWSet( /* WSet *s1, WSet *s2 */ );
/* Create a return a new WSet representing the minimum arrows over the set of
   arrows 's1' \union 's2'.

   SIDE-EFFECT: The new WSet is also registered in the WSet hash table in the
   global variable 'WSetTbl' (if there is not an instance of it there already).
*/

#ifdef MACRO_FUNCTION

WSet *FastMinWSet( /* WSet *s1, WSet *s2 */ );
/* Same semantics as MinWSet() above, only some tests are made first to
   optimize the computation and invoke the slower MinWSet only when necessary.
*/
#endif MACRO_FUNCTION

#ifdef MEASURE
#define FastMinWSet(_s1,_s2) \
    ((_s1)==(_s2) ? (EqSets++,(_s1)) \
     : (_s1)==DefaultSet ? (SecondSetDef++,(_s2)) \
     : (_s2)==DefaultSet ? (FirstSetDef++,(_s1)) \
     : (MinWSetCnt++,MinWSet((_s1),(_s2))))
#else
#define FastMinWSet(_s1,_s2) \
    ((_s1)==(_s2) ? (_s1) \
     : (_s1)==DefaultSet ? (_s2) \
     : (_s2)==DefaultSet ? (_s1) \
     : MinWSet((_s1),(_s2)))
#endif MEASURE

int InitAccessTable( /* int subj_cnt */ );
/* Initialize the hash table for storing AccessBlock's. This function must be
   called before any call to FindOrAddAccess() below. 'Subj_cnt' should be the
   number of subjects in the picture (the vectors in the table will have this
   many entries).

   Returns the number of bits in each access vector.
*/

ULint FindOrAddAccess( /* BoolVector vector */ );
/* Looks for 'vector' in the AccessBlock hash table. If it is not found, an
   entry for it is added to the table. In either case, the integer index of
   the block is returned.
*/
#endif RELS_H

void DisplayAccessBlocks( /* int subj_cnt */ );
/* Displays the access blocks added to the AccessTable by calls to
   FindOrAddAccess() above. This also generates the '(init-sem # #)' call
   which declares how many subjects (namely, 'subj_cnt') and how many objects
   (namely, the number of entries in the AccessTable) there are.
*/
