/* ID-TABLE.C

   Module for creating and managing a hash table of identifiers.

   $Header: id-table.c,v 1.1 91/08/01 20:06:34 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "box-type.h"
#include "elts.h"
#include "id-table.h"

/* MACRO DEFINITIONS ======================================================= */

/* hash table size */
#define ID_HASH_SZ (71)

/* Radix of the ASCII character set */
#define CharSetRadix 256

/* LOCAL FUNCTIONS ======================================================== */

static int HashID(name,kind)
  String name;
  IdKind kind;
{
    register int result = 0;
    for (; *name != '\0'; name++) {
	result = ((result * CharSetRadix) + (int)(*name)) % ID_HASH_SZ;
    }
    result = (result + (int)kind) % ID_HASH_SZ;
    return(result);
}

/* GLOBAL DEBUGGING FUNCTIONS ============================================== */

#ifdef OLD_DEBUG

void DisplayIdHashTable(t)
  IdHashTable t;
{
    register int i;
    TableEntryList *curr;

    fprintf(stderr,"\nId Hash Table:\n");
    StepIndex(i,0,ID_HASH_SZ) {
	if (t[i] != (TableEntryList *)NULL) {
	    fprintf(stderr,"  %2d: ",i);
	    StepLinkedList(curr,t[i]) {
		fprintf(stderr,"'%s' ",curr->entry->name);
		switch (curr->entry->kind) {
		  case BoxTypeId: fputs("(type)",stderr); break;
		  case AttrNameId: fputs("(attr)",stderr); break;
		  case PermNameId: fputs("(perm)",stderr); break;
		}
		fprintf(stderr,"%s",curr->next ? ", " : "\n");
	    }
	}
    }
}

#endif OLD_DEBUG

/* GLOBAL FUNCTIONS ======================================================== */

IdHashTable NewIdHashTable()
{
    register int i;
    IdHashTable result = AllocPtrArray(TableEntryList,ID_HASH_SZ);

    StepIndex(i,0,ID_HASH_SZ) {
	result[i] = (TableEntryList *)NULL;
    }
    return(result);
}

TableEntry *FindTableId(t,kind,name)
  IdHashTable t;
  IdKind kind;
  String name;
/* IMPLEMENTATION: This function makes use of the fact that HashID will never
   place two strings with the same name but different IdKind in the same
   bucket. Therefore, it is enough just to search for a matching name in the
   bucket.
*/
{
    TableEntryList *curr;

    StepLinkedList(curr,t[HashID(name,kind)]) {
	if (name==curr->entry->name || SameString(name,curr->entry->name)) {
	    return(curr->entry);
	}
    }
    return((TableEntry *)NULL);
}

TableEntry *AddTableId(t,kind,name,val)
  INOUT IdHashTable t;
  IdKind kind;
  String name;
  Generic *val;
{
    int hash_index;
    TableEntryList *list;
    TableEntry *entry;

    hash_index = HashID(name,kind);
    list = AllocOne(TableEntryList);
    list->next = t[hash_index];
    t[hash_index] = list;
    entry = list->entry = AllocOne(TableEntry);
    CopyString(entry->name,name);
    entry->kind = kind;
    switch (kind) {
      case BoxTypeId:  entry->u.box_type = (BoxType *)val; break;
      case AttrNameId: entry->u.attr     =    (Attr *)val; break;
      case PermNameId: entry->u.perm_val = (PermVal *)val; break;
    }
    return(entry);
}
