/* EXTRACT-ELTS.C

   Uses extract library to extract boxes, arrows, and inside relations from
   the parse tree.

   $Header: extract-elts.c,v 1.4 91/11/14 16:32:32 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "error.h"
#include "parser.g"
#include "convert.h"
#include "extract.h"

#include "box-type.h"
#include "extract-elts.h"
#include "sysname-types.h"
#include "elts.h"
#include "id-table.h"
#include "pict.h"

#define MAX_PNAMES 5		/* maximum # of prop names in any ENTRY */

/* LOCAL FUNCTIONS ======================================================== */

/* Conversion Functions ---------------------------------------------------- */

static void ConvertParity(p_val_ptr,val_ptr,line_no)
  PropVal *p_val_ptr;
  Generic *val_ptr;
  int line_no;
{
    ArrowParity *result_ptr = (ArrowParity *)val_ptr;

    if (SameString("pos",IdValOf(p_val_ptr))) {
	*result_ptr = Pos;
    } else if (SameString("neg",IdValOf(p_val_ptr))) {
	*result_ptr = Neg;
    } else {
	ParseErrorS(line_no,
		    "arrow parity '%s' should be either 'pos' or 'neg'",
		    IdValOf(p_val_ptr));
    }
}

static void ConvertAttribute(p_val_ptr,val_ptr,line_no)
  PropVal *p_val_ptr;
  Generic *val_ptr;
  int line_no;
{
    int member_cnt;
    ListEntry *curr;
    String id,end;
    Attr *result_ptr = (Attr *)val_ptr;

    if (p_val_ptr->prop_val_type != ListPValType) {
	ParseError(line_no,"attributes must be a list of lists");
	return;
    }
    member_cnt = 0;
    result_ptr->type.list_nesting = 0;
    StepLinkedList(curr,p_val_ptr->val.list_head) {
	if (++member_cnt < 4) {
	    if (curr->list_prop_val_ptr->prop_val_type != IdPValType) {
		ParseErrorI(line_no,
		"attribute descriptor contains a non-identifier in field %d",
			    member_cnt);
		return;
	    }
	    id = curr->list_prop_val_ptr->val.id_val;
	}
	switch(member_cnt) {
	  case 1:		/* attribute name */
	    result_ptr->name = id;
	    break;
	  case 2:		/* attribute type */
	    /* NOTE: We can't change 'id' permanently because it points into
	       the ID hash table. We change it temporarily to make the string
	       comparisons, and then change it back afterwards */
	    /* compute proper list nesting */
	    for (end=id+strlen(id)-5; end>=id; end-=5) {
		if (!strncmp(end,"-list",5)) {
		    (result_ptr->type.list_nesting)++;
		}
	    }
	    /* temporarily change '-' of *first* "-list" to '\0' */
	    for (end=id; *end; end++) {
		if (!strncmp(end,"-list",5)) { break; }
	    }
	    if (*end == '-') {
		*end = '\0';	/* temporarily change to string terminator */
	    } else {
		end = (String)NULL; /* set to NULL to indicate no change */
	    }
	    /* match the proper base type */
	    if (SameString(id,"integer")) {
		result_ptr->type.prim = IntVal;
	    } else if (SameString(id,"identifier")) {
		result_ptr->type.prim = IdVal;
	    } else if (SameString(id,"string")) {
		result_ptr->type.prim = StringVal;
	    } else if (SameString(id,"boolean")) {
		result_ptr->type.prim = BoolVal;
	    } else if (SameString(id,"box-type")) {
		result_ptr->type.prim = BoxTypeVal;
	    } else {
		ParseErrorS(line_no,"'%s' not a legal attribute type",id);
	    }
	    /* change '\0' back to '-' if necessary */
	    if (end) {
		*end = '-';
	    }
	    break;
	  case 3:		/* mandatory or optional? */
	    if (SameString(id,"mandatory")) {
		result_ptr->required = True;
	    } else if (SameString(id,"optional")) {
		result_ptr->required = False;
	    } else {
		ParseError(line_no,
	     "attribute's 'required' field must be 'mandatory' or 'optional'");
	    }
	    break;
	  case 4:		/* default value for optional attribute*/
	    if (result_ptr->required) {
		ParseError(line_no,
			   "default value specified for mandatory attribute");
	    }
	    /* ignore default value */
	    break;
	  default:
	    ParseError(line_no,"attribute descriptor too long");
	    return;
	} /* end switch() */
    } /* end StepLinkedList() */
    if (member_cnt < 3) {
	ParseError(line_no,"attribute descriptor too short");
	return;
    }
}

/* Box Functions ----------------------------------------------------------- */

static Box *ExtractBox(box_entry,pict,bt)
  Entry *box_entry;
  Pict *pict;
  OUT BoxType **bt;
/* Return a pointer to a new Box containing attributes according to the Entry
   'box_entry', or NULL if errors were found in the Entry. Set '*bt' to point
   to the BoxType of this box.
*/
{
    Box *result;
    String type_name;
    TableEntry *tbl;

    /* allocate a box and initialize optional attributes */
    result = NewBox();

    /* install attribute names */
    StartNewPNameList();
    AddPNameDesignator("sysname",True,ConvertInt,
		       (Generic *)&(result->sysname),NULL_LIST);
    AddPNameDesignator("type",True,ConvertId,
		       (Generic *)&(type_name),NULL_LIST);

    /* extract info from entry */
    if (MatchPNames(box_entry)) goto box_error;

    /* no errors found; check box-type and sysname */
    if ((tbl=FindTableId(pict->table,BoxTypeId,type_name)) == NULL) {
	ParseErrorS(EntryLineNumberOf(box_entry),
		    "unknown box-type '%s'",type_name);
	goto box_error;
    }
    if (FindBoxSysname(pict->b_sysnames,result->sysname) != NULL) {
	ParseErrorI(EntryLineNumberOf(box_entry),
		    "box with sysname %d already exists",
		    (int)result->sysname);
	goto box_error;
    }

    /* set box-type information */
    *bt = tbl->u.box_type;
    result->box_type_val = (*bt)->box_type_val;

#ifdef OLD_DEBUG
    fprintf(stderr,"\nExtracted BOX, sysname = %d, type = %s\n"
	    result->sysname,type_name);
#endif OLD_DEBUG
    return(result);

box_error:
    FreeBox(result);
    return((Box *)NULL);
}

static void InsertBox(b,pict,bt)
  Box *b;
  Pict *pict;
  BoxType *bt;
{
    int ix;

    if (b != (Box *)NULL) {
	/*
	 * insert the box in the list */
	ix = (int)bt->role;
	pict->s[ix].cnt++;
	b->next = pict->s[ix].u.boxes;
	pict->s[ix].u.boxes = b;
	/*
	 * register this box's sysname*/
	AddBoxSysname(pict->b_sysnames,b->sysname,b);
    }
}

static void ExtractBoxEntries(p_tree,pict)
  Entry *p_tree;
  INOUT Pict *pict;
/* Traverse the list of entries in 'p_tree', creating Boxes for each BOX entry
   in the tree. Add these Boxes to the list 'pict->boxes'. This function
   assumes all BoxTypes have been read and processed, and that the 'role'
   field of each BoxType has been set.
*/
{
    Entry *curr_ent;
    BoxType *bt;

    StepLinkedList (curr_ent,p_tree) {
	if (EntryTypeOf(curr_ent) == BoxEntry) {
	    InsertBox(ExtractBox(curr_ent,pict,&bt),pict,bt);
	}
    }
}

/* Arrow Functions --------------------------------------------------------- */

static FullArrow *ExtractArrow(arrow_entry,pict)
  Entry *arrow_entry;
  Pict *pict;
{
    static FullArrow arrow;	/* this is the only FullArrow allocated */

    int to,from;		/* sysnames of head/tail boxes */
    ListDesc perm_list,*perms;
    String perm_name;
    int line_no = EntryLineNumberOf(arrow_entry);
    TableEntry *tbl;

    /* initialize */
    perms = &perm_list;

    /* register attributes */
    StartNewPNameList();
    AddPNameDesignator("from",True,ConvertInt,(Generic *)&from,NULL_LIST);
    AddPNameDesignator("to",True,ConvertInt,(Generic *)&to,NULL_LIST);
    AddPNameDesignator("parity",True,ConvertParity,
		       (Generic *)&(arrow.parity),NULL_LIST);
    AddPNameDesignator("permissions",False,ConvertId,
		       (Generic *)NULL,perms);

    /* convert attribute values */
    if (MatchPNames(arrow_entry)) { return((FullArrow *)NULL); }

    /* convert box sysnames to Box pointers */
    if ((arrow.from=FindBoxSysname(pict->b_sysnames,(BSysname)from)) == NULL) {
	ParseErrorI(EntryLineNumberOf(arrow_entry),
		    "unknown 'from' box %d",from);
	return((FullArrow *)NULL);
    }
    if ((arrow.to=FindBoxSysname(pict->b_sysnames,(BSysname)to)) == NULL) {
	ParseErrorI(EntryLineNumberOf(arrow_entry),
		    "unknown 'to' box %d",to);
	return((FullArrow *)NULL);
    }

    /* parse arrow permission(s) */
    arrow.perms = 0x0;
    if (ValFoundP("permissions")) {
	/* proccess all "permissions" values, forming a PermSet */
	while (NextListEntryPtrOf(perms) != NULL) {
	    if (MatchNextListElement(perms,(Generic *)&(perm_name))) {
		return((FullArrow *)NULL);
	    }
	    if ((tbl=FindTableId(pict->table,PermNameId,perm_name)) == NULL) {
		ParseErrorS(line_no,"unknown arrow permission '%s'",perm_name);
		return((FullArrow *)NULL);
	    }
	    arrow.perms |= tbl->u.perm_val->perm_set;
	}
    }
#ifdef OLD_DEBUG
    fprintf(stderr,"\nExtracted ARROW, from = %d, to = %d\n",from,to);
#endif OLD_DEBUG
    return(&arrow);
}

static void InsertArrow(a,pict)
  FullArrow *a;
  Pict *pict;
/* If 'a' is non-NULL, insert the information in the FullArrow 'a' (as an
   Arrow) into the sysname hash table 'pict->a_sysnames'.
*/
{  
    if (a != (FullArrow *)NULL) {
	/*
	 * add the arrow to the hash table */
	ASysname a_sysname = CantorPair(a->from->sysname,a->to->sysname);
	(void)AddArrowSysname(pict->a_sysnames,a_sysname,a);
	/*
	 * fill in permissions of incident boxes */
	a->from->a_incidence |= a->perms;
	a->to->a_incidence |= a->perms;
    }
}

/* Inside Functions -------------------------------------------------------- */

static void *ExtractInside(inside_entry,pict)
  Entry *inside_entry;
  Pict *pict;
{
    int from,to;
    Box *from_box,*to_box;
    ListDesc children_list,*children;

    /* initialize */
    children = &children_list;

    /* install attribute names */
    StartNewPNameList();
    AddPNameDesignator("parent",True,ConvertInt,(Generic *)&to,NULL_LIST);
    AddPNameDesignator("children",True,ConvertInt,(Generic *)NULL,children);

    /* extract attribute values */
    if (MatchPNames(inside_entry)) { return; }
    
    /* make sure head box exists */
    if ((to_box=FindBoxSysname(pict->b_sysnames,(BSysname)to)) == NULL) {
	ParseErrorI(EntryLineNumberOf(inside_entry),
		    "unknown parent box %d",to);
	return;
    }

    /* extract child boxes */
    while (NextListEntryPtrOf(children) != NULL) {
	/*
	 * get next child sysname */
	if (MatchNextListElement(children,(Generic *)&from)) {
	    return;
	}
	/*
	 * make sure tail box exists */
	if ((from_box=FindBoxSysname(pict->b_sysnames,(BSysname)from))==NULL) {
	    ParseErrorI(EntryLineNumberOf(inside_entry),
			"unknown child box %d",from);
	    return;
	}
	/*
	 * assert box containment */
	AssertBoxContainment(to_box,from_box);
#ifdef OLD_DEBUG
	fprintf(stderr,"\nExtracted INSIDE containment relation\n");
	fprintf(stderr,"  from = %d, to = %d\n",from,to);
#endif OLD_DEBUG
    }
}

/* Box Type Functions ------------------------------------------------------ */

static BoxType *ExtractBoxType(type_entry)
  Entry *type_entry;
{
    BoxType *result = NewBoxType(EntryLineNumberOf(type_entry));
    ListDesc attr_list,*attr_list_ptr;
    Attr *attr;
    AttrList *attr_cell;

    /* initialize so we don't have to dereference attr_list everywhere */
    attr_list_ptr = &attr_list;

    /* initialize list of PNameDesignators */
    StartNewPNameList();
    AddPNameDesignator("type-name",True,ConvertId,
		       (Generic *)&result->name,NULL_LIST);
    AddPNameDesignator("supertype",False,ConvertId,
		       (Generic *)&result->supertype,NULL_LIST);
    AddPNameDesignator("attributes",False,ConvertAttribute,
		       (Generic *)NULL,attr_list_ptr);

    /* extract the data from the Entry */
    if (MatchPNames(type_entry)) {
	goto type_error;
    }

    /* set 'supertype' to NULL if this type has no parent */
    if (!ValFoundP("supertype")) {
	result->supertype = (String)NULL;
    }

    /* extract attribute information */
    if (ValFoundP("attributes")) {
	while (NextListEntryPtrOf(attr_list_ptr) != NULL) {
	    attr = AllocOne(Attr);
	    if (MatchNextListElement(attr_list_ptr,(Generic *)attr)) {
		Dealloc(attr);
	    } else {
		/* insert the new attribute at the head of the attr list */
		attr_cell = AllocOne(AttrList);
		attr_cell->next = result->attr_head;
		result->attr_head = attr_cell;
		attr_cell->attr = attr;
	    }
	}
    }
#ifdef OLD_DEBUG
    DisplayBoxType(result);
#endif OLD_DEBUG
    return(result);

type_error:
    Dealloc(result);
    return((BoxType *)NULL);
}

static void InsertAttributes(a_list,line_no,pict)
  AttrList *a_list;		/* list of attributes to install */
  int line_no;			/* error message line number */
  Pict *pict;
{
    Attr *attr;

    StepLinkedList(a_list,a_list) {
	attr = a_list->attr;
	if (FindTableId(pict->table,AttrNameId,attr->name) != NULL) {
	    ParseErrorS(line_no,"attribute name '%s' already used",attr->name);
	} else {
	    /* install the attribute in the hash table */
	    (void)AddTableId(pict->table,AttrNameId,
			     attr->name,(Generic *)attr);
	}
    }
}

static void InsertType(bt,pict)
  BoxType *bt;
  INOUT Pict *pict;

{
    static BoxTypeSet curr_box_type_val = 1;
    BoxTypeList *tl;

    if (bt != (BoxType *)NULL) {
	if (FindTableId(pict->table,BoxTypeId,bt->name) != NULL) {
	    ParseErrorS(bt->line_no,"box type '%s' already exists",bt->name);
	    Dealloc(bt);
	} else {
	    /*
	     * set the 'val' field of the box-type */
	    bt->box_type_val = curr_box_type_val;
	    curr_box_type_val <<= 1;
	    Assert(curr_box_type_val != 0);
	    /*
	     * add the type name to the IdHashTable */
	    (void)AddTableId(pict->table,BoxTypeId,bt->name,(Generic *)bt);
	    /*
	     * add the BoxType to the front of the pict->box_types list */
	    tl = AllocOne(BoxTypeList);
	    tl->next = pict->box_types;
	    pict->box_types = tl;
	    tl->bt = bt;
	    /*
	     * add the attribute names assoc. w/ 'bt' to the IdHashTable */
	    InsertAttributes(bt->attr_head,bt->line_no,pict);
	}
    }
}

/* Permission Functions ---------------------------------------------------- */

static Boolean ExtractTypeSet(line_no,list_ptr,pict,set)
  int line_no;
  ListDesc *list_ptr;
  Pict *pict;
  OUT PermSet *set;
{
    String box_type_name;
    TableEntry *tbl;

    *set = 0;			/* turn off all bits */
    while (NextListEntryPtrOf(list_ptr) != NULL) {
	if (MatchNextListElement(list_ptr,(Generic *)(&box_type_name))) {
	    return(True);
	}
	if ((tbl=FindTableId(pict->table,BoxTypeId,box_type_name)) == NULL) {
	    ParseErrorS(line_no,"unknown box-type '%s'",box_type_name);
	    return(True);
	}
	*set |= tbl->u.box_type->box_type_val;
    }
    return(False);
}

static PermVal *ExtractPerm(perm_entry,pict,perm_name)
  Entry *perm_entry;
  Pict *pict;
  OUT String *perm_name;
{
    PermVal *result;
    String p_name;
    ListDesc domain_list,*domain_list_ptr;
    ListDesc range_list,*range_list_ptr;
    ListDesc syn_list,*syn_list_ptr;
    int line_no = EntryLineNumberOf(perm_entry);

    /* initialize pointers */
    domain_list_ptr = &domain_list;
    range_list_ptr = &range_list;
    syn_list_ptr = &syn_list;

    /* initialize list of PNameDesignators */
    StartNewPNameList();
    AddPNameDesignator("perm-name",True,ConvertId,
		       (Generic *)(&p_name),NULL_LIST);
    AddPNameDesignator("domain",True,ConvertId,
		       (Generic *)NULL,domain_list_ptr);
    AddPNameDesignator("range",True,ConvertId,
		       (Generic *)NULL,range_list_ptr);
    AddPNameDesignator("synonyms",False,ConvertId,
		       (Generic *)NULL,syn_list_ptr);

    /* extract the data from the Entry */
    if (MatchPNames(perm_entry)) { return((PermVal *)NULL); }

    /* extract "domain" and "range" sets */
    result = AllocOne(PermVal);
    if (ExtractTypeSet(line_no,domain_list_ptr,pict,&(result->domain)))
	goto perm_error;
    if (ExtractTypeSet(line_no,range_list_ptr,pict,&(result->range)))
	goto perm_error;

    /* install abbreviations if necessary */
    if (ValFoundPOf(DesigPtrOf(syn_list_ptr))) {
	String syn_name,name;	/* name of synonym */
	static void InsertPerm();
	while (NextListEntryPtrOf(syn_list_ptr) != NULL) {
	    if (MatchNextListElement(syn_list_ptr,(Generic *)(&name)))
		goto perm_error;
	    CopyString(syn_name,name);
	    InsertPerm(result,syn_name,False,line_no,pict);
	}
    }
    CopyString(*perm_name,p_name);
    return(result);

perm_error:
    Dealloc(result);
    return((PermVal *)NULL);
}

static void InsertPerm(perm,name,new_perm,line_no,pict)
  PermVal *perm;
  String name;
  Boolean new_perm;
  int line_no;
  INOUT Pict *pict;
/* Adds the permission 'perm' named 'name' to the IdHashTable 'pict->table' so
   it can be looked up by name.

   If the flag 'new_perm' is True, then 'perm' is considered to be a new
   permission, so it is given a PermSet value and an index. It is also added
   to the list of permissions for the picture 'pict->perm_list'.

   SIDE-EFFECT: If 'new_perm' is True, the values 'pict->perms',
   'pict->perm_cnt', and 'pict->perm_list' are changed.
*/
{
    PermList *cell;
    TableEntry *tbl;

    if (FindTableId(pict->table,PermNameId,name) != NULL) {
	ParseErrorS(line_no,"permission '%s' already exists",name);
	Dealloc(perm);
	Dealloc(name);
    } else {
	/*
	 * add the permission 'name' to the IdHashTable */
	tbl = AddTableId(pict->table,PermNameId,name,(Generic *)perm);
	/*
	 * add the permission to the picture if 'new_perm' */
	if (new_perm) {
	    /*
	     * set 'perm_set' and 'index' fields */
	    perm->perm_set = pict->perms;
	    perm->index = pict->perm_cnt++;
	    pict->perms <<= 1;
	    Assert(pict->perms != 0); /* check for overflow */
	    /*
	     * add the permission to the list of perms */
	    cell = AllocOne(PermList);
	    cell->perm_name = tbl->name;
	    SpliceIntoList(pict->perm_list,cell);
	}
    }
}

/* Extract Other Entries --------------------------------------------------- */

static void ExtractOtherEntries(p_tree,pict)
  Entry *p_tree;
  INOUT Pict *pict;
/* Traverse the list of entries in 'p_tree', processing ARROW, INSIDE, EDITOR,
   and TYPE entries in the tree. Create Arrows, Ranges, and BoxType's as
   necessary, and update the 'arrows', 'range', and 'box_types' fields of
   'pict' as necessary.
*/
{
    Entry *curr_ent;
    String p_name;
    PermVal *perm;

    StepLinkedList (curr_ent,p_tree) {
	switch (EntryTypeOf(curr_ent)) {
	  case ArrowEntry:
	    InsertArrow(ExtractArrow(curr_ent,pict),pict);
	    break;
	  case InsideEntry:
	    ExtractInside(curr_ent,pict);
	    break;
	  case TypeEntry:
	    InsertType(ExtractBoxType(curr_ent),pict);
	    break;
	  case PermEntry:
	    if ((perm=ExtractPerm(curr_ent,pict,&p_name)) != NULL) {
		InsertPerm(perm,p_name,True,EntryLineNumberOf(curr_ent),pict);
	    }
	    break;
	}
    }
#ifdef OLD_DEBUG
    DisplayIdHashTable(pict->table);
#endif OLD_DEBUG
}

/* GLOBAL FUNCTIONS ======================================================= */

void ConfigureHashTable()
{
    /* Install Entries */
    AddEntryName("ARROW",      ArrowEntry);
    AddEntryName("BOX",        BoxEntry);
    AddEntryName("INSIDE",     InsideEntry);
    AddEntryName("BOXTYPE",    TypeEntry);
    AddEntryName("PERMISSION", PermEntry);

    /* Install ARROW Property Names */
    AddPropName("from",        ArrowPName,    IntPValType);
    AddPropName("to",          ArrowPName,    IntPValType);
    AddPropName("parity",      ArrowPName,    IdPValType);
    AddPropName("permissions", ArrowPName,    IdListPValType);

    /* Install BOX Property Names */
    AddPropName("sysname",     ObjectPNames,  IntPValType);
    AddPropName("type",        BoxPName,      IdPValType);

    /* Install INSIDE Property Names */
    AddPropName("children",    InsidePName,   IntListPValType);
    AddPropName("parent",      InsidePName,   IntPValType);

    /* install TYPE property names */
    AddPropName("type-name",   TypePName,     IdPValType);
    AddPropName("supertype",   TypePName,     IdPValType);
    AddPropName("attributes",  TypePName,     AllPValTypes);

    /* install PERMISSION property names */
    AddPropName("perm-name",   PermPName,     IdPValType);
    AddPropName("domain",      PermPName,     IdListPValType);
    AddPropName("range",       PermPName,     IdListPValType);
    AddPropName("synonyms",    PermPName,     IdListPValType);
}

Boolean Extract(p_tree,pict)
  Entry *p_tree;
  Pict *pict;
{
    InitExtract(MAX_PNAMES);
    ExtractBoxEntries(p_tree,pict);
    ExtractOtherEntries(p_tree,pict);
    return(MakeBoolean(parse_error_cnt > 0));
}
