/* BOX-TYPE.C

   Module for creating BoxType objects.

   $Header: box-type.c,v 1.1 91/08/01 20:03:01 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "error.h"

#include "box-type.h"
#include "elts.h"
#include "id-table.h"
#include "pict.h"

/* LOCAL DEBUGGING FUNCTIONS ============================================== */

#ifdef OLD_DEBUG

static void DisplayTypeTree(tl)
  BoxTypeList *tl;
{
    fprintf(stderr,"\nBox Type Tree:\n");
    StepLinkedList(tl,tl) {
	fprintf(stderr,"  Parent('%s') = '%s'\n",tl->bt->name,
		(tl->bt->parent ? tl->bt->parent->name : "NONE"));
    }
}
#endif OLD_DEBUG

/* LOCAL FUNCTIONS ========================================================= */

static void SetRecursiveRole(bt,role)
  BoxType *bt;
  BoxRole role;
{
    BoxTypeList *curr;

    bt->role = role;
    StepLinkedList(curr,bt->children) {
	SetRecursiveRole(curr->bt,role);
    }
}

static Boolean SetRole(role,pict)
  BoxRole role;
  Pict *pict;
{
    TableEntry *entry;
    String role_name = (role == SubjectRole) ? "subject" : "object";

    if ((entry=FindTableId(pict->table,BoxTypeId,role_name)) == NULL) {
	ParseErrorS(-1,"no box-type name '%s'",role_name);
	return(True);
    }
    SetRecursiveRole(entry->u.box_type,role);
    return(False);
}

/* GLOBAL FUNCTIONS ======================================================== */

BoxType *NewBoxType(line_no)
  int line_no;
{
    BoxType *result = AllocOne(BoxType);
    result->attr_head = (AttrList *)NULL;
    result->line_no = line_no;
    result->parent = (BoxType *)NULL;
    result->children = (BoxTypeList *)NULL;
    return(result);
}

Boolean BuildTypeTree(pict)
  Pict *pict;
{
    BoxType *bt;
    BoxTypeList *curr,*last,*child_entry;
    AttrList **attr;
    TableEntry *entry;
    Boolean error_flag = False;

    /* reverse the list pict->box_types so error messages are generated in
       order of increasing line number */
    curr = pict->box_types;
    pict->box_types = (BoxTypeList *)NULL;
    while (curr != (BoxTypeList *)NULL) {
	last = curr;
	curr = curr->next;
	last->next = pict->box_types;
	pict->box_types = last;
    }

    /* make each type point to its supertype (and children) */
    StepLinkedList(curr,pict->box_types) {
	bt = curr->bt;
	if (bt->supertype != (String)NULL) {
	    /* set bt->parent and children fields */
	    if ((entry=FindTableId(pict->table,BoxTypeId,bt->supertype))
		== (TableEntry *)NULL) {
		ParseErrorS(bt->line_no,"supertype '%s' does not exist",
			    bt->supertype);
		error_flag = True;
	    } else {
		/*
		 * point to parent (found in table 'entry' above) */
		bt->parent = entry->u.box_type;
		/*
		 * give parent a child entry back to this type */
		child_entry = AllocOne(BoxTypeList);
		child_entry->bt = bt;
		child_entry->next = entry->u.box_type->children;
		entry->u.box_type->children = child_entry;
		/*
		 * make end of attribute list point to parent's list */
		for (attr = &(bt->attr_head); /* move to end of list */
		     *attr != (AttrList *)NULL;
		     attr = &((*attr)->next))
		    /* empty loop */;
		*attr = bt->parent->attr_head;
	    }
	}
    }

    /* set the 'role' of each BoxType */
    if (SetRole(SubjectRole,pict)) error_flag = True;
    if (SetRole(ObjectRole,pict)) error_flag = True;
#ifdef OLD_DEBUG
    DisplayTypeTree(pict->box_types);
#endif OLD_DEBUG
    return(error_flag);
}
