/* AMBIG.C

   Compute semantic relations between subjects and objects.

   $Header: ambig.c,v 1.7 91/10/28 11:45:25 heydon Exp $

   Written for the Miro project at Carnegie Mellon by Allan Heydon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "sysname-types.h"
#include "box-type.h"
#include "elts.h"
#include "pict.h"
#include "ambig.h"
#include "rels.h"
#include "rels.g"
#include "top.g"

/* MACROS ================================================================== */

/* indices for array sides */
#define SUBJ ((int)SubjectRole)
#define OBJ ((int)ObjectRole)

/* LOCAL VARIABLES ========================================================= */

/* The following variables are used as private globals because they would
   otherwise be passed down a call stack unchanged. This way, we don't waste
   time copying the same sets of values all the way down the call stack.
*/
static int SubjCnt;		/* number of subject boxes */
static Box **SubjBoxes;		/* array of subject boxes by level */
static int SubjLevels;		/* index of top-most non-super-box level */
static ASysnameTable ArrowTbl;	/* sysname hash table of arrows */
static Boolean AtomOutput;	/* print only atom pairs? */
static PermSet CurrPerm;	/* current permission being checked */
static BoxTypeSet DomainTypes;	/* box types in domain of current perm */
static BoxTypeSet RangeTypes;	/* box types in range of current perm */
static BoolVector BoolVect;	/* vector of permission values */

/* LOCAL DEBUGGING FUNCTIONS =============================================== */

#ifdef DEBUG
#define MAX_COLS 12

static void DisplayBoxLevels(side,pict)
  int side;
  Pict *pict;
{
    int i,col;
    Box *curr;

    fprintf(stderr,"\n%s Boxes:\n",(side==SUBJ) ? "Subject" : "Object");
    StepIndex(i,0,pict->s[side].levels) {
	fprintf(stderr,"  Level %d:",i);
	col = 0;
	StepLinkedList(curr,pict->s[side].u.level[i]) {
	    if (col == 0) fputs("\n    ",stderr);
	    fprintf(stderr,"%5d ",curr->sysname);
	    col = (col + 1) % MAX_COLS;
	}
	fputc('\n',stderr);
    }
}
#endif DEBUG

/* LOCAL FUNCTIONS ========================================================= */

static void AddSuperBox(side,pict)
  int side;
  Pict *pict;
{
    Box *super = NewBox();
    register Box *curr;

    /* add the new box to the front of the list */
    super->sysname = SUPER_BOX_SYSNAME;
    SpliceIntoList(pict->s[side].u.boxes,super);
    pict->s[side].cnt++;

    /* assert that roots are contained in this box */
    StepLinkedList(curr,super->next) {
	if (curr->parents == NULL) { AssertBoxContainment(super,curr); }
    }
}

static Arrow *AddDefaultArrow(pict)
  Pict *pict;
{
    FullArrow a;
    ASysname a_sysname;

    /* fill in the arrow */
    a.from = pict->s[SUBJ].u.boxes;
    a.to = pict->s[OBJ].u.boxes;
    a.parity = Neg;
    a.perms = pict->perms - 1;	/* all permissions */

    /* add it to the picture */
    a_sysname = CantorPair(a.from->sysname,a.to->sysname);
    return(AddArrowSysname(pict->a_sysnames,a_sysname,&a));
}

static Arrow *AddDefault(pict)
  Pict *pict;
/* Creates the super Subject and Object boxes, and then negative default arrow
   between them. Returns the new default arrow created.
*/
{
    /* add super boxes to each side of the picture */
    AddSuperBox(SUBJ,pict);
    AddSuperBox(OBJ,pict);

    /* add default arrow */
    return(AddDefaultArrow(pict));
}

static void ResetIncidence(side,pict)
  int side;
  Pict *pict;
/* Consider each (non-super) box on side 'side' of the picture 'pict'. Turn on
   all bits in the 'a_incidence' field for each box having > 1 parent.

   Pre-condition: All boxes (except the super box -- the first box appearing
   in the list 'pict->s[side].u.boxes) must have at least 1 parent.
*/
{
    register Box *curr;
    StepLinkedList(curr,pict->s[side].u.boxes->next) {
	if (curr->parents->next != (BoxList *)NULL) {
	    curr->a_incidence = ALL_PERMS;
	}
    }
}

static void FreeBoxList(list)
  BoxList *list;
/* Deallocates the space used by the BoxList structures in 'list' (but *not*
   the space of the actual Boxes they point to).
*/
{
    BoxList *curr;
    while (list != NULL) {
	curr = list;
	list = curr->next;
	Dealloc(curr);
    }
}

static void AssignIndices(side,pict)
  int side;
  Pict *pict;
/* Assigns indices to the boxes 'pict->s[side].u.boxes' in the range
   [0,pict->s[side].cnt). This routine also has the side-effect of removing
   and deallocating the space used by the 'u.children' field of each Box on
   side 'side'.
*/
{
    register int i=0;
    register Box *curr;

    StepLinkedList(curr,pict->s[side].u.boxes) {
	FreeBoxList(curr->u.children);
	curr->u.index = i++;
    }
}

static int AssignLevel(b)
  Box *b;
/* Returns the level of 'b', where the level of a box is defined to be 1 plus
   the maximum levels of its children, or 0 if the box is a leaf.

   This routine also checks for cycles in the containment graph. If it detects
   one, it prints an error to stderr and aborts.
*/
{
    int max = -1,child;
    BoxList *curr;

    if (b->level >= 0) { return(b->level); } /* base case; already known */
    if (b->level == SCANNING_LEVEL) {
	fprintf(stderr,"%s: cycle detected in containment graph at box %u\n",
		Argv0,b->sysname);
	exit(ERROR_EXIT);
    }
    b->level = SCANNING_LEVEL;
    StepLinkedList(curr,b->u.children) {
	if ((child=AssignLevel(curr->b)) > max) { max = child; }
    }
    return(b->level=(max+1));
}

static void AssignLevels(side,pict)
  int side;
  Pict *pict;
/* Assigns a level to each box in 'pict' (i.e., fills in the 'level' field of
   each Box).
*/
{
    /* recursively assign levels to the boxes */
    pict->s[side].levels = AssignLevel(pict->s[side].u.boxes) + 1;
}

static void FormBoxLevelArray(side,pict)
  int side;
  INOUT Pict *pict;
/* Fills in the fields 'pict->s[side].levels' and 'pict->s[side].u.level'. The
   latter is constructed as an array of boxes indexed by level (i.e., all
   boxes in pict->s[side].u.level[i]' have 'level' == 'i').

   IMPLEMENTATION NOTE: The old lists of boxes are destroyed; the new lists by
   levels are formed simply by manipulating the 'next' pointers of the Boxes.
*/
{
    int i;
    int num_levels;		/* total number of levels for 'side' */
    Box **level;		/* array of boxes indexed by level */
    register Box *curr,*head;

    /* build levels array */
    head = pict->s[side].u.boxes;
    num_levels = pict->s[side].levels;
    level = pict->s[side].u.level = AllocPtrArray(Box,num_levels);
    StepIndex(i,0,num_levels) { level[i] = NULL; }
    while (head != NULL) {
	curr = head; head = head->next;
	SpliceIntoList(level[curr->level],curr);
    }
#ifdef OLD_DEBUG
    DisplayBoxLevels(side,pict);
#endif OLD_DEBUG
}

static void ComputeWSet(s,o,o_grp)
  Box *s,*o;			/* subject, object boxes */
  INOUT WSetGrp o_grp;		/* current WSetGrp of WBlock of 'o' */
/* Fill in the proper array entry of 'o_grp' for the boxes 's' and 'o'
   (namely, the entry indexed by 's->u.index'). In the computatation,
   witness sets for 'o' and the parents of 's' may be read from 'o_grp' (these
   are guaranteed to have been previously computed). Also, the witness sets
   for 's' and the parents of 'o' are guaranteed to exist in the 'blk' field
   of the Box structures in the list 'o->parents'.
*/
{
    int s_index = s->u.index;
    ASysname a_sysname;
    Arrow *a;
    WSet *new_set,*s_parents,*o_parents,*temp_set;
    BoxList *parent;

    /* compute 'new_set' witness set */
    a_sysname = CantorPair(s->sysname,o->sysname);
    if ((a=FindArrowSysname(ArrowTbl,a_sysname,CurrPerm)) != NULL) {
	/* witness set determined directly from arrow */
	new_set = FindOrAddWSet(NewWSetFromArrow(a,CurrPerm));
#ifdef MEASURE
	WSetFromArrowCnt++;
#endif MEASURE
    } else if (!(s->a_incidence & CurrPerm)) {
	/* 's' has 1 parent and no CurrPerm arrows: copy WSet from parent */
	new_set = o_grp[s->parents->b->u.index];
#ifdef MEASURE
	WSetInheritedCnt++;
#endif MEASURE
    } else {
	/* compute min of W(Parents(s),o) sets */
	parent = s->parents; s_parents = o_grp[parent->b->u.index];
	StepLinkedList(parent,parent->next) {
	    temp_set = o_grp[parent->b->u.index];
	    s_parents = FastMinWSet(s_parents,temp_set);
	}
	/* compute min of W(s,Parents(o)) sets */
	parent = o->parents; o_parents = parent->b->blk->w_set_grp[s_index];
	StepLinkedList(parent,parent->next) {
	    temp_set = parent->b->blk->w_set_grp[s_index];
	    o_parents = FastMinWSet(o_parents,temp_set);
	}
	/* compute overall result */
	new_set = FastMinWSet(s_parents,o_parents);
#ifdef MEASURE
	WSetComputedCnt++;
#endif MEASURE
    }
    o_grp[s_index] = CopyWSet(new_set);
}

static WBlock *ComputeWSets(obj)
  Box *obj;
/* Create, compute, and return the WBlock for the object Box 'obj'. The 'blk'
   fields for each of the boxes in the list 'obj->parents' are guaranteed to
   have been computed.
*/
{
    int i;
    Box *curr;
    WBlock *result = NewWBlock(SubjCnt,(WSet *)NULL);

    /* compute WSet's top-down by level */
    for (i=SubjLevels; i >= 0; i--) {
	StepLinkedList(curr,SubjBoxes[i]) {
	    ComputeWSet(curr,obj,result->w_set_grp);
	}
    }
    return(result);
}

static WBlock *ComputeWBlock(obj)
  Box *obj;
/* Compute and return the WBlock for the object Box 'obj'. Since the
   computation of this block depends on the blocks associated with the parents
   of 'objs', this call may invoke recursive computations to compute the
   WBlocks for ancestors of 'obj'.
*/
{
    BoxList *parent;
    Box *b;
    WBlock *blk;

    /* make sure WBlocks have been computed for all parents */
    StepLinkedList(parent,obj->parents) {
	if (parent->b->blk == UnvisitedBlk) {
	    b = parent->b;
	    b->blk = ComputeWBlock(b);
	}
    }

    /* decide whether to copy or compute the current block */
    if (!(obj->a_incidence & CurrPerm)) {
	/* 'obj' has 1 parent and no CurrPerm arrows: copy from parent */
#ifdef MEASURE
	WBlkInheritedCnt++;
#endif MEASURE
	blk = obj->parents->b->blk;
	return(CopyWBlock(blk));
    } else {
	/* return the WBlock values computed for each subject */
#ifdef MEASURE
	WBlkComputedCnt++;
#endif MEASURE
	return(ComputeWSets(obj));
    }
}

static void PrintBlockRelations(o)
  Box *o;
/* IMPLEMENTATION NOTE: We repeat a great deal of code here to minimize the
   number of tests within loops.
*/
{
    int i,s_index;
    register Box *s;
    WSetGrp o_grp;
    WSet *w_set;

    /* print result if both boxes atomic and if necessary */
    if (AtomOutput) {
	if (o->level == 0) {
	    o_grp = o->blk->w_set_grp;
	    if (PrintCompactMatrix) {
	      StepLinkedList(s,SubjBoxes[0]) {	/* loop over level 0 subj's */
		  w_set = o_grp[(s_index=s->u.index)];
		  BoolVect[s_index] = (Boolean)PosWSet(w_set);
		  if (AmbigWSet(w_set)) {
		      (void)DisplayParity(stderr,w_set,s->sysname,o->sysname);
		  }
	      }
	      fprintf(OutFile,"(obj %d %d '%s)\n",o->sysname,
		      FindOrAddAccess(BoolVect),PermName);
	    } else if (PrintFullMatrix) {
	      StepLinkedList(s,SubjBoxes[0]) {	/* loop over level 0 subj's */
		  w_set = o_grp[s->u.index];
		  (void)DisplayParity(OutFile,w_set,s->sysname,o->sysname);
	      }
	    } else {
	      StepLinkedList(s,SubjBoxes[0]) {	/* loop over level 0 subj's */
		  w_set = o_grp[s->u.index];
		  if (AmbigWSet(w_set)) {
		      (void)DisplayParity(stderr,w_set,s->sysname,o->sysname);
		  }
	      }
	    }
	}
  } else {
	if (o->box_type_val & RangeTypes) {
	    o_grp = o->blk->w_set_grp;
	    if (PrintCompactMatrix) {
	      StepIndex(i,0,SubjLevels) {	   /* loop over all levels */
	      StepLinkedList(s,SubjBoxes[i]) {     /* loop on level i subj's */
	      if (s->box_type_val & DomainTypes) { /* subj right type? */
		  w_set = o_grp[(s_index=s->u.index)];
		  BoolVect[s_index] = (Boolean)PosWSet(w_set);
		  if (AmbigWSet(w_set)) {
		      (void)DisplayParity(stderr,w_set,s->sysname,o->sysname);
		  }
	      }
	      }
	      }
	      fprintf(OutFile,"(obj %d %d '%s)\n",o->sysname,
		      FindOrAddAccess(BoolVect),PermName);
	    } else if (PrintFullMatrix) {
	      StepIndex(i,0,SubjLevels) {	   /* loop over all levels */
	      StepLinkedList(s,SubjBoxes[i]) {     /* loop on level i subj's */
	      if (s->box_type_val & DomainTypes) { /* subj right type? */
		  w_set = o_grp[s->u.index];
		  (void)DisplayParity(OutFile,w_set,s->sysname,o->sysname);
	      }
	      }
	      }
	    } else {
	      StepIndex(i,0,SubjLevels) {	   /* loop over all levels */
	      StepLinkedList(s,SubjBoxes[i]) {     /* loop on level i subj's */
	      if (s->box_type_val & DomainTypes) { /* subj right type? */
		  w_set = o_grp[s->u.index];
		  if (AmbigWSet(w_set)) {
		      (void)DisplayParity(stderr,w_set,s->sysname,o->sysname);
		  }
	      }
	      }
	      }
	    }
	}
    }
}

static void ComputeWBlocks(obj)
  Box *obj;
/* Compute the WBlocks for the object Box 'obj' and all of its *descendants*
   in the containment DAG. Since the computation for 'obj' depends on the
   blocks for the parents of 'obj', this call may invoke computations of
   WBlocks for some of the *ancestors* of 'objs' as well.

   If necessary, the relations between the objs and all (relevant) subjects
   are printed (namely, if PrintFullMatrix is True or if the relation between
   the boxes is ambiguous).
*/
{
    BoxList *child;

    /* test if the block for this box has already been computed */
    if (obj->blk == DeletedBlk) return;

    /* compute the block for this object box if necessary */
    if (obj->blk == UnvisitedBlk) {
	obj->blk = ComputeWBlock(obj);
	PrintBlockRelations(obj);
    }

    /* compute the blocks for each of this box's children */
    StepLinkedList(child,obj->u.children) {
	ComputeWBlocks(child->b);
    }

    /* free the space allocated for the WBlock of 'obj' */
    FreeWBlock(obj->blk,SubjCnt); /* line (A) (referenced below) */
    obj->blk = DeletedBlk;
}

void InitBoolVector(size,vector)
  int size;
  INOUT BoolVector vector;
{
    register int i;
    StepIndex(i,0,size) { vector[i] = False;}
}

void DisplaySubjects(box)
  Box *box;
{
    StepInitializedLinkedList(box) {
	fprintf(OutFile,"(subj %d %d)\n",box->sysname,box->u.index);
    }
}

/* GLOBAL FUNCTIONS ======================================================= */

Options *NewOptions()
{
    Options *result = AllocOne(Options);
    result->fp = stdout;
    result->matrix = False;
    result->full_matrix = False;
    result->lisp_output = False;
    result->atom_output = False;
    result->perm_list = (PermList *)NULL;
    return(result);
}

int CheckAmbig(options,pict)
  Options *options;
  Pict *pict;
{
    PermList *curr_perm;	/* current permission being checked */
    TableEntry *tbl;		/* hash table entry for that permission */
    Arrow *default_arrow;	/* the "default" arrow itself */
    Box *super_object;		/* the "super" Object box */
    BoxList *child;		/* current child of 'super_object' */
    int vect_sz;		/* size of each boolean vector */
    register Box *curr_box;

    /* set rels.c global variables */
    OutFile = options->fp;
    PrintCompactMatrix = options->matrix;
    PrintFullMatrix = options->full_matrix;
    PrintLisp = options->lisp_output;
    AmbigEntryCnt = 0;
    WSetTbl = NewWSetTable();

    /* add super boxes and default arrow to picture */
    default_arrow = AddDefault(pict);

    /* assign levels to each side of the picture */
    AssignLevels(SUBJ,pict);
    AssignLevels(OBJ,pict);

    /* reset 'a_incidence' field to 0 if necessary */
    ResetIncidence(SUBJ,pict);
    ResetIncidence(OBJ,pict);

    /* assign indices (and free children structures) for Subjects */
    AssignIndices(SUBJ,pict);
    if (PrintCompactMatrix) { DisplaySubjects(pict->s[SUBJ].u.boxes); }
    FormBoxLevelArray(SUBJ,pict);

    /* set private global variables */
    SubjCnt = pict->s[SUBJ].cnt;           /* used as size of all WBlock's */
    SubjBoxes = pict->s[SUBJ].u.level;     /* array of subj boxes by level */
    SubjLevels = pict->s[SUBJ].levels - 2; /* subtract 2 to skip super-box */
    ArrowTbl = pict->a_sysnames;
    AtomOutput = options->atom_output;

    /* get super Object box */
    super_object = pict->s[OBJ].u.boxes;

    /* form WBlock for super Object box & 'DefaultSet' */
    DefaultSet = FindOrAddWSet(NewWSetFromArrow(default_arrow,ALL_PERMS));
    super_object->blk = NewWBlock(SubjCnt,DefaultSet);

    /* initialize overrides cache of elts.c module */
    InitOverridesCache();

    /* initialize the AccessTable and BoolVect if necessary */
    if (PrintCompactMatrix) {
	vect_sz = InitAccessTable(SubjCnt);
	BoolVect = AllocArray(Boolean,vect_sz);
    }

    /* compute access matrices by permission */
    if (options->perm_list == NULL) { options->perm_list = pict->perm_list; }
    StepLinkedList(curr_perm,options->perm_list) {
	/*
	 * find current permission and initialize private permission globals */
	tbl = FindTableId(pict->table,PermNameId,curr_perm->perm_name);
	if (tbl == NULL) {
	    fprintf(stderr,"%s: unknown permission %s\n",
		    Argv0,curr_perm->perm_name);
	    exit(ERROR_EXIT);
	}
	PermName = curr_perm->perm_name;
	CurrPerm = tbl->u.perm_val->perm_set;
	DomainTypes = tbl->u.perm_val->domain;
	RangeTypes = tbl->u.perm_val->range;
	if (PrintCompactMatrix) { InitBoolVector(vect_sz,BoolVect); }
	/*
	 * compute the relations for this permission: iterate over children */
	StepLinkedList(child,super_object->u.children) {
	    ComputeWBlocks(child->b);
	}
	/* NOTE: At this point, all WBlock's will have been freed on the
	 * upward unravelling of recursive calls in line (A) of
	 * ComputeWBlocks(). However, we still have to reset the 'blk' fields
	 * of each Object box to UnvisitedBlk */
	if (curr_perm->next != NULL) {
	    StepLinkedList(curr_box,super_object->next) {
		Assert(curr_box->blk == DeletedBlk);
		curr_box->blk = UnvisitedBlk;
	    }
	}
    }
#ifdef MEASURE
    FreeWBlock(super_object->blk,SubjCnt);
#endif MEASURE
    if (PrintCompactMatrix) {
	DisplayAccessBlocks(SubjCnt);
    }

#ifdef MEASURE
    fprintf(stderr,"\nComputeWBlock() Performance:\n");
    fprintf(stderr,"  Inherited =   %7d (%d each)\n",
	    WBlkInheritedCnt,SubjCnt-1);
    fprintf(stderr,"  Computed =    %7d\n",WBlkComputedCnt);
    fprintf(stderr,"  Total Comp. = %7d\n",(SubjCnt-1)*WBlkComputedCnt);
    fprintf(stderr,"\nComputeWSet() Performance:\n");
    fprintf(stderr,"  From Arrow =  %7d\n",WSetFromArrowCnt);
    fprintf(stderr,"  Inherited =   %7d\n",WSetInheritedCnt);
    fprintf(stderr,"  Computed =    %7d\n",WSetComputedCnt);
    fprintf(stderr,"  Total =       %7d\n",
	    WSetFromArrowCnt + WSetInheritedCnt + WSetComputedCnt);
    fprintf(stderr,"\nFastMinWSet() Performance:\n");
    fprintf(stderr,"  Equal =       %7d\n",EqSets);
    fprintf(stderr,"  Deflt Arrow = %7d\n",FirstSetDef+SecondSetDef);
    fprintf(stderr,"  Computed =    %7d\n",MinWSetCnt);
    fprintf(stderr,"\nWSetTable Cache Performance:\n");
    fprintf(stderr,"  Hits =        %7d\n",WSetHitCnt);
    fprintf(stderr,"  Misses =      %7d\n",WSetMissCnt);
    fprintf(stderr,"\nFastArrowOverrides() Performance:\n");
    fprintf(stderr,"  Eql Arrows =  %7d\n",EqlArrowCnt);
    fprintf(stderr,"  Bad Levels =  %7d\n",WrongLevelCnt);
    fprintf(stderr,"  Neither    =  %7d\n",ArrowPairHitCnt+ArrowCompareCnt);
    fprintf(stderr,"\nArrowOverrides() Performance:\n");
    fprintf(stderr,"  Hash Hits =   %7d\n",ArrowPairHitCnt);
    fprintf(stderr,"  Computed =    %7d\n",ArrowCompareCnt);
    fprintf(stderr,"\nAncestorBox() Performance:\n");
    fprintf(stderr,"  Total calls = %7d\n",AncestorBoxCallCnt);
    fprintf(stderr,"  Equal boxes = %7d\n",EqlBoxesCnt);
    fprintf(stderr,"  Wrong level = %7d\n",UneqlLevelsCnt);
#endif MEASURE
    return(AmbigEntryCnt);
}
