# iff2ipql.awk
#
#/*****************************************************************************
#                Copyright Carnegie Mellon University 1992
#
#                      All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of CMU not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
# CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
# ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
# WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
# ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
# SOFTWARE.
#*****************************************************************************/
#
#
# Gawk(1) program to generate IPQL commands to add objects in an IFF file.
#
# $Header: iff2ipql.awk,v 1.13 91/11/21 01:50:12 heydon Exp $
#
# Written by Allan Heydon for the Miro project at Carnegie Mellon
#
# SYNTAX
#   gawk -f iff2ipql.awk path=pathname nonatoms IFFfile
#
# SYNOPSIS
#   Generate IPQL commands for adding boxes, syntax arrows, and direct
#   containment arrows to the IPQL data-structures. The file 'nonatoms' should
#   contain a list of sysnames, one per line, of nonatomic boxes in 'IFFfile'.
#   The output lines have the form:
#
#   (box <sysname> "<name>" <type> <atomic?> &rest <args>)
#     A box with sysname <sysname>, name <name>, and type <type>. The
#     <atomic?> value should be either t or nil.
#
#   (syn <from> <to> '<perm> <positive?>)
#     A syntax arrow from the box with sysname <from> to the box with sysname
#     <to> having permission <perm> and parity <positive?> (which should be
#     either t or nil).
#
#   (con <from> <to> <direct?>)
#     A containment arrow from the box with sysname <from> to the box with
#     sysname <to> having permission <perm>. The <direct?> value should be
#     either t or nil.
#
# These commands are written to the files path".box.cl", path".syn.cl", and
# path".con.cl", respectively. Thus, the value passed for the variable 'path'
# should be the relative path of where the file should be placed *plus* the
# prefix of the filename to use.
#
# BUGS
#   There are restrictions on the input iff file.
#
# FILES
#   ~miro/bin/iff2ipql		script using this awk program
#
# SEE ALSO
#   fs2iff(1), iff2ciff(1), iff2ipql(1)

# WRITE-HEADER FUNCTIONS ======================================================

function write_header(outfile) {
  print "(require 'objs)" > outfile;
  print "(require 'ipql)" > outfile;
  print "(use-package 'ipql)" > outfile;
  print "" > outfile;
}

# BEGIN =======================================================================

BEGIN {
  # read nonatomic box sysnames
  while (getline nonatom > 0) {
    nonatoms[nonatom] = 1;
  }

  # set field separator so "=", ";" separate fields
  FS = "[ \t]*[;=][ \t]*"

  # define separator character for lists
  PERM_SEP_CHAR = "[ \t]*,[ \t]*"

  # initialize output file names to empty
  boxfile = "";
  synfile = "";
  confile = "";
}

# BOX =========================================================================

/^[ \t]*>[ \t]*BOX/ {
  if (boxfile == "") {
    boxfile = path "-box.cl";
    printf("(in-package 'constraints)\n\n") > boxfile
    write_header(boxfile);
  }
  name = "\"\"";
  rest = "";
  match($1,"^[ \t]*>[ \t]*BOX[ \t]*");
  $1 = substr($1,RLENGTH+1);
  for (i=1; i < NF; i = i+2) {
         if ($i == "type")    { type = $(i+1); }
    else if ($i == "sysname") { sysname = $(i+1); }
    else if ($i == "name")    { name = $(i+1); }
    else if ($i != "loc" && $i != "size" && $i != "role") {
      value = $(i+1);
      # handle boolean values as special cases
      if (value == "true") { value = "t"; }
      else if (value == "false") { value = "nil"; }
      # convert identifiers to strings
      else if (value ~ /^[a-zA-Z].*$/) value = "\"" value "\""
      rest = " :" $i " " value rest;
    }
  }
  printf("(box %d %s '%s %s%s)\n",sysname,name,type,
         ((sysname in nonatoms)?"nil":"t"),rest) > boxfile;
  next;
}

# ARROW =======================================================================

/^[ \t]*>[ \t]*ARROW/ {
  if (synfile == "") {
    synfile = path "-syn.cl";
    write_header(synfile);
  }
  match($1,"^[ \t]*>[ \t]*ARROW[ \t]*");
  $1 = substr($1,RLENGTH+1);
  for (i=1; i < NF; i = i+2) {
    if      ($i == "from")        { from = $(i+1);    }
    else if ($i == "to")          { to = $(i+1);      }
    else if ($i == "parity")      { parity = $(i+1);  }
    else if ($i == "permissions") { permset = $(i+1); }
  }
  if (substr(permset,1,1) == "{" && substr(permset,length(permset)) == "}") {
    permset = substr(permset,2,length(permset)-2);
    perm_cnt = split(permset,perms,PERM_SEP_CHAR);
    for (i=1; i<=perm_cnt; i++) {
      printf("(syn %d %d '%s %s)\n",from,to,perms[i],
             (parity=="pos")?"t":"nil") > synfile;
    }
  } else {
    print "Permissions string '" permset "' of improper form.";
    exit(1);
  }
  next;
}

# INSIDE ======================================================================

/^[ \t]*>[ \t]*INSIDE/ {
  if (confile == "") {
    confile = path "-con.cl";
    write_header(confile);
  }
  match($1,"^[ \t]*>[ \t]*INSIDE[ \t]*");
  $1 = substr($1,RLENGTH+1);
  for (i=1; i < NF; i = i+2) {
    if ($i == "parent")        { pindex = $(i+1); }
    else if ($i == "children") { clist = $(i+1);  }
  }
  if (substr(clist,1,1) == "{" && substr(clist,length(clist)) == "}") {
    clist = substr(clist,2,length(clist)-2);
    child_cnt = split(clist,children,",");
    for (i=1; i <= child_cnt; i++) {
      printf("(con %d %d t)\n", children[i], pindex) > confile;
    }
  } else {
    print "Children string '" clist "' of improper form.";
    exit(1);
  }
  next;
}
