:
#! /bin/sh
#
# This program runs an audit.  The scheme used is:
#	checksum file list in host/Cmap{filesys}
#	master file list in host/Fmap(filesys)
#	ignore file list in host/Imap(filesys)
# where map(filesys) is the file system name with all '/'
# deleted.  Put an extra hex number after 'C', 'F', 'I' if
# setuid, setgid, device, or any combo.  The default file system
# list is defined
# below as the value of BINLIST; if a file List is found
# in the directory containing the master lists, the contents
# of that file are used as the list of file systems; if file
# systems are named on the command line, those are used.  The
# options -a, -d, -g, -ga, -sg, -su, and -u mean the same as
# for audit (qv.)
#
# Options:
#
# -1	do not send letter; write output to stdout
# -a<name1>,<name2>,...
#	Copies of the audit report will be mailed to <name1>, <name2>, ...
# -c	Checksum all audited files
# -d	Do not recurse.
# -g	Generate the file against which future audits are to be compared.
#	No auditing is done.
# -ga	After the audit is complete, replace the master file with the
#	results of the current scan of the file system.
# -h<host>
#	Run the audit for host <host>.  A listing will be generated
#	remotely, moved to the local host, and the analysis done locally.
# -n<type>
#	If any of the file systems to be scanned are mounted on a volume
#	of type <type>, do not perform the scan.
# -o<type>
#	If any of the file systems to be scanned are not mounted on a volume
#	of type <type>, do not perform the scan.  His is the inverse of -n.
# -q	uiet mode; don't say anything if there are no changes
# -r<file>
#	If <file> is too old (as specified in the makefile), delete it;
#	otherwise, ignore changes/additions/deletions to the files in it
# -sb	Audit block special device files only.
# -sc	Audit character special device files only.
# -sG	Audit setgid non-directory files only (necessary hack for Suns running
#	SunOS 4)
# -sg	Audit setgid files only.
# -su	Audit setuid files only.
# -t<sec>
#	Time out interval.
# -u	With this option, every user who owns a file that has been added,
#	deleted, or changed will be mailed a message describing the change.
#	This message looks like an audit report.  If a file's ownership
#	changes, both the new and old owner will be told about it.
# -v	Debug mode; print shell input lines as they are read
# -x	Debug mode; print commands and their arguments as they are executed
# 
# Matt Bishop
# Research Institute for Advanced Computer Science
# NASA Ames Research Center
# Moffett Field, CA  94035
#
# mab@riacs.edu, ...!{decvax!decwrl,ihnp4!ames}!riacs!mab
#
# (c) Copyright 1986 by Matt Bishop and the Research Institute for
#			Advanced Computer Science
#
# version number
VERSION="RIACS Audit Package version 3.1.3 Tue May 19 12:59:43 PDT 1992 (Matt.Bishop@dartmouth.edu)"
#
#	system and shell variables
PATH=/usr/ucb:/bin:/usr/bin:%%DESTBIN%%	# execution path
export PATH				# for subprograms
#	relevant programs
MAILER=%%MAILER%%		# program to send mail
SUBJECT=%%SUBJECT%%		# yes if mailer supports command-line subject
NOTIFY=%%AUDITOR%%		# who gets error messages
AUDIT=audit			# audit program
#	relevant files
NOCHANGE=%%TMPDIR%%/Ac1$$0	# temp file for change message
#	setup for host
HOST=`hostname | sed 's/\..*//'` # name of the current host
HOSTROOT=%%LISTDIR%%		# all host directories in this directory
HOSTMAP=$HOSTROOT/Equiv		# file to normalize host names
#	prefixes for various file types
MASTERP=F			# prefix for master file
CHKSUMP=C			# prefix for checksum file
IGNOREP=I			# prefix for ignore file
PATTRNP=P			# prefix for pattern master file
TMPMASP=T			# prefix for temporary master list
TMPIGN=%%LISTDIR%%/Reject	# name of reject file
#	options to be passed to audit program
ONEOUTPUT=no			# output to stdout, not letter
OPTIONS=			# options to be passed to audit program
QUIET=no			# quiet mode
DEBUG=				# in debug mode errors go to output!
SETBLK=no			# audit block specials only
SETCHR=no			# audit character specials only
SETGID=no			# audit setgid files only
SETUID=no			# audit setuid files only
GAOPT=no			# -ga not given
GOPT=no				# -g not given
#	default list of directories to be audited
BINLIST="/bin /usr/bin"		# list of directories to be audited
#
# on interrupt, do some clean up and quit
#
trap "rm -f $NOCHANGE; exit 1" 1 2 3 15
#
# parse the command line
#
FLAG=				# option waiting for argument
PPRE=
LIST=
CHECKSUMALL=no
COMMAND=$0
NUMBER=0
CARRY=nothing
for I
do
	COMMAND="$COMMAND $I"
	#
	# if you are expecting an argument, pick it up
	#
	if test -n "$FLAG"
	then
		case $FLAG in
		a)	NOTIFY="$NOTIFY $I"; OPTIONS="$OPTIONS$I";;
		c)	OPTIONS="$OPTIONS$I";;
		h)	HOST=$I; OPTIONS="$OPTIONS$I";;
		r)	TMPIGN="$I";;
		esac
		FLAG=
		continue
	fi
	case $I in
	'&')				# in background
		BACKGROUND=yes;;
	-1)				# output to stdout
		OPTIONS="$OPTIONS $I"
		ONEOUTPUT=yes;;
	-a*)				# new list of auditors
		OPTIONS="$OPTIONS $I";
		CARRY=`expr "$I" : '-a\(.*\)'`;
		if test -z "$CARRY"
		then
			FLAG=a
		else
			NOTIFY=$CARRY
		fi;;
	-c)				# checksum audited files -- TRANSLATE
		CHECKSUMALL=yes
		OPTIONS="$OPTIONS -C";;
	-d)				# recurse -- tack it on
		OPTIONS="$OPTIONS $I";;
	-g)				# generate master file -- tack it on
		GOPT=yes
		OPTIONS="$OPTIONS $I";;
	-ga)				# audit, replace master file -- tack it on
		GAOPT=yes
		OPTIONS="$OPTIONS $I";;
	-h*)				# new host name
		OPTIONS="$OPTIONS $I";
		CARRY=`expr "$I" : '-h\(.*\)'`;
		if test -z "$CARRY"
		then
			FLAG=h
		else
			HOST="$CARRY"
		fi;;
	-n*|-o*)				# volume type
		OPTIONS="$OPTIONS $I";
		CARRY=`expr "$I" : '-[no]\(.*\)'`;
		if test -z "$CARRY"
		then
			FLAG=c
		fi;;
	-q)				# quiet mode
		QUIET=yes;;
	-r*)				# reject file
		OPTIONS="$OPTIONS $I";
		CARRY=`expr "$I" : '-r\(.*\)'`;
		if test -z "$CARRY"
		then
			FLAG=r
		else
			TMPIGN="$CARRY"
		fi;;
	-sb)				# block special files
		if test "$SETBLK" = no
		then
			NUMBER=`expr "$NUMBER" + 8`;
			OPTIONS="$OPTIONS $I";
			SETBLK=yes;
		fi;;
	-sc)				# character special files
		if test "$SETCHR" = no
		then
			NUMBER=`expr "$NUMBER" + 4`;
			OPTIONS="$OPTIONS $I";
			SETCHR=yes;
		fi;;
	-s[gG])				# setgid files
		if test "$SETGID" = no
		then
			NUMBER=`expr "$NUMBER" + 2`;
			OPTIONS="$OPTIONS $I";
			SETGID=yes;
		fi;;
	-su)				# setuid files
		if test "$SETUID" = no
		then
			NUMBER=`expr "$NUMBER" + 1`;
			OPTIONS="$OPTIONS $I";
			SETUID=yes;
		fi;;
	-t*)				# timeout
		OPTIONS="$OPTIONS $I";
		CARRY=`expr "$I" : '-t\(.*\)'`;
		if test -z "$CARRY"
		then
			FLAG=c
		fi;;
	-u)				# notify users -- tack it on
		OPTIONS="$OPTIONS $I";;
	-v)				# verbose debug mode
		DEBUG="${DEBUG}v";
		OPTIONS="$OPTIONS $I";
		set -$DEBUG;;
	-x)				# debug mode
		DEBUG="${DEBUG}x";
		OPTIONS="$OPTIONS $I";
		set -$DEBUG;;
	*)				# new file system -- tack it on
		LIST="$LIST $I";;
	esac;
done
#
# set up the debugging option
#
if test "X$DEBUG" != X
then
	DEBUG="-$DEBUG"
fi
#
#	set up the directories
#
DIRMASTER=$HOSTROOT
DIRIGNORE=$HOSTROOT
DIRENVIRON=$HOSTROOT
#
#	canonicalize the host name
#
if test -n "$HOST"
then
	X=
	if test -r $HOSTMAP
	then
		X=`grep "^$HOST[ 	]" $HOSTMAP | awk '{ print $2 }'`
	fi
	if test -n "$X"
	then
		NUMFIELDS=`echo "$X" | awk -F, '{ print NF }'`
		if test "$NUMFIELDS" -eq 0
		then
			DIRMASTER=$HOSTROOT/$X
			DIRIGNORE=$HOSTROOT/$X
			DIRENVIRON=$HOSTROOT/$X
		elif test "$NUMFIELDS" -eq 1
		then
			X1=`echo "$X" | awk -F, '{ print $1 }'`
			DIRMASTER=$HOSTROOT/$X1
			DIRIGNORE=$HOSTROOT/$X1
			DIRENVIRON=$HOSTROOT/$X1
		elif test "$NUMFIELDS" -eq 2
		then
			X1=`echo "$X" | awk -F, '{ print $1 }'`
			X2=`echo "$X" | awk -F, '{ print $2 }'`
			DIRMASTER=$HOSTROOT/$X1
			DIRIGNORE=$HOSTROOT/$X2
			DIRENVIRON=$HOSTROOT/$X2
		elif test "$NUMFIELDS" -eq 3
		then
			X1=`echo "$X" | awk -F, '{ print $1 }'`
			X2=`echo "$X" | awk -F, '{ print $2 }'`
			X3=`echo "$X" | awk -F, '{ print $3 }'`
			DIRMASTER=$HOSTROOT/$X1
			DIRIGNORE=$HOSTROOT/$X2
			DIRENVIRON=$HOSTROOT/$X3
		elif test "$NUMFIELDS" -ge 4
		then
			X1=`echo "$X" | awk -F, '{ print $1 }'`
			X2=`echo "$X" | awk -F, '{ print $2 }'`
			X3=`echo "$X" | awk -F, '{ print $3 }'`
			X4=`echo "$X" | awk -F, '{ print $4 }'`
			DIRMASTER=$HOSTROOT/$X1
			DIRIGNORE=$HOSTROOT/$X2
			DIRENVIRON=$HOSTROOT/$X4
		fi
	else
		DIRMASTER=$HOSTROOT/$HOST
		DIRIGNORE=$HOSTROOT/$HOST
		DIRENVIRON=$HOSTROOT/$HOST
	fi
fi
#
# set up the prefix, if any
#
if test "$NUMBER" -gt 0
then
   	PPRE=`echo $NUMBER | \
		awk '{ printf "%s\n", substr("123456789ABCDEF", $0, 1); }'`
fi
#
# set up the proper files
#
CHKSUM=$DIRMASTER/$CHKSUMP$PPRE
MASTER=$DIRMASTER/$MASTERP$PPRE
PATTRN=$DIRMASTER/$PATTRNP$PPRE
TMPMAS=$DIRMASTER/$TMPMASP$PPRE
IGNORE=$DIRIGNORE/$IGNOREP$PPRE
ENVIRON=$DIRENVIRON/Environ
#
# now verify the master file directory exists; die if not
#
if test ! -d "$DIRMASTER"
then
	echo "master file directory $DIRMASTER does not exist; aborting" 1>&2
	exit 0
fi
#
# Do the systems named on the command line
#
if test "$LIST"
then
	for DIR in $LIST
	do
		X=`echo $DIR | sed 's;/;;g'`
		if test "$CHECKSUMALL" = no
		then
			CHKSUMFLAG="-c$CHKSUM$X"
		fi
		if test "$BACKGROUND" = yes
		then
			$AUDIT $OPTIONS -r$TMPIGN -e$ENVIRON -m$MASTER$X \
				-p$PATTRN$X $CHKSUMFLAG -z$TMPMAS$X \
					-i$IGNORE$X -f$DIR &
		else
			$AUDIT $OPTIONS -r$TMPIGN -e$ENVIRON -m$MASTER$X \
				-p$PATTRN$X $CHKSUMFLAG -z$TMPMAS$X \
					-i$IGNORE$X -f$DIR
		fi
		if test $? -eq 0 -a "$QUIET" = "no"
		then
			if test "$GAOPT" = yes
			then
				echo "The audit for the file system ${HOST}:$DIR " >> $NOCHANGE
				echo "completed at `date`." >>$NOCHANGE
				echo "No changes were found and no errors occurred." >>$NOCHANGE
				echo "The master file was also updated." >>$NOCHANGE
				SUBJTITLE="audit: no changes to ${HOST}:$DIR"
			elif test "$GOPT" = yes
			then
				echo "A master file for the file system ${HOST}:$DIR " >> $NOCHANGE
				echo "was generated on `date`." >>$NOCHANGE
				echo "No errors occurred." >>$NOCHANGE
				SUBJTITLE="audit: master for ${HOST}:$DIR generated"
			else
				echo "The audit for the file system ${HOST}:$DIR " >> $NOCHANGE
				echo "completed at `date`." >>$NOCHANGE
				echo "No changes were found and no errors occurred." >>$NOCHANGE
				SUBJTITLE="audit: no changes to ${HOST}:$DIR"
			fi
			if test "$ONEOUTPUT" = yes
			then
				echo ' ' ;
				echo "** $SUBJTITLE **"
				echo ' ' ;
				cat $NOCHANGE
			elif test "$SUBJECT" = "yes"
			then
				( echo "$VERSION" ;\
				  echo ' ' ;\
				  cat $NOCHANGE ) | $MAILER -s "$SUBJTITLE" $NOTIFY
			else
				( echo "$VERSION" ;\
				  echo ' ' ;\
				  echo "** $SUBJTITLE **" ;\
				  echo ' ' ;\
				  cat $NOCHANGE ) | $MAILER $NOTIFY
			fi
			rm -f $NOCHANGE
		fi
	done
else
	echo "No file systems specified; nothing done" 1>&2
fi
#
# all done
#
exit 0
