/*
 * This program takes two arguments with an optional third:
 * 	auditpmn [ -m [ .n ] ] pref newfile patfile [ mismatchfile ]
 * It writes on the standard output those lines in newfile for which the
 * first field does not match any of the patterns in the first field of
 * patfile.  If mismatchfile is present and writeable it writes those
 * lines of newfile for which the first field matches at least one first
 * field in patfile but some other field does not; if mismatchfile is not
 * given or is unwriteable, it writes those lines to stdout.
 *   If the option -m is present, auditpnm groups lines in sets of m; if
 * the first field of any line in that set does not match any of the patterns,
 * the entire group of lines is printed.  Similarly, if there is a match
 * for all lines but one or more does not match a pattern, the entire group
 * is printed, with the pattern line following each.
 *
 * The first field of the pattern file uses standard ed(1) syntax;
 * the other fields are strings, and a "*" string matches anything.
 *
 * Matt Bishop
 * Department of Mathematics and Computer Science
 * Dartmouth College
 * Hanover, NH  03755
 *
 * Matt.Bishop@dartmouth.edu, ...!decvax!dartvax!Matt.Bishop
 */
#include <stdio.h>

/*
 * version number
 */
static char *version = "RIACS Audit Package version 3.1.3 Tue May 19 12:59:43 PDT 1992 (Matt.Bishop@dartmouth.edu)";

/*
 * wild card; match any field except the first
 */
#define	WC	"*"
#define MAXGROUP 3

/*
 * file information
 */
char *newfile;		/* name of new file */
char newbuf[BUFSIZ];	/* buffer for line from new file */
FILE *nfp;		/* pointer to new file */
char *patfile;		/* name of pattern file */
char *pref;		/* refix for mismatched files */
int plineno;		/* current line number of pattern file */
char patbuf[BUFSIZ];	/* buffer for line from pattern file */
FILE *pfp;		/* pointer to pattern file */
char *mmfile;		/* name of mismatch file */
char mbuf[BUFSIZ];	/* buffer for line from pattern file */
FILE *mfp;		/* pointer to pattern file */
char group[MAXGROUP][BUFSIZ];	/* buffers for grouping */
int startpt;		/* start at this character */
int groupsz = 1;	/* size of group */

/*
 * program name
 */
char *progname;		/* name of program */

/*
 * forward references to internal functions
 */
char *getfield();	/* get next field */

/*
 * forward references to library functions
 */
extern int strcmp();	/* string compare */
extern char *strcpy();	/* string copy */

main(argc, argv)
int argc;
char **argv;
{
	int succ;		/* 1 when a match is made */
	char nbuf[BUFSIZ];	/* work buffer for newfile line */
	char pbuf[BUFSIZ];	/* work buffer for newfile line */
	char *nb, *nf;		/* ptrs to new buffer, new field */
	char *pb, *pf;		/* ptrs to pattern buffer, pattern field */
	int mismatch;
	int argnxt;
	int prfile, prout, grpct;
	int i;

	/*
	 * get the name of the program
	 */
	progname = argv[0];

	/*
	 * get any argument switches
	 */
	argnxt = 1;
	if (argv[1][0] == '-'){
		if (argv[1][1] == '.'){
			if (sscanf(argv[1], "-.%d", &startpt) != 1){
				fprintf(stderr,
			"Usage: %s [ -grp.ign ] newfile patternfile [ file ]\n",
								    progname);
				exit(1);
			}
		}
		else if (argv[1][strlen(argv[1])-1] == '.'){
			if (sscanf(argv[1], "-%d.", &groupsz) != 1){
				fprintf(stderr,
			"Usage: %s [ -grp.ign ] newfile patternfile [ file ]\n",
								    progname);
				exit(1);
			}
		}
		else if (sscanf(argv[1], "-%d.%d", &groupsz, &startpt) != 2){
			fprintf(stderr,
			"Usage: %s [ -grp.ign ] newfile patternfile [ file ]\n",
								    progname);
			exit(1);
		}
		argnxt++;
	}
	if (groupsz >= MAXGROUP){
		fprintf(stderr, "Grouping size must be under %d\n", MAXGROUP);
		exit(1);
	}
	if (groupsz == 0)
		groupsz = 1;

	/*
	 * get the old and new file names,
	 * reporting any usage problems
	 */
	if ((argc - argnxt) != 3 && (argc - argnxt) != 4){
		fprintf(stderr,
		"Usage: %s [ -grp.ign ] prefix newfile patternfile [ file ]\n",
								    progname);
		exit(1);
	}
	pref = argv[argnxt++];
	newfile = argv[argnxt++];
	patfile = argv[argnxt++];
	mmfile = ((argc - argnxt) == 1) ? argv[argnxt] : NULL;

	/*
	 * open the new file if it isn't stdin
	 */
	if (strcmp(newfile, "-") == 0)
		nfp = stdin;
	else if ((nfp = fopen(newfile, "r")) == NULL){
		perror(newfile);
		exit(1);
	}
	/*
	 * open the pattern file
	 */
	if (strcmp(patfile, "-") == 0){
		fprintf(stderr, "%s: patternfile cannot be standard input\n",
				progname);
		exit(1);
	}
	else if ((pfp = fopen(patfile, "r")) == NULL){
		perror(patfile);
		exit(1);
	}

	/*
	 * open the mismatched file
	 */
	if (mmfile == NULL)
		mfp = stdout;
	else{
		if (strcmp(mmfile, "-") == 0)
			mfp = stdout;
		if ((mfp = fopen(mmfile, "a")) == NULL){
			perror(mmfile);
			mfp = stdout;
		}
	}

	/*
	 * the analysis loop
	 * get any required new lines up here
	 */
	grpct = 0;
	prfile = prout = 0;
	while(fgets(newbuf, BUFSIZ, nfp) != NULL){
		if (grpct == groupsz){
			prfile = prout = 0;
			grpct = 0;
		}
		(void) strcpy(group[grpct++], newbuf);
		succ = 0;
		mismatch = 0;
		rewind(pfp);
		while(!succ && fgets(patbuf, BUFSIZ, pfp) != NULL){
			(void) strcpy(pbuf, patbuf);
			(void) strcpy(nbuf, &newbuf[startpt]);
			nb = nbuf;
			pb = pbuf;
			nf = getfield(&nb);
			pf = getfield(&pb);
			if (!match(nf, pf))
				continue;
			succ = 1;
			do{
				nf = getfield(&nb);
				pf = getfield(&pb);
				if (*nf != '\0' || *pf != '\0'){
					if (strcmp(pf, WC) != 0 &&
							strcmp(nf, pf) != 0){
						succ = 0;
						if (mismatch++ == 0)
							(void) strcpy(mbuf, patbuf);
					}
				}
			} while(succ == 1 && *nf != '\0' && *pf != '\0');
		}
		if (succ){
			if (prfile)
				fprintf(mfp, "%s%s", pref, newbuf);
			if (prout)
				printf("%s", newbuf);
		}
		else{
			if (mismatch){
				if (!prfile){
					prfile = 1;
					for(i = 0; i < grpct; i++)
						fprintf(mfp, "%s%s",
							pref, group[i]);
				}
				else
					fprintf(mfp, "%s%s", pref, newbuf);
				fprintf(mfp, "(pat) %s", mbuf);
				if (--mismatch > 0)
					fprintf(mfp, "(and) %d %s\n",
					    mismatch, mismatch == 1 ?
							  "other" : "others");
			}
			else{
				if (!prout){
					prout = 1;
					for(i = 0; i < grpct; i++)
						printf("%s", group[i]);
				}
				else
					printf("%s", newbuf);
			}
		}
	}

	/*
	 * close the files
	 * and quit with successful error code
	 */
	(void) fclose(nfp);
	(void) fclose(pfp);
	exit(0);
}

/*
 * get the next field
 */
char *getfield(s)
register char **s;		/* beginning of rest of line */
{
	register char *p;

	/*
	 * go until you find a tab, newline, or NUL
	 */
	for(p = *s; **s && **s != '\t' && **s != '\n'; (*s)++);
	/*
	 * if a tab, reset s to point to the next field (ie, skip the tab)
	 */
	if (**s == '\t'){
		*(*s)++ = '\0';
		while(**s == '\t')
			(*s)++;
	}
	/*
	 * if a newline, clobber it with a nul
	 */
	if (**s == '\n')
		**s = '\0';
	/*
	 * return a pointer to the field
	 */
	return(p);
}

/*****
 * pattern matching
 *****
 */
#ifdef SYSV
#	define	INIT		register char *sp = instring;
#	define	GETC()		(*sp++)
#	define	PEEKC()		(*sp)
#	define	UNGETC(c)	(--sp)
#	define	RETURN(c)	return;
#	define	ERROR(c)	regerr(c)
#	include <regexp.h>
#endif

int match(str, pat)
char *str;
char *pat;
{
#ifdef BSD4
	char *p;		/* points to returned error message */
	char *re_comp();	/* pattern compiler */

	/*
	 * compile the pattern; croak on error
	 */
	if ((p = re_comp(pat)) != NULL){
		fprintf(stderr, "%s: %s (line %d)\n", progname, p, plineno);
		exit(1);
	}
	/*
	 * try to match
	 */
	switch(re_exec(str)){
	case 1:			/* yup */
		return(1);
	case 0:			/* nope */
		return(0);
	case -1:		/* scrod again! */
		fprintf(stderr, "%s: %s (line %d)\n", progname, p, plineno);
		exit(1);
	}
	/* NOTREACHED */
#endif
#ifdef SYSV
	char patbuf[10*BUFSIZ];	/* buffer for compiled pattern */

	/*
	 * compile the pattern
	 */
	(void) compile(pat, patbuf, &patbuf[10*BUFSIZ], '\0');
	/*
	 * matches only if ENTIRE string matched!
	 */
	return (step(str, patbuf) != 0 && loc1 == str && *loc2 == '\0');
#endif
}

#ifdef SYSV
regerr(n)
int n;
{
	fprintf(stderr, "%s: ", progname);
	switch(n){
	case 11:
		fprintf(stderr, "range endpoint too large");
		break;
	case 16:
		fprintf(stderr, "bad number");
		break;
	case 25:
		fprintf(stderr, "\\digit out of range");
		break;
	case 36:
		fprintf(stderr, "illegal or missing delimiter");
		break;
	case 41:
		fprintf(stderr, "(internal error) no remembered search string");
		break;
	case 42:
		fprintf(stderr, "\\( \\) imbalance");
		break;
	case 43:
		fprintf(stderr, "too many \\(");
		break;
	case 44:
		fprintf(stderr, "more than 2 numbers given in \\{ \\}");
		break;
	case 45:
		fprintf(stderr, "} expected after \\");
		break;
	case 46:
		fprintf(stderr, "first number exceeds second in \\{ \\}");
		break;
	case 49:
		fprintf(stderr, "[ ] imbalance");
		break;
	case 50:
		fprintf(stderr, "regular expression too long");
		break;
	default:
		fprintf(stderr, "unknown regular expression error #%d", n);
		break;
	}
	fprintf(stderr, " in %s at line %d\n", patfile, plineno);
	exit(1);
}
#endif
