/*
 *	auditmerge <old> <new>
 * This program copies lines from <new> to the standard output with the
 * following modifications:
 * * if there is a line in <old> with the same first field as in the line
 *   of <new>, and any other fields in the line of <old> have *s, the
 *   corresponding fields in the line of <new> are replaced.
 *
 * Matt Bishop
 * Department of Mathematics and Computer Science
 * Dartmouth College
 * Hanover, NH  03755
 *
 * Matt.Bishop@dartmouth.edu, ...!decvax!dartvax!Matt.Bishop
 */
#include <stdio.h>

/*
 * version number
 */
static char *version = "RIACS Audit Package version 3.1.3 Tue May 19 12:59:43 PDT 1992 (Matt.Bishop@dartmouth.edu)";

/*
 * macros
 */
#define	WC	"*"	/* wild card character for fields in oldfile */

/*
 * which file do we get a line from
 */
int getnew = 1;		/* get a line from the new file */
int getold = 1;		/* get a line from the old file */

/*
 * buffers for lines from file
 */
char newbuf[BUFSIZ];	/* buffer for line from new file */
char oldbuf[BUFSIZ];	/* buffer for line from old file */
char mergebuf[BUFSIZ];	/* buffer for newfile made by putting in WCs */

/*
 * pointers to files
 */
FILE *nfp = NULL;	/* pointer to new file */
FILE *ofp = NULL;	/* pointer to old file */

/*
 * forward references to input functions
 */
char *mgets();		/* like fgets but deletes trailing newline */

/*
 * forward references to library functions
 */
char *strcpy();		/* copy a string */

main(argc, argv)
int argc;
char **argv;
{
	register int i;		/* used to walk argument list */
	char *oldfile;		/* name of old file */
	char *newfile;		/* name of new file */
	char toldbuf[BUFSIZ];
	char tnewbuf[BUFSIZ];

	/*
	 * process the options
	 */
	if (argc != 3){
		fprintf(stderr, "Usage: %s oldfile newfile\n", argv[0]);
		exit(1);
	}

	/*
	 * get the old and new file names,
	 */
	oldfile = argv[1];
	newfile = argv[2];

	/*
	 * open the old file if it isn't stdin
	 */
	if (strcmp(oldfile, "-") == 0)
		ofp = stdin;
	else if ((ofp = fopen(oldfile, "r")) == NULL){
		perror(oldfile);
		exit(1);
	}
	/*
	 * open the new file if it isn't stdin
	 */
	if (strcmp(newfile, "-") == 0)
		nfp = stdin;
	else if ((nfp = fopen(newfile, "r")) == NULL){
		perror(newfile);
		exit(1);
	}

	/*
	 * the analysis loop
	 * get any required new lines up here
	 */
	while(getnext()){
		/*
		 * compare the lines
		 */
		(void) strcpy(toldbuf, oldbuf);
		(void) strcpy(tnewbuf, newbuf);
		i = mstrcmp(toldbuf, tnewbuf);
		/*
		 * if you're merging, print the merge of the
		 * old and new file lines
		 */
		if (mergebuf[0] == '\0'){
			if (i > 0 && getnew)
				printf("%s\n", newbuf);
		}
		else
			printf("%s\n", mergebuf);
		/* if (i == 0) lines the same; get next one from both */
		/* else if (i > 0) line in new, not in old; get next new one */
		/* else	line in old, not in new; get next old one */
		getnew = (i >= 0);
		getold = (i <= 0);
	}

	/*
	 * if old file hit EOF,
	 *  dump the rest of the new file
	 */
	while(mgets(newbuf, BUFSIZ, nfp) != NULL)
		(void) printf("%s\n", newbuf);
	/*
	 * close the files
	 * and quit with successful error code
	 */
	(void) fclose(ofp);
	(void) fclose(nfp);
	exit(0);
}

/*
 * get the next line from the appropriate file(s)
 * 1 on success
 * 0 if ANYONE returns EOF
 */
getnext()
{
	/*
	 * get line from old file
	 */
	if (getold && mgets(oldbuf, BUFSIZ, ofp) == NULL)
		return(0);
	/*
	 * get line from new file
	 */
	if (getnew && mgets(newbuf, BUFSIZ, nfp) == NULL)
		return(0);
	return(1);
}

/*
 * just like fgets, but deletes trailing newline
 */
char *mgets(buf, n, fp)
char *buf;				/* buffer for input */
int n;					/* size of buffer */
FILE *fp;				/* file toget inp[ut from */
{
	register int c;		/* input character */
	register char *b;	/* used to put chars into buffer */

	/*
	 * load up the buffer
	 */
	b = buf;
	while((c = getc(fp)) != EOF && c != '\n'){
		*b++ = c;
		/*
		 * check for buffer overflow
		 */
		if (b >= &buf[n-1]){
			/*
			 * got it -- terminate buffer and return
			 */
			*b = '\0';
			return(buf);
		}
	}
	/*
	 * end buffer and return, handling EOF properly
	 */
	*b = '\0';
	return((b == buf && c == EOF) ? NULL : buf);
}

/*
 * get the next field
 */
char *getfield(s)
register char **s;		/* beginning of rest of line */
{
	register char *p;

	/*
	 * go until you find a tab, newline, or NUL
	 */
	for(p = *s; **s && **s != '\t' && **s != '\n'; (*s)++);
	/*
	 * if a tab, reset s to point to the next field (ie, skip the tab)
	 */
	if (**s == '\t'){
		*(*s)++ = '\0';
		while(**s == '\t')
			(*s)++;
	}
	/*
	 * if a newline, clobber it
	 */
	if (**s == '\n')
		**s = '\0';
	/*
	 * return a pointer to the field
	 */
	return(p);
}

/*
 * like strcmp, but builds the merge array
 */
int mstrcmp(o, n)
char *o, *n;		/* old, new lines */
{
	int dif;			/* do first fields match? */
	register char *of, *nf;		/* point to fields in old, new bufs */
	char *m = mergebuf;		/* used to walk the merge buffer */

	/*
	 * clobber anything in the merge buffer
	 */
	*m = '\0';

	/*
	 * check first field
	 */
	of = getfield(&o);
	nf = getfield(&n);
	if ((dif = strcmp(of, nf)) != 0)
		return(dif);
	/*
	 * put first field in the buffer
	 */
	while(*nf != '\0')
		*m++ = *nf++;
	
	/*
	 * first fields same -- merge them
	 */
	do{
		/*
		 * get the fields
		 */
		of = getfield(&o);
		nf = getfield(&n);
		if (*of != '\0' || *nf != '\0'){
			/*
			 * wildcard matches everything
			 */
			*m++ = '\t';
			if (strcmp(of, WC) == 0)
				*m++ = WC[0];
			else{
				while(*nf != '\0')
					*m++ = *nf++;
			}
		}
	} while(*of != '\0' || *nf != '\0');

	/*
	 * end the merge string
	 */
	*m = '\0';

	/*
	 * matching first fields
	 */
	return(0);
}
