/*
 * This program works in two modes.
 *   When invoked like
 * 	auditgrep file leader
 * where "file" is a file name and "leader" any string, this program
 * reads strings from the standard input and prints lines in the file
 * the first field of which are identical to the strings (no pattern
 * matching done!)  If present, "leader" followed by a blank will be
 * prepended to each line of the output.
 *   When invoked like	
 *	auditgrep file1 leader1 file2 leader2
 * where "file1"  and "file2" are file names and "leader1" and "leader2"
 * any strings, this program reads strings from the standard input and
 * prints lines in file1 and file2 the first fields of which are identical
 * to the strings (no pattern matching done!)  For each string, all lines
 * in file1 will be printed, then all lines in file2. "leader1" must
 * be present (but it may be empty); "leader2" may be omitted.  Lines from
 * file1 have "leader1" followed by a blank prepended; lines from file2
 * have "leader2" followed by a blank prepended.
 *   "file", "file1", and "file2" consist of fields separated by tab
 * characters ('\t', ^I, ASCII HT, char 011 or 0x09).
 *   To compare on a field other than the first, use "-fn", where "n" is
 * the number of the field to make the comparison on; the field at the
 * beginning of the line is field number 1, and omitting this argument
 * implies "-f1".
 *
 * Matt Bishop
 * Research Institute for Advanced Computer Science
 * NASA Ames Research Center
 * Moffett Field, CA  94035
 *
 * mab@riacs.edu, ...!{decvax!decwrl,ihnp4!ames}!riacs!mab
 *
 * (c) Copyright 1986 by Matt Bishop and the Research Institute for
 *			Advanced Computer Science
 */ 
#include <stdio.h>
#include <ctype.h>

/*
 * version number
 */
static char *version = "RIACS Audit Package version 3.1.3 Tue May 19 12:59:43 PDT 1992 (Matt.Bishop@dartmouth.edu)";

/*
 * global variables
 */
char *progname;			/* program name (for error messages) */
char word[2*BUFSIZ];		/* string to be checked for */
int field = 0;			/* number of field for comparisons */

main(argc, argv)
int argc;
char **argv;
{
	char *a;			/* used to get field number */
	char *lead1 = NULL;		/* leader for lines from first file */
	char *lead2 = NULL;		/* leader for lines from second file */
	FILE *s1fp = NULL;		/* pointer to first file */
	FILE *s2fp = NULL;		/* pointer to second file */

	/*
	 * get the program name
	 */
	progname = argv[0];

	/*
	 * any options?
	 */
	while (argc > 1 && argv[1][0] == '-'){
		switch(argv[1][1]){
		case 'f':			/* field number */
			field = 0;
			for(a = &argv[1][2]; isdigit(*a); a++)
				field = field * 10 + *a - '0';
			if (*a != '\0'){
				fprintf(stderr, "%s: bad option %s\n",
							progname, argv[1]);
				exit(1);
			}
			if (--field <= 0){
				fprintf(stderr, "%s: bad field number %d\n",
							progname, field);
				exit(1);
			}
			break;
		default:			/* huh? */
			fprintf(stderr, "%s: bad option %s\n",
							progname, argv[1]);
			exit(1);
		}
		argc--;
		argv++;
	}

	/*
	 * need at least 1 arg
	 */
	if (argc < 2){
		fprintf(stderr, "%s: too few args\n", progname);
		exit(1);
	}

	/*
	 * open the file to be scanned
	 */
	if ((s1fp = fopen(argv[1], "r")) == NULL){
		perror(argv[1]);
		exit(1);
	}

	/*
	 * list any leader
	 */
	if (argc > 2){
		if ((lead1 = argv[2]) == '\0')
			lead1 = NULL;
	}

	/*
	 * open the second file to be scanned
	 */
	if (argc > 3 && (s2fp = fopen(argv[3], "r")) == NULL){
		perror(argv[3]);
		exit(1);
	}

	/*
	 * list any leader
	 */
	if (argc > 4){
		if ((lead2 = argv[4]) == '\0')
			lead2 = NULL;
	}


	/*
	 * read the standard input, looking for the
	 * requisite lines in the input
	 */
	while(gets(word) != NULL){
		/*
		 * begin the search at the beginning
		 * of the file to be scanned
		 */
		rewind(s1fp);
		/*
		 * scan the file
		 */
		lookfor(s1fp, lead1);
		/*
		 * now do the second file
		 */
		if (s2fp != NULL){
			/*
			 * begin the search at the beginning
			 * of the file to be scanned
			 */
			rewind(s2fp);
			/*
			 * scan the file
			 */
			lookfor(s2fp, lead2);
		}
	}

	/*
	 * close the file being scanned
	 */
	(void) fclose(s1fp);

	/*
	 * that's all folks!
	 */
	exit(0);
}

/*
 * this routine scans the file s1fp points to
 * for a line beginning with word[] and word[]
 * being followed by a '\t' character; it prints
 * the line and then returns
 * if no such line is found, return nothing
 * the file is NOT closed
 */
lookfor(sfp, lead)
FILE *sfp;			/* file to be scanned */
char *lead;			/* leader for output line */
{
	char buf[BUFSIZ];		/* buffer for line of file */
	register char *w;		/* pointer used to walk down word[] */
	register char *b;		/* pointer used to walk down buf[] */
	register int i;			/* counter in a for loop */

	/*
	 * read a line from the file to be scanned
	 */
	while(fgets(buf, BUFSIZ, sfp) != NULL){
		/*
		 * find the correct field
		 * if none, continue
		 */
		b = buf;
		for(i = 0; *b && i < field; i++)
			while(*b && *b++ != '\t');
		if (*b == '\0')
			continue;
		/*
		 * compare word[] to the leading chars in the field
		 */
		for(w = word; *w && *w == *b; w++, b++);
		/*
		 * was the field word[]?
		 */
		if (*w == '\0' && (*b == '\t' || *b == '\0')){
			/*
			 * print the leader, if any
			 */
			if (lead != NULL)
				printf("%s ", lead);
			/*
			 * print the line
			 */
			printf("%s", buf);
		}
	}
}

