/*
 *
 *			   IPSEC for Linux
 *		         Preliminary Release
 * 
 *	 Copyright (C) 1996, 1997, John Ioannidis <ji@hol.gr>
 * 
 * Changes by Angelos D. Keromytis and Niels Provos
 * ported from OpenBSD 2.2 by Petr Novak, <pn@i.cz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <linux/config.h>
#include <asm/segment.h>
#include <asm/system.h>
#include <asm/io.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/config.h>

#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/icmp.h>
#include <linux/udp.h>
#include <net/ip.h>
#include <net/protocol.h>
#include <net/route.h>
#include <net/tcp.h>
#include <net/udp.h>
#include <net/sock.h>
#include <net/icmp.h>

#include <net/checksum.h>

#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/miscdevice.h>

#include <linux/skbuff.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>

#include <net/netlink.h>
#include <unistd.h>
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_radij.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"
#include "ipsec_ipe4.h"
#ifdef CONFIG_IPSEC_AH
#include "ipsec_ah.h"
#endif

#ifdef DEBUG_IPSEC_NETLINK

int debug_netlink = 0;

#endif

/*
 * IPSEC <> netlink interface.
 */

#define SENDERR(_x) do { len = -(_x); goto errlab; } while (0)

int 
ipsec_callback(struct sk_buff *skb)
{
	/*
	 * this happens when we write to /dev/ipsec (c 36 10)
	 */
	int len = skb->len;
	u_char *dat = (u_char *)skb->data;
	struct encap_msghdr *emp;
	struct flow *flow, *flow2, *flow3, *flow4;
	struct sockaddr_encap encapdst, encapgw, encapnetmask;
	struct in_addr alts, altm;
	struct tdb *tdbp, *tdbp2;
	int error;
	__u32 spi;

	emp = (struct encap_msghdr *)dat;
	if (emp->em_version != PFENCAP_VERSION_1)
	{
		return -EINVAL;
	}
	
	memset(&encapdst, 0, sizeof(struct sockaddr_encap));
	memset(&encapnetmask, 0, sizeof(struct sockaddr_encap));
	memset(&encapgw, 0, sizeof(struct sockaddr_encap));

	switch (emp->em_type)
	{
	      case EMT_SETSPI:
		tdbp = gettdb(emp->em_spi, emp->em_dst, emp->em_sproto);
		if (tdbp == NULL)
		{
#ifdef DEBUG_IPSEC_NETLINK
			if (debug_netlink & DB_NL_TDBCB)
				printk("ipsec_callback: could not find a TDB for 0x%08x, 0x%08x, allocating\n", (u_int)ntohl(emp->em_spi), (u_int)ntohl(emp->em_dst.s_addr));
#endif
			tdbp = (struct tdb *)kmalloc(sizeof (*tdbp), GFP_ATOMIC);

			if (tdbp == NULL)
			  SENDERR(ENOBUFS);
			
			memset((caddr_t)tdbp, 0, sizeof(*tdbp));
			
			tdbp->tdb_spi = emp->em_spi;
			tdbp->tdb_dst = emp->em_dst;
			tdbp->tdb_sproto = emp->em_sproto;
			puttdb(tdbp);
		}
		else
		{
			if (tdbp->tdb_xform)
				(*tdbp->tdb_xform->xf_zeroize)(tdbp);
		}
		
		tdbp->tdb_src = emp->em_src;

		/* Check if this is an encapsulating SPI */
		if (emp->em_osrc.s_addr != 0)
		{
			tdbp->tdb_flags |= TDBF_TUNNELING;
			tdbp->tdb_osrc = emp->em_osrc;
			tdbp->tdb_odst = emp->em_odst;

			/* TTL */
			switch (emp->em_ttl)
			{
				case IP4_DEFAULT_TTL:
					tdbp->tdb_ttl = 0;
					break;

				case IP4_SAME_TTL:
					tdbp->tdb_flags |= TDBF_SAME_TTL;
					break;

				default:
					/* Get just the least significant bits */
					tdbp->tdb_ttl = emp->em_ttl & 0xff;
					break;
			}
		}

		/* Clear the invalid flag */
		tdbp->tdb_flags &= (~TDBF_INVALID);

		error = tdb_init(tdbp, emp);
		if (error)
		{
			tdb_delete(tdbp, 0);
			SENDERR(EINVAL);
		}

		break;
	      case EMT_DELSPI:
		if (len != EMT_DELSPI_FLEN)
			SENDERR(EINVAL);

		if ((tdbp = gettdb(emp->em_gen_spi, emp->em_gen_dst,
				emp->em_gen_sproto)) == NULL)
			SENDERR(ENOENT);

		error = tdb_delete(tdbp, 0);
		if (error)
			SENDERR(EINVAL);

		break;
		
	      case EMT_DELSPICHAIN:
		if (len != EMT_DELSPICHAIN_FLEN)
			SENDERR(EINVAL);

		if ((tdbp = gettdb(emp->em_gen_spi, emp->em_gen_dst,
				emp->em_gen_sproto)) == NULL)
			SENDERR(ENOENT);

		error = tdb_delete(tdbp, 1);
		if (error)
			SENDERR(EINVAL);

		break;

	      case EMT_GRPSPIS:
		if (len != EMT_GRPSPIS_FLEN)
			SENDERR(EINVAL);

		if ((tdbp = gettdb(emp->em_rel_spi, emp->em_rel_dst,
				emp->em_rel_sproto)) == NULL)
		{
			printk("ipsec_callback: groupspis: no TDB for 0x%08x, %08x\n", (u_int)ntohl(emp->em_rel_dst.s_addr), (u_int)ntohl(emp->em_rel_spi));
			SENDERR(ENOENT);
		}

		if ((tdbp2 = gettdb(emp->em_rel_spi2, emp->em_rel_dst2,
				emp->em_rel_sproto2)) == NULL)
		{
			printk("ipsec_callback: groupspis: no TDB for 0x%08x, %08x\n", (u_int)ntohl(emp->em_rel_dst2.s_addr), (u_int)ntohl(emp->em_rel_spi2));
			SENDERR(ENOENT);
		}
		
		tdbp->tdb_onext = tdbp2;
		tdbp2->tdb_inext = tdbp;

		error = 0;

		break;

	      case EMT_RESERVESPI:
		if (len != EMT_RESERVESPI_FLEN)
			SENDERR(EINVAL);

		spi = reserve_spi(emp->em_gen_spi, emp->em_gen_dst,
			emp->em_gen_sproto, &error);

		if (spi == 0)
			SENDERR(error);

		emp->em_gen_spi = spi;

		netlink_post(NETLINK_IPSEC, skb);

		error = 0;

		break;

	      case EMT_ENABLESPI:
		if (len != EMT_ENABLESPI_FLEN)
			SENDERR(EINVAL);

		tdbp = gettdb(emp->em_ena_spi, emp->em_ena_dst, 
			  emp->em_ena_sproto);

		if (tdbp == NULL)
			SENDERR(ENOENT);

		flow = flow2 = flow3 = flow4 = (struct flow *) NULL;

		emp->em_ena_isrc.s_addr &= emp->em_ena_ismask.s_addr;
		emp->em_ena_idst.s_addr &= emp->em_ena_idmask.s_addr;

		flow3 = find_global_flow(emp->em_ena_isrc, emp->em_ena_ismask,
			     	     emp->em_ena_idst, emp->em_ena_idmask,
			     	     emp->em_ena_protocol, emp->em_ena_sport,
			     	     emp->em_ena_dport);
		if (flow3 != (struct flow *) NULL)
			if (!(emp->em_ena_flags & ENABLE_FLAG_REPLACE))
				SENDERR(EEXIST);

		/* Check for 0.0.0.0/255.255.255.255 if the flow is local */
		if (emp->em_ena_flags & ENABLE_FLAG_LOCAL)
		{
			alts.s_addr = INADDR_ANY;
			altm.s_addr = INADDR_BROADCAST;
			flow4 = find_global_flow(alts, altm, emp->em_ena_idst,
				         emp->em_ena_idmask,
					 emp->em_ena_protocol,
				    	 emp->em_ena_sport, emp->em_ena_dport);
			if (flow4 != (struct flow *) NULL)
			{
				if (!(emp->em_ena_flags & ENABLE_FLAG_REPLACE))
					SENDERR(EEXIST);
				else if (flow3 == flow4)
					SENDERR(EINVAL);
			}
		}

		flow = get_flow();
		if (flow == (struct flow *) NULL)
			SENDERR(ENOBUFS);

		flow->flow_src.s_addr = emp->em_ena_isrc.s_addr;
		flow->flow_dst.s_addr = emp->em_ena_idst.s_addr;
		flow->flow_srcmask.s_addr = emp->em_ena_ismask.s_addr;
		flow->flow_dstmask.s_addr = emp->em_ena_idmask.s_addr;
		flow->flow_proto = emp->em_ena_protocol;
		flow->flow_sport = emp->em_ena_sport;
		flow->flow_dport = emp->em_ena_dport;

		if (emp->em_ena_flags & ENABLE_FLAG_LOCAL)
		{
			flow2 = get_flow();
			if (flow2 == (struct flow *) NULL)
			{
				kfree(flow);
				SENDERR(ENOBUFS);
			}

			flow2->flow_src.s_addr = INADDR_ANY;
			flow2->flow_dst.s_addr = emp->em_ena_idst.s_addr;
			flow2->flow_srcmask.s_addr = INADDR_BROADCAST;
			flow2->flow_dstmask.s_addr = emp->em_ena_idmask.s_addr;
			flow2->flow_proto = emp->em_ena_protocol;
			flow2->flow_sport = emp->em_ena_sport;
			flow2->flow_dport = emp->em_ena_dport;

			put_flow(flow2, tdbp);
		}

		put_flow(flow, tdbp);

		/* Setup the encap fields */
		encapdst.sen_len = SENT_IP4_LEN;
		encapdst.sen_family = AF_ENCAP;
		encapdst.sen_type = SENT_IP4;
		encapdst.sen_ip_src.s_addr = flow->flow_src.s_addr;
		encapdst.sen_ip_dst.s_addr = flow->flow_dst.s_addr;
		encapdst.sen_proto = flow->flow_proto;
		encapdst.sen_sport = flow->flow_sport;
		encapdst.sen_dport = flow->flow_dport;

		encapgw.sen_len = SENT_IPSP_LEN;
		encapgw.sen_family = AF_ENCAP;
		encapgw.sen_type = SENT_IPSP;
		encapgw.sen_ipsp_dst.s_addr = tdbp->tdb_dst.s_addr;
		encapgw.sen_ipsp_spi = tdbp->tdb_spi;
		encapgw.sen_ipsp_sproto = tdbp->tdb_sproto;

		encapnetmask.sen_len = SENT_IP4_LEN;
		encapnetmask.sen_family = AF_ENCAP;
		encapnetmask.sen_type = SENT_IP4;
		encapnetmask.sen_ip_src.s_addr = flow->flow_srcmask.s_addr;
		encapnetmask.sen_ip_dst.s_addr = flow->flow_dstmask.s_addr;

		if (flow->flow_proto)
		{
			encapnetmask.sen_proto = 0xff;

			if (flow->flow_sport)
				encapnetmask.sen_sport = 0xffff;

			if (flow->flow_dport)
				encapnetmask.sen_dport = 0xffff;
		}

		/* If this is set, delete any old route for this flow */
		if (emp->em_ena_flags & ENABLE_FLAG_REPLACE)
			ipsec_delroute(&encapdst, &encapnetmask);

		/* Add the entry in the routing table */
		error = ipsec_makeroute(&encapdst, &encapnetmask, &encapgw)==NULL;
	    
		if (error)
		{
			encapdst.sen_len = SENT_IP4_LEN;
			encapdst.sen_family = AF_ENCAP;
			encapdst.sen_type = SENT_IP4;
			encapdst.sen_ip_src.s_addr = flow3->flow_src.s_addr;
			encapdst.sen_ip_dst.s_addr = flow3->flow_dst.s_addr;
			encapdst.sen_proto = flow3->flow_proto;
			encapdst.sen_sport = flow3->flow_sport;
			encapdst.sen_dport = flow3->flow_dport;

			encapgw.sen_len = SENT_IPSP_LEN;
			encapgw.sen_family = AF_ENCAP;
			encapgw.sen_type = SENT_IPSP;
			encapgw.sen_ipsp_dst.s_addr = flow3->flow_sa->tdb_dst.s_addr;
			encapgw.sen_ipsp_spi = flow3->flow_sa->tdb_spi;
			encapgw.sen_ipsp_sproto = flow3->flow_sa->tdb_sproto;

			encapnetmask.sen_len = SENT_IP4_LEN;
			encapnetmask.sen_family = AF_ENCAP;
			encapnetmask.sen_type = SENT_IP4;
			encapnetmask.sen_ip_src.s_addr = flow3->flow_srcmask.s_addr;
			encapnetmask.sen_ip_dst.s_addr = flow3->flow_dstmask.s_addr;

			if (flow3->flow_proto)
			{
				encapnetmask.sen_proto = 0xff;
		    
				if (flow3->flow_sport)
					encapnetmask.sen_sport = 0xffff;

				if (flow->flow_dport)
					encapnetmask.sen_dport = 0xffff;
			}
		
			/* Try to add the old entry back in */
			ipsec_makeroute(&encapdst, &encapnetmask, &encapgw);
	    
			delete_flow(flow, tdbp);
			if (flow2)
				delete_flow(flow2, tdbp);
			SENDERR(error);
		}

		/* If this is a "local" packet flow */
		if (emp->em_ena_flags & ENABLE_FLAG_LOCAL)
		{
			encapdst.sen_ip_src.s_addr = INADDR_ANY;
			encapnetmask.sen_ip_src.s_addr = INADDR_BROADCAST;

			if (emp->em_ena_flags & ENABLE_FLAG_REPLACE)
				ipsec_delroute(&encapdst, &encapnetmask);

			error = ipsec_makeroute(&encapdst, &encapnetmask, &encapgw)==NULL;

			if (error)
			{
				/* Delete the first entry inserted */
				encapdst.sen_ip_src.s_addr = emp->em_ena_isrc.s_addr;
				encapnetmask.sen_ip_src.s_addr = emp->em_ena_ismask.s_addr;

				ipsec_delroute(&encapdst, &encapnetmask);

				/* Setup the old entries */
				encapdst.sen_len = SENT_IP4_LEN;
				encapdst.sen_family = AF_ENCAP;
				encapdst.sen_type = SENT_IP4;
				encapdst.sen_ip_src.s_addr = flow3->flow_src.s_addr;
				encapdst.sen_ip_dst.s_addr = flow3->flow_dst.s_addr;
				encapdst.sen_proto = flow3->flow_proto;
				encapdst.sen_sport = flow3->flow_sport;
				encapdst.sen_dport = flow3->flow_dport;

				encapgw.sen_len = SENT_IPSP_LEN;
				encapgw.sen_family = AF_ENCAP;
				encapgw.sen_type = SENT_IPSP;
				encapgw.sen_ipsp_dst.s_addr = flow3->flow_sa->tdb_dst.s_addr;
				encapgw.sen_ipsp_spi = flow3->flow_sa->tdb_spi;
				encapgw.sen_ipsp_sproto = flow3->flow_sa->tdb_sproto;
		   
				encapnetmask.sen_len = SENT_IP4_LEN;
				encapnetmask.sen_family = AF_ENCAP;
				encapnetmask.sen_type = SENT_IP4;
				encapnetmask.sen_ip_src.s_addr = flow3->flow_srcmask.s_addr;
				encapnetmask.sen_ip_dst.s_addr = flow3->flow_dstmask.s_addr;

				if (flow3->flow_proto)
				{
					encapnetmask.sen_proto = 0xff;
			
					if (flow3->flow_sport)
						encapnetmask.sen_sport = 0xffff;

					if (flow->flow_dport)
						encapnetmask.sen_dport = 0xffff;
				}
		
				ipsec_makeroute(&encapdst, &encapnetmask, &encapgw);

				encapdst.sen_ip_src.s_addr = INADDR_ANY;
				encapnetmask.sen_ip_src.s_addr = INADDR_BROADCAST;

				ipsec_makeroute(&encapdst, &encapnetmask, &encapgw);

				delete_flow(flow, tdbp);
				delete_flow(flow2, tdbp);
				SENDERR(error);
			}
		}

		/*
		 * If we're here, it means we've successfully added the new
		 * entries, so free the old ones.
		 */
		if (flow3)
			delete_flow(flow3, flow3->flow_sa);

		if (flow4)
			delete_flow(flow4, flow4->flow_sa);

		error = 0;

		break;

	case EMT_DISABLESPI:
		if (len != EMT_DISABLESPI_FLEN)
			SENDERR(EINVAL);

		tdbp = gettdb(emp->em_ena_spi, emp->em_ena_dst, 
				emp->em_ena_sproto);
		if (tdbp == NULL)
			SENDERR(ENOENT);

		emp->em_ena_isrc.s_addr &= emp->em_ena_ismask.s_addr;
		emp->em_ena_idst.s_addr &= emp->em_ena_idmask.s_addr;

		flow = find_flow(emp->em_ena_isrc, emp->em_ena_ismask,
				emp->em_ena_idst, emp->em_ena_idmask,
				emp->em_ena_protocol, emp->em_ena_sport,
				emp->em_ena_dport, tdbp);
		if (flow == (struct flow *) NULL)
			SENDERR(ENOENT);

		flow2 = (struct flow *)NULL;

		if (emp->em_ena_flags & ENABLE_FLAG_LOCAL)
		{
			alts.s_addr = INADDR_ANY;
			altm.s_addr = INADDR_BROADCAST;

			flow2 = find_flow(alts, altm, emp->em_ena_idst,
					emp->em_ena_idmask, emp->em_ena_protocol,
					emp->em_ena_sport, emp->em_ena_dport, tdbp);
			if (flow2 == (struct flow *) NULL)
				SENDERR(ENOENT);

			if (flow == flow2)
				SENDERR(EINVAL);
		}

		/* Setup the encap fields */
		encapdst.sen_len = SENT_IP4_LEN;
		encapdst.sen_family = AF_ENCAP;
		encapdst.sen_type = SENT_IP4;
		encapdst.sen_ip_src.s_addr = flow->flow_src.s_addr;
		encapdst.sen_ip_dst.s_addr = flow->flow_dst.s_addr;
		encapdst.sen_proto = flow->flow_proto;
		encapdst.sen_sport = flow->flow_sport;
		encapdst.sen_dport = flow->flow_dport;

		encapnetmask.sen_len = SENT_IP4_LEN;
		encapnetmask.sen_family = AF_ENCAP;
		encapnetmask.sen_type = SENT_IP4;
		encapnetmask.sen_ip_src.s_addr = flow->flow_srcmask.s_addr;
		encapnetmask.sen_ip_dst.s_addr = flow->flow_dstmask.s_addr;

		if (flow->flow_proto)
		{
			encapnetmask.sen_proto = 0xff;

			if (flow->flow_sport)
				encapnetmask.sen_sport = 0xffff;

			if (flow->flow_dport)
				encapnetmask.sen_dport = 0xffff;
		}

		/* Delete the entry */
		ipsec_delroute(&encapdst, &encapnetmask);
	    
		if (emp->em_ena_flags & ENABLE_FLAG_LOCAL)
		{

			encapdst.sen_ip_src.s_addr = INADDR_ANY;
			encapnetmask.sen_ip_src.s_addr = INADDR_BROADCAST;

			ipsec_delroute(&encapdst, &encapnetmask);

			delete_flow(flow2, tdbp);
		}

		delete_flow(flow, tdbp);

		break;

	      default:
		printk("ipsec_callback: unknown message type %d\n", emp->em_type);
	}
	
      errlab:
	return len;
}
