/*
 * The author of this code is John Ioannidis, ji@tla.org,
 * 	(except when noted otherwise).
 *
 * This code was written for BSD/OS in Athens, Greece, in November 1995.
 *
 * Ported to NetBSD, with additional transforms, in December 1996,
 * by Angelos D. Keromytis, kermit@forthnet.gr.
 *
 * Copyright (C) 1995, 1996, 1997 by John Ioannidis and Angelos D. Keromytis.
 *	
 * Permission to use, copy, and modify this software without fee
 * is hereby granted, provided that this entire notice is included in
 * all copies of any software which is or includes a copy or
 * modification of this software.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTY. IN PARTICULAR, NEITHER AUTHOR MAKES ANY
 * REPRESENTATION OR WARRANTY OF ANY KIND CONCERNING THE
 * MERCHANTABILITY OF THIS SOFTWARE OR ITS FITNESS FOR ANY PARTICULAR
 * PURPOSE.
 */

/*
 * IP-inside-IP processing
 */

#ifndef lint
static char rcsid[] = "$Id: ip_ip4.c,v 1.1 1995/11/29 14:33:49 ji Exp $";
#endif

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/domain.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <sys/time.h>
#include <sys/kernel.h>
#include <machine/cpu.h>

#include <net/if.h>
#include <net/route.h>
#include <net/netisr.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/in_pcb.h>
#include <netinet/in_var.h>
#include <netinet/ip_var.h>
#include <netinet/ip_icmp.h>

#include <sys/socketvar.h>
#include <net/raw_cb.h>
#include <net/encap.h>

#include <netinet/ip_ipsp.h>
#include <netinet/ip_ip4.h>
#include <netinet/ip_md5.h>


MD5_CTX IP4_ctx;	/* Randomization of ip_id */

extern int ticks;
#ifdef EXT_CLOCK
extern int clock_count;
#endif

/*
 * ip4_input gets called when we receive an encapsulated packet,
 * either because we got it at a real interface, or because AH or ESP
 * were being used in tunnel mode (in which case the rcvif element will 
 * contain the address of the encapX interface associated with the tunnel.
 */

void
ip4_input(register struct mbuf *m, int iphlen)
{
	struct ip *ipo, *ipi;
	struct ifqueue *ifq = NULL;
	int s;
	/*
	 * Strip IP options, if any.
	 */

	if (iphlen > sizeof (struct ip))
	{
		ip_stripoptions(m, (struct mbuf *)0);
		iphlen = sizeof (struct ip);
	}
	
	/*
	 * Make sure next IP header is in the first mbuf.
	 *
	 * Careful here! we are receiving the packet from ipintr;
	 * this means that the ip_len field has been adjusted to
	 * not count the ip header, and is also in host order.
	 */

	ipo = mtod(m, struct ip *);

	if (m->m_len < iphlen + sizeof (struct ip))
	{
		if ((m = m_pullup(m, iphlen + sizeof (struct ip))) == 0)
		{
			ip4stat.ip4s_hdrops++;
			return;
		}
		ipo = mtod(m, struct ip *);
	}
	ipi = (struct ip *)((caddr_t)ipo + iphlen);
	
	/*
	 * XXX - Should we do anything to the inner packet?
	 * Does arriving at the far end of the tunnel count as one hop
	 * (thus requiring ipi->ip_ttl to be decremented)?
	 */

	if (ipi->ip_v != IPVERSION)
	{
		ip4stat.ip4s_notip4++;
		return;
	}
	
	/*
	 * Interface pointer is already in first mbuf; chop off the 
	 * `outer' header and reschedule.
	 */

	m->m_len -= iphlen;
	m->m_pkthdr.len -= iphlen;
	m->m_data += iphlen;
	
	/* XXX -- interface pointer stays the same (which is probably
	 * the way it should be.
	 */

	ifq = &ipintrq;

	s = splimp();			/* isn't it already? */
	if (IF_QFULL(ifq))
	{
		IF_DROP(ifq);
		m_freem(m);
		ip4stat.ip4s_qfull++;
		splx(s);
		return;
	}
	IF_ENQUEUE(ifq, m);
	schednetisr(NETISR_IP);
	splx(s);
	
	return;
}

int
ipe4_output(struct mbuf *m, struct sockaddr_encap *gw, struct tdb *tdb, struct mbuf **mp)
{
	MD5_CTX ctx;
	struct ip *ipo, *ipi;
	ushort ilen;
	u_char dat[16];

	ipi = mtod(m, struct ip *);
	ilen = ntohs(ipi->ip_len);

	M_PREPEND(m, sizeof (struct ip), M_DONTWAIT);
	if (m == 0)
	  return ENOBUFS;

	ipo = mtod(m, struct ip *);
	
	ipo->ip_v = IPVERSION;
	ipo->ip_hl = 5;
	ipo->ip_tos = ipi->ip_tos;
	ipo->ip_len = htons(ilen + sizeof (struct ip));
	/* ipo->ip_id = htons(ip_id++); */
	MD5Update(&IP4_ctx, (u_char *)&ip_id, sizeof(ip_id));
	ctx = IP4_ctx;
	MD5Final(dat, &ctx);
	ip_id = (*(u_short *)dat);
	ip_id++;
	MD5Update(&IP4_ctx, (u_char *)&ip_id, sizeof(ip_id));
	MD5Update(&IP4_ctx, (u_char *)&ticks, sizeof(ticks));
#ifdef EXT_CLOCK
	MD5Update(&IP4_ctx, (u_char *)&clock_count, sizeof(clock_count));
#endif
	ctx = IP4_ctx;
	MD5Final(dat, &ctx);
	ipo->ip_id = (*(u_short *)dat);
	ipo->ip_off = ipi->ip_off & ~(IP_MF | IP_OFFMASK); /* keep C and DF */
	ipo->ip_ttl = ipi->ip_ttl;	/* already decremented if fwding */
	ipo->ip_p = IPPROTO_IPIP;
	ipo->ip_sum = 0;
	ipo->ip_src = gw->sen_ipsp_src;
	ipo->ip_dst = gw->sen_ipsp_dst;
	
/*	printf("ip4_output: [%x->%x](l=%d, p=%d)", 
	       ntohl(ipi->ip_src.s_addr), ntohl(ipi->ip_dst.s_addr),
	       ilen, ipi->ip_p);
	printf(" through [%x->%x](l=%d, p=%d)\n", 
	       ntohl(ipo->ip_src.s_addr), ntohl(ipo->ip_dst.s_addr),
	       ipo->ip_len, ipo->ip_p);*/

	*mp = m;
	return 0;

/*	return ip_output(m, NULL, NULL, IP_ENCAPSULATED, NULL);*/
}

int
ipe4_attach()
{
	return 0;
}

int
ipe4_init(struct tdb *tdbp, struct xformsw *xsp, struct mbuf *m)
{
	MD5Init(&IP4_ctx);
#ifdef EXT_CLOCK
	MD5Update(&IP4_ctx, (u_char *)&clock_count, sizeof(clock_count));
#endif
	MD5Update(&IP4_ctx, (u_char *)&ticks, sizeof(ticks));
	printf("ipe4_init: setting up\n");
	tdbp->tdb_xform = xsp;
	if (m)
	  m_freem(m);
	return 0;
}

int
ipe4_zeroize(struct tdb *tdbp)
{
	/* Nothing much really - we don't need any state */
	return 0;
}



struct mbuf *
ipe4_input(struct mbuf *m, struct tdb *tdb)
{
	printf("ipe4_input: should never be called\n");
	if (m)
	  m_freem(m);
	return NULL;
}
