/* Copyright (c) 1995,1996 NEC Corporation.  All rights reserved.            */
/*                                                                           */
/* The redistribution, use and modification in source or binary forms of     */
/* this software is subject to the conditions set forth in the copyright     */
/* document ("COPYRIGHT") included with this distribution.                   */

/* This file contains all of the socket creation and configuration functions */
/* that are used by the daemon.  There are a couple of utility functions     */
/* here becuase they deal with the sockets themselves...                     */
#include "socks5p.h"
#include "daemon.h"
#include "validate.h"
#include "msgids.h"
#include "socket.h"
#include "proxy.h"
#include "sema.h"
#include "log.h"
#include "sigfix.h"

/* Two server modes (threading & preforking) work in a master/slave mode,    */
/* these macros make decisions based on that status easier to read           */
#define ISMASTER() (!iamachild && (servermode == PREFORKING || servermode == THREADED))
#define ISSLAVE()  (iamachild  && (servermode == PREFORKING || servermode == THREADED))

static int acceptor  = 0;  /* the pid of the accepting process if threaded   */
static int nconns    = 0;  /* the number of proxies the daemon has had...    */
static int nchildren = 0;  /* the number of children the daemon has had...   */
static int iamachild = 0;  /* Am I a child, or the parent...                 */

static sig_atomic_t hadfatalsig = 0;  /* Has a sighup/sigusr1 occured?       */
static sig_atomic_t hadsigint   = 0;  /* Has a sigint         occured?       */

/* A function to collect dead children before they become zombies...  Also,  */
/* It keeps track of how many children are around for preforking...          */
/*                                                                           */
/* On a non-posix system, I suppose calling wait3 or waitpid in a signal     */
/* handler (according to APUE) might not be cool.  Someone will have to      */
/* tell me if this is ever the case, and I'll write code so that reaping is  */
/* done elsewhere.                                                           */
static RETSIGTYPE gravedigger(void) {
    int oerrno = errno, wval, wstatus;

    for (;;) {
#ifdef HAVE_WAITPID
	wval = waitpid(-1, &wstatus, WNOHANG);
#else
        wval = wait3(&wstatus, WNOHANG, NULL);
#endif

	switch (wval) {
	    case -1:
		if (errno == EINTR) continue;
	    case 0:
		errno = oerrno;
		return;
	    default:
		if (servermode == THREADED && wval == acceptor) acceptor = 0;
		nchildren--;
	}
    }
}

/* Indicate that we had a sighup, so we exit & clean up later                */
static RETSIGTYPE markdone(void) {
    hadsigint = 1;
}

/* Indicate that we had a sighup, so we can re-read the config file later.   */
static RETSIGTYPE die(void) {
    hadfatalsig = 1;
}

/* fork(), and do the right thing...the right thing is...handle errors and   */
/* incremenent nchildren...pretty simple...                                  */
static int DoFork(void) {
    int pid;
  
    if (servermode == THREADED) {
	if (nchildren >= MAXCLIENTS) {
	    errno = EAGAIN;
	    return -1;
	}
    } else if (nchildren >= nservers) {
	errno = EAGAIN;
	return -1;
    }
  
    switch (pid = fork()) {
	case 0:
	    Signal(SIGHUP,  SIG_IGN);
	    Signal(SIGINT,  SIG_DFL);
	    Signal(SIGUSR1, die);

	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(15), 0, "Child: Starting");
	    iamachild = 1;
	    return 0;
	case -1:
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR,     0, "Fork failed: %m");
	    return -1;
	default:
	    nchildren++;
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(15), 0, "Parent: %d child%s", nchildren, (nchildren != 1)?"ren":"");
	    return pid;
    }
}

/* Make a socket with the correct protocol (p), bind it to the right port    */
/* (n, the name or port, the default), and set the function to be called the */
/* socket becomes active (func)...store all this in an fdrec structure for   */
/* convenience's sake...if for any reason something fails and this enry      */
/* should become invalid, the r->fd should be set to -1, so other places     */
/* know to ignore it...                                                      */
static int MakeSocket(int start, S5IOHandle *infd) {
    static u_short port = 0;
    S5NetAddr bndaddr;
    int on = 1;
    char *tmp;

    ReadConfig();

    if (!start) {
	time_t now = time(NULL);
	char tbuf[1024];

	MUTEX_LOCK(lt_mutex);
	strftime(tbuf, sizeof(tbuf), "%c", localtime(&now));
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_INFO, MSGID_SERVER_RESTART, "Socks5 restarting at %s", tbuf);
	MUTEX_UNLOCK(lt_mutex);
    }

    if (*infd != S5InvalidIOHandle) {
	return 0;
    }

    if (!port) {
	MUTEX_LOCK(env_mutex);
	tmp = getenv("SOCKS5_BINDPORT");

        if (tmp) {
	    lsName2Port(tmp, "tcp", &port);
	    MUTEX_UNLOCK(env_mutex);
	} else {
	    MUTEX_UNLOCK(env_mutex);
	    lsName2Port("socks", "tcp", &port);
	}

	if (port == INVALIDPORT) {
	    port = htons(SOCKS_DEFAULT_PORT);
	}

	S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(12), 0, "Socks5 attempting to run on port: %d", ntohs(port));
    } 

    memset((char *)&bndaddr, 0, sizeof(S5NetAddr));
    bndaddr.sin.sin_family       = AF_INET;
    bndaddr.sin.sin_addr.s_addr  = htonl(INADDR_ANY);
    lsAddrSetPort(&bndaddr, port);
  
    if ((*infd = socket(AF_INET, SOCK_STREAM, 0)) == S5InvalidIOHandle) {
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, 0, "Socket failed for %s:%d: %m", ADDRANDPORT(&bndaddr));
	goto cleanup;
    }

    if (setsockopt(*infd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(int)) < 0) {
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, 0,  "Turning on address reuse failed for %s:%d: %m", ADDRANDPORT(&bndaddr));
	goto cleanup;
    }
  
    if (bind(*infd, (ss *)&bndaddr, sizeof(ssi)) < 0) {
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, MSGID_SERVER_SOCKS_BIND, "Bind failed for %s:%d: %m", ADDRANDPORT(&bndaddr));
	goto cleanup;
    }
  
    if (listen(*infd, 5) < 0) {
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, 0, "Listen failed for %s:%d: %m", ADDRANDPORT(&bndaddr));
	goto cleanup;
    }

    /* If we bound, we're the owner, so put our pid in the pidfile.          */
#ifndef DONT_STORE_PID
    {
	char abuf[64], *myfl, *ofile = NULL;
	S5IOHandle fd = S5InvalidIOHandle;
	pid_t pid = getpid();
	struct stat sbuf;
      
	MUTEX_LOCK(env_mutex);
	myfl = getenv("SOCKS5_PIDFILE");
	myfl = myfl?myfl:SRVPID_FILE;

	if (bndaddr.sin.sin_port != htons(SOCKS_DEFAULT_PORT)) {
	    if ((ofile = malloc(strlen(myfl)+7))) {
		sprintf(ofile, "%s-%d", myfl, ntohs(bndaddr.sin.sin_port));
		myfl = ofile;
	    } else {
		myfl = NULL;
	    }
	}
      
	if (myfl) {
	    int flags = O_WRONLY | O_CREAT | O_TRUNC;
	    /* Open exclusively if the file doesn't exist or if it does, it  */
	    /* is a link, and someone else owns it                           */
	    if (lstat(myfl, &sbuf) || (S_ISLNK(sbuf.st_mode) && geteuid() != sbuf.st_uid)) flags |= O_EXCL;
	    fd = open(myfl, flags, 0666);
	}

	if (fd == S5InvalidIOHandle) {
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_WARNING, 0, "Error: Failed to open pid file: %s: %m", myfl?myfl:"(null)");
	} else {
	    sprintf(abuf, "%d\n", (int)pid);
	    WRITEFILE(fd, abuf, strlen(abuf));
	    close(fd);
	}
	
	MUTEX_UNLOCK(env_mutex);
	if (ofile) free(ofile);
    }
#endif

    return 0;
    
  cleanup:
    if (*infd != S5InvalidIOHandle) CLOSESOCKET(*infd);
    *infd = S5InvalidIOHandle;
    return -1;
}

/* This is called whenever an error occurs that requires a signal to fix.    */
/* If the server is preforking or threaded, the parent begins in this state, */
/* and only children come out of it... If it is normal, this state is        */
/* reached by having a fatal call to accept...                               */
/*                                                                           */
/* Children come here sometimes when they've had fatal signals and need to   */
/* be killed off...                                                          */
static int GetSignals(void * asem, S5IOHandle *infd) {
    Sigset_t set;

    /* If we're a child and we had a fatal signal, kill ourselves off, since */
    /* there's nothing else to do.  Note that in the threaded case, this     */
    /* doesn't necessarily mean this process will die, just the accepting    */
    /* thread...                                                             */
    if (iamachild && hadfatalsig) {
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(10), 0, "%s exiting: SIGUSR1", (servermode == THREADED)?"Main thread":"Child");
	THREAD_EXIT(0);
    }

    /* Block appropriate signals here so that we don't get interrupted while */
    /* we're forking or setting things up...                                 */
    for (set = SigBlock(SIGHUP); ; ) {
	/* Do our thing if everything is ok...                               */
	if (*infd != S5InvalidIOHandle) {
	    switch (servermode) {
		case THREADED:
		    if (acceptor == 0) acceptor = DoFork();
		    if (iamachild) goto done;
		    break;
		case PREFORKING:
		    while (DoFork() > 0);
		    if (iamachild) goto done;
		    break;
	    }
	}

	/* Wait for any signal to arrive, esp SIGCHLD and SIGHUP.  SIGHUP    */
	/* will cause a re-read of the config file, everything else: loop.   */
	if (!hadfatalsig && !hadsigint) {
	    SigPause();
	}
	
	if (hadsigint) {
	    time_t now = time(NULL);
	    char tbuf[1024];

	    MUTEX_LOCK(lt_mutex);
	    strftime(tbuf, sizeof(tbuf), "%c", localtime(&now));
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_INFO, MSGID_SERVER_STOP, "Socks5 Exiting at: %s", tbuf);
	    MUTEX_UNLOCK(lt_mutex);

	    if (ISMASTER()) { semdestroy(asem); }
	    exit(0);
	}

	if (!hadfatalsig) {
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(15), 0, "Parent reaped? (%d child%s)", nchildren, (nchildren != 1)?"ren":"");
	    continue;
	}
	
	hadfatalsig = 0;
	MakeSocket(0, infd);

	switch (servermode) {
	    case NORMAL:
		/* We remade the socket, so try using it...                  */
		if (*infd != S5InvalidIOHandle) goto done;
		continue;
	    case PREFORKING:
		/* Send SIGUSR1 to our children to make them restart.        */
		/* Reset the semaphore for accept access.                    */
		acceptor = -getpid();
		/* fall through                                              */
	    case THREADED:
		/* Kill whoever is accepting (all of them if preforking);    */
		/* and reset the semaphore...                                */
		if (acceptor && acceptor != -1) kill(acceptor, SIGUSR1);
		if (ISMASTER()) { semreset(asem, 1); } 
		acceptor = 0;
		continue;
	}
    }

  done:
    SigUnblock(set);
    return 0;
}

static void DoWork(S5IOHandle sd) {
    int eval;

    if (servermode == NORMAL && DoFork()) {
	CLOSESOCKET(sd);
	return;
    }

    eval = HandlePxyConnection(sd);

    if (servermode == THREADED) {
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(10), 0, "Accept: Thread exiting...");
	nconns--;
	THREAD_EXIT(0);
    }

    if (servermode == PREFORKING) {
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(10), 0, "Accept: Done with connection...");
	return;
    }
    
    exit(eval);
}

/* Get a connection from our input socket.  Then based on the socket's       */
/* data, handle the connection correctly, and go on waiting for more         */
/* connetions...                                                             */
static void GetNetConnection(void) {
    Sigset_t set = SigBlock(ISMASTER()?SIGUSR1:SIGHUP);
    S5IOHandle in = S5InvalidIOHandle, afd;
    S5NetAddr source;
    void *asem = NULL;
    int rval, aerrno;

#ifndef USE_THREADS
    if (servermode == THREADED) {
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(0), 0, "Warning: Attempt to run server in threaded mode when threads were not a compile time option");
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(0), 0, "Warning: Running as a normal standalone server");
	servermode = NORMAL;
    }
#endif

    Signal(SIGUSR1, SIG_IGN);
    Signal(SIGHUP,  die);
    Signal(SIGCHLD, gravedigger);
    Signal(SIGINT,  markdone);

    rval = MakeSocket(1, &in);

    if (servermode == SINGLESHOT && rval < 0) {
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(0), 0, "Accept: Failed to make listening socket");
	exit(-1);
    }

    if (ISMASTER()) {
	asem = semcreate(1);
	GetSignals(asem, &in);
    }

    if (ISMASTER()) {
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(0), 0, "Error: Master reached slave code in GetNetConnection");
	exit(EXIT_ERR);
    }

    for (;;) {
	int len = sizeof(S5NetAddr);

	/* If an important signal has arrived or the acc fd is corrupted,    */
	/* got into the signal waiting state (and possibly exit - if child). */
	if (hadfatalsig || hadsigint || in == S5InvalidIOHandle) {
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(2), 0, "Accept: Processing exception");
	    GetSignals(asem, &in);
	    hadfatalsig = 0;
	}
	
	if (SigPending(ISSLAVE()?SIGUSR1:SIGHUP)) {
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(2), 0, "Accept: Waiting for a pending fatal signal...");
	    while (!hadfatalsig) SigPause();
	}
	
	/* If the child process is threaded and it has concurrent threads    */
	/* larger than the nthreads, send a HUP signal to the parent so that */
	/* the parent can spawn another threaded child to accept more        */
  	/* connections...                                                    */
	if (servermode == THREADED && nconns >= nthreads) {
	    static int hupsend = 0;

	    if (!hupsend) {
	        S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(2), 0, "Accept: Sending HUP to parent and exit.");
	    	hupsend = 1;
	    	kill(getppid(), SIGHUP);
		THREAD_EXIT(0);
	    }
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(2), 0, "Accept: HUP already sent");
	    continue;
	}

	/* If the child process is preforked and it has handled connections  */
	/* larger than the nthreads, exit so that parent can spawn another   */
	/* child to accept connections. The reason of doing that is to clean */
	/* up potential memory leaks caused by preforked child...            */
	if (servermode == PREFORKING && nconns >= nthreads) {
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(2), 0, "Accept: Child exiting for cleanup");
	    exit(0);
	}

	/* For some reason (thanks to Rich Stevens for pointing this out),   */
	/* System 5 is unhappy about a bunch of people calling accept.  So   */
	/* we'll add some locks around it to synchronize access...           */
	if (ISSLAVE() && !hadfatalsig) {
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(15), 0, "Accept: Acquiring semaphore");

	    SigUnblock(set);
	    if (semacquire(asem)) THREAD_EXIT(-1);
	    set = SigBlock((servermode == NORMAL)?SIGHUP:SIGUSR1);		    
	}
	
	/* Try to accept a connection.  We may receive a signal (HUP or      */
	/* USR1) here, if that happens, accept should return -1, with errno  */
	/* set to EINTR.  We'll handle that later, after we release the      */
	/* semaphore.  If we've already received a signal, don't bother      */
	/* accepting the connection, just set pri->in to -1 and errno to     */
	/* EINTR, to simulate having received the signal here.               */
	if (hadfatalsig) {
	    afd    = S5InvalidIOHandle;
	    aerrno = EINTR;
	} else {
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(10), 0, "Accept: Waiting on accept or a signal");

	    SigUnblock(set);
	    afd = accept(in, &source.sa, &len);
	    if (afd == S5InvalidIOHandle) aerrno = errno;
	    set = SigBlock((servermode == NORMAL)?SIGHUP:SIGUSR1);
	}

	/* Since we've got the connection, release the semaphore.            */
	/*                                                                   */
	/* We don't have to worry about releasing a reset semaphore, if we   */
	/* received a signal in the meantime, since semreset *should* create */
	/* a whole new semaphore, so we'll be releasing one which no one     */
	/* else looks at anymore.                                            */
	if (ISSLAVE() && !hadfatalsig) {
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(15), 0, "Accept: Releasing semaphore");
	    semrelease(asem);
	}

	/* Do the work according to the protocol to be passed on pri->in.    */
	/* When we're done, clean things up so we can do it all again...     */
	if (afd == S5InvalidIOHandle) {
	    /* We have to make sure the error wasn't too serious.  If it     */
	    /* was, quit, unless we are the parent, in which caes we we wait */
	    /* for a HUP to tell us things are fixed.                        */
	    if (aerrno == EINTR || aerrno == ECONNRESET) continue;
#ifdef EPROTO
	    if (aerrno == EPROTO) continue;
#endif

	    errno = aerrno;
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, MSGID_SERVER_SOCKS_ACCEPT, "Accept: Accept failed: %m");
	    if (ISSLAVE()) exit(EXIT_ERR);

	    CLOSESOCKET(in);
	    in = S5InvalidIOHandle;
	} else if (servermode == THREADED) {
#ifdef USE_THREADS
#ifdef HAVE_PTHREAD_H
	    THREAD_T tid;
	    ATTR_T attr;
	    sigset_t set, oset;

	    sigemptyset(&set);
	    sigaddset(&set, SIGUSR1);

	    ATTR_INIT(attr);
	    ATTR_SCOPE(attr, PTHREAD_SCOPE_SYSTEM);
	    ATTR_DETACH(attr, PTHREAD_CREATE_DETACHED);
	    THREAD_SIGMASK(SIG_BLOCK, set, oset);

	    if (pthread_create(&tid, &attr, (void *(*)P((void *)))DoWork, (void *)afd) < 0) {
		S5LogUpdate(S5LogDefaultHandle, S5_LOG_ERROR, 0, "Accept: Thread creation failed failed: %m");
		CLOSESOCKET(afd);
	    } else nconns++;

	    THREAD_SIGMASK(SIG_UNBLOCK, set, oset);
#endif
#endif
	    afd = S5InvalidIOHandle;
	} else {
	    if (servermode == SINGLESHOT) CLOSESOCKET(in);
    	    nconns++;
	    DoWork(afd);
	}
    }
}

/* Setup a connection which has already been set up for us by inetd.         */
/* Basically, we're just filling in the right structures and calling the     */
/* work function, HandleProxyConnection...                                   */
void GetStdioConnection(void) {
    ReadConfig();
    fclose(stdout);
    fclose(stderr);
    DoWork(STDIN_FILENO);
}

void GetConnection() {
    Signal(SIGPIPE, SIG_IGN);

    if (servermode == INETD) GetStdioConnection();
    else                     GetNetConnection();
}
