/*
   This file is part of TALER
   Copyright (C) 2022 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_account_kyc_set_status.c
 * @brief Implementation of the account_kyc_set_status function for Postgres
 * @author Iván Ávalos
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_account_kyc_set_status.h"
#include "pg_helper.h"

enum GNUNET_DB_QueryStatus
TMH_PG_account_kyc_set_status (
  void *cls,
  const char *merchant_id,
  const struct TALER_MerchantWireHashP *h_wire,
  const char *exchange_url,
  uint64_t exchange_kyc_serial,
  const struct TALER_ExchangeSignatureP *exchange_sig,
  const struct TALER_ExchangePublicKeyP *exchange_pub,
  struct GNUNET_TIME_Timestamp timestamp,
  bool kyc_ok,
  enum TALER_AmlDecisionState aml_decision)
{
  struct PostgresClosure *pg = cls;
  uint32_t aml32 = (uint32_t) aml_decision;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (merchant_id),
    GNUNET_PQ_query_param_auto_from_type (h_wire),
    GNUNET_PQ_query_param_string (exchange_url),
    GNUNET_PQ_query_param_uint64 (&exchange_kyc_serial),
    GNUNET_PQ_query_param_timestamp (&timestamp),
    GNUNET_PQ_query_param_bool (kyc_ok),
    exchange_pub
    ? GNUNET_PQ_query_param_auto_from_type (exchange_pub)
    : GNUNET_PQ_query_param_null (),
    exchange_sig
    ? GNUNET_PQ_query_param_auto_from_type (exchange_sig)
    : GNUNET_PQ_query_param_null (),
    GNUNET_PQ_query_param_uint32 (&aml32),
    GNUNET_PQ_query_param_end
  };

  check_connection (pg);
  PREPARE (pg,
           "upsert_account_kyc",
           "INSERT INTO merchant_kyc"
           "(kyc_timestamp"
           ",kyc_ok"
           ",exchange_kyc_serial"
           ",account_serial"
           ",exchange_url"
           ",exchange_pub"
           ",exchange_sig"
           ",aml_decision)"
           " SELECT $5, $6, $4, account_serial, $3, $7, $8, $9"
           " FROM merchant_instances"
           " JOIN merchant_accounts USING (merchant_serial)"
           " WHERE merchant_id=$1"
           "  AND h_wire=$2"
           " ON CONFLICT(account_serial,exchange_url) DO "
           "UPDATE"
           " SET exchange_kyc_serial=$4"
           "    ,kyc_timestamp=$5"
           "    ,kyc_ok=$6"
           "    ,exchange_pub=$7"
           "    ,exchange_sig=$8"
           "    ,aml_decision=$9");
  return GNUNET_PQ_eval_prepared_non_select (pg->conn,
                                             "upsert_account_kyc",
                                             params);
}
