// 
//   Copyright (C) 2008 Free Software Foundation, Inc.
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

// This file incorporates work covered by the following copyright and permission
// notice:

/* GStreamer
 * Copyright (C) 2007 David Schleef <ds@schleef.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "gnashconfig.h"
#endif

#include <string.h>
#include <assert.h>

#include "gstbuffersrc.h"


GST_DEBUG_CATEGORY (buffer_src_debug);
#define GST_CAT_DEFAULT buffer_src_debug

static const GstElementDetails buffer_src_details = GST_ELEMENT_DETAILS ("BufferSrc",
    "FIXME",
    "FIXME",
    "autogenerated by makefilter");

enum
{
  PROP_0
};

static GstStaticPadTemplate gst_buffer_src_template =
GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

static void gst_buffer_src_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_buffer_src_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);
static void gst_buffer_src_dispose (GObject * object);
static GstFlowReturn gst_buffer_src_create (GstBaseSrc *src, guint64 offset, guint size, GstBuffer **buf);
static gboolean gst_buffer_src_start (GstBaseSrc * psrc);
static gboolean gst_buffer_src_stop (GstBaseSrc * psrc);
static gboolean gst_buffer_is_seekable (GstBaseSrc *src);
static gboolean gst_buffer_get_size (GstBaseSrc *src, guint64 *size);
static gboolean gst_buffer_check_get_range (GstBaseSrc *src);

GST_BOILERPLATE (GstBufferSrc, gst_buffer_src, GstBaseSrc, GST_TYPE_BASE_SRC);


static void
gst_buffer_src_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (buffer_src_debug, "buffersrc", 0, "buffersrc element");

  gst_element_class_set_details (element_class, &buffer_src_details);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_buffer_src_template));

}

static void
gst_buffer_src_class_init (GstBufferSrcClass * klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;
  GstBaseSrcClass *basesrc_class = (GstBaseSrcClass *) klass;

  gobject_class->set_property = gst_buffer_src_set_property;
  gobject_class->get_property = gst_buffer_src_get_property;
  gobject_class->dispose = gst_buffer_src_dispose;

  basesrc_class->create = gst_buffer_src_create;
  basesrc_class->is_seekable = gst_buffer_is_seekable;
  basesrc_class->start = gst_buffer_src_start;
  basesrc_class->stop = gst_buffer_src_stop;
  basesrc_class->get_size = gst_buffer_get_size;
  basesrc_class->check_get_range = gst_buffer_check_get_range;
}

static void
gst_buffer_src_dispose (GObject * obj)
{
  GstBufferSrc *buffersrc = GST_BUFFER_SRC (obj);

  if (buffersrc->queue) {
    gst_buffer_src_flush (buffersrc);
    g_queue_free (buffersrc->queue);
    buffersrc->queue = NULL;
  }
  if (buffersrc->caps) {
    gst_caps_unref (buffersrc->caps);
    buffersrc->caps = NULL;
  }
  if (buffersrc->mutex) {
    g_mutex_free (buffersrc->mutex);
    buffersrc->mutex = NULL;
  }
  if (buffersrc->cond) {
    g_cond_free (buffersrc->cond);
    buffersrc->cond = NULL;
  }


  G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
gst_buffer_src_init (GstBufferSrc * buffersrc, GstBufferSrcClass * klass)
{
  UNUSEDPAR(klass);

  buffersrc->mutex = g_mutex_new ();
  buffersrc->cond = g_cond_new ();
  buffersrc->queue = g_queue_new ();
  buffersrc->total_size = 0;
}

static void
gst_buffer_src_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  UNUSEDPAR(value);
  GstBufferSrc *buffersrc = GST_BUFFER_SRC (object);

  GST_OBJECT_LOCK (buffersrc);
  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
  GST_OBJECT_UNLOCK (buffersrc);
}

static void
gst_buffer_src_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  UNUSEDPAR(value);
  GstBufferSrc *buffersrc = GST_BUFFER_SRC (object);

  GST_OBJECT_LOCK (buffersrc);
  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
  GST_OBJECT_UNLOCK (buffersrc);
}

static gboolean
gst_buffer_src_start (GstBaseSrc * psrc)
{
  UNUSEDPAR(psrc);

#if 0
  /* FIXME: I don't know if this makes sense */
  buffersrc->end_of_stream = FALSE;
  buffersrc->flush = FALSE;
#endif

  return TRUE;
}

static gboolean
gst_buffer_src_stop (GstBaseSrc * psrc)
{
  UNUSEDPAR(psrc);

  return TRUE;
}


static GstBuffer*
gst_buffer_src_find_offset (GstBufferSrc * buffersrc, guint64* offset)
{
  guint i;

  guint64 bytes = 0;
  guint64 prev_bytes = 0;
  
  for (i = 0; i < g_queue_get_length(buffersrc->queue); ++i) {
    GstBuffer* buf = g_queue_peek_nth(buffersrc->queue, i);
    assert(buf);
    
    prev_bytes = bytes;
    bytes += GST_BUFFER_SIZE(buf);

    if (bytes > *offset) {
      *offset = *offset - prev_bytes;
      return buf;
    }
  }
 
  return NULL;
}

static GstFlowReturn
gst_buffer_src_create (GstBaseSrc *src, guint64 offset, guint size,
		                 GstBuffer **buf)
{  
  GstBufferSrc *buffersrc = GST_BUFFER_SRC (src);
  int ret = GST_FLOW_ERROR;

  g_mutex_lock (buffersrc->mutex);

  while (1) {
    if (!g_queue_is_empty (buffersrc->queue)) {

      guint64 buf_offset = offset;
      GstBuffer* bigbuf = gst_buffer_src_find_offset (buffersrc, &buf_offset);
    
      if (!bigbuf) {
        ret = GST_FLOW_UNEXPECTED;  // EOS
        break;
      }

      guint avail_size = GST_BUFFER_SIZE(bigbuf) - buf_offset;

      guint subsize = MIN(avail_size, size);

      *buf = gst_buffer_create_sub (bigbuf, buf_offset, subsize);

      GST_BUFFER_OFFSET(*buf) = offset;
      GST_BUFFER_OFFSET_END(*buf) = offset + subsize;

      gst_buffer_set_caps (*buf, buffersrc->caps);

      ret = GST_FLOW_OK;
      break;
    }
    if (buffersrc->end_of_stream) {
      buffersrc->end_of_stream = FALSE;
      ret = GST_FLOW_UNEXPECTED;
      break;
    }
    if (buffersrc->flush) {
      buffersrc->flush = FALSE;
      /* FIXME: I don't really know how to do this */
      break;
    }
    g_cond_wait (buffersrc->cond, buffersrc->mutex);
  }

  g_mutex_unlock (buffersrc->mutex);
  return ret;
}

static gboolean 
gst_buffer_is_seekable  (GstBaseSrc *src)
{
  UNUSEDPAR(src);

  return TRUE;
}

static gboolean 
gst_buffer_get_size (GstBaseSrc *src, guint64 *size)
{
  GstBufferSrc* buffersrc = (GstBufferSrc*)src;
  
  g_mutex_lock (buffersrc->mutex);

  *size = buffersrc->total_size;

  g_mutex_unlock (buffersrc->mutex);
  
  return TRUE;
}

static gboolean
gst_buffer_check_get_range (GstBaseSrc* src)
{
  UNUSEDPAR(src);
  return TRUE;
}


static gboolean
plugin_init (GstPlugin * plugin)
{
  GST_DEBUG_CATEGORY_INIT (buffer_src_debug, "buffersrc", 0, "Application source");

  if (!gst_element_register (plugin, "buffersrc", GST_RANK_PRIMARY,
          gst_buffer_src_get_type ()))
    return FALSE;

  return TRUE;
}

GST_PLUGIN_DEFINE_STATIC (GST_VERSION_MAJOR, GST_VERSION_MINOR,
    "buffersrc", "Element application source",
    plugin_init, VERSION, "LGPL", "Gnash's buffer source", "Gnash")


/* external API */

/**
 * gst_buffer_src_push_buffer_unowned:
 *  @buffersrc:
 *  @buffer:
 * 
 * Adds a buffer to the queue of buffers that the buffersrc element will
 * push to its source pad, when requested.
 *
 * This function does NOT take ownership of the buffer. The caller is
 * responsible for destroying the buffer, but can only do so after it has
 * cleared the pipeline and the buffer_src's internal queue has been flushed.
 *
 * This function will create sub-buffers of the passed buffer as needed, so
 * its size does not matter.
 */
void
gst_buffer_src_push_buffer_unowned (GstBufferSrc * buffersrc, GstBuffer * buffer)
{
  g_return_if_fail (buffersrc);
  g_return_if_fail (GST_IS_BUFFER_SRC (buffersrc));

  g_mutex_lock (buffersrc->mutex);

  g_queue_push_tail (buffersrc->queue, buffer);
  buffersrc->total_size += GST_BUFFER_SIZE(buffer);

  g_cond_signal (buffersrc->cond);
  g_mutex_unlock (buffersrc->mutex);
}


/**
 * gst_buffer_src_set_caps:
 * @buffersrc:
 * @caps:
 *
 * Set the capabilities on the buffersrc element.  This function takes
 * ownership of the caps structure.
 */
void
gst_buffer_src_set_caps (GstBufferSrc * buffersrc, GstCaps * caps)
{
  g_return_if_fail (buffersrc);
  g_return_if_fail (GST_IS_BUFFER_SRC (buffersrc));

  gst_caps_replace (&buffersrc->caps, caps);
}

/**
 * gst_buffer_src_flush:
 * @buffersrc:
 *
 * Flushes all queued buffers from the buffersrc element.
 */
void
gst_buffer_src_flush (GstBufferSrc * buffersrc)
{
  GstBuffer *buffer;

  g_return_if_fail (buffersrc);
  g_return_if_fail (GST_IS_BUFFER_SRC (buffersrc));

  g_mutex_lock (buffersrc->mutex);

  while ((buffer = g_queue_pop_head (buffersrc->queue))) {
    gst_buffer_unref (buffer);
  }
  buffersrc->flush = TRUE;

  g_cond_signal (buffersrc->cond);
  g_mutex_unlock (buffersrc->mutex);
}

/**
 * gst_buffer_src_end_of_stream:
 * @buffersrc:
 *
 * Indicates to the buffersrc element that the last buffer queued in the
 * element is the last buffer of the stream.
 */
void
gst_buffer_src_end_of_stream (GstBufferSrc * buffersrc)
{
  g_return_if_fail (buffersrc);
  g_return_if_fail (GST_IS_BUFFER_SRC (buffersrc));

  g_mutex_lock (buffersrc->mutex);

  buffersrc->end_of_stream = TRUE;

  g_cond_signal (buffersrc->cond);
  g_mutex_unlock (buffersrc->mutex);
}
