// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <cc++/process.h>
#include "server.h"

#ifndef	WIN32
#include <sys/utsname.h>
#endif

namespace server {
using namespace ost;
using namespace std;

#if defined(_MSC_VER) && _MSC_VER >= 1300
#if defined(_WIN64_) || defined(__WIN64__)
#define RLL_SUFFIX ".x64"
#elif defined(_M_IX86)
#define RLL_SUFFIX ".x86"
#else
#define RLL_SUFFIX ".xlo"
#endif
#endif

#if defined(__MINGW32__) | defined(__CYGWIN32__)
#define RLL_SUFFIX ".dso"
#endif

#ifdef  W32
#ifndef RLL_SUFFIX
#define RLL_SUFFIX ".rll"
#endif
#endif

#ifndef RLL_SUFFIX
#define RLL_SUFFIX ".dso"
#endif

#ifdef	WIN32
#define	PROMPT_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Audio"
#define	SCRIPT_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Scripts"
#define	VAR_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Files"
#define	LOG_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Logs"
#define	CONFIG_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Config"
#define	RUN_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Logs"
#define	LIBEXEC_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Libxec"

#ifdef	_DEBUG
#define	LIBDIR_FILES	"C:\\Program Files\\GNU Telephony\\CAPE Framework\\debug"
#define	MODULE_FILES	LIBDIR_FILES
#define	DRIVER_FILES	LIBDIR_FILES
#define	PHRASE_FILES	LIBDIR_FILES	
#else	
#define	LIBDIR_FILES	"C:\\Program Files\\Common Files\\GNU Telephony\\Runtime"
#define	MODULE_FILES	"C:\\Program Files\\Common Files\\GNU Telephony\\Bayonne Plugins"
#define	DRIVER_FILES	"C:\\Program Files\\Common Files\\GNU Telephony\\Bayonne Drivers"
#define	PHRASE_FILES	"C:\\Program Files\\Common Files\\GNU Telephony\\Bayonne Languages"
#endif

#else
#define	MODULE_FILES	LIBDIR_FILES
#define	DRIVER_FILES	LIBDIR_FILES
#define	PHRASE_FILES	LIBDIR_FILES
#endif

static Keydata::Define paths[] = {
	{"libexec", LIBEXEC_FILES},
	{"datafiles", VAR_FILES},
	{"logfiles", LOG_FILES},
	{"runfiles", RUN_FILES},
	{"config", CONFIG_FILES},
	{"modules", MODULE_FILES},
	{"drivers", DRIVER_FILES},
	{"phrases", PHRASE_FILES},
	{"scripts", SCRIPT_FILES},
	{"prompts", PROMPT_FILES},
	{NULL, NULL}};

static Keydata::Define server[] = {
	{"user", "bayonne"},
	{"group", "bayonne"},
	{"language", "none"},
	{"location", "us"},
	{"voice", "none/prompts"},
	{"state", "init"},
	{"logging", "warn"},
	{NULL, NULL}};

static Keydata::Define engine[] = {
	{"timeslots", "32"},
	{"priority", "0"},
	{"scheduler", "default"},
	{"events", "32"},
	{"server", "localhost"},
	{"userid", ""},
	{"secret", ""},
	{NULL, NULL}};

#ifdef	WIN32
Keydata keypaths(paths, "/bayonne/paths");
Keydata keyserver(server, "/bayonne/server");
Keydata keyengine(engine, "/bayonne/engine");
#else
Keydata keypaths(paths, "/bayonne/server/paths");
Keydata keyserver(server, "/bayonne/server/server");
Keydata keyengine(engine, "/bayonne/server/engine");
#endif
Keydata keyoptions;

static void commonConfig(void)
{
	char path[256];
	char lang[32];
	const char *env;
	char *p;
	const char *cp;

#ifdef	WIN32
        char nodename[MAX_COMPUTERNAME_LENGTH + 1];
        DWORD namelen = MAX_COMPUTERNAME_LENGTH + 1;
        SYSTEM_INFO sysinfo;
        OSVERSIONINFO osver;
	const char *cpu = "x86";
	const char *tmp = Process::getEnv("TEMP");
	if(!tmp)
		tmp = Process::getEnv("TMP");
	if(!tmp)
		tmp = "C:\\TEMP";
	if(!keypaths.getLast("tmp"))
		keypaths.setValue("tmp", tmp);
	if(!keypaths.getLast("tmpfs"))
		keypaths.setValue("tmpfs", tmp);
	Dir::create(tmp);

        osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
        GetVersionEx(&osver);
        GetComputerName(nodename, &namelen);
        keyserver.setValue("node", nodename);
        GetSystemInfo(&sysinfo);
        switch(sysinfo.wProcessorArchitecture)
        {
        case PROCESSOR_ARCHITECTURE_INTEL:
		cpu="x86";
                break;
        case PROCESSOR_ARCHITECTURE_PPC:
                cpu = "ppc";
                break;
        case PROCESSOR_ARCHITECTURE_MIPS:
                cpu = "mips";
                break;
        case PROCESSOR_ARCHITECTURE_ALPHA:
                cpu = "alpha";
                break;
        }
	keyserver.setValue("node", nodename);
	keyserver.setValue("platform", "w32");
	keyserver.setValue("cpu", cpu);

	snprintf(path, sizeof(path), "%s/bayonne.key", keypaths.getLast("datafiles"));
	keypaths.setValue("keysave", path);
	snprintf(path, sizeof(path), "%s/bayonne.tmp", keypaths.getLast("datafiles"));
	keypaths.setValue("keytemp", path);
#else
	struct utsname uts;
	char hostname[128];
	uname(&uts);
	memset(hostname, 0, sizeof(hostname));
	gethostname(hostname, sizeof(hostname) - 1);
	hostname[sizeof(hostname) - 1] = 0;
	char *hp = strchr(hostname, '.');
	if(hp)
		*hp = 0;

	if(!keyserver.getLast("node"))
		keyserver.setValue("node", hostname);
	
	keyserver.setValue("platform", uts.sysname);
	keyserver.setValue("cpu", uts.machine);

	keypaths.setValue("tmp", "/tmp/.bayonne");
	purgedir("/tmp/.bayonne");
	Dir::create("/tmp/.bayonne");

	if(canModify("/dev/shm"))
	{
		keypaths.setValue("tmpfs", "/dev/shm/.bayonne");
		purgedir("/dev/shm/.bayonne");
		Dir::create("/dev/shm/.bayonne");
	}
	else
		keypaths.setValue("tmpfs", "/tmp/.bayonne");

	snprintf(path, sizeof(path), "%s/bayonne.keys", keypaths.getLast("runfiles"));
	keypaths.setValue("keysave", path);
	snprintf(path, sizeof(path), "%s/.bayonne.keys", keypaths.getLast("runfiles"));
	keypaths.setValue("keytemp", path);
#endif
	env = Process::getEnv("LANG");
	if(env)
	{
		snprintf(lang, sizeof(lang), "%s", env);
		p = strchr(lang, '.');
		if(p)
			*p = 0;
		p = strchr(lang, '_');
		if(p)
		{
			p[1] = tolower(p[1]);
			p[2] = tolower(p[2]);
			if(isalpha(p[1]) && isalpha(p[2]) && !p[3])
				keyserver.setValue("location", ++p);
		}
		lang[0] = tolower(lang[0]);
		lang[1] = tolower(lang[1]);
		if(isalpha(lang[0]) && isalpha(lang[1]))
			keyserver.setValue("language", lang);
	}
	snprintf(path, sizeof(path), "%s/tones.conf", keypaths.getLast("config"));
	TelTone::load(path);
	keypaths.setValue("tones", path);

#ifdef	WIN32
#define	EVENTS	"%s/%s.evt"
#define	ERRORS	"%s/%s.err"
#define	OUTPUT	"%s/%s.out"
#define	CALLER	"%s/%s.cdr"
#define	STATS	"%s/%s.sta"
#else
#define	EVENTS	"%s/%s.events"
#define	ERRORS	"%s/%s.errors"
#define	OUTPUT	"%s/%s.output"
#define	CALLER	"%s/%s.calls"
#define	STATS	"%s/%s.stats"
#endif

	snprintf(path, sizeof(path), ERRORS,
		keypaths.getLast("logfiles"), keyserver.getLast("node"));
	runtime.setValue("errlog", path);

        snprintf(path, sizeof(path), EVENTS,
                keypaths.getLast("logfiles"), keyserver.getLast("node"));
        runtime.setValue("evtlog", path); 

        snprintf(path, sizeof(path), OUTPUT,
                keypaths.getLast("logfiles"), keyserver.getLast("node"));
       	runtime.setValue("output", path);

        snprintf(path, sizeof(path), CALLER,
                keypaths.getLast("logfiles"), keyserver.getLast("node"));
        runtime.setValue("calls", path);

        snprintf(path, sizeof(path), STATS,
                keypaths.getLast("logfiles"), keyserver.getLast("node"));
        runtime.setValue("stats", path);

	runtime.setValue("include", keypaths.getLast("scripts"));
	keypaths.setValue("include", keypaths.getLast("scripts"));
	runtime.setValue("tmp", keypaths.getLast("tmp"));
	runtime.setValue("tmpfs", keypaths.getLast("tmpfs"));
	runtime.setValue("prompts", keypaths.getLast("prompts"));
	runtime.setValue("voice", keyserver.getLast("voice"));
	runtime.setValue("logfiles", keypaths.getLast("logfiles"));
	runtime.setValue("runfiles", keypaths.getLast("runfiles"));
	runtime.setValue("servername", "bayonne2");
	runtime.setValue("serverversion", VERSION);
	
	cp = keyengine.getLast("server");
	if(cp)
		runtime.setValue("server", cp);
	cp = keyengine.getLast("proxy");
	if(cp)
		runtime.setValue("proxy", cp);

	runtime.setValue("userid", keyengine.getLast("userid"));
	runtime.setValue("secret", keyengine.getLast("secret"));
	runtime.setValue("location", keyserver.getLast("location"));

	cp = keyengine.getLast("autostack");
	if(cp)
		Thread::setStack(atoi(cp) * 1024);

	cp = keyengine.getLast("stepping");
	if(cp)
		Bayonne::step_timer = atoi(cp);

	cp = keyengine.getLast("reset");
	if(cp)
		Bayonne::reset_timer = atoi(cp);

	cp = keyengine.getLast("exec");
	if(cp)
		Bayonne::exec_timer = atoi(cp) * 1000;

	cp = keyengine.getLast("autostep");
	if(cp)
		Script::autoStepping = atoi(cp);

	cp = keyengine.getLast("faststep");
	if(cp)
		Script::fastStepping = atoi(cp);

	cp = keyengine.getLast("paging");
	if(!cp)
		cp = keyengine.getLast("pagesize");
	if(cp)
	{
		Script::pagesize = atoi(cp);
		Script::symlimit = Script::pagesize - sizeof(Script::Symbol) - 32;
	}

	cp = keyengine.getLast("symsize");
	if(cp)
		Script::symsize = atoi(cp);	

	Script::fastStart = false;
}

void testConfig(char *argv0)
{
#ifdef	WIN32
#ifdef	_DEBUG
	keypaths.setValue("phrases", "Debug");
	keypaths.setValue("drivers", "Debug");
	keypaths.setValue("modules", "Debug");
#else
        keypaths.setValue("phrases", "Release");
        keypaths.setValue("drivers", "Release");
        keypaths.setValue("modules", "Release"); 
#endif
	keypaths.setValue("prompts", "../audio");
	keypaths.setValue("scripts", "../scripts");
	keypaths.setValue("config", "../config");
	Process::setEnv("DRIVERCONFIG", "../config/driver.conf", true);
	Process::setEnv("MODULECONFIG", "../config/module.conf", true);
	Process::setEnv("SPANCONFIG", "../config/spans.conf", true);
#else
	char pbuf[256];
	char *p = strrchr(argv0, '/');
	if(p)
	{
		*p = 0;
		chdir(p);
		getcwd(pbuf, sizeof(pbuf));
	}
	else
		getcwd(pbuf, sizeof(pbuf));

	keypaths.setValue("libexec", pbuf);
	keypaths.setValue("modules", "../modules");
	keypaths.setValue("drivers", "../modules");
	keypaths.setValue("phrases", "../modules/languages/.libs");
	keypaths.setValue("prompts", SOURCE_FILES "/audio");
	keypaths.setValue("scripts", SOURCE_FILES "/scripts");
	keypaths.setValue("config", SOURCE_FILES "/config");
#endif
	keypaths.setValue("datafiles", "tempfiles");
	keypaths.setValue("logfiles", "tempfiles");
	keypaths.setValue("runfiles", "tempfiles");
	keypaths.setValue("modexec", "../scripts");
	Dir::create("tempfiles");
	keyserver.setValue("state", "test");
	keyserver.setValue("logging", "debug");
	commonConfig();
}	

void loadConfig(void)
{
#ifdef	WIN32
	char pbuf[256];
	const char *config = keypaths.getLast("config");

	snprintf(pbuf, sizeof(pbuf), "%s/server.conf", config); 
	keypaths.loadFile(pbuf, "paths");
	keyserver.loadFile(pbuf, "server");
	keyengine.loadFile(pbuf, "engine");

	snprintf(pbuf, sizeof(pbuf), "%s/driver.conf", config);
	Process::setEnv("DRIVERCONFIG", pbuf, true); 
	snprintf(pbuf, sizeof(pbuf), "%s/module.conf", config);
	Process::setEnv("MODULECONFIG", pbuf, true);

//	Process::setGroup(keyserver.getLast("group"));
        Dir::create(keypaths.getLast("datafiles"));
        Dir::create(keypaths.getLast("runfiles"));
        Dir::create(keypaths.getLast("logfiles"));
#else
	char homepath[256];
	const char *home = NULL;
	bool homed = false;

	if(getuid())
		home = Process::getEnv("HOME");

	if(!isDir(keypaths.getLast("config")))
		keypaths.setValue("config", "/etc/bayonne");

	if(!isDir(keypaths.getLast("scripts")))
		keypaths.setValue("scripts", "/etc/bayonne.d");

	if(home)
		snprintf(homepath, sizeof(homepath), "%s/.bayonne", home);
	else
		strcpy(homepath, "tempfiles");

	if(getuid() && home)
	{
		keypaths.load("~bayonne/paths");
		keyserver.load("~bayonne/server");
		keyengine.load("~bayonne/engine");
	}

	umask(007);

	Process::setGroup(keyserver.getLast("group"));
		
	Dir::create(keypaths.getLast("datafiles"));
	Dir::create(keypaths.getLast("runfiles"));
	Dir::create(keypaths.getLast("logfiles"));

	if(!getuid())
		goto live;

	if(!canModify(keypaths.getLast("datafiles")))
	{
		homed = true;
		keypaths.setValue("datafiles", homepath);
	}

	if(!canModify(keypaths.getLast("runfiles")))
	{
		homed = true;
		keypaths.setValue("runfiles", homepath);
	}

	if(!canModify(keypaths.getLast("logfiles")))
	{
		homed = true;
		keypaths.setValue("logfiles", homepath);
	}

	if(homed)
	{
		umask(077);
	        Dir::create(homepath, File::attrPrivate);
	}
live:
#endif
	chdir(keypaths.getLast("datafiles"));
	keyserver.setValue("state", "live");
	commonConfig();
}

void parseConfig(char **argv)
{
	char buffer[64];
	char path[256];
	const char *prefix;
	char *option;
	char *script = NULL;
	char *argv0 = *(argv++);
	char *cmd = *(argv++);
	char *driver = NULL;
	char *ext;
	unsigned scrcount = 0;
	unsigned timeslots = atoi(keyengine.getLast("timeslots"));
	unsigned minslots = 0;

	if(**argv != '-')
		driver=*(argv++);

retry:
	if(!*argv)
		goto skip;

	option = *argv;
        if(!strcmp("--", option))
        {
                ++argv;
                goto skip;
        }

        if(!strnicmp("--", option, 2))
                ++option;

	if(!strnicmp(option, "-driver=", 8))
	{
		driver = option + 8;
		++argv;
		goto retry;
	} 

	if(!strnicmp(option, "-logging=", 9))
        {
                keyserver.setValue("logging", option + 9);
                ++argv;
                goto retry;
        }


        if(!strnicmp(option, "-libexec=", 9))
        {
                keypaths.setValue("libexec", option + 9);
                ++argv;
                goto retry;
        }
            

	if(!strnicmp(option, "-vvv", 4))
	{
		keyserver.setValue("logging", "debug");
		++argv;
		goto retry;
	}

        if(!strnicmp(option, "-vv", 4))
        {
                keyserver.setValue("logging", "notice");
                ++argv;
                goto retry;
        }

        if(!stricmp(option, "-v"))
        {
                keyserver.setValue("logging", "error");
                ++argv;
                goto retry;
        }

	if(!strnicmp(option, "-sip=", 5))
	{
		runtime.setValue("slots.sip", option + 5);
		keyoptions.setValue("sip.stack", option + 5);
		minslots += atoi(option + 5);
		++argv;
		goto retry;
	} 

        if(!strnicmp(option, "-h323=", 6))
        {
		runtime.setValue("slots.h323", option + 6);
                keyoptions.setValue("h323.stack", option + 6);
		minslots += atoi(option + 6);
                ++argv;
                goto retry;            
        } 

	if(!strnicmp(option, "-voice=", 7))
	{
		keyserver.setValue("voice", option + 7);
		++argv;
		goto retry;
	} 

	if(!strnicmp(option, "-location=", 10))
	{ 
		Process::setEnv("LANG", option + 10, true);
		keyserver.setValue("location", option + 10);
		++argv;
		goto retry;
	}

        if(!strnicmp(option, "-timeslots=", 11))
        {
		timeslots = atoi(option + 11);
                ++argv;
                goto retry;
        }

        if(!stricmp(option, "-libexec"))
        {
                ++argv;
                if(!*argv)
                {
                        cerr << "bayonne: " << cmd << ": -libexec: missing argument" << endl;
                        exit(-1);
                }
                keypaths.setValue("libexec", *(argv++));
                goto retry;
        }

	if(!stricmp(option, "-logging"))
        {
                ++argv;
                if(!*argv)
                {
			cerr << "bayonne: " << cmd << ": -logging: missing argument" << endl;
                        exit(-1);
                }
                keyserver.setValue("logging", *(argv++));
                goto retry;
        }

        if(!stricmp(option, "-location"))  
        {
                ++argv;
                if(!argv) 
                {
        		cerr << "bayonne: " << cmd << ": -location: missing argument" << endl;
                        exit(-1);
                }      
		Process::setEnv("LANG", *argv, true);
                keyserver.setValue("location", *(argv++));
                goto retry;
        }


	if(!stricmp(option, "-voice"))
	{
		++argv;
		if(!argv)
		{
			cerr << "bayonne: " << cmd << ": -voice: missing argument" << endl;
			exit(-1);
		}
		keyserver.setValue("voice", *(argv++));
		goto retry;
	}

        if(!stricmp(option, "-driver"))
        {
                ++argv;
                if(!*argv)
                {
			cerr << "bayonne: " << cmd << ": -driver: missing argument" << endl;
                        exit(-1);
                }
                driver = *(argv++);
                goto retry;
        }

	if(!stricmp(option, "-sip"))
	{
		++argv;
		if(!*argv)
		{
			cerr << "bayonne: " << cmd << ": -sip: missing timeslots" << endl;
			exit(-1);
		}
		runtime.setValue("slots.sip", *argv);
		keyoptions.setValue("sip.stack", *(argv++));
		goto retry;
	}

        if(!stricmp(option, "-h323"))      
        {
                ++argv;
                if(!*argv)
                {            
        		cerr << "bayonne: " << cmd << ": -h323: missing timeslots" << endl;                              
			exit(-1);
                }
		runtime.setValue("slots.h323", *argv);
                keyoptions.setValue("h323.stack", *(argv++));
                goto retry;
        }
           
        if(!stricmp(option, "-timeslots"))
        {
                ++argv;
                if(!*argv)
                {
			cerr << "bayonne: " << cmd << ": -timeslots: missing argument" << endl;
                        exit(-1);
                }
		timeslots = atoi(*(argv++));
                goto retry;
	}

skip:
	if(!driver)
	{
		cerr << "bayonne: driver missing" << endl;
		exit(-1);
	}

#ifdef	WIN32
	Process::setEnv("DISPLAY", "yes", true);
	if(!stricmp(driver, "console"))
	{
		Process::setEnv("DISPLAY", "", true);
		driver = "soundcard";
	}
#else
	if(!stricmp(driver, "console"))
	{
		Process::setEnv("DISPLAY", "", true);
		driver = "soundcard";
	}
#endif

	ext = strchr(driver, ':');
	if(ext)
	{
		*(ext++) = 0;
		minslots += atoi(ext);
		snprintf(buffer, sizeof(buffer), "slots.%s", driver);
		runtime.setValue(buffer, ext);
	}

	if(timeslots < minslots)
		timeslots = minslots;

	snprintf(buffer, sizeof(buffer), "%s.stack", driver);
	keyoptions.setValue(buffer, "0");
		
	keyserver.setValue("driver", driver);
	snprintf(path, 10, "%d", timeslots);
	keyserver.setValue("timeslots", path);
 
	prefix = keypaths.getLast("drivers");
#ifdef	WIN32
	snprintf(path, sizeof(path), "%s/%s" RLL_SUFFIX, prefix, driver);
#else
	if(!strnicmp(prefix, "..", 2))
		snprintf(path, sizeof(path), "%s/%s/.libs/%s.ivr", prefix, driver, driver);
	else
		snprintf(path, sizeof(path), "%s/%s.ivr", prefix, driver);
#endif
	keyoptions.setValue("driver", path);

	if(!canAccess(path))
	{
		cerr << "server exiting; " << path << " missing" << endl;
		exit(-1);
	}
	
	while(*argv)
	{
		ext = strrchr(*argv, '.');
		if(!ext)
		{
			prefix = keypaths.getLast("modules");
#ifdef	WIN32
			snprintf(path, sizeof(path), "%s/%s" RLL_SUFFIX, prefix, *argv);
#else
			if(!strnicmp(prefix, "..", 2))
				snprintf(path, sizeof(path), "%s/%s/.libs/%s.dso", prefix, *argv, *argv);
			else
				snprintf(path, sizeof(path), "%s/%s.dso", prefix, *argv);
#endif			
			if(!canAccess(path))
				cerr << "bayonne: " << *argv << ": cannot access" << endl;
			else
				keyoptions.setValue("modules", path);
		}
		else if(!stricmp(ext, ".scr"))
		{
			prefix = strrchr(*argv, '/');
			if(!prefix)
				prefix = strrchr(*argv, '\\');
			if(!prefix)
				prefix = strrchr(*argv, ':');

			if(prefix)
				++prefix;

			if(!canAccess(*argv))
				cerr << "bayonne: " << *argv << ": cannot access" << endl;
			else
			{
				if(!scrcount++)
					script = *argv;
				else
					script = NULL;

				runtime.setValue(prefix, *argv);
			}
		}
		else
		{
			if(canAccess(*argv))
				keyoptions.setValue("configs", *argv);
			else
				cerr << "bayonne: " << *argv << ": cannot access" << endl;
		}
		++argv;
	}

	Bayonne::allocate(timeslots, dynamic_cast<ScriptCommand *>(&runtime));

	if(!script)
	{
		runtime.setValue("scripts", keypaths.getLast("scripts"));
		return;
	}
	
	option = strrchr(script, '/');
	if(!option)
		option = strrchr(script, '\\');
	if(!option)
		option = strrchr(script, ':');
	if(option)
		script = ++option;
	setString(buffer, sizeof(buffer), script);
	script = buffer;
	ext = strrchr(script, '.');
	if(ext)
		*ext = 0;

	runtime.setValue("startup", script);
}		

void dumpConfig(Keydata &keydef)
{
	unsigned count = keydef.getCount();
	char **keys = new char *[count + 1];
	count = keydef.getIndex(keys, count + 1);
	while(count--)
	{
		cout << *keys << " = " << keydef.getLast(*keys) << endl;
		++keys;
	}
}

void purgedir(const char *dir)
{
	const char *cp;
	if(isDir(dir) && canAccess(dir))
	{
		DirTree dt(dir, 16);
		while(NULL != (cp = dt.getPath()))
			remove(cp);
		remove(dir);
	}
}

} // namespace
