/****************************************************************************
  This file is part of the Freedom Remailer.  It is mostly borrowed from
  Matt Ghio's statistics functions as given in his remailer distribution,
  but significant changes have been made by John B. Fleming and Johannes
  Kroeger.  Changes are
  (C) 1995-1997  John B. Fleming (jfleming@indiana.edu)
  (C) 1997-1998  Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/file.h>
#include "freedom.h"

void
updatestats (enum stats_flag flag)
{
  char month[24][5];
  struct tm *curtime;
  time_t now;
  FILE *file;
  struct flock lock;
  int lockfd, hour, currenthour, x;
  int Tm, Tp;
  int date[24], Hm[24], Hp[24];

  now = time (NULL);
  curtime = localtime (&now);
  currenthour = curtime->tm_hour;

  /* Don't manipulate the stats file before locking stats.lock exclusively.  */
  lockfd = open ("stats.lock", O_WRONLY | O_CREAT | O_TRUNC, 0600);
  lock.l_type = F_WRLCK;
  lock.l_start = 0;
  lock.l_whence = 0;
  lock.l_len = 0;
  fcntl (lockfd, F_SETLKW, &lock);

  if (!(file = fopen (STATS_FILE, "r")))
    {
      /* STATS_FILE does not exist; make a new empty stats file.  */
      file = fopen (STATS_FILE, "w+");
      fputs ("0\n", file);
      fputs ("0 0\n", file);
      for (x = 0; x < 24; x++)
	/* First day, no messages, tmp for month.  */
	fputs ("tmp 0 0 0\n", file);
      rewind (file);
    }
  fscanf (file, "%d", &hour);
  fscanf (file, "%d %d", &Tm, &Tp);
  for (x = 0; x < 24; x++)
    fscanf (file, "%s %d %d %d", month[x], &date[x], &Hm[x], &Hp[x]);
  fclose (file);

  x = hour;
  while (x != currenthour)
    {
      if (x > 0)
	{
	  strcpy (month[x], month[x - 1]);
	  date[x] = date[x - 1];
	}
      else
	{
	  strftime (month[0], 5, "%b", curtime);
	  date[0] = curtime->tm_mday;
	}
      Hm[x] = Hp[x] = 0;
      x++;
      if (x > 23)
	x = 0;
    }

  if (hour != currenthour)
    {
      Hm[hour] = Tm;
      Hp[hour] = Tp;
      Tm = Tp = 0;
    }

  if (flag == STATS_MESSAGE)
    Tm++;
  else if (flag == STATS_PGP)
    Tm++, Tp++;

  file = fopen (STATS_FILE, "w");
  fprintf (file, "%d\n", currenthour);
  fprintf (file, "%d %d\n", Tm, Tp);
  for (x = 0; x < 24; x++)
    fprintf (file, "%s %d %d %d\n", month[x], date[x], Hm[x], Hp[x]);
  fclose (file);

  /* Unlock and close stats.lock.  */
  lock.l_type = F_UNLCK;
  fcntl (lockfd, F_SETLKW, &lock);
  close (lockfd);
}

void
mailstats (const char *to)
{
  char sendmail[BUFSIZ], month[24][5];
  struct tm *curtime;
  time_t now;
  FILE *file;
  struct flock lock;
  int lockfd, hour, currenthour, x;
  int Tm, Tp, x0, y, mmax = 0;
  int date[24], Hm[24], Hp[24];
  float mscale;

  now = time (NULL);
  curtime = localtime (&now);
  currenthour = curtime->tm_hour;

  /* Don't manipulate the stats file before locking stats.lock exclusively.  */
  lockfd = open ("stats.lock", O_WRONLY | O_CREAT | O_TRUNC, 0600);
  lock.l_type = F_WRLCK;
  lock.l_start = 0;
  lock.l_whence = 0;
  lock.l_len = 0;
  fcntl (lockfd, F_SETLKW, &lock);

  if ((file = fopen (STATS_FILE, "r")))
    {
      fscanf (file, "%d", &hour);
      fscanf (file, "%d %d", &Tm, &Tp);
      for (x = 0; x < 24; x++)
	fscanf (file, "%s %d %d %d", month[x], &date[x], &Hm[x], &Hp[x]);
      fclose (file);

      x = hour;
      while (x != currenthour)
	{
	  if (x > 0)
	    {
	      strcpy (month[x], month[x - 1]);
	      date[x] = date[x - 1];
	    }
	  else
	    {
	      strftime (month[0], 5, "%b", curtime);
	      date[0] = curtime->tm_mday;
	    }
	  Hm[x] = Hp[x] = 0;
	  x++;
	  if (x > 23)
	    x = 0;
	}

      if (hour != currenthour)
	{
	  Hm[hour] = Tm;
	  Hp[hour] = Tp;
	  Tm = Tp = 0;
	}

      sprintf (sendmail, "%s", SENDMAIL);
      file = popen (sendmail, "w");
      fprintf (file, "From: %s <%s>\n", REMAILER_NAME, REMAILER_ADDR);
      fprintf (file, "Subject: Cypherpunk Remailer Statistics\n"
	       "To: %s\n\n", to);
      fprintf (file, "Statistics for last 24 hours from anonymous remailer\n"
	       "%s\n\n", REMAILER_NAME);
      fprintf (file, "Number of messages per hour from "
	       "%s %2d %02d:00 to %s %2d %02d:59\n\n",
	       month[23], date[23], currenthour, month[0], date[0],
	       (currenthour + 23) % 24);
      for (x = 0; x < 24; x++)
	mmax = (Hm[x] > mmax) ? Hm[x] : mmax;
      mscale = (mmax > 67) ? 67.0 / mmax : 1.0;
      for (x0 = 0; x0 < 24; x0++)
	{
	  x = (x0 + currenthour) % 24;
	  fprintf (file, "%02d:00 (%3d) ", x, Hm[x]);
	  if (Hm[x] > 0)
	    {
	      for (y = 0; y < Hp[x] * mscale; y++)
		fputs ("*", file);
	      for (y = 0; y < (Hm[x] - Hp[x]) * mscale; y++)
		fputs ("+", file);
	      Tm += Hm[x];
	      Tp += Hp[x];
	    }
	  fputs ("\n", file);
	}
      fprintf (file, "\nTotal messages received in last 24 hours:\t%5d\n", Tm);
      if (ALLOW_PGP)
	fprintf (file, "Number of PGP encrypted messages:\t\t%5d\n", Tp);
      fprintf (file, "Number of plaintext messages:\t\t\t%5d\n", Tm - Tp);
      pclose (file);
    }

  /* Unlock and close stats.lock.  */
  lock.l_type = F_UNLCK;
  fcntl (lockfd, F_SETLKW, &lock);
  close (lockfd);
}

void
mailconf (const char *to)
{
  FILE *p, *infile;
  char sendmail[BUFSIZ];

  sprintf (sendmail, "%s", SENDMAIL);
  p = popen (sendmail, "w");
  fprintf (p, "From: %s <%s>\n", REMAILER_NAME, REMAILER_ADDR);
  fprintf (p, "Subject: Capabilities of the %s\n", REMAILER_NAME);
  fprintf (p, "To: %s\n\n", to);

  fprintf (p, "Remailer-Type: Freedom %s\n\n", VERSION);

  if (SIZE_LIMIT)
    fprintf (p, "Maximum message size:\t\t\t\t%d bytes\n", SIZE_LIMIT);
  else
    fputs ("Maximum message size:\t\t\t\tunlimited\n", p);

  if (USE_STATS)
    fputs ("Message statistics autoresponder:\t\tenabled\n", p);
  else
    fputs ("Message statistics autoresponder:\t\tdisabled\n", p);

  if (ALLOW_LTIME)
    fputs ("Queuing messages with Latent-Time:\t\tenabled\n\n", p);
  else
    fputs ("Queuing messages with Latent-Time:\t\tdisabled\n\n", p);

#ifdef USE_RX
  fputs ("Blocked header lines (regular expressions):\n", p);
#else /* !USE_RX */
  fputs ("Blocked header lines:\n", p);
#endif /* !USE_RX */
  if ((infile = fopen (HDRFILTER, "r")))
    {
      print_list (infile, p);
      fclose (infile);
    }
  fputs ("\n", p);

  if (ALLOW_PGP == 2)
    fputs ("Unencrypted incoming messages:\t\t\tnot supported\n", p);
  else
    fputs ("Unencrypted incoming messages:\t\t\tsupported\n", p);

  if (ALLOW_PGP)
    {
      fputs ("PGP-encrypted incoming messages:\t\tsupported\n", p);
      fputs ("Conventional encryption with Encrypt-Key:\tsupported\n", p);
      fputs ("Subject encryption with Encrypt-Subject:\tsupported\n", p);
      fputs ("Encryption to next hop with Encrypt-To:\t\tsupported\n\n", p);
      fputs ("Cypherpunk remailer keys available for Encrypt-To:\n", p);
      print_list (repgplist, p);
      fputs ("\n", p);
    }
  else
    {
      fputs ("PGP-encrypted incoming messages:\t\tnot supported\n", p);
      fputs ("Conventional encryption with Encrypt-Key:\tnot supported\n", p);
      fputs ("Subject encryption with Encrypt-Subject:\tsupported\n", p);
      fputs ("Encryption to next hop with Encrypt-To:\t\tnot supported\n\n", p);
    }

  if (ALLOW_GPG)
    fputs ("GNUPG-encrypted incoming messages:\t\tsupported\n\n", p);
  else
    fputs ("GNUPG-encrypted incoming messages:\t\tnot supported\n\n", p);

  if (USE_MIX == 2)
    fputs ("Transparent remixing to next hop:\t\tenabled\n", p);
  else
    fputs ("Transparent remixing to next hop:\t\tdisabled\n", p);

  if (USE_MIX)
    {
      fputs ("Reordering through Mixmaster pool:\t\tenabled\n", p);
      fputs ("Remixing to next hop with Remix-To:\t\tenabled\n\n", p);
      fputs ("Mixmaster remailer keys available for Remix-To:\n", p);
      print_list (remixlist, p);
      fputs ("\n", p);
    }
  else
    {
      fputs ("Reordering through Mixmaster pool:\t\tdisabled\n", p);
      fputs ("Remixing to next hop with Remix-To:\t\tdisabled\n\n", p);
    }

  if (ALLOW_POST)
    {
      if (ALLOW_POST == 1)
	fputs ("News posting with Anon-Post-To:\t\tvia local news server\n\n",
	       p);
      else if (ALLOW_POST == 2)
	fprintf (p, "News posting with Anon-Post-To:\t\tvia %s\n\n", MAIL2NEWS);
#ifdef USE_RX
      fputs ("Blocked newsgroups (regular expressions):\n", p);
#else /* !USE_RX */
      fputs ("Blocked newsgroups:\n", p);
#endif /* !USE_RX */
      if ((infile = fopen (GROUP_BLOCK, "r")))
	{
	  print_list (infile, p);
	  fclose (infile);
	}
      fputs ("\n", p);
    }
  else
    fputs ("News posting with Anon-Post-To:\t\t\tdisabled\n\n", p);

  if (ALLOW_WWW)
    {
      fputs ("Retrieving WWW files with Get-URL:\t\tenabled\n\n", p);
#ifdef USE_RX
      fputs ("Blocked URLs (regular expressions):\n", p);
#else /* !USE_RX */
      fputs ("Blocked URLs:\n", p);
#endif /* !USE_RX */
      if ((infile = fopen (URL_BLOCK, "r")))
	{
	  print_list (infile, p);
	  fclose (infile);
	}
    }
  else
    fputs ("Retrieving WWW files with Get-URL:\t\tdisabled\n", p);

  pclose (p);
}
