/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997 Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/file.h>
#include "config.h"

int main(int argc, char *argv[])
{
  FILE *file1, *file2, *p;
  char tempfilename[BUFSIZ], bFilename[BUFSIZ], dFilename[BUFSIZ],
       eFilename[BUFSIZ], cFilename[BUFSIZ], qFilename[BUFSIZ],
       mFilename[BUFSIZ], kFilename[BUFSIZ], uFilename[BUFSIZ],
       input[BUFSIZ], send_to[BUFSIZ], message_from[BUFSIZ], filename[17];
  int lockfd, recursing, trimmings = 0;
  enum message_type anon;
#ifdef SIZE_LIMIT
  struct stat tempfilestat;
#endif /* SIZE_LIMIT */

  umask (007);
  setuid(geteuid());
  setgid(getegid());

  lockfd = open(REMAILER_DIR "/lock", O_CREAT | O_WRONLY, 0660);

  if (argc > 1) {
    if (strcmp(argv[1], "-q") == 0) {
      if (DEBUG) {
	printf("Running the latent message queue.\n");
	fflush(stdout);
      }
      runlatentqueue();
      exit(0);
    } else {
      printf("%s: Usage: %s [-q]\n", argv[0], argv[0]);
      printf("\t-q:\trun queue of latent messages\n");
      printf("\tWith no options, reads mail on stdin and processes.\n");
      exit(1);
    }
  }

  sprintf(filename, genid());
  sprintf(qFilename, "%s/q%s", QUEUE_DIR, filename);
  sprintf(mFilename, "%s/m%s", QUEUE_DIR, filename);
  sprintf(tempfilename, "%s/mail%s", REMAILER_DIR, filename);
  sprintf(bFilename, "%sb", tempfilename);
  sprintf(dFilename, "%sd", tempfilename);
  sprintf(eFilename, "%se", tempfilename);
  sprintf(cFilename, "%sc", tempfilename);
  sprintf(kFilename, "%sk", tempfilename);
  sprintf(uFilename, "%su", tempfilename);

  /* copy message coming in on stdin to a temporary file */
  file1 = fopen(tempfilename, "w");
  if (file1 == NULL) {
    fprintf(stderr, "%s: can't write file %s\n", argv[0], tempfilename);
    fflush(stderr);
    exit(1);
  } else {
    if (DEBUG) {
      printf("\n\nWriting new message to %s\n", tempfilename);
      fflush(stdout);
    }
  }
  while (fgets(input, sizeof(input), stdin)) {
    fputs(input, file1);
  }
  fclose(file1);

#ifdef SIZE_LIMIT
  stat(tempfilename, &tempfilestat);
  if (tempfilestat.st_size > SIZE_LIMIT) {
    fprintf(stderr, "%s: message too large\n", argv[0]);
    fflush(stderr);
    if (DEBUG) {
      printf("Message too large (%lu bytes).\nDeleting %s\n",
	     tempfilestat.st_size, tempfilename);
      fflush(stdout);
    }
    unlink(tempfilename);
    exit(0);
  }
#endif /* SIZE_LIMIT */

#ifndef DEFAULT_LATENCY
#define DEFAULT_LATENCY	"+0:00"
#endif /* DEFAULT_LATENCY */
  if (flock(lockfd, LOCK_EX | LOCK_NB) < 0) {
    fprintf(stderr, "%s: already running\n", argv[0]);
    fflush(stderr);
    if (DEBUG) {
      printf("Queuing message with latency %s.\n", DEFAULT_LATENCY);
      fflush(stdout);
    }
    queuelatent(tempfilename, qFilename, mFilename, DEFAULT_LATENCY);
    unlink(tempfilename);
    exit (0);
  }

  recursing = 1;
  while (recursing) {
    recursing = 0;

    anon = scan_message(tempfilename, send_to, message_from);
    if (anon == NULL_MESSAGE) {
      if (DEBUG) {
	printf("Discarding message.\n");
	fflush(stdout);
      }
      unlink(tempfilename);
    } else if ((anon == PGP_MESSAGE) && (USE_PGP)) {
      if (DEBUG) {
	printf("Decrypting message.\n");
	fflush(stdout);
      }
      if (USE_STATS)
	updatestats(STATS_PGP);
      pgpdecrypt(tempfilename, dFilename);
      rename(tempfilename, eFilename);
      file1 = fopen(tempfilename, "w");
      file2 = fopen(dFilename, "r");
      if (file2 != NULL) {
	fputs("PGP-Recursive: Yes\n\n", file1);
	while (fgets(input, sizeof(input), file2)) {
	  fputs(input, file1);
	}
	fclose(file2);
	file2 = fopen(eFilename, "r");
	while(fgets(input, sizeof(input), file2)) {
	  if (strstr(input, "-----END PGP MESSAGE-----") != NULL) {
	    while(fgets(input, sizeof(input), file2)) {
	      fputs(input, file1);
	    }
	  }
	}
	fclose(file2);
      }
      fclose(file1);
      unlink(eFilename);
      unlink(dFilename);
      recursing = 1;
    } else if (anon == ANON_MESSAGE) {
      if (DEBUG) {
	printf("Anonymous message detected.\n");
	fflush(stdout);
      }
      if (USE_STATS) { /* do stats whether or not blocked */
        updatestats(STATS_MESSAGE);
      }
      if (blocked(message_from, SOURCE_BLOCK)) {
	if (DEBUG) {
	  printf("Blocked From: %s\nNot remailing.\n", message_from);
	  fflush(stdout);
	}
	unlink(tempfilename);
      } else {
	remove_headers(tempfilename, bFilename, send_to);
	if (scan_subject(bFilename) == -1) {
	  if (DEBUG) {
	    printf("Not remailing.\n");
	    fflush(stdout);
	  }
	  unlink(tempfilename);
	  unlink(bFilename);
	} else {
	  if (DEBUG) {
	    printf("Remailing.\n");
	    fflush(stdout);
	  }
	  sprintf(input, "%s", SENDMAIL);
	  p = (FILE *) popen(input, "w");
	  fprintf(p, "From: %s <%s>\n", ANON_NAME, ANON_ADDR);
	  fprintf(p, DISCLAIMER);
	  if (scan_subject(bFilename) == 0)
	    fprintf(p, "Subject: None\n");
	  file1 = fopen(bFilename, "r");
	  if (file1 != NULL) {
	    while (fgets(input, sizeof(input), file1)) {
	      fputs(input, p);
	    }
	    fclose(file1);
	  }
	  pclose(p);
	  unlink(tempfilename);
	  unlink(bFilename);
	}
      }
    }
#ifndef MAIL2NEWS
      else if ((anon == ANON_POST) && (ALLOW_POST)) {
      if (DEBUG) {
	printf("Anonymous post detected.\n");
	fflush(stdout);
      }
      if (USE_STATS)
	updatestats(STATS_POST);
      if (blocked(message_from, SOURCE_BLOCK)) {
	if (DEBUG) {
	  printf("Blocked From: %s\nNot posting.\n", message_from);
	  fflush(stdout);
	}
        unlink(tempfilename);
      } else {
	remove_headers(tempfilename, bFilename, send_to);
	if (scan_subject(bFilename) == -1) {
	  if (DEBUG) {
	    printf("Not posting.\n");
	    fflush(stdout);
	  }
	  unlink(tempfilename);
	  unlink(bFilename);
	} else {
	  if (DEBUG) {
	    printf("Posting.\n");
	    fflush(stdout);
	  }
	  sprintf(input, "%s -h", INEWS);
	  p = (FILE *) popen(input, "w");
	  fprintf(p, "Organization: %s\n", ORGANIZATION);
	  fprintf(p, "From: %s <%s>\n", ANON_NAME, ANON_ADDR);
	  fprintf(p, DISCLAIMER);
	  if (!scan_subject(bFilename))
	    fprintf(p, "Subject: None\n");
	  file1 = fopen(bFilename, "r");
	  if (file1 != NULL) {
	    while (fgets(input, sizeof(input), file1)) {
              fputs(input, p);
	    }
	    fclose(file1);
	  }
	  pclose(p);
	  unlink(tempfilename);
	  unlink(bFilename);
	}
      }
    }
#endif /* !MAIL2NEWS */
      else if ((anon == STATS_REQ) && (USE_STATS)) {
      if (DEBUG) {
	printf("Statistics were requested.\n");
	fflush(stdout);
      }
      updatestats(STATS_STAT);
      if (blocked(message_from, DEST_BLOCK)) {
	if (DEBUG) {
	  printf("Blocked From: %s\nNot mailing.\n", message_from);
	  fflush(stdout);
	}
      } else {
	if (DEBUG) {
	  printf("Mailing.\n");
	  fflush(stdout);
	}
	mailstats(message_from);
      }
      unlink(tempfilename);
    } else if (anon == HELP_REQ) {
      if (DEBUG) {
	printf("Help was requested.\n");
	fflush(stdout);
      }
      if (USE_STATS)
	updatestats(STATS_HELP);
      if (blocked(message_from, DEST_BLOCK)) {
	if (DEBUG) {
	  printf("Blocked From: %s\nNot mailing.\n", message_from);
	  fflush(stdout);
	}
      } else {
	if (DEBUG) {
	  printf("Mailing.\n");
	  fflush(stdout);
	}
	mailhelp(message_from);
      }
      unlink(tempfilename);
    } else if ((anon == KEY_REQ) && (USE_PGP)) {
      if (DEBUG) {
	printf("PGP key was requested.\n");
	fflush(stdout);
      }
      if (USE_STATS)
	updatestats(STATS_KEY);
      if (blocked(message_from, DEST_BLOCK)) {
	if (DEBUG) {
	  printf("Blocked From: %s\nNot mailing.\n", message_from);
	  fflush(stdout);
	}
      } else {
	if (DEBUG) {
	  printf("Mailing.\n");
	  fflush(stdout);
	}
	mailkey(message_from);
      }
      unlink(tempfilename);
    } else if (anon == CUTMARKS) {
      if (DEBUG) {
	printf("Cutmarks: %s detected.\nCutting message.\n", send_to);
	fflush(stdout);
      }
      cut(tempfilename, cFilename, send_to, "cutmarks:", 1);
      trimmings = 1;
    } else if (anon == ENCRYPT_SUB) {
      if (DEBUG) {
	printf("Encrypt-Subject: %s detected.\nEncrypting subject.\n", send_to);
	fflush(stdout);
      }
      encrypt_sub(tempfilename, send_to);
      recursing = 1;
    } else if ((anon == ENCRYPT_KEY) && (USE_PGP)) {
      if (DEBUG) {
	printf("Encrypt-Key: %s detected.\nEncrypting message.\n", send_to);
	fflush(stdout);
      }
      cut(tempfilename, kFilename, "**", "encrypt-key:", 0);
      encrypt_key(kFilename, tempfilename, send_to);
      recursing = 1;
    } else if (anon == LATENT_TIME) {
      if (DEBUG) {
	printf("Latent-Time: %s detected.\nQueuing message.\n", send_to);
	fflush(stdout);
      }
      if (USE_STATS)
        updatestats(STATS_LATENT);
      queuelatent(tempfilename, qFilename, mFilename, send_to);
      unlink(tempfilename);
    } else if ( ((anon == URL_REQUEST) || (anon == HTML_REQUEST))
		&& (ALLOW_WWW) ) {
      if (DEBUG) {
	if (anon == URL_REQUEST) printf("WWW page requested.\n");
	else if (anon == HTML_REQUEST) printf("HTML source requested.\n");
	fflush(stdout);
      }
      if (USE_STATS)
	updatestats(STATS_WWW);
      remove_headers(tempfilename, bFilename, send_to);
      chop(send_to);
      if (blocked(send_to + 3, DEST_BLOCK)) {
	if (DEBUG) {
	  printf("Blocked %s\nNot mailing.\n", send_to);
	  fflush(stdout);
	}
	unlink(tempfilename);
	unlink(bFilename);
      } else {
	mailurl(message_from, send_to, bFilename, uFilename, anon);
	unlink(tempfilename);
	unlink(bFilename);
      }
    } else if (trimmings) {
      if (DEBUG) {
	printf("\nProcessing cutmarks leftovers:\n");
	fflush(stdout);
      }
      rename (cFilename, tempfilename);
      trimmings = 0;
      recursing = 1;
    } else {
      if (DEBUG) {
	printf("Non-anonymous message detected.\n");
	fflush(stdout);
      }
      file1 = fopen(MAIL_SPOOL, "a");
      file2 = fopen(tempfilename, "r");
      if (file2 != NULL) {
	if (fgets(input, sizeof(input), file2)) {
	  if (strncmp(input, "Cut-Recursive: ", 15) != 0) {
	    if (DEBUG) {
	      printf("Saving to spool.\n");
	      fflush(stdout);
	    }
	    fputs(input, file1);
	    while (fgets(input, sizeof(input), file2)) {
	      fputs(input, file1);
	    }
	  } else {
	    if (DEBUG) {
	      printf("Discarding cutmarks leftovers.\n");
	      fflush(stdout);
	    }
	  }
	}
	fclose(file2);
      }
      fclose(file1);
      unlink(tempfilename);
    }
    if (trimmings)
      recursing = 1;
  }
/*
  if (DEBUG) {
    printf("Running the latent message queue.\n\n\n");
    fflush(stdout);
  }
  runlatentqueue();
*/
  exit(0);
}
