#!/usr/common/bin/perl -w

#  This is version 1.1 of Crowds
# 
#  The authors of this software are Mike Reiter and Avi Rubin
#               Copyright (c) 1997 by AT&T.
#  Permission to use, copy, and modify this software without fee is
#  hereby granted, provided that this entire notice is included in all
#  copies of any software which is or includes a copy or modification
#  of this software and in all copies of the supporting documentation
#  for such software.
# 
#  SOME PARTS OF CROWDS MAY BE RESTRICTED UNDER UNITED STATES EXPORT
#  REGULATIONS.
# 
#  THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR
#  IMPLIED WARRANTY.  IN PARTICULAR, NEITHER THE AUTHORS NOR AT&T MAKE
#  ANY REPRESENTATION OR WARRANTY OF ANY KIND CONCERNING THE
#  MERCHANTABILITY OF THIS SOFTWARE OR ITS FITNESS FOR ANY PARTICULAR
#  PURPOSE.

# This package contains the blender, which is used to manage the
# membership in the crowd. Complete installation instructions
# can be found at
#    http://www.research.att.com/projects/crowds/manual.html
#
# The blender processes requests to join the crowd, and if successful,
# notifies the new member of the current crowd members. Each member
# also receives an announcement about new members at various times
# during the day. Unsuccesful join attempts are logged to
# logfile.blender.

require 5.004;
use strict;
use IO::Socket;
use Carp;
use BlockCipher;
use TimeStamp;
use Random;
use Sys::Hostname;
use Msg ('INTERNAL');

#-- My port number
use vars qw($PORT);
*PORT = \8124;

#--- Configuration
my @join_time = ( 1.00,  1.15,  1.30,  1.45,  2.00,  2.15,  2.30,  2.45,
		  3.00,  3.15,  3.30,  3.45,  4.00,  4.15,  4.30,  4.45,
		  5.00,  5.15,  5.30,  5.45,  6.00,  6.15,  6.30,  6.45,
		  7.00,  7.15,  7.30,  7.45,  8.00,  8.15,  8.30,  8.45,
		  9.00,  9.15,  9.30,  9.45, 10.00, 10.15, 10.30, 10.45,
		 11.00, 11.15, 11.30, 11.45, 12.00, 12.15, 12.30, 12.45,
		 13.00, 13.15, 13.30, 13.45, 14.00, 14.15, 14.30, 14.45,
		 15.00, 15.15, 15.30, 15.45, 16.00, 16.15, 16.30, 16.45,
		 17.00, 17.15, 17.30, 17.45, 18.00, 18.15, 18.30, 18.45,
		 19.00, 19.15, 19.30, 19.45, 20.00, 20.15, 20.30, 20.45,
		 21.00, 21.15, 21.30, 21.45, 22.00, 22.15, 22.30, 22.45,
		 23.00, 23.15, 23.30, 23.45 );

my $rng;  #my random number generator
my $admin_cipher = read_passphrase();
my $joins = 0;

#--- Define the cipher used for each account
my %Accounts = ();
my %acct = ();
read_accounts();

#--- Identify the time of the next join commit
my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
my $time = $hour.".".$min;
my $next_join_index = scalar(@join_time) - 1;
while ($time < $join_time[$next_join_index]) {
    $next_join_index--;
}
$next_join_index++;
$next_join_index %= scalar(@join_time);
my $zone = `date`;
$zone =~ s!.{20}(\w\w\w).*!$1!s;

logmsg("server started on port $PORT");
logit("server started on port $PORT");

Msg->new_server(hostname(), $PORT, \&new_connection, INTERNAL());
Msg->event_loop(10, \&commit_join);

exit();

sub new_connection {
    my ($conn, $client_host, $client_port) = @_;
    my $env = { 'peer_addr' => $client_host,
		'peer_port' => $client_port };

    return (\&handle_connection, $env);
}

sub handle_connection {
    my ($conn, $env, $msg, $err) = @_;
    my (%Members, $member);
    my %SharedKeys;

    return if (!$msg);

    my ($enc_time,$account,$new_port,$firewall) = split(':', $msg);	
    $firewall = 0 unless defined $firewall;  #backwards compatibility

    # If we get an account that is not in the accts hash,
    # check to see if this account has been recently added
    if (!defined $acct{$account}) {
        dbmopen %Accounts, "accounts", 0666 or die "dbmopen: $!\n";
        if (defined $Accounts{$account}) {
            my $passphrase = $admin_cipher->cbc_decrypt($Accounts{$account});
            $acct{$account} = BlockCipher::new($passphrase);
        }
        dbmclose %Accounts;
    }

    #--- See if the account is registered
    if (!defined $account || !defined $acct{$account}) {
	#--- TODO: return an error message?
	logmsg("Invalid message from $env->{peer_addr}:$env->{peer_port}");
	logit("Invalid message from $env->{peer_addr}:$env->{peer_port}");
	$conn->disconnect();
	return;
    }

    logmsg("connection from $env->{peer_addr}:$env->{peer_port}");
    logit("connection from $env->{peer_addr}:$env->{peer_port}");

    #Check timestamp for this user
    if (TimeStamp::check_time($acct{$account}->cbc_decrypt($enc_time)) eq 0) {
	logmsg("Old timestamp from $env->{peer_addr}:$env->{peer_port}");
	logit("Old timestamp from $env->{peer_addr}:$env->{peer_port}");
	$conn->send_later('1234567890'); 
	return;
    } 

    #--- Add the member to the database
    dbmopen(%Members, "members", 0600) or die "dbmopen: $!";
    my $new_member = $env->{peer_addr}.':'.$new_port;
    $Members{$new_member} = $account unless $firewall;

    #--- Generate keys for this client to share with all the other members
    foreach $member (keys(%Members)) {
	$SharedKeys{$member} =
	    join('', unpack("H*", $rng->get_rand_string(8)));
    }

    #--- Report the current membership back to the client
    my $reply = '';
    foreach $member (keys(%Members)) {
	$reply .= $member.",".$Members{$member}.",".$SharedKeys{$member}.";";
    }
    chop($reply);

    #add commit time to message
    my $ctime = sprintf("%2.2f %s", $join_time[$next_join_index], $zone);
    $ctime =~ s/\./:/;
    $reply = $ctime."=".$reply;

    #add time to message
    $reply = time()."=".$reply;

    $reply = $acct{$account}->cbc_encrypt($reply);
    $conn->send_later($reply);

    #mark that a join has occurred
    $joins = 1;

    #--- Send an update to the other members
    my @temp = keys(%Members);
    while (defined($member = pop(@temp))) {
	next if ($member eq $new_member);

	my ($mbr_inetaddr, $mbr_port) = split(':', $member);

	if ($conn = Msg->connect($mbr_inetaddr, $mbr_port+1,
				 undef, undef, INTERNAL())) {
	    my $update = time()               .",".
		         $new_member          .",".
			 $account             .",".
			 $SharedKeys{$member} .",".
			 $firewall;
	    $update = $acct{$Members{$member}}->cbc_encrypt($update);
	    $update = push_header($update);
	    $conn->send_later($update);
	}
	else {
	    #--- Apparently this member is no longer alive (?)
	    logit("Deleted ".$Members{$member});
	    delete $Members{$member};
	}
    }

    dbmclose(%Members);
}


#--- COMMIT_JOIN
#--- Sends a "commit" message to all members if it is time.
sub commit_join {
    my %Members;
    my ($conn, $member);

    my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
    my $time = $hour.".".$min;
    if ($time == $join_time[$next_join_index]) {
	$next_join_index++;
	$next_join_index %= scalar(@join_time);

	if ($joins) {
	    $joins = 0;

	    dbmopen(%Members, "members", 0600) or die "database open failed";
	    my @temp = keys(%Members);

	    while (defined($member = pop(@temp))) {

		my ($mbr_inetaddr, $mbr_port) = split(':', $member);

		if ($conn = Msg->connect($mbr_inetaddr, $mbr_port+1,
					 undef, undef, INTERNAL())) {
		    my $msg = time().",COMMIT,COMMIT,COMMIT,COMMIT";
		    $msg = $acct{$Members{$member}}->cbc_encrypt($msg);
		    $msg = push_header($msg);
		    $conn->send_now($msg);
		} else {
		    #--- Apparently this member is no longer alive (?)
		    logit("Deleted ".$Members{$member});
		    delete $Members{$member};
		}
	    }

	    dbmclose(%Members);
	}
    }
}


#--- PUSH_HEADER
#--- Pushes a message header on the message
sub push_header {
    my ($msg) = @_;
    return 'crowds/1.1:blender:'.$msg;
}

sub logmsg { print "$0 $$: @_ at ".scalar(localtime)."\n"; }

sub logit { 
  open LOGS , ">>logfile.blender"                or die "open: $!";
  print LOGS "@_ at ", scalar localtime, "\n" ;
  close LOGS;
}


#--- read_accounts
#--- Reads in accounts from the "accounts" file
sub read_accounts {
    my $keyword;
    my $name;
    my $pw;

    undef %acct;

    #read in accounts and decrypt them
    dbmopen %Accounts, "accounts", 0666 or die "dbmopen: $!\n";

    $keyword = $Accounts{KEYWORD};
    $keyword = $admin_cipher->cbc_decrypt($keyword);
    if ($keyword ne "KEYWORD") {
        print "Incorrect admin passphrase.\n";
        exit;
    }

    while (($name,$pw) = each %Accounts) {
        if ($name ne "KEYWORD") {
            my $passphrase = $admin_cipher->cbc_decrypt($pw);
            $acct{$name} = BlockCipher::new($passphrase);
        }
    }

    dbmclose %Accounts;

}


sub read_passphrase {
    my $passphrase;
    my $my_cipher;

    system "stty -echo";
    print "Enter admin passphrase: ";
    chop($passphrase = <STDIN>);
    print "\n";
    system "stty echo";

    #--- Initialize the cipher
    $my_cipher = BlockCipher::new($passphrase);

    #--- Initialize the random number generator
    my $seed = $passphrase.`netstat`.`ps`;
    $rng = Random::new($seed);
    $rng->generate(1000);

    return $my_cipher;
}
