#  This is version 1.1 of Crowds
# 
#  The authors of this software are Mike Reiter and Avi Rubin
#               Copyright (c) 1997 by AT&T.
#  Permission to use, copy, and modify this software without fee is
#  hereby granted, provided that this entire notice is included in all
#  copies of any software which is or includes a copy or modification
#  of this software and in all copies of the supporting documentation
#  for such software.
# 
#  SOME PARTS OF CROWDS MAY BE RESTRICTED UNDER UNITED STATES EXPORT
#  REGULATIONS.
# 
#  THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR
#  IMPLIED WARRANTY.  IN PARTICULAR, NEITHER THE AUTHORS NOR AT&T MAKE
#  ANY REPRESENTATION OR WARRANTY OF ANY KIND CONCERNING THE
#  MERCHANTABILITY OF THIS SOFTWARE OR ITS FITNESS FOR ANY PARTICULAR
#  PURPOSE.

# This package contains routines to manage the stream cipher that
# is used to encrypt web requests and replies. The stream cipher
# is a basic shift counter with three states. It was designed by
# Jim Reeds at AT&T. A random seed (or key) is used to fill a 128
# byte table. Then, the counters provide random indices into the
# table.
#
# The new routine initializes the stream counters and the buffer.
# The fill routine picks random values out of the table and appends
# them to the stream buffer. Then, the counters are advanced. Each
# counter value becomes a linear combination of some or all of the
# other counters indices to the table. So the values in the stream
# determine the future counter values.
#
# Encryption and decryption is done by simple xor with the cipher
# buffer.
#
# In future releases, we may plug in a more rigorous stream cipher.

package StreamCipher;

use strict;

sub new {
    my ($seed) = @_;
    my $i;

    my $stream = { table    => [ split //, $seed ],
		   buffer   => "",
		   counter1 => 0,
		   counter2 => 0,
		   counter3 => 0 };

    # Build a copy of "table" with identical entries, except
    # converted to integer format
    $stream->{indices} = [ ];
    for ($i = 0; $i < scalar(@{$stream->{table}}); ++$i) {
	$stream->{indices}[$i] =
	    hex(join('',unpack("H*", ${$stream->{table}}[$i])));
    }

    return bless $stream, 'StreamCipher';
}


sub fill {
    my ($stream, $size) = @_;
    my $tsize = scalar(@{$stream->{table}});
    my $ctr1 = $stream->{counter1};
    my $ctr2 = $stream->{counter2};
    my $ctr3 = $stream->{counter3};

    while (length($stream->{buffer}) < $size) {
        $ctr1 = ($ctr2 ^ $ctr3) % $tsize;
        $ctr2 = $stream->{indices}[$ctr1] % $tsize;
	$ctr1 = ($ctr1 ^ $ctr2) % $tsize;
        $ctr3 = $stream->{indices}[$ctr1] % $tsize;
	$stream->{buffer} .= $stream->{table}[$ctr2] ^ $stream->{table}[$ctr3];
    }

    $stream->{counter1} = $ctr1;
    $stream->{counter2} = $ctr2;
    $stream->{counter3} = $ctr3;
}


sub encrypt {
    my ($stream, $msg) = @_;
    my $length = length($msg);
    $stream->fill($length);
    $msg ^= substr($stream->{buffer}, 0, $length);
    $stream->{buffer} = substr($stream->{buffer}, $length);
    return $msg;
}


sub decrypt {
    my ($stream, $msg) = @_;
    my $length = length($msg);
    $stream->fill($length);
    $msg ^= substr($stream->{buffer}, 0, $length);
    $stream->{buffer} = substr($stream->{buffer}, $length);
    return $msg;
}

1;

