/*  Hey Emacs, switch to -*-C-*-, please 
------------

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.]

The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This
implementation was a non-trivial and unpaid effort.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <idea.h>
#include <rand.h>

#include "pgplib.h"
#include "pgp_m.h"


/* This constant has been taken from PGP src/crypto.c */
#define	RAND_PREFIX_LENGTH	8

/*
 * Decrypt a conventional encrypted package, given the MD5
 * of the password.  Done in a PGP-compatible way.  Note that
 * this packet might be part of a larger setup.  Therefore, we
 * return data in a separate buffer.
 */

int
conventional_key_decrypt(u_char	*inbuf,
			 u_char	digest[16],
			 u_char	**outbuf)
{   
    IDEA_KEY_SCHEDULE 	ks;
    int 		num;
    int			inputlen;
    /* variables used by IDEA, to mimic the PGP behaviour */
    u_char 		iv[8];
    
    if ( ((inbuf[0] & CTB_TYPE_MASK) >> 2) != CTB_CKE_TYPE ) {
	fprintf(stderr, "Wrong type\n");
	return(-1);
    }

    inputlen = packet_length(inbuf) - 5;
    /* This get's us beyond the header */
    inbuf += 5;

    if ( inputlen > MAX_ENC_PACKET ) {
	fprintf(stderr, "Sorry, file too big\n");
	return(-1);
    }
    /* 8+2=10 bytes of header for iv */
    *outbuf = malloc(inputlen-10);
    if ( *outbuf == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }

    /* Let us get ready to IDEA-decrypt the data */
    idea_set_encrypt_key(digest,&ks);
    
    /* We are ready to decrypt the data.  As described in PGFORMAT.DOC
     * and in PGP/src/idea.c.  We decrypt the first 10 bytes to verify
     * that the key we have is correct.
     */
    memset(iv, 0, 8);
    num = 0;

    idea_cfb64_encrypt(inbuf, *outbuf, RAND_PREFIX_LENGTH +2, &ks, iv, 
		       &num, IDEA_DECRYPT);
    assert ( num == 2 );

    /* Verify that it has been decrypted correctly */
    if ( (*outbuf)[RAND_PREFIX_LENGTH] != (*outbuf)[RAND_PREFIX_LENGTH-2]  
	 ||  
	 (*outbuf)[RAND_PREFIX_LENGTH+1] != (*outbuf)[RAND_PREFIX_LENGTH-1] ) {
	/* No it didn't */
	free(*outbuf);
	*outbuf = NULL;
	return(0);
    }
    /* The key was correct.  We have used 10 bytes of iv so far
     * (that is 8+2) and Zimmerman's idea is to restart from here.  
     */

    memmove(iv + 6, iv, 2);
    memmove(iv, inbuf + 2, 6);
    num = 0;

    inbuf += 10;
    idea_cfb64_encrypt(inbuf, *outbuf, inputlen - 10, &ks, iv,
		       &num, IDEA_DECRYPT);
    return (inputlen-10);
}



/*
 * Take a hash and genereate a Conventional packet.
 * Before encryption we  compress, if we are asked to do so.  In other 
 * words, if you don't like that ZIP leaves your data in /tmp,
 * don't use it.
 * Return -1 on error, 0 if OK.
 *
 * We use RAND_bytes.  Remember to feed it well before using this
 * routine.
 *
 */

int
conventional_key_encrypt(u_char	*inbuf,
			 int	inbuflen,
			 u_char digest[16],
			 int	do_compress,
			 int	*outbuflen,
			 u_char	**outbuf)
{

    IDEA_KEY_SCHEDULE	ks;
    u_char	*buf, *tmp;
    int		header_len;
    u_char	header[5];
    int		buf_len, num;
    u_long	ul;
    u_char 	iv[8];
    /* scratch */
    int 	i;

    if (inbuf == NULL || inbuflen == 0 || digest == NULL 
	|| outbuflen == NULL || outbuf == NULL)
    {
	fprintf(stderr, "Error in parameter\n");
	return(-1);
    }
    
    *outbuflen = 0;
    *outbuf = NULL;

    if ( do_compress ) {
	i = compress_buffer(inbuf, inbuflen, &buf, &buf_len);
	if ( i != 0 ) {
	    fprintf(stderr, "Failed to compress buffer\n");
	    return(-1);
	}
	/* ensure that inbuflen holds how much we shall encrypt */
	inbuflen = buf_len;
	inbuf = buf;
    }
    else
	buf_len = inbuflen;

    /* 
     * We have have a buffer buf, that contains some data to be encrypted.
     * To the data we add 64bit random + 16bit keycheck = 10 bytes.
     */
    buf_len += 10;
    header_len = make_ctb(CTB_CKE_TYPE, buf_len, 1, header);
    buf_len += header_len;
    assert(header_len == 5 );
    
    *outbuf = calloc(1, buf_len);
    if ( *outbuf == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }
    tmp = *outbuf;
    memcpy(tmp, header, header_len);

    assert(tmp[0] = 0xa6);

    /* The CTB and length is not included in the lenght */
    ul = htonl(buf_len - header_len);
    memcpy(tmp+1, &ul, header_len - 1);

    RAND_bytes(tmp+header_len, 8);
    
    /* Include keycheck material */
    tmp[ header_len + RAND_PREFIX_LENGTH ] = 
	tmp[ header_len + RAND_PREFIX_LENGTH - 2];
    tmp[ header_len + RAND_PREFIX_LENGTH + 1] =
	tmp[ header_len + RAND_PREFIX_LENGTH - 1];

    /* Get ready to IDEA-encrypt the header*/
    idea_set_encrypt_key(digest,&ks);

    memset(iv, 0, 8);
    num = 0;

    idea_cfb64_encrypt(tmp+header_len, tmp+header_len, 
		       RAND_PREFIX_LENGTH +2, &ks, iv, &num, IDEA_ENCRYPT);
    assert(num == 2 );
    
    /* Now, as Zimmerman says, resync the stream as we cross a semantic
     * border.  Refill iv with the encphered data.  We have used 2 (10-8) 
     * bytes of iv, save the last 2.
     */
    memmove(iv+6, iv, 2);
    memcpy(iv, tmp+5+2, 6);
    
    /* Then encrypt the rest, that is the text. */

    idea_cfb64_encrypt(inbuf, tmp+header_len+RAND_PREFIX_LENGTH+2, 
		       inbuflen, &ks, iv, &num, IDEA_ENCRYPT);
    *outbuflen = buf_len;
    return(0);

}
