#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include <bn.h>
#include <sha.h>

/*------------------------------------*/
/* pgp format to SSLeay bignum */
BIGNUM *PGP_mpiBN(unsigned char **buf)
{
  int len;
  BIGNUM *N;

  len = *(*buf)++ * 256, len += *(*buf)++, len = (len + 7) / 8;
  N = BN_bin2bn(*buf, len, NULL);  /* allocates space and convert */
  *buf += len;                  /* bypass this number */
  return N;
}

/*------------------------------------*/
/* hash for new keyid - SHA hardcoded */
static unsigned char newkeyid[20];
/* these two are used by lookup5 */
keyid_t PGP_keyid;
unsigned char PGP_keypfx[3];

void PGP_gkid(unsigned char *buf, int len)
{
  int i;
  SHA_CTX hctx;

  SHA1_Init(&hctx);
  PGP_keypfx[0] = 0x99, PGP_keypfx[1] = len >> 8, PGP_keypfx[2] = len & 0xff;
  SHA1_Update(&hctx, PGP_keypfx, 3);
  SHA1_Update(&hctx, buf, len);
  SHA1_Final(newkeyid, &hctx);

  for (i = 0; i < 8; i++) {
    PGP_keyid <<= 8;
    PGP_keyid |= newkeyid[12 + i];
  }
}

/*------------------------------------*/
#define MAXLEN 32768

void *PGP_gtpkt(FILE * fp, unsigned char ctb, int *gaz)
{
  unsigned long len;
  int neof = 0, i, k;
  unsigned char *bufptr, *bp;
  unsigned long gazouta;

  gazouta = 0;
  if (ctb < 0xc0) {
    i = 1 << (ctb & 3);         /* length of length */
    for (k = 0, len = 0; k < i; k++)
      len = (len << 8) + fgetc(fp);
  } else {
    if ((len = fgetc(fp)) >= 0xc0) {
      if (len >= 0xe0)
        if (len == 0xff)
          for (k = 0, len = 0; k < 4; k++)
            len = (len << 8) + fgetc(fp);
        else
          len = 1 << (len & 31), neof = 1;  /* noteof */
      else
        len = ((len & 31) << 8) + 192 + fgetc(fp);
    }
  }
  if (len > MAXLEN || !len)
    return NULL;
  bufptr = malloc(neof ? MAXLEN + 1 : len + 1);  /* +1 to add final null */

  bp = bufptr;
  for (;;) {
    gazouta += len;
    if (gazouta > MAXLEN) {
      if (0 >= (i = fread(bp, 1, MAXLEN - gazouta, fp)))
        exit(-2);
      gazouta += i - len;
      len = 0, neof = 0;
    }
    while (len) {
      if (0 >= (i = fread(bp, 1, len, fp)))
        exit(-2);
      len -= i, bp += i;
    }
    if (!neof) {
      bufptr[gazouta] = 0;
      if (gaz)
        *gaz = gazouta;
      return bufptr;
    }
    neof = 0;
    if ((len = fgetc(fp)) >= 0xc0) {
      if (len >= 0xe0)
        len = 1 << (len & 31), neof += len;  /* noteof */
      else
        len = ((len & 31) << 8) + 192 + fgetc(fp);
    }
  }
}

/*------------------------------------*/
int main(int argc, char *argv[])
{
  unsigned char *bp0, *bp, t, *sp = NULL;
  unsigned long i, ll;
  int j;
  keyid_t dsakid = 0, dhkid = 0;
  FILE *outf = stdout, *ringf = NULL;
  extern char *optarg;

  while ((i = getopt(argc, argv, "o:r:")) != -1) {
    if (i == 'r')
      ringf = fopen(optarg, "rb");
    else if (i == 'o') {
      outf = fopen(optarg, "wb");
      if (outf == NULL)
        exit(-1);
    } else {
      fprintf(stderr, "usage: ringscan [-r ring]\n");
      exit(-1);
    }
  }
  bp0 = malloc(2048);
  if (!ringf && (bp = getenv("PGPPATH"))) {  /* if no ringf, try PGPPATH */
    strncpy(bp0, bp, 2000);
    strcat(bp0, "/pubring.pkr");
    ringf = fopen(bp0, "rb");
  }
  if (!ringf && (bp = getenv("HOME"))) {  /* no, try default from HOME */
    strncpy(bp0, bp, 2000);
    strcat(bp0, "/.pgp/pubring.pkr");
    ringf = fopen(bp0, "rb");
  }
  if (!ringf)
    return -3;                  /* no keyring */
  free(bp0);

  while (!feof(ringf)) {
    t = fgetc(ringf);
    if (!(bp = PGP_gtpkt(ringf, t, &j)))
      break;
    bp0 = bp;
    t = (t & 0x7c) >> 2;
    switch (t) {
    case 2:                    /* sig */
      fprintf(outf, "sig\n");
      continue;
    case 12:                   /* validity */
      fprintf(outf, "val %02x\n", *bp);
      continue;
    case 13:                   /* uid - should match */
      fprintf(outf, "uid %s\n", bp);
      continue;
    case 5:
    case 6:
    case 7:
    case 14:
      switch (t) {
      case 5:
        sp = "KEY";
        break;
      case 6:
        sp = "key";
        break;
      case 7:
        sp = "SUB";
        break;
      case 14:
        sp = "sub";
        break;
      }
      j = *bp++;                /* vers */
      /* timestamp */
      ll = *bp++ << 24, ll += *bp++ << 16, ll += *bp++ << 8, ll += *bp++;
      if (j != 4)
        bp += 2;                /* old valid days */
      switch (*bp++) {
      case 17:
        BN_free(PGP_mpiBN(&bp)), BN_free(PGP_mpiBN(&bp));
        BN_free(PGP_mpiBN(&bp)), BN_free(PGP_mpiBN(&bp));
        PGP_gkid(bp0, bp - bp0);
        fprintf(outf, "%s %016qX DSA\n", sp, PGP_keyid);
        break;
      case 16:
      case 20:
        BN_free(PGP_mpiBN(&bp)), BN_free(PGP_mpiBN(&bp));
        BN_free(PGP_mpiBN(&bp));
        PGP_gkid(bp0, bp - bp0);
        fprintf(outf, "%s %016qX DH\n", sp, PGP_keyid);
        break;
      case 1:
      case 2:
      case 3:
        {
          BIGNUM *N;
          N = PGP_mpiBN(&bp), BN_free(PGP_mpiBN(&bp));

          PGP_gkid(bp0, bp - bp0);
          dsakid = PGP_keyid;
          j = 64;
          dhkid = 0;
          while (j)
            dhkid = (dhkid << 1) + BN_is_bit_set(N, --j);
          fprintf(outf, "%s %016qX RSA %016qX (v4)\n", sp, dsakid, PGP_keyid);
          break;
        }
      }
      break;
    default:
      fprintf(outf, "??? %d\n", t);
    }
    if (bp0)
      free(bp0);
  }
  return -4;                    /* not found */
}
