/*

    ``gpgp'' Gnome/GTK Front for PGP
    Copyright (C) 1998  Max Valianskiy

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: pgman.c,v 1.41 1998/09/06 13:33:56 maxcom Stab $ 

*/

#include "../config.h"
#include "../version.h"

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <gnome.h>
#include <string.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include "gpgp.h"
#include "pgman.h"

/* malloc blocks */
#define BLK_SIZE 500000

#define MY_COMMENT "GnomePGP version " VERSION "  http://maxcom.ml.org/gpgp"

char* pgman_error=NULL;

static void show_error(int fd);
static GList* read_signatures(char* ring, char* uid);

GList* get_keylist(char* ring)
{
 char buf[1000];
 GList* list;
 int pip[2];

 FILE* fp;

 pgman_error=NULL;

 pipe(pip);
 
 g_snprintf(buf,sizeof(buf),"gpgm --batch --keyring %s --list-keys --no-default-keyring %s  2>&%d", ring, gpgp_options.optstring, pip[1]);

 fp = popen(buf,"r");
 
 if (!fp)
   {
    pgman_error=g_strerror(errno);
    return NULL;
   }
 
 close(pip[1]);
 
 if (!fgets(buf, sizeof(buf), fp) )
   {
    pgman_error=_("unexpected end of gpg output");
    pclose(fp);
    show_error(pip[0]);
    return NULL;
   }
 
  if (!fgets(buf, sizeof(buf), fp) )
   {
    pgman_error=_("unexpected end of gpg output");
    pclose(fp);
    show_error(pip[0]);
    return NULL;
   }

 list = g_list_alloc();
 
 while ( fgets(buf, sizeof(buf), fp) )
   {
    char** str = g_malloc ( sizeof(char*) * 5 );
    char tmp1[1000];
    char tmp2[1000];
    char tmp3[1000];
    char tmp4[1000];

    memset(str, 0, sizeof(char*)*5);

    sscanf(buf, "%s %s %s %[^\n]", tmp1, tmp2, tmp3, tmp4);

    if (strcmp(tmp1,"pub")==0)
      str[0]=_("public");
     else
       if (strcmp(tmp1,"sec")==0)
         str[0]=_("secret");
	else
	 continue;
 
    str[1]=g_strdup(tmp3);
    str[2]=g_strdup(tmp4);

    g_list_append(list, str);
   }

 if (WEXITSTATUS(pclose(fp))!=0)
   {
     pgman_error=_("gpg returned error");
     g_list_free(list);
     show_error(pip[0]);
     return NULL;
   }
   else
   {
    show_error(pip[0]);
    return list;
   }
}

void pgp_add_key(char* file, char* ring)
{
  char buf[1000];
  int pip[2];
  int res;

  pipe(pip);

  g_snprintf(buf,sizeof(buf),"gpgm --keyring %s %s --no-default-keyring --batch --import %s 2>&%d", ring, gpgp_options.optstring, file, pip[1]);
  res=system(buf);
  close(pip[1]);
  show_error(pip[0]);

  if (res==0)
     pgman_error=NULL;
    else
     pgman_error=_("gpg returned error");
}

void pgp_del_key(char* ring, char* uid)
{
  char buf[1000];
  int pip[2];
  int res;

  pipe(pip);

  g_snprintf(buf,sizeof(buf),"gpg --keyring %s --no-default-keyring --batch --yes %s --delete-key \"=%s\" 2>&%d", ring, gpgp_options.optstring, uid, pip[1]);
  res=system(buf);
  close(pip[1]);
  show_error(pip[0]);
  if (res==0)
     pgman_error=NULL;
    else
     pgman_error=_("gpg returned error");
}

void pgp_export_key(char* ring, char* uid, char* file)
{
  char buf[1000];
  int pip[2];
  int res;

  pipe(pip);

  g_snprintf(buf,sizeof(buf),"gpgm --keyring %s --no-default-keyring --batch --armor %s  --export \"=%s\" >> %s 2>&%d ", ring, gpgp_options.optstring, uid, file, pip[1]);
  
  res=system(buf);
  close(pip[1]);
  show_error(pip[0]);
  if (res==0)
     pgman_error=NULL;
    else
     pgman_error=_("gpg returned error");
}

struct show_error_pack
  {
   FILE* fp;
   GtkWidget* dialog;
  };

static void error_ok_cb(GtkWidget* widget, struct show_error_pack* pack);

static
void show_error(int fd)
{
 FILE* fp = fdopen(fd, "r");
 GtkWidget* dialog;
 GtkWidget* less;
 struct show_error_pack* pack=g_malloc(sizeof(struct show_error_pack));
 int ch;

 if (fp==NULL)
  {
   gnome_error_dialog(_("Internal Error: show_error can't open fd"));
   return;
  }
 
 ch = fgetc(fp); /* fdopen do not set EOF flag, so feof(fp) always
                        returns FALSE here */

 if (ch==EOF)
    fclose(fp);
   else
    {
     ungetc(ch,fp);
     dialog = gnome_dialog_new(_("Gnu Privacy Guard"), GNOME_STOCK_BUTTON_OK, NULL);
     gtk_widget_set_usize(dialog, 450, 250);
     pack->dialog=dialog;
     pack->fp=fp;
     gnome_dialog_button_connect (GNOME_DIALOG (dialog), 0, GTK_SIGNAL_FUNC (error_ok_cb), pack);
 
     less = gnome_less_new();
     gtk_box_pack_start( GTK_BOX(GNOME_DIALOG(dialog)->vbox), less, FALSE, TRUE, GNOME_PAD_SMALL);
     gtk_widget_show(less);
     gtk_widget_realize(GTK_WIDGET(GNOME_LESS(less)->text)); /* ??? */
     gnome_less_show_filestream(GNOME_LESS(less),fp);
     gtk_widget_show(dialog);
    }
}

static 
void error_ok_cb(GtkWidget* widget, struct show_error_pack* pack)
{
 gtk_widget_destroy(pack->dialog);
 fclose(pack->fp);
 g_free(pack);
}

void get_key_options(char* ring, char* uid, struct key_options* options)
{
  char buf[1000];
  int pip[2];
  int res;
  FILE* fp;
  char type[1000];
  char bits[1000];
  char keyid[1000];
  char date[1000];
  char finger[1000];

  options->signatures=NULL;

  pipe(pip);

  g_snprintf(buf,sizeof(buf),"gpgm --keyring %s --keyring %s --no-default-keyring --batch --with-colons %s --fingerprint \"=%s\" 2>&%d ", ring, gpgp_options.defpub, gpgp_options.optstring, uid, pip[1]);
  
  if (!(fp=popen(buf, "r")))
    {
     pgman_error=g_strerror(errno);
     close(pip[0]);
     close(pip[1]);
     return;
    }
  
  if (!fgets(buf, sizeof(buf), fp) )
   {
    pgman_error=_("unexpected end of gpg output");
    close(pip[1]);
    show_error(pip[0]);
    pclose(fp);
    return;
   }

  sscanf(buf, "%[^:] %*c %*[^:] %*c %[^:] %*c %*[^:] %*c %[^:] %*c %[^:]", type, bits, keyid, date);

  options->bits=g_strdup(bits);
  options->keyid=g_strdup(keyid);
  options->date=g_strdup(date);

  if (strcmp(type,"pub")==0)
     {
      options->type=_("public");

      if (!fgets(buf, sizeof(buf), fp) )
        {
         pgman_error=_("unexpected end of gpg output");
         close(pip[1]);
         show_error(pip[0]);
         pclose(fp);
         return;
        }
      sscanf(buf, "%*[^:] %*[^0-1A-Z] %[^:]", finger);
 
      options->fingerprint=g_strdup(finger);
     } 
     else
       if (strcmp(type,"sec")==0)
        {
         options->type=_("secret");
	 options->fingerprint="";
	}
	else
          {
           pgman_error=_("unexpected gpg output");
           close(pip[1]);
           show_error(pip[0]);
           pclose(fp);
           return;
          }

  res=WEXITSTATUS(pclose(fp));

  close(pip[1]);
  show_error(pip[0]);
  if (res==0)
    {
     pgman_error=NULL;
     options->signatures=read_signatures(ring, uid);
    }
    else
     pgman_error=_("gpg returned error");
}

static
GList* read_signatures(char* ring, char* uid)
{
 char buf[1000]; 
 GList* list;
 int pip[2];

 FILE* fp;

 pgman_error=NULL;

 pipe(pip);
 
 g_snprintf(buf,sizeof(buf),"gpgm --batch --keyring %s --keyring %s --list-keys --verbose --no-default-keyring %s  \"=%s\" 2>&%d", ring, gpgp_options.defpub, gpgp_options.optstring, uid, pip[1]);
 
 fp = popen(buf,"r");

 if (!fp)
   {
    pgman_error=g_strerror(errno);
    return NULL;
   }
 
 close(pip[1]);
 
 list = g_list_alloc();
 
 while ( fgets(buf, sizeof(buf), fp) )
   {
    char** str = g_malloc ( sizeof(char*) * 6 );
    char tmp1[1000];
    char tmp2[1000];
    char tmp3[1000];
    char tmp4[1000];

    memset(str, 0, sizeof(char*)*6);

    sscanf(buf, "%s %s %s %[^\n]", tmp1, tmp2, tmp3, tmp4);

    if (strcmp(tmp1,"pub")==0)
      str[0]=_("public");
     else
       if (strcmp(tmp1,"sec")==0)
         str[0]=_("secret");
	else
	 if (strcmp(tmp1,"sig")==0)
           str[0]=_("signature");
	  else
	   if (strcmp(tmp1, "sub")==0)
	    str[0]=_("subkey");
	   else 
	    continue;
 
    str[1]=g_strdup(tmp3);
    str[2]=g_strdup(tmp2);
    str[3]=g_strdup(tmp4);

    g_list_append(list, str);
   }

 if (WEXITSTATUS(pclose(fp))!=0)
   { 
     pgman_error=_("gpg returned error");
     g_list_free(list);
     show_error(pip[0]);
     return NULL;
   }
   else
   {
    show_error(pip[0]);
    return list;
   }
}

/***************************/
/*   SIGN_ACTION           */
/***************************/

struct action_pack
  {
   int inpip[2];
   int outpip[2];
   int errpip[2];
   char* data;
   int old_datalen;
   int datalen;
   char** new;
   int* newlen;
   int curlen;
   int read_key;
   int write_key;
   GnomeAppProgressKey key; 
   PgmanActionFunc func;
   gpointer data2;
  };

static
void action_finish(struct action_pack* pack, int cancel)
{
 int status;

 if (pack->datalen) gdk_input_remove(pack->write_key);
 gdk_input_remove(pack->read_key);

 if (pack->inpip[1]) close(pack->inpip[1]);
 close(pack->outpip[0]);
 close(pack->inpip[0]);

 wait(&status);
 
 if (WEXITSTATUS(status)!=0 && !pgman_error)
   pgman_error=_("gpg returned error");
       
 if (gpgp_options.use_status && !cancel) gnome_app_progress_done(pack->key);

 pack->func(pack->data2);

 show_error(pack->errpip[0]);

 g_free(pack);

 if (gpgp_options.use_status)
    gnome_app_flash(gpgp_options.app, _("Operation complete"));
}

static
void action_error(struct action_pack* pack)
{
 action_finish(pack,0);
}

static
void cancel_cb(struct action_pack* pack)
{ 
 pgman_error=_("canceled by user");
 action_finish(pack,1);
}

static
void action_read_cb(struct action_pack* pack, gint source, GdkInputCondition cond)
{
    if (!pack->curlen) /* realloc */
      {
       *pack->new = g_realloc(*pack->new,*pack->newlen+BLK_SIZE);
       pack->curlen = BLK_SIZE;
      } /* endif */

    switch (read(pack->outpip[0], *pack->new+(*pack->newlen) , 1)) 
	      {
	       case 1:
	              (*pack->newlen)++;
	              pack->curlen--;
		      break;
	       case 0: /* EOF */
		      action_finish(pack,0);
		      break;
	       case -1: /* error */
                       pgman_error=g_strerror(errno);
		       action_error(pack);
		       return;
	      } /* end switch */

}

static
void action_write_cb(struct action_pack* pack, gint source, GdkInputCondition cond)
{
   if (write (pack->inpip[1], pack->data, 1)!=1)
      {
       pgman_error=g_strerror(errno);
       if (gpgp_options.use_status) gnome_app_progress_done(pack->key);
       action_error(pack);
       return;
      } /* endif */
		    
    pack->data++;
    pack->datalen--;

    if (!pack->datalen)
      {
       gdk_input_remove(pack->write_key);
       close(pack->inpip[1]);
       pack->inpip[1]=0;
       pack->write_key=0;
      } /* endif */
}

static
gdouble action_percent_cb(struct action_pack* pack)
{
   return ((float) pack->old_datalen-pack->datalen)/(float) pack->old_datalen;
}

void pgp_sign_action(char* options, char* data, int datalen, char** new, int* newlen, PgmanActionFunc func, gpointer data2)
{ 
 char buf[1000];
 struct action_pack* pack=g_malloc(sizeof(struct action_pack));

 pgman_error=NULL;

 pack->data=data;
 pack->datalen=datalen;
 pack->old_datalen=datalen;
 pack->new=new;
 pack->newlen=newlen;
 pack->func=func;
 pack->data2=data2;

 if (gpgp_options.use_status) 
    pack->key=gnome_app_progress_timeout(gpgp_options.app,_("Processing data"),100, (GnomeAppProgressFunc) action_percent_cb, (GnomeAppProgressCancelFunc) cancel_cb, pack);

 g_snprintf(buf,sizeof(buf),"gpg --batch --no-default-keyring --keyring %s %s --secret-keyring %s %s", gpgp_options.defpub, gpgp_options.optstring, gpgp_options.defsec, options);

 if (pipe(pack->inpip)==-1 || pipe(pack->outpip)==-1 || pipe(pack->errpip)==-1)
   {
    pgman_error=g_strerror(errno);
    return;
   }

 switch (fork())
    {
     case 0:{
             int res;
             /* child */
             dup2(pack->inpip[0], 0);
	     dup2(pack->outpip[1], 1);
	     dup2(pack->errpip[1], 2);
	     close(pack->inpip[0]);
	     close(pack->inpip[1]);
	     close(pack->outpip[0]);
	     close(pack->outpip[1]);
	     close(pack->errpip[0]);
	     close(pack->errpip[1]);

             /* I'm too lazy to do it right way. FIXME */
	     res=WEXITSTATUS(system(buf)); /* why WEXIT? */

	     close(0);
	     close(1);
	     close(2);
	     _exit(res); /* gnome is not fork safe??? */
	    }
     case -1: /* error */
             pgman_error=g_strerror(errno);
	     close(pack->inpip[0]);
	     close(pack->inpip[1]);	     
	     close(pack->outpip[0]);	     
	     close(pack->outpip[1]);
	     close(pack->errpip[0]);
	     close(pack->errpip[1]);
	     return;
     default: /* father */
             {
	      *pack->new=g_malloc(BLK_SIZE);
	      *pack->newlen=0;

	      pack->curlen = BLK_SIZE;

	      close(pack->outpip[1]);
	      close(pack->errpip[1]);

              pack->read_key=gdk_input_add(pack->outpip[0], GDK_INPUT_READ, (GdkInputFunction) action_read_cb, pack);
     	      pack->write_key=gdk_input_add(pack->inpip[1], GDK_INPUT_WRITE, (GdkInputFunction) action_write_cb, pack);
	     } 
     }
}	     

/***************************/
/* UNSIGN_ACTION           */
/***************************/

struct action_pack2
  {
   int inpip[2];
   int outpip[2];
   int errpip[2];
   int status[2];
   char* data;
   int old_datalen;
   int datalen;
   char** new;
   int* newlen;
   int curlen;
   int read_key;
   int write_key;
   int status_key;
   GnomeAppProgressKey key; 
   PgmanActionFunc func;
   gpointer data2;
   char* area; /* shm area */
   int pid; /* server pid */
   int id; /* shm id */
  };

static
void action_finish2(struct action_pack2* pack, int cancel)
{
 int status;

 if (pack->datalen) gdk_input_remove(pack->write_key);
 gdk_input_remove(pack->read_key);
 gdk_input_remove(pack->status_key);

 if (pack->inpip[1]) close(pack->inpip[1]);
 close(pack->outpip[0]);
 close(pack->inpip[0]);
 close(pack->status[0]);

 wait(&status);
 
 if (WEXITSTATUS(status)!=0 && !pgman_error)
   pgman_error=_("gpg returned error");
       
 if (gpgp_options.use_status && !cancel) gnome_app_progress_done(pack->key);

 pack->func(pack->data2);

 show_error(pack->errpip[0]);

 g_free(pack);

 if (gpgp_options.use_status)
    gnome_app_flash(gpgp_options.app, _("Operation complete"));
}

static
void action_error2(struct action_pack2* pack)
{
 action_finish2(pack,0);
}

static
void cancel2_cb(struct action_pack2* pack)
{ 
 pgman_error=_("canceled by user");
 action_finish2(pack,1);
}

static
void action_read2_cb(struct action_pack2* pack, gint source, GdkInputCondition cond)
{
   if (!pack->curlen) /* realloc */
     {
      *pack->new = g_realloc(*pack->new,*pack->newlen+BLK_SIZE);
      pack->curlen = BLK_SIZE;
     } /* endif */

    switch (read(pack->outpip[0], *pack->new+(*pack->newlen) , 1)) 
	      {
	       case 1:
	              (*pack->newlen)++;
	              pack->curlen--;
		      break;
	       case 0: /* EOF */
		      action_finish2(pack,0);
		      break;
	       case -1: /* error */
                       pgman_error=g_strerror(errno);
		       g_message("read error");
		       action_error2(pack);
		       return;
	      } /* end switch */

}

static void do_get_string_cb(gchar* string, struct action_pack2* pack);
static void do_get_yes_cb(gint reply, struct action_pack2* pack);
static void do_action_finish(struct action_pack2* pack);

static void
do_get_string(int mode, const char *keyword, struct action_pack2* pack)
{
   if (strcmp(keyword, "passphrase.enter")==0)
      keyword=_("Enter passphrase");

   if( mode == 1 ) /* get string */
       gnome_question_dialog(keyword, (GnomeReplyCallback) do_get_string_cb, pack);
    else if( mode == 3 ) /* get password */
       gnome_request_password_dialog(keyword, (GnomeStringCallback) do_get_string_cb, pack);
    else /* get yes/no */
       gnome_request_string_dialog(keyword, (GnomeStringCallback) do_get_yes_cb, pack);
}

static
void do_get_string_cb(gchar* string, struct action_pack2* pack)
{
 int n = pack->area[0] << 8 | pack->area[1];
 int len;

 if (!string)
   {
    pgman_error=_("canceled by user");
    string="";
   }
 
 len = strlen(string);

 memcpy(pack->area+n+2, string, len );
 pack->area[n] = len >> 8;
 pack->area[n+1] = len;

 do_action_finish(pack);
}

static
void do_get_yes_cb(gint reply, struct action_pack2* pack)
{
 int n = pack->area[0] << 8 | pack->area[1];
 
 pack->area[n] = 0;
 pack->area[n+1] = 1;
 pack->area[n+2] = reply;

 do_action_finish(pack);
}

static
void do_action_finish(struct action_pack2* pack)
{
 pack->area[3] = 1; 
 kill(pack->pid,SIGUSR1);
 if (pgman_error) action_finish2(pack, 0);
}

static
void action_status2_cb(struct action_pack2* pack, gint source, GdkInputCondition cond)
{
 char buf[1000];

 if (fgets(buf, sizeof(buf), fdopen(pack->status[0],"r")))
   {
    if(memcmp(buf,"[GNUPG:] ",9)==0)
      { 
       int word=0;
       int is_info = 0,is_get = 0;
       char* p=strtok(buf+9," \n");
       char* p2;

       for(;p;p=strtok(NULL," \n")) 
           {
	    word++;
	    if(word==1 && !strcmp(p,"SHM_INFO")) 
	        {
	         if(!pack->area)
		   is_info=1;
		    else
		   g_print("duplicate SHM_INFO ignored\n");
		}
	       else 
	        if(is_info&&(p2=strchr(p,'='))) 
	         {
		    int val;
		    *p2++ = 0;
		    val = atoi(p2);
		    if( !strcmp(p, "pv" ) ) 
		       {
			if( atoi(p2) != 1 )
			   {
			    pgman_error=(N_("invalid protocol"));
			    action_error2(pack);
			    return;
			   }
			is_info = 2;
		       }
		    else if( !strcmp(p, "pid" ) )
			pack->pid = val;
		    else if( !strcmp(p, "shmid" ) )
			pack->id = val;
		  }
		else if( word == 1 && !strcmp(p,"SHM_GET") )
		    is_get = 1;
		else if( word == 1 && !strcmp(p,"SHM_GET_BOOL") )
		    is_get = 2;
		else if( word == 1 && !strcmp(p,"SHM_GET_HIDDEN") )
		    is_get = 3;
		else if( word == 2 && is_get )	
		   {
		    do_get_string(is_get,p,pack);
		    return;
		   }
		else if( word == 1 )
		    g_print("Status: %s\n", p);
	      }
	    if( is_info ) 
	       {
		if( is_info < 2 )
		    pgman_error=_("SHM info without protocol version");
		if( pack->pid == -1 )
		    pgman_error=_("SHM info without server's pid");
		if( pack->id == -1 )
		    pgman_error=_("SHM info without id");

		if (pgman_error) 
		  {
		   action_error2(pack);
		   return;
		  }
		
		pack->area = shmat( pack->id, 0, 0 );
		if( pack->area == (void*)-1 )
		  {
		   pgman_error=g_strerror(errno);
 		   action_error2(pack);
		   return;
	          }
	       }
      }
   }
   else /* eof */
   {
/*    pgman_error=g_strerror(errno);
    action_error2(pack);*/
    return;
   } /* end if */
}

static
void action_write2_cb(struct action_pack2* pack, gint source, GdkInputCondition cond)
{
   if (write (pack->inpip[1], pack->data, 1)!=1)
      {
       g_message("write error");
       pgman_error=g_strerror(errno);
       if (gpgp_options.use_status) gnome_app_progress_done(pack->key);
       action_error2(pack);
       return;
      } /* endif */
		    
    pack->data++;
    pack->datalen--;

    if (!pack->datalen)
      {
       gdk_input_remove(pack->write_key);
       close(pack->inpip[1]);
       pack->inpip[1]=0;
       pack->write_key=0;
      } /* endif */
}

static
gdouble action_percent2_cb(struct action_pack2* pack)
{
   return ((float) pack->old_datalen-pack->datalen)/(float) pack->old_datalen;
}

void pgp_unsign_action(char* data, int datalen, char** new, int* newlen, PgmanActionFunc func, gpointer data2)
{ 
 char buf[1000];
 struct action_pack2* pack=g_malloc(sizeof(struct action_pack2));

 pgman_error=NULL;

 pack->data=data;
 pack->datalen=datalen;
 pack->old_datalen=datalen;
 pack->new=new;
 pack->newlen=newlen;
 pack->func=func;
 pack->data2=data2;
 pack->area=NULL;

 if (gpgp_options.use_status) 
    pack->key=gnome_app_progress_timeout(gpgp_options.app,_("Processing data"),100, (GnomeAppProgressFunc) action_percent2_cb, (GnomeAppProgressCancelFunc) cancel2_cb, pack);

 if (pipe(pack->inpip)==-1 || pipe(pack->outpip)==-1 || pipe(pack->errpip)==-1 || pipe(pack->status)==-1)
   {
    pgman_error=g_strerror(errno);
    return;
   }

 g_snprintf(buf,sizeof(buf),"gpg --decrypt %s --no-greeting --run-as-shm-coprocess 0 --status-fd %d --no-default-keyring --keyring %s --secret-keyring %s", gpgp_options.optstring, pack->status[1], gpgp_options.defpub, gpgp_options.defsec);

 switch (fork())
    {
     case 0:{
             int res;
             /* child */
             dup2(pack->inpip[0], 0);
	     dup2(pack->outpip[1], 1);
	     dup2(pack->errpip[1], 2);
	     close(pack->inpip[0]);
	     close(pack->inpip[1]);
	     close(pack->outpip[0]);
	     close(pack->outpip[1]);
	     close(pack->errpip[0]);
	     close(pack->errpip[1]);
	     close(pack->status[0]);

             /* I'm too lazy to do it right way. FIXME */
	     res=WEXITSTATUS(system(buf)); /* why WEXIT? */

	     close(0);
	     close(1);
	     close(2);
	     _exit(res); /* gnome is not fork safe??? */
	    }
     case -1: /* error */
             pgman_error=g_strerror(errno);
	     close(pack->inpip[0]);
	     close(pack->inpip[1]);	     
	     close(pack->outpip[0]);	     
	     close(pack->outpip[1]);
	     close(pack->errpip[0]);
	     close(pack->errpip[1]);
	     return;
     default: /* father */
             {
	      *pack->new=g_malloc(BLK_SIZE);
	      *pack->newlen=0;

	      pack->curlen = BLK_SIZE;

	      close(pack->outpip[1]);
	      close(pack->errpip[1]);
	      close(pack->status[1]);

              pack->read_key=gdk_input_add(pack->outpip[0], GDK_INPUT_READ, (GdkInputFunction) action_read2_cb, pack);
              pack->status_key=gdk_input_add(pack->status[0], GDK_INPUT_READ, (GdkInputFunction) action_status2_cb, pack);
	      pack->write_key=gdk_input_add(pack->inpip[1], GDK_INPUT_WRITE, (GdkInputFunction) action_write2_cb, pack);
	     } 
     }
}	     

void update_gpgp_options()
{
 char buf[1000]="", buf2[1000];

 if (gpgp_options.comment)
   {
    sprintf(buf2, "%s %s", buf, "--comment \"" MY_COMMENT "\"");
    strcpy(buf, buf2);
   }
   else
   {
    sprintf(buf2, "%s %s", buf, "--no-comment");
    strcpy(buf, buf2);
   }

 if (gpgp_options.optstring) g_free(gpgp_options.optstring);

 gpgp_options.optstring=g_strdup(buf);
}
