#include "lucre.h"

/* The following routines are used for doing deposits when you don't have
    a bank account. */

/* Place a payment into the recdb.  If the depflags are EC_M_DEPFLAGS_CASH,
    this payment SHOULD NOT YET BE SEALED.  Otherwise, it SHOULD be sealed.
    This may result in more than one entry being added to the recdb.  Use
    EC_W_recdb_get_all_seqno() to get the list of all of the payments in
    the recdb, and deal with them one at a time.  If the payment is inserted
    into the recdb successfully, it is freed. */
EC_Errno EC_W_noaccount_recdb_put(EC_W_Wallet wallet, EC_M_Payment payment,
    EC_M_Depflags depflags)
{
    EC_Errno err = EC_ERR_NONE;
    EC_M_Pcoins pcoins;
    EC_M_Payment_hdr phdr;
    UInt32 c;

    /* If this is a received payment, just put the payment normally. */
    if (depflags == EC_M_DEPFLAGS_RCVD) {
	return EC_W_recdb_put(wallet, payment, depflags, NULL);
    }

    if (!wallet || !payment || depflags != EC_M_DEPFLAGS_CASH)
	return EC_ERR_INTERNAL;

    /* We're going to have to split up the payment into lots of 1-coin
	payments, all with similar payment headers. */
    pcoins = payment->pcoins;
    phdr = payment->payment_hdr;
    if (pcoins->cryptcoins) {
	/* Ummm... why was this crypted already? */
	return EC_ERR_INTERNAL;
    }
    for(c=0;c<pcoins->numcoins;++c) {
	EC_M_Pcoins npcoins = NULL;
	EC_M_Payment_hdr npay_hdr = NULL;
	EC_M_Payment npmt = NULL;
	EC_M_Onl_coin *onl_coin = EC_G_malloc(sizeof(EC_M_Onl_coin));
	if (!onl_coin) {
	    return EC_ERR_INTERNAL;
	}
	onl_coin[0] = EC_M_clone_onl_coin(pcoins->onl_coin[c]);
	if (!onl_coin[0]) {
	    EC_G_free(onl_coin);
	    return EC_ERR_INTERNAL;
	}
	npcoins = EC_M_new_pcoins(NULL, 1, onl_coin);
	if (!npcoins) {
	    EC_G_free(onl_coin);
	    return EC_ERR_INTERNAL;
	}
	npay_hdr = EC_M_clone_payment_hdr(phdr);
	if (!npay_hdr) {
	    EC_M_free_pcoins(npcoins);
	    return EC_ERR_INTERNAL;
	}

	/* Fix the payment header */
	npay_hdr->ncoins = 1;
	npay_hdr->amount = 1 << (npcoins->onl_coin[0]->keyversion
				    & EC_M_KEYVER_VALMASK);
	err = EC_M_snap_payment_hdr(npay_hdr);
	if (err) {
	    EC_M_free_pcoins(npcoins);
	    EC_M_free_payment_hdr(npay_hdr);
	    return err;
	}

	/* Create the payment */
	npmt = EC_M_new_payment(npay_hdr, npcoins);
	if (!npmt) {
	    EC_M_free_pcoins(npcoins);
	    EC_M_free_payment_hdr(npay_hdr);
	    return EC_ERR_INTERNAL;
	}

	/* Seal the payment */
	err = EC_W_seal_payment(wallet, npmt);
	if (err) {
	    EC_M_free_payment(npmt);
	    return err;
	}

	/* Put it into the database */
	err = EC_W_recdb_put(wallet, npmt, EC_M_DEPFLAGS_RCVD, NULL);
	if (err) {
	    EC_M_free_payment(npmt);
	    return err;
	}
    }
    EC_M_free_payment(payment);
    return EC_ERR_NONE;
}

/* For each seqno in the recdb, call this to fill in the msg you will send
    to the moneychanger. */
EC_Errno EC_W_noaccount_deposit_1(EC_W_Wallet wallet, EC_M_Msg msg,
    UInt32 seqno, UInt32 minpayments)
{
    EC_Errno err = EC_ERR_NONE;
    EC_M_Dep dep = NULL;
    EC_M_Payment_hdr phdr = NULL;
    EC_M_Pcoins pcoins = NULL;
    EC_M_Payment pmt = NULL;
    EC_M_Withdraw3 wd3 = NULL;

    /* Get the dep message from the recdb. */
    dep = EC_W_recdb_get(wallet, seqno);
    if (!dep) {
	return EC_ERR_INTERNAL;
    }

    /* Make a payment out of it. */
    err = EC_M_examine_dep(dep, NULL, &phdr, &pcoins);
    EC_M_free_dep(dep);
    if (err) {
	return err;
    }
    /* Fix some flags */
    phdr->flags = EC_M_DEPFLAGS_NONE;
    pmt = EC_M_new_payment(phdr, pcoins);
    if (!pmt) {
	EC_M_free_payment_hdr(phdr);
	EC_M_free_pcoins(pcoins);
	return EC_ERR_INTERNAL;
    }
    phdr = NULL; pcoins = NULL;

    /* Create a withdrawal of the same amount. */
    err = EC_W_create_withdraw3(wallet, &wd3, pmt->payment_hdr->amount,
	minpayments);
    if (err) {
	EC_M_free_payment(pmt);
	return err;
    }

    /* Compile them up. */
    err = EC_M_compile_withdraw3(wd3, msg);
    EC_M_free_withdraw3(wd3);
    if (err) {
	EC_M_free_payment(pmt);
	return err;
    }
    err = EC_M_compile_payment(pmt, msg);
    EC_M_free_payment(pmt);
    if (err) {
	return err;
    }

    return EC_ERR_NONE;
}

/* When you get a reply, call this to read the results.  Use the same seqno
    as above.  *changeamt is filled in to be the amount of cash received
    for the payment. */
EC_Errno EC_W_noaccount_deposit_2(EC_W_Wallet wallet, EC_M_Msg msg,
    UInt32 seqno, UInt32 *changeamt)
{
    EC_Errno err = EC_ERR_NONE;
    UInt32 amt = 0;
    EC_M_Fieldtype fld;
    EC_M_Rectype rec;

    while (!err) {
        err = EC_M_examine_msg(&fld, &rec, msg);
        if (err) break;
        if (fld == EC_M_FIELD_NONE) break;
        if (fld == EC_M_FIELD_SOR && rec == EC_M_REC_WITHDRAW4) {
            EC_M_Withdraw4 wd4 = NULL;
            if (!err) err = EC_M_decompile_withdraw4(&wd4, msg);
            if (!err) err = EC_W_read_withdraw4(wallet, wd4, &amt);
            EC_M_free_withdraw4(wd4);
        } else if (fld == EC_M_FIELD_SOR) {
            err = EC_W_handle_common(wallet, msg);
        } else {
            /* We didn't understand the field; skip it */
            err = EC_M_transfer_field(msg, NULL);
            continue;
	}
    }
    if (err) {
	return err;
    }
    if (changeamt) *changeamt = amt;
    err = EC_W_recdb_del_seqno(wallet, seqno);

    return err;
}
