#ifndef EC_INCL_EPOCKET_H
#define EC_INCL_EPOCKET_H
/* beginning of epocket.h */


	/* public ecashlib includes */
#include "eerrno.fh"
#include "epocket.fh"
#include "epocktid.fh"
#include "eprotocl.fh"
#include "eaccid.fh"
#include "eaddr.fh"
#include "einfo.fh"
#include "erecord.fh"
#include "eacth.fh"
#include "eamount.fh"
#include "emsg.fh"


	/* debug info headers */

#ifdef DC_DEBUG_INFO
#include "epocket.ih"
#endif




	/* EC_Pocket functions */


/*
 * Create a new EC_Pocket.
 *
 * parameters:
 *	pocketID_lnk: the EC_PocketID for this pocket
 *	*!b the pocket ID must be known _before_ EC_pocket_begin_open_acc()
 *	*!b the pocket ID consists of an EC_AccountID and a "base_dir" string
 *	mint_addr_lnk: address of the mint for this pocket
 *	acct_info_str: info about bank account for this pocket
 *	rnd_data_str: random data used for recovery
 *	pocket_password_str: a password to encrypt the data in the files.  If
 *		the files already exist then this must the password that was
 *		used to encrypt them.  Else, EC_pocket_new() will fail and
 *		return a NULL pointer.  If the files don't already exist
 *		then this will be the password used to encrypt them.
 *		*! Note that currently only the sec key is encrypted...
 *	acct_password_str: the password used to protect the account at the
 *		mint.  It is used to seed the generation of the RSA key pair.
 *		This argument is only necessary when creating the RSA key
 *		pair.  That is: the first time that EC_pocket_new() is
 *		called on a given EC_PocketID.  On successive calls of
 *		EC_pocket_new(), the RSA key pair will be read from
 *		the pocket's persistent storage.
 *
 * notes:
 *	*!note:  For now only a single pocket can exist per base_dir,
 *	and only a single EC_Pocket can be instantiated at a time.
 *	If the EC_AccountID does not correspond to the EC_AccountID
 *	stored in the files in base_dir, or if you try to new a second
 *	EC_Pocket when one already exists, you will get an error.
 *
 *	*!note:  For now, only a single EC_Pocket can exist per bank
 *	account.  If you try to use the same bank account from different
 *	hard-drives or different base_dirs, you may experience problems
 *
 *	*!design:  If we evolve some better (i.e. more abstract) way to
 *	interface ecashlib with permanent storage, EC_PocketID will lose
 *	the base_dir member at that same time.
 *
 *	A failure return value can result if ecashlib is unable to open
 *	or create the directory specified in the pocketID_lnk.
 *
 * return values:
 *	pointer to the new EC_Pocket on success, NULL pointer on failure
 */
EC_EXPORT EC_Pocket *EC_pocket_new(const EC_PocketID *pocketID_lnk, 
	const EC_Address *mint_addr_lnk, const Char *acct_info_str, 
	const Char *pocket_password_str, const Char *rnd_data_str,
	const Char *acct_password_str);


/*
 * Free an EC_Pocket.
 *
 * return values:
 *	standard EC_Errno
 */
EC_EXPORT EC_Errno EC_pocket_free(EC_Pocket *this_pocket_ptr);


/*
 * Set a different password for this pocket.
 *
 * parameters:
 *	new_password_str: new password
 *	old_password_str: old password
 *	
 * return values:
 *	standard EC_Errno
 *	EC_ERR_BAD_VALUE: old_password_str wasn't the right old password.
 */
EC_EXPORT EC_Errno EC_pocket_password(EC_Pocket *this_pocket_lnk,
	const Char *new_password_str, const Char *old_password_str);


/*
 * Get information about a pocket.
 *
 * return values:
 *	pointer to the new EC_Info on success, NULL pointer on failure
 */
EC_EXPORT EC_Info *EC_pocket_get_info(const EC_Pocket *this_pocket_lnk);


/*
 * Get a record list for a pocket.
 *
 * return values:
 *	pointer to the new EC_RecordList on success, NULL pointer on failure
 */
EC_EXPORT EC_RecordList *EC_pocket_get_rlist(const EC_Pocket *this_pocket_lnk);



	/* Pocket "Complex Action" functions */

/*
 * These are Account functions which initiate an complex action--an
 * action that cannot be completed without some networking services
 * being performed by the application on behalf of the ecashlib.
 * When an "EC_pocket_begin_foo()" function is called an EC_ActionHandler 
 * will be created to handle the complex action.
 */


/*
 * Open the account for this pocket.
 *
 * parameters:
 *	acct_password_str: password needed to open the account at the mint
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 *
 * notes:
 *	If this pocket already has an open account, 
 *	EC_pocket_begin_open_account() will return a NULL pointer.
 *!! Shouldn't it return an EC_ActionHandler which will enter an error
 * state?
 *	Also note that the issuer may refuse to open an account for this
 *	EC_AccountID for some reason.
 */
EC_EXPORT EC_ActionHandler *EC_pocket_begin_open_account(
	EC_Pocket *this_pocket_lnk, const Char *acct_password_str);


/*
 * Withdraw cash.
 *
 * parameters:
 *	amount: amount to withdraw
 *	min_payments: after withdrawal at least this many payments can
 *		can be made before 
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
EC_EXPORT EC_ActionHandler *EC_pocket_begin_withdrawal(
	EC_Pocket *this_pocket_lnk, EC_Amount amount, UInt32 min_payments);


/*
 * Deposit cash.
 *
 * parameters:
 *	amount: amount to deposit
 *	min_payments: after deposit at least this many payments can
 *		can be made before needing a withdrawal
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
EC_EXPORT EC_ActionHandler *EC_pocket_begin_deposit(EC_Pocket *this_pocket_lnk,
	EC_Amount amount, UInt32 min_payments);


/*
 * Make payment.
 *
 * parameters:
 *	protocol_lnk: protocol to use for payment
 *	amount: amount to pay
 *	description_str: payment description or NULL
 *	recip_accID_lnk, recip_addr_lnk: account ID and address of recipient
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
EC_EXPORT EC_ActionHandler *EC_pocket_begin_make_payment(
	EC_Pocket *this_pocket_lnk,
	EC_Protocol protocol_lnk, EC_Amount amount,
	const Char *description_str, const EC_AccountID *recip_accID_lnk,
	const EC_Address *recip_addr_lnk);


/*
 * Accept incoming payment.
 *
 * parameters:
 *	payment: the EC_Msg which contains the payment
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
EC_EXPORT EC_ActionHandler *EC_pocket_begin_accept_payment(
	EC_Pocket *this_pocket_lnk,
	const EC_Msg *payment_lnk);


/*
 * Request payment.
 *
 * parameters:
 *	protocol_lnk: protocol to use for payment
 *	amount: amount to request
 *	description_str: payment description or NULL
 *	requestee_accID_lnk, requestee_addr_lnk: ID and address of
 *		requestee
 *	return_addr_lnk: the address which the payment should be sent
 *		to if different from the account's normal address.  This
 *		is an optional parameter which will be ignored if a NULL
 *		pointer or the null EC_Address is passed.
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
EC_EXPORT EC_ActionHandler *EC_pocket_begin_request_payment(
	EC_Pocket *this_pocket_lnk,
	EC_Protocol protocol_lnk, EC_Amount amount,
	const Char *description_str, const EC_AccountID *requestee_accID_lnk,
	const EC_Address *requestee_addr_lnk, 
	const EC_Address *return_addr_lnk);


/*
 * Cancel a payment from a record.
 *
 * parameters:
 *	payment_lnk: a record of the payment
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
EC_EXPORT EC_ActionHandler *EC_pocket_begin_cancel_payment(
	const EC_Pocket *this_pocket_lnk,
	const EC_PaymentRecord *payment_lnk);


/*
 * Resend a payment from a record.
 *
 * parameters:
 *	payment_lnk: a record of the payment
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
EC_EXPORT EC_ActionHandler *EC_pocket_begin_resend_payment(
	const EC_Pocket *this_pocket_lnk,
	const EC_PaymentRecord *payment_lnk);



/* end of epocket.h */
#endif

