/*
 * Copyright (c) 1992 by Sun Microsystems, Inc.
 */

#include <sys/param.h>
#include <sys/kmem.h>
#include <sys/vfs.h>
#include <sys/vnode.h>
#include <sys/cmn_err.h>
#include <sys/debug.h>
#include <sys/systm.h>
#include <sys/t_lock.h>
#include <sys/errno.h>
#include <sys/thread.h>
#include <sys/tiuser.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <rpc/auth.h>
#include <rpc/clnt.h>
#include <vm/page.h>
#include <sys/pathname.h>

#include <fist.h>
#include FIST_HEADER


/* Hash table {this_vnode, hidden_vnode} functions */

/*
 * Get our vp based on hidden_vp.
 * Return NULL if not found. HT must be locked.
 */
vnode_t *
fist_ht_find_vp(const vnode_t * hidden_vp, const vfs_t * this_vfsp)
{
  u_int index = fisttablehash(hidden_vp);
  fist_bucket_t *bp;

  for (bp = vfstofwi(this_vfsp)->fwi_buckets[index];
       bp;
       bp = bp->next) {
    if (bp->hidden_vp == hidden_vp) {
      fist_dprint(5, "HT found entry with vnode 0x%x, hidden 0x%x\n",
		  bp->this_vp, bp->hidden_vp);
      return bp->this_vp;
    }
  }
  fist_dprint(5, "HT did not find entry with hidden 0x%x\n", hidden_vp);
  return NULL;
}

/*
 * delete entry from locked HT, based on hidden_vp.
 */
void
fist_ht_del_vp(const vnode_t * hidden_vp, const vfs_t * this_vfsp)
{
  u_int index = fisttablehash(hidden_vp);
  fist_bucket_t **bp, *aux;

  for (bp = &vfstofwi(this_vfsp)->fwi_buckets[index];
       *bp;
       *bp = (*bp)->next) {
    if ((*bp)->hidden_vp == hidden_vp) {
      aux = *bp;
      fist_dprint(5, "HT deleted entry with vnode 0x%x, hidden 0x%x\n",
		  aux->this_vp, aux->hidden_vp);
      *bp = (*bp)->next;
      kmem_free(aux, sizeof(fist_bucket_t));
      return;
    }
  }
  fist_dprint(4, "PANIC: vnode not found in HT!!!\n");
}

/*
 * Insert our vp based on hidden_vp.
 * Assume entry isn't already in HT. HT must be locked.
 * NOTE: hidden_vp can be gotten from this_vp.
 */
void
fist_ht_insert_vp(vnode_t * hidden_vp, vnode_t * this_vp)
{
  u_int index = fisttablehash(hidden_vp);
  fist_bucket_t *newbp;
  vfs_t *this_vfsp = this_vp->v_vfsp;

  newbp = kmem_alloc(sizeof(fist_bucket_t), KM_SLEEP);
  if (!newbp) {
    printk("fist_ht_insert_vp NO MORE MEMORY!\n");
    return;
  }
  newbp->hidden_vp = hidden_vp;
  newbp->this_vp = this_vp;
  newbp->next = vfstofwi(this_vfsp)->fwi_buckets[index];
  vfstofwi(this_vfsp)->fwi_buckets[index] = newbp;
}


/* check if HT is empty: 1==empty, 0=not empty */
int
fist_ht_empty_assert(const vfs_t * this_vfsp)
{
  int index;

  for (index = 0; index < FIST_HT_SIZE; ++index)
    if (vfstofwi(this_vfsp)->fwi_buckets[index] != NULL)
      return 0;
  return 1;
}
