/*********************************************************************
**
**     File name:               ssh_mp_int.c
**
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 handle multiple precision integers
**
**     Author/Date:             Tadayoshi Kohno, 22 November 1997
**
**     References:              draft-ylonen-ssh-protocol-00.txt
**
**     Notes:
**	The functions in this file implement a "multi-precision integer"
**	to meet the definition of the multi-precision integer datatype
**	in T. Ylonen's internet draft on ssh: draft-ylonen-ssh-protocol-00.txt
**
**	Callers currently need to malloc the memory for the num element
**	of the MP_Int struct.  This is a "bad thing," but is a hold over.
**	I'll think about changing it once things are more stable.
**
**     Functions:
**	mp_int_new		make a new MP_Int
**	mp_int_to_stream	put an MP_Int on a stream of bytes
**	mp_int_from_stream	take an MP_Int from a stream of bytes.		
**
**	make_rsakey		make an RSA key from MP_Ints
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_mp_int.c,v 1.2 1998/05/09 17:39:23 kohno Exp $";
#endif

#include "ssh_mp_int.h"
#include "ssh_util.h"

/*********************************************************************
**
**     Function:                mp_int_new
**
**     Purpose:                 mark an MP_Int as unused
**
**     Entry (pre) conditions:  memory for MP_Int itself already allocated
**
**     Parameters:              number		number to mark as new
**
**     Return value:            0
**
**     Error codes:             -1		not used
**
**     Side effects:            number->data_bytes = 0
**				number->num = NULL
**
**     Author/Date:             Tadayoshi Kohno, 22 November 1997
**
**     Notes:
**	Note that we're NOT allocating memory for number-num here.
**	That's currently up to the user.
**
*********************************************************************/

int mp_int_new
(
	MP_Int * number		/* MP_Int number */
)
{
	number->nbits = number->data_bytes = 0;
	number->num = (uint8_t *) NULL;
	return(0);
}

/*********************************************************************
**
**     Function:                mp_int_to_stream
**
**     Purpose:                 pack MP_Int into a stream
**
**     Entry (pre) conditions:  number and stream valid
**
**     Parameters:              stream		output stream
**				number		number to output
**				stream_size	size of stream we have
**
**     Return value:            length of output on stream
**
**     Error codes:             -1	error
**				< 2	impossible because to MP_Int
**					definition (see internet draft)
**
**     Side effects:            stream now contains the stream
**				represention of the MP_Int number
**				as per the SSH Internet Draft for
**				protocol v1.5
**
**     Author/Date:             Tadayoshi Kohno, 22 November 1997
**
**     Notes:
**
*********************************************************************/

int mp_int_to_stream
(
	uint8_t * stream,	/* output stream */
	MP_Int number,		/* input number */
	int stream_size		/* size of stream to write to */
)
{
	uint16_t num_bits;	/* number of bits */

	/*
	**	Make sure we're not going to overflow a buffer
	**	(XXX is this too slow?)
	*/
	if (number.data_bytes + sizeof(num_bits) > (uint32_t) stream_size)
	{
		return(-1);
	}

	/*
	**	First lets pack the number of bits
	*/
	num_bits = htons(number.nbits);
	my_bcopy((void *) &num_bits, (void *) stream, sizeof(num_bits));

	/*
	**	And now pack the number itself
	*/
	my_bcopy((void *) number.num, (void *) (stream + sizeof(num_bits)),
		number.data_bytes);

	return(number.data_bytes + sizeof(num_bits));
}

/*********************************************************************
**
**     Function:                mp_int_from_stream
**
**     Purpose:                 unpack an MP_Int from the stream
**
**     Entry (pre) conditions:  mp_int_new called first on number
**				number and stream valid memory addrs
**
**     Parameters:              stream		input stream
**				number		output MP_Int number
**
**     Return value:            number of bytes read
**
**     Error codes:             -1		malloc() failed
**
**     Side effects:            *number now contains MP_Int represented
**					by the stream
**
**     Author/Date:             Tadayoshi Kohno, 22 November 1997
**
**     Notes:
**
*********************************************************************/

int mp_int_from_stream
(
	const uint8_t * stream,	/* stream to decode */
	MP_Int * number		/* output number */
)
{
	uint16_t num_bits;	/* number of bits (network order) */

	/*
	**	First free the memory if it was allocated before
	*/
	if (number->num)
	{
		free(number->num);
		mp_int_new(number);
	}

	/*
	**	Now lets compute the number of bits
	*/
	my_bcopy((void *) stream, (void *) &num_bits, sizeof(num_bits));
	number->nbits = ntohs(num_bits);

	/*
	**	And the number of bytes
	*/
	number->data_bytes = (uint16_t) (number->nbits + 7) / 8;

	/*
	**	And now lets copy the data :)
	**	Note that we're assuming a correctly formatted MP_Int
	**	and thus are assuming we're not going to read passed
	**	the endo of the stream
	*/
	if ((number->num =
		(uint8_t *) malloc(sizeof(uint8_t) * (number->data_bytes + 1)))
			== NULL)
	{
		return(-1);
	}

	my_bcopy((void *) (stream + sizeof(num_bits)), (void *) number->num,
		number->data_bytes);

	return(number->data_bytes + sizeof(number->nbits));
}


/*********************************************************************
**
**     Function:                make_rsakey
**
**     Purpose:                 convert MP_Ints to an RSA public key that
**				will work with Eric Young's SSLeay RSA
**				library
**
**     Entry (pre) conditions:  mod, exp valid MP_Ints
**				memory for key, key->{n,e} initialized
**
**     Parameters:              mod		modulus
**				exp		exponent
**
**				*key		RSA key
**
**     Return value:            0
**
**     Error codes:             -1		error converting MP_Ints
**						to BIGNUMs
**
**     Side effects:            *key contains RSA public key
**
**     Author/Date:             Tadayoshi Kohno, 25 November 1997
**
**     Notes:
**	This function is to provide compatibility between my MP_Ints
**	and Eric Young's BIGNUMs
**
*********************************************************************/

int make_rsakey
(
	MP_Int mod,		/* MP_Int modulus */
	MP_Int exp,		/* MP_Int exponent */
	RSA * key		/* RSA key */
)
{
	if (BN_bin2bn(mod.num, mod.data_bytes, key->n) == NULL)
	{
		return(-1);
	}

	if (BN_bin2bn(exp.num, exp.data_bytes, key->e) == NULL)
	{
		return(-1);
	}
	return(0);
}

