/*********************************************************************
**
**     File name:               ssh_connect.c
**
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 connect with remote host
**
**     Author/Date:             Tadayoshi Kohno, 21 November 1997
**
**     References:                  
**	Internet Draft draft-ylonen-ssh-protocol-00.txt
**
**     Notes:
**	Lots of things to think about with this implementation of the
**	SSH v1.5 protocol.
**
**	Some things I would like to work on are:
**	timing out read/write/send/recv.  Is this easy under NT?
**
**	maybe also put locks around send()/recv() so that we don't get
**	gobledygook (handled by pseudo-semaphores at the frame level)
**
**	try to compile and build this thing on a unix box -- may take some
**	changes but I would like this to be able to work on multiple
**	platforms (at least the ssh protocol-level stuff).
**
**	figure out VC++ and name these files .c and compile as C, not C++.
**
**	avoid too many copyings.  optimize for performance (after first
**	step of meeting protocol specs).
**
**	Many more todos listed throughout.  It would be a good idea to go
**	through this again sometime and clean them all out :).
**
**	I think I'm going to add another function like ssh_prepaire_client()
**	to handle prepratory stuff (rather than roll it all into
**	ssh_connect_client()) (see ssh_client).
**
**     Functions:
**	ssh_connect_client			connect to server
**	ssh_disconnect_client_confirm		confirm a disconnection
**	ssh_disconnect_client_active		close an active connection
**
**	static ssh_connect_client_init		initialize connection
**	static ssh_connect_client_user		sends login to server
**	static ssh_connect_client_passwd	send password to server
**	static ssh_connect_client_rsa		authenticate using rsa
**
**	static ssh_session_id_gen		generate session id
**	static ssh_session_key_gen		generate session key
**	static ssh_session_key_encrypt		encrypt session key
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_connect.c,v 3.15 1998/05/21 16:19:41 kohno Exp $";
#endif

#include <assert.h>
#include <time.h>

#include "ssh.h"
#include "ssh_connect.h"

#include "ssh_proto.h"
#include "ssh_crypt.h"
#include "ssh_packet.h"
#include "ssh_msg.h"
#include "ssh_smsg.h"
#include "ssh_cmsg.h"
#include "ssh_comm.h"
#include "ssh_ident.h"
#include "ssh_random.h"

#include "ssh_mp_int.h"

#include "ssh_util.h"

/*
**	Helper functions for ssh_connect_client()
*/
static int ssh_connect_client_init(socket_type sockfd,
	struct ssh_struct * ssh_info,
	const char * applic_ver, int * at_major, uint8_t cipher_choice,
	uint32_t * supported_auth_mask, uint8_t * session_id);

static int ssh_connect_client_user(socket_type sockfd,
	struct ssh_struct * ssh_info, const char * user);

static int ssh_connect_client_passwd(socket_type sockfd,
	struct ssh_struct * ssh_info, const char * passwd);

static int ssh_connect_client_rsa(socket_type sockfd,
	struct ssh_struct * ssh_info,
	const char * passphrase, const char * identity_file,
	const uint8_t * session_id);

static void ssh_session_id_gen(uint8_t * session_id,
	MP_Int host_key_public_modulus,	MP_Int server_key_public_modulus,
	const uint8_t * cookie);

static void ssh_session_key_gen(struct ssh_struct * ssh_info,
	uint8_t * session_key);

static int ssh_session_key_encrypt(struct ssh_struct * ssh_info,
	MP_Int * encoded_session_key,
	const uint8_t * session_key, const uint8_t * session_id,
	MP_Int server_key_public_exponent, MP_Int server_key_public_modulus,
	MP_Int host_key_public_exponent, MP_Int host_key_public_modulus);

/*********************************************************************
**
**     Function:                ssh_connect_client
**
**     Purpose:                 handle ssh-level connection details
**
**     Entry (pre) conditions:  socket established
**				ssh_presetup_client called
**
**     Parameters:              sockfd		socket file descriptor
**				ssh_info	connection-specific information
**				applic_ver	application version to send
**				user		user to login as
**				auth_mode	authentication mode of choice
**				passwd		password/passphrase
**				cipher_choice	desired encryption mode
**				indentity_file	location for RSA identity
						(if auth_mode uses RSA)
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno passed from called functions --
**				general errors can occur in handshaking,
**				sending user information, authenticating.
**
**     Side effects:            connection established with server
**				(over the sockfd) using SSH.
**
**				encryption is set to cipher_choice, or
**				SSH_DEFAULT_CIPHER if cipher_choice is
**				unavailable.
**
**				user authenticated with server
**				using auth_mode
**
**				passwd bzero'd (but somewhere down the
**				list of called function might exist
**				a packet with the passwd xxx)
**
**     Author/Date:             Tadayoshi Kohno, 23 November 1997
**     Modified:		Tadayoshi Kohno, 14 March 1998
**					(check for NULL pointers)
**
**     Notes:
**	There might be a time when we need to know what version we're running
**	at (1 or [hopefully someday] 2).  When that happens, we may need
**	to make at_major static to this file and make an access function to
**	grab it.
**
**	Also, it might be nice for calling function to know what cipher
**	mode was chosen.
**
*********************************************************************/
int ssh_connect_client
(
	socket_type sockfd,		/* socket to talk over */
	struct ssh_struct * ssh_info,	/* connection-specific information */
	const char * applic_ver,	/* version of client application */
	const char * user,		/* user to authenticate */
	int auth_mode,			/* authentication mode */
	const char * passwd,		/* password/passphrase to use */
	uint8_t cipher_choice,		/* choice of cipher */
	const char * identity_file	/* location for user RSA keys */
)
{
	/*
	**	Characteristics about this session
	*/
	uint8_t session_id[SSH_SESSION_ID_SIZE+1];	/* session id */

	int at_major;			/* what major version we're at */

	int err;			/* error value */
	int internal;			/* another error value */

	uint32_t server_auth_mask;	/* server's supported auth */

	/*
	**	Make sure our pointers are valid (14 March 1998)
	*/
	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	if (applic_ver == (char *) NULL
		|| user == (char *) NULL
		|| passwd == (char *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	/*
	**	First lets generate a session id, session key, and start
	**	talking with the server
	*/
	if (ssh_connect_client_init(sockfd, ssh_info, applic_ver, &at_major,
		cipher_choice, &server_auth_mask, session_id))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error in ssh_connect_client_init",
			"ssh_connect_client");
		return(S_BAD);
	}

	/*
	**	Now lets "Declare the User Name" (to quote the internet draft)
	**	and authenticate the user if we need to
	*/
	if ((err = ssh_connect_client_user(sockfd, ssh_info, user))
		== SSH_NEED_AUTH)
	{
		/*
		**	User needs authentication, so let's authenticate
		*/
		if (!((1 << auth_mode) && server_auth_mask))
		{
			ssh_errno_set(SSH_ERRNO_AUTHMODE);
			return(S_BAD);
		}
		switch (auth_mode)
		{
		case SSH_AUTH_PASSWORD:
			/*
			**	Let's authenticate with a password
			*/
			if ((internal = ssh_connect_client_passwd(sockfd,
				ssh_info, passwd)) == SSH_NOT_AUTHENTICATED)
			{
				ssh_debugger_new(&(ssh_info->debug_info),
					"password didn\'t clear",
					"ssh_connect_client");
				return(S_BAD);
			} else if (internal == SSH_AUTH_ERR)
			{
				ssh_debugger_new(&(ssh_info->debug_info),
					"error on authentication",
					"ssh_connect_client");
				return(S_BAD);
			}
			break;

		case SSH_AUTH_RSA:
			/*
			**	Let's authenticate using RSA
			*/
			ssh_debugger_new(&(ssh_info->debug_info),
				"using RSA authentication",
				"ssh_connect_client");

			if ((internal = ssh_connect_client_rsa(sockfd,
				ssh_info, passwd, identity_file, session_id))
				== SSH_NOT_AUTHENTICATED)
			{
				ssh_debugger_new(&(ssh_info->debug_info),
					"RSA Authentication Failed",
					"ssh_connect_client");
				return(S_BAD);
			} else if (internal == SSH_AUTH_ERR)
			{
				ssh_debugger_new(&(ssh_info->debug_info),
					"Error on RSA authentication",
					"ssh_connect_client");
				return(S_BAD);
			}
			break;

		default:
			ssh_debugger_new(&(ssh_info->debug_info),
				"selected auth mode not supported yet",
				"ssh_connect_client");
			return(S_BAD);
			break;
		}
	} else if (err == SSH_USER_ERR)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"majorly bonked on user id",
			"ssh_connect_client");
		return(S_BAD);
	}

	/*
	**	Wipe out some memory-related stuff (maybe also do this
	**		before each
	**	return on error (??) xxx
	*/
	my_bzero(session_id, SSH_SESSION_ID_SIZE);

	/* removed as per samorris@cs.colorado.edu suggestion (8 May 1998)
		my_bzero(passwd, strlen(passwd));
	*/

	/*
	**	Success!! We've been authenticated!
	*/
	ssh_debugger_new(&(ssh_info->debug_info), "User Authenticated",
		"ssh_connect_client");
	
	return(S_GOOD);
}


/*********************************************************************
**
**     Function:                ssh_disconnect_client_confirm
**
**     Purpose:                 confirm disconnection with server
**
**     Entry (pre) conditions:  connection with server established
**				server sent an SSH_SMSG_EXITSTATUS
**
**     Parameters:              sockfd		socket to server
**XXX				ssh_info	connection-specific info
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		not used
**
**     Side effects:            server's disconnection confirmed
**
**     Author/Date:             Tadayoshi Kohno, 8 December 1997
**
**     References:                  
**
**     Notes:
**	call this after server closes a connection with
**	SSH_SMSG_EXITSTATUS.
**
*********************************************************************/

int ssh_disconnect_client_confirm
(
	socket_type sockfd,		/* socket to server */
	struct ssh_struct * ssh_info	/* connection-specific information */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data in packet */
	uint32_t data_len;		/* length of data in packet */

	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	/*
	**	yes, we're really disconnecting
	*/
	ssh_debugger_new(&(ssh_info->debug_info),
		"sending confirmation of disconnection",
		"ssh_disconnect_client_confirm");

	/*
	**	form the data portion of the packet
	*/
	(void) ssh_cmsg_exit_confirmation_encode(data, &data_len);
	
	/*
	**	send the packet (after forming it and encrypting it).
	*/
	if (ssh_send(sockfd, ssh_info, data, data_len,
		SSH_CMSG_EXIT_CONFIRMATION))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending data to server",
			"ssh_disconnect_client_confirm");

		return(S_BAD);
	}
	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_disconnect_client_active
**
**     Purpose:                 close and active connection
**
**     Entry (pre) conditions:  sockfd open and active connection
**				made with server
**
**     Parameters:              sockfd		socket to server
**				ssh_info	connection-specific information
**				msg		reason for disconnecting
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error with send
**						(ssh_errno set)
**
**     Side effects:            termination message sent to server
**
**     Author/Date:             Tadayoshi Kohno, 8 December 1997
**
**     Notes:
**	Use this to close an active connection with the server.
**
*********************************************************************/

int ssh_disconnect_client_active
(
	socket_type sockfd,		/* socket to server */
	struct ssh_struct * ssh_info,	/* connection-specific information */
	const char * msg		/* reason to disconnect */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data in packet */
	uint32_t data_len;		/* length of data in packet */

	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	/*
	**	Form the data protion of the packet
	*/
	(void) ssh_msg_disconnect_encode(data, &data_len, (uint8_t *) msg);

	/*
	**	send the packet (encrypted, packed, ...)
	*/
	if (ssh_send(sockfd, ssh_info, data, data_len, SSH_MSG_DISCONNECT))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending data to server",
			"ssh_disconnect_client_active");
		return(S_BAD);
	}
	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                static ssh_connect_client_init
**
**     Purpose:                 handles first part of client
**				connecting to server (protocol
**				exchange, session_{id,key}
**				generation, ...
**
**     Entry (pre) conditions:  sockfd valid
**
**     Parameters:              sockfd		socket to server
**				ssh_info	connection-specific information
**				applic_ver	application version string
**
**				at_major	final choice of version
**
**				cipher_choice	choice of cipher
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno values may be passed from called
**				functions. Additionally, this function
**				may set:
**					SSH_ERRNO_VERSIONS
**					SSH_ERRNO_TYPE
**
**     Side effects:            client connected to server.
**				session id and key generated.
**				encryption started (using cipher_choice or
**				SSH_DEFAULT_CIPHER).
**									
**
**     Author/Date:             Tadayoshi Kohno, 23 November 1997
**
**     Notes:
**	This is a mondo-huge function.  I think I will chop them into
**	smaller, helper functions.
**
**	This function pretty much follows the outline described in
**	draft-ylonen-ssh-protocol-00.txt.
**
**	Basically, it
**		* determines a version to run at (now only 1.5)
**		* get information (keys, cookie, ...) from server
**		* computes a session id from the server's information
**
**		* computes a random session key
**		* encrypts the session key
**		* sends encrypted session key and cipher choice, ... to server
**		* starts encryption (all subsequent packets encrypted)
**		* receives confirmation/success message from server.
**
*********************************************************************/

static int ssh_connect_client_init
(
	socket_type sockfd,		/* socket to talk to */
	struct ssh_struct * ssh_info,	/* connection-specific information */
	const char * applic_ver,	/* version of this applicatoin */
	int * at_major,			/* what major number we're running at */
	uint8_t cipher_choice,		/* first choice of cipher type */
	uint32_t * supported_auth_mask,	/* supported authentications? */	
	uint8_t * session_id		/* session id */
)
{
	/*
	**	Important identification stuff
	*/
	uint8_t session_key[SSH_SESSION_KEY_SIZE+1];/* session key */

	/*
	**	Protocol exchange information
	*/
	int serv_major;		/* server's major number */
	int serv_minor;		/* server's minor number */
	char serv_version[SSH_VERSION_LEN + 1];	/* server's version */

	/*
	**	Decode Packet Stuff
	*/
	uint8_t packet[SSH_MAX_PACKET];	/* binary packet form packet */
	uint8_t data[SSH_MAX_PACKET];	/* data in packet */
	uint32_t data_len;		/* length of data in packet */
	uint8_t type;			/* message type of packet */

	/*
	**	Parse SSH_SMSG_PUBLIC_KEY stuff
	*/
	uint8_t cookie[SSH_COOKIE_SIZE];/* anti-spoofing cookie */
	uint32_t server_key_bits;	/* server key bits */
	uint32_t host_key_bits;		/* host key bits */
	uint32_t protocol_flags;	/* protocol flags */
	uint32_t supported_cipher_mask;	/* cipher mask */
	uint32_t supported_authentications_mask;/* auth. mask */

	MP_Int server_key_public_exponent;	/* public key exp */
	MP_Int server_key_public_modulus;	/* public key mod */
	MP_Int host_key_public_exponent;	/* host key exp */
	MP_Int host_key_public_modulus;		/* host key mod */

	/*
	**	Encode packet stuff
	*/
	uint32_t packet_len;		/* packet length */
	uint8_t cipher_type;		/* type of cipher */
	MP_Int encoded_session_key;	/* encoded session key */

	/*
	**	And some misc. stuff
	*/
	int count;			/* misc counter */
	int err;			/* error value (when we save) */

	/*
	**	And lets initialize some "stuff"
	*/
	(void) mp_int_new(&server_key_public_exponent);
	(void) mp_int_new(&server_key_public_modulus);
	(void) mp_int_new(&host_key_public_exponent);
	(void) mp_int_new(&host_key_public_modulus);
	(void) mp_int_new(&encoded_session_key);


	/*
	**	First lets get the protocol information from the server
	*/
	if (ssh_proto_get(sockfd, ssh_info, &serv_major, &serv_minor,
		serv_version))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"couldn\'t grab protocol info",
			"ssh_connect_client_init");
		return(S_BAD);
	}

	/*
	**	Make sure we can handle this version of the protocol
	**	(right now only 1.5 is really implemented).
	*/
	if (serv_major != SSH_PROTO_MAJOR)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"ssh versions do not match",
			"ssh_connect_client_init");

		ssh_errno_set(SSH_ERRNO_VERSIONS);
		return(S_BAD);
	}
	*at_major = SSH_PROTO_MAJOR;

	/*
	**	Now that we know what version to run and have done prepratory
	**	work (assuming we'll be doing 2.0 and downgrading to 1.5, but
	**	lets just do 1.5 now), lets send our version stuff.
	*/
	if (ssh_proto_put(sockfd, ssh_info, SSH_PROTO_MAJOR, SSH_PROTO_MINOR,
		applic_ver))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"couldn\'t put protocol info",
			"ssh_connect_client");
		return(S_BAD);
	}

	/*
	**	Now Switching to packet-based transmissions and get server's
	**	SSH_SMSG_PUBLIC_KEY
	*/
	if (recv(sockfd, (char *) packet, SSH_MAX_PACKET, SR_FLAGS) <= 0)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error on recv first packet",
			"ssh_connect_client_init");

		ssh_errno_set(SSH_ERRNO_RECV);
		return(S_BAD);
	}

	if (ssh_packet_unpack(ssh_info, packet, &data_len, &type, data,
		SSH_MAX_PACKET))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error unpacking first packet",
			"ssh_connect_client_init");
		return(S_BAD);
	}
	
	/*
	**	Now that the packet is unpacked and in place, lets
	**	try decoding it :)
	*/
	if (type != SSH_SMSG_PUBLIC_KEY)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"expected SSH_SMSG_PUBLIC_KEY",
			"ssh_connect_client_init");

		ssh_errno_set(SSH_ERRNO_TYPE);
		return(S_BAD);
	}

	if (ssh_smsg_public_key_decode(data, data_len,
		cookie, &server_key_bits, &server_key_public_exponent,
		&server_key_public_modulus, &host_key_bits,
		&host_key_public_exponent, &host_key_public_modulus,
		&protocol_flags, &supported_cipher_mask,
		&supported_authentications_mask))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error decoding first packet",
			"ssh_connect_client_init");
		return(S_BAD);
	}

	/*
	**	Now that we know what the server sent us, lets first compute
	**	our session id (uses Eric Young's md5 stuff).
	*/
	ssh_session_id_gen(session_id, host_key_public_modulus,
		server_key_public_modulus, cookie);

	/*
	**	Now let's start computing the session key.
	*/
	ssh_session_key_gen(ssh_info, session_key);

	/*
	**	And now lets encrypt the session key
	*/
	if (ssh_session_key_encrypt(ssh_info, &encoded_session_key,
		session_key, session_id, server_key_public_exponent,
		server_key_public_modulus, host_key_public_exponent,
		host_key_public_modulus))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error in encrypting session key",
			"ssh_connect_client_init");
		return(S_BAD);
	}

	/*
	**	Now encoded_session_key is the one encrypted session key
	**	and we can send a SSH_CMSG_SESSION_KEY to the server.
	*/

	/*
	**	First check to make sure cipher type is supported
	*/

	if ((1 << cipher_choice) & supported_cipher_mask)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Setting Cipher Type to User\'s Choice",
			"ssh_connect_client_init");
		cipher_type = cipher_choice;
	} else
	{
		if ((1 << SSH_DEFAULT_CIPHER) & supported_cipher_mask)
		{
			cipher_type = SSH_DEFAULT_CIPHER;
			ssh_debugger_new(&(ssh_info->debug_info),
				"Setting Cipher type to default",
				"ssh_connect_client_init");
		} else
		{
			ssh_debugger_new(&(ssh_info->debug_info),
				"Can\'t Find Appropriate Cipher Type",
				"ssh_connect_client_init");

			ssh_errno_set(SSH_ERRNO_FINDCIPHER);
			return(S_BAD);
		}
	}

	protocol_flags = protocol_flags;	/* xxx work with this */

	/*
	**	Now lets pack up the message and send it out
	*/

	ssh_cmsg_session_key_encode(data, &data_len,
		cipher_type, cookie, encoded_session_key, protocol_flags);

	ssh_packet_pack(ssh_info, packet, &packet_len, SSH_CMSG_SESSION_KEY,
		data, data_len, SSH_MAX_PACKET);

	if ((count = send(sockfd, (char *) packet, packet_len, SR_FLAGS))
		!= (int) packet_len)
	{
		ssh_debug_int_new(&(ssh_info->debug_info),
			"send failed with sent %d", count,
			"ssh_connect_client_init");

		ssh_errno_set(SSH_ERRNO_SEND);
		return(S_BAD);
	}


	/*
	**	From now on all communication is encrypted.
	*/
	if (ssh_set_cipher(ssh_info, cipher_type, session_key))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error setting cipher",
			"ssh_connect_client_init");
		return(S_BAD);
	}

	/*
	**	Lets try to get back an SSH_SMSG_SUCCESS
	*/
	if ((err = ssh_recv(sockfd, ssh_info, data, &data_len))
		!= SSH_SMSG_SUCCESS)
	{
		if (err != SSH_MSG_NOTYPE)
		{
			ssh_errno_set(SSH_ERRNO_TYPE);
		}
		return(S_BAD);
	}


	*supported_auth_mask = supported_authentications_mask;

	/*
	**	Wipe out the session key (even though it is only valid for this
	**	session).
	*/
	my_bzero(session_key, SSH_SESSION_KEY_SIZE);

	ssh_debugger_new(&(ssh_info->debug_info),
		"Right On!  Passed first phase of connecting!",
		"ssh_connect_client_init");

	return(S_GOOD);
}


/*********************************************************************
**
**     Function:                static ssh_connect_client_user
**
**     Purpose:                 tell the server who is logging in
**
**     Entry (pre) conditions:  connection via ...client_init()
**				established
**
**     Parameters:              sockfd		socket to server
**				ssh_info	connection specific information
**				user		user to login as
**
**     Return value:            SSH_NEED_AUTH	user needs authentication
**				SSH_NO_NEED_AUTH	user doesn't need "
**
**     Error codes:             SSH_USER_ERR	error
**				ssh_errno set to SSH_ERRNO_USER
**
**				ssh_errno may be passed from called func()
**
**     Side effects:            packet containing login name sent to
**				server.
**
**				server's reply read and return value
**				set accordingly.
**
**     Author/Date:             Tadayoshi Kohno, 23 November 1997
**
**     Notes:
**
*********************************************************************/

static int ssh_connect_client_user
(
	socket_type sockfd,		/* socket to talk over */
	struct ssh_struct * ssh_info,	/* connection-specific information */
	const char * user		/* user to login as */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data in packet */
	uint32_t data_len;		/* length of data in packet */
	uint8_t type;			/* message type of packet */

	/*
	**	form the data portion of the packet
	*/
	(void) ssh_cmsg_user_encode(data, &data_len, (const uint8_t *) user);

	/*
	**	send the packet (with encryption, ...)
	*/
	if (ssh_send(sockfd, ssh_info, data, data_len, SSH_CMSG_USER))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending data to server",
			"ssh_client_connect_user");

		return(SSH_USER_ERR);
	}

	/*
	**	Now let's see if we need authentication.
	*/
	type = ssh_recv(sockfd, ssh_info, data, &data_len);

	if (type == SSH_SMSG_SUCCESS)
	{
		return(SSH_NO_NEED_AUTH);
	} else if (type == SSH_SMSG_FAILURE)
	{
		return(SSH_NEED_AUTH);
	}

	ssh_errno_set(SSH_ERRNO_USER);
	return(SSH_USER_ERR);
}

/*********************************************************************
**
**     Function:                static ssh_connect_client_passwd
**
**     Purpose:                 send password to authenticate user
**
**     Entry (pre) conditions:  ...client_user() passed
**				sockfd valid
**
**     Parameters:              sockfd		socket to server
**				ssh_info	connection-specific information
**				passwd		user's password
**
**     Return value:            SSH_AUTHENTICATED	authenticated
**				SSH_NOT_AUTHENTICATED	not "
**				ssh_errno to SSH_ERRNO_PASSWD_FAILED
**
**     Error codes:             SSH_AUTH_ERR		error
**				ssh_errno set to SSH_ERRNO_AUTH
**
**				ssh_errno may be passed from encoding,
**				decoding, packing, and unpacking functions.
**
**     Side effects:            password sent to server (using encryption
**				if chosen).
**
**     Author/Date:             Tadayoshi Kohno, 23 November 1997
**
**     Notes:
**	This function's return value tell whether the password cleared
**	or not.
**
*********************************************************************/

static int ssh_connect_client_passwd
(
	socket_type sockfd,		/* socket to talk over */
	struct ssh_struct * ssh_info,	/* connection-specific information */
	const char * passwd		/* passwd to use */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data in packet */
	uint32_t data_len;		/* length of data in packet */
	uint8_t type;			/* message type of packet */

	/*
	**	form the data portion of the packet
	*/
	(void) ssh_cmsg_auth_password_encode(data, &data_len,
		(const uint8_t *) passwd);
	
	/*
	**	send the packet (after forming it and encrypting it).
	*/
	if (ssh_send(sockfd, ssh_info, data, data_len, SSH_CMSG_AUTH_PASSWORD))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending data to server",
			"ssh_client_connect_passwd");

		return(SSH_USER_ERR);
	}

	/*
	** xxx bzero data up to data_len bytes to clear passwd
	*/

	/*
	**	Have we been authenticated?
	*/
	type = ssh_recv(sockfd, ssh_info, data, &data_len);

	if (type == SSH_SMSG_SUCCESS)
	{
		return(SSH_AUTHENTICATED);
	} else if (type == SSH_SMSG_FAILURE)
	{
		ssh_errno_set(SSH_ERRNO_PASSWD_FAILED);
		return(SSH_NOT_AUTHENTICATED);
	}

	ssh_errno_set(SSH_ERRNO_AUTH);
	return(SSH_AUTH_ERR);
}

/*********************************************************************
**
**     Function:                static ssh_connect_client_rsa
**
**     Purpose:                 authenticate user using RSA
**
**     Entry (pre) conditions:  ...client_user() passed
**				sockfd valid
**
**     Parameters:              sockfd		socket to server
**				ssh_info	connection-specific information
**				passphrase	passphrase for identity file
**				identity_file	file for RSA keys
**									
**
**     Return value:            SSH_AUTHENTICATED	authenticated
**				SSH_NOT_AUTHENTICATED	not "
**					ssh_errno set to xxx
**
**     Error codes:             SSH_AUTH_ERR	error
**				ssh_errno set to xxx
**
**				ssh_errno may be passed from encoding,
**				decoding, packing, and unpacking.
**
**     Side effects:            user authenticated using RSA authentication
**
**     Author/Date:             Tadayoshi Kohno, 9 December 1997
**
**     Notes:
**
*********************************************************************/

static int ssh_connect_client_rsa
(
	socket_type sockfd,		/* socket to talk over */
	struct ssh_struct * ssh_info,	/* connection-specific information */
	const char * passphrase,	/* passphrase for identity_file */
	const char * identity_file,	/* user's RSA key file */
	const uint8_t * session_id	/* session id */
)
{
	/*
	**	Our RSA key information
	*/
	RSA * private_key;		/* our private key */

	MP_Int public_key_modulus;	/* n = p * q */
	MP_Int public_key_exponent;	/* e = (p - 1)(q - 1) */

	MP_Int private_key_modulus;	/* same as public */
	MP_Int private_key_exponent;	/* d = mult. inv. of e mod phi(n) */

	MP_Int encrypted_challenge;	/* our encrypted challenge */

	/*
	**	real challenge and MD5 reply
	*/
	uint8_t challenge[1000];	/* our challenge */
	int challenge_len;		/* length of the challenge */

	MD5_CTX md5_context;			/* MD5 context */
	uint8_t md5_digest[MD5_DIGEST_LEN];	/* MD5 digest for reply */

	/*
	**	Packet data
	*/
	uint8_t data[SSH_MAX_PACKET];	/* data in packet */
	uint32_t data_len;		/* length of data in packet */
	uint8_t type;			/* message type of packet */

	
	/*
	**	Initialize our MP_Ints
	*/
	mp_int_new(&public_key_modulus);
	mp_int_new(&public_key_exponent);
	mp_int_new(&private_key_modulus);
	mp_int_new(&private_key_exponent);
	mp_int_new(&encrypted_challenge);

	private_key = RSA_new();
	private_key->n = BN_new();
	private_key->d = BN_new();
	private_key->e = BN_new();

	if (identity_file == (char *) NULL
		|| session_id == (uint8_t *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(SSH_AUTH_ERR);
	}

	/*
	**	First lets read our identity file
	**	(currently this may be redirected through ylonen's _or_ my
	**	file formats)
	*/
	if (ssh_identity_load(ssh_info, identity_file, passphrase,
		&public_key_modulus, &public_key_exponent,
		&private_key_modulus, &private_key_exponent) == S_BAD)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error reading RSA keys",
			"ssh_connect_client_rsa");
		return(SSH_AUTH_ERR);
	}

	/*
	**	Send our public key to the server
	*/
	(void) ssh_cmsg_auth_rsa_encode(data, &data_len, public_key_modulus);
	if (ssh_send(sockfd, ssh_info, data, data_len, SSH_CMSG_AUTH_RSA))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending data to server",
			"ssh_client_connect_rsa");

		return(SSH_AUTH_ERR);
	}


	/*
	**	Get back a challenge.
	*/
	type = ssh_recv(sockfd, ssh_info, data, &data_len);

	if (type != SSH_SMSG_AUTH_RSA_CHALLENGE)
	{
		ssh_errno_set(SSH_ERRNO_RSA_FAILED);
		return(SSH_NOT_AUTHENTICATED);
	}

	if (ssh_smsg_auth_rsa_challenge_decode(data, data_len,
		&encrypted_challenge) != S_GOOD)
	{
		return(SSH_AUTH_ERR);
	}

	/*
	**	Decrypt the challenge
	*/

	/* move to a function */
	if (BN_bin2bn(private_key_modulus.num, private_key_modulus.data_bytes,
		private_key->n) == NULL)
	{
		ssh_errno_set(111);
		return(SSH_AUTH_ERR);
	}

	if (BN_bin2bn(private_key_exponent.num,
		private_key_exponent.data_bytes, private_key->d) == NULL)
	{
		ssh_errno_set(111);
		return(SSH_AUTH_ERR);
	}
	/* end move to function */

	if ((challenge_len
		= RSA_private_decrypt(encrypted_challenge.data_bytes,
		encrypted_challenge.num, challenge, private_key,
		RSA_PKCS1_PADDING)) == -1)
	{
		ssh_errno_set(SSH_ERRNO_CHALLENGE_DECRYPT);

		/* move these to a function too */
		my_bzero(private_key_exponent.num,
			private_key_exponent.data_bytes);
		my_bzero(private_key_modulus.num,
			private_key_modulus.data_bytes);

		RSA_free(private_key);
		/* end move these to a function too */
		return(SSH_AUTH_ERR);
	}
	/* move these to a function too */
	my_bzero(private_key_exponent.num, private_key_exponent.data_bytes);
	my_bzero(private_key_modulus.num, private_key_modulus.data_bytes);

	RSA_free(private_key);
	/* end move these to a function too */

	ssh_debug_int_new(&(ssh_info->debug_info),
		"challenge length is (should be 32) %d", challenge_len,
		"ssh_connect_client_rsa");

	/*
	**	Compute MD5 of the encrypted challenge
	*/
	my_bcopy(session_id, challenge + challenge_len, SSH_SESSION_ID_SIZE);
	challenge_len = challenge_len + SSH_SESSION_ID_SIZE;

	MD5_Init(&md5_context);
	MD5_Update(&md5_context, (uint8_t *) challenge, challenge_len);
	MD5_Final(md5_digest, &md5_context);

	/*
	**	Return the MD5 and get back a confirmation
	*/
	(void) ssh_cmsg_auth_rsa_response_encode(data, &data_len, md5_digest);

	if (ssh_send(sockfd, ssh_info, data, data_len,
		SSH_CMSG_AUTH_RSA_RESPONSE))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending data rsa response to server",
			"ssh_client_connect_rsa");

		return(SSH_AUTH_ERR);
	}

	/*
	**	Have we been authenticated?
	*/
	type = ssh_recv(sockfd, ssh_info, data, &data_len);

	if (type == SSH_SMSG_SUCCESS)
	{
		return(SSH_AUTHENTICATED);
	} else if (type == SSH_SMSG_FAILURE)
	{
		ssh_errno_set(SSH_ERRNO_RSA_FAILED);
		return(SSH_NOT_AUTHENTICATED);
	} else
	{
		ssh_debug_int_new(&(ssh_info->debug_info),
			"Got Unexpected Type: %d", type,
			"ssh_connect_client_rsa");
	}

	ssh_errno_set(SSH_ERRNO_TYPE);
	return(SSH_AUTH_ERR);
}

/*********************************************************************
**
**     Function:                static ssh_session_id_gen
**
**     Purpose:                 generate session id from server data
**
**     Entry (pre) conditions:  server data valid
**
**     Parameters:              *session_id	computed session id
**
**				host_key_public_modulus		host key
**				server_key_public_modulus	server key
**				cookie				server cookie
**
**     Side effects:            session_id set to MD5 of concatenated
**				public keys and cookie.
**
**     Author/Date:             Tadayoshi Kohno, 27 November 1997
**
**     Notes:
**	This function (as does other functions in my code) makes
**	very heavy use of Eric Young's cryptographic libraries.  Here
**	I use have calls to his MD5 code.
**
**	See draft-ylonen-ssh-protocol-00.txt for an explanation of
**	how to compute the session id.
**
*********************************************************************/

static void ssh_session_id_gen
(
	uint8_t * session_id,			/* session id to compute */
	MP_Int host_key_public_modulus,		/* host key modulus */
	MP_Int server_key_public_modulus,	/* server key modulus */
	const uint8_t * cookie			/* 8 byte cookie */
)
{	
	MD5_CTX md5_context;			/* md5 context */
	uint8_t tmp_buf[STR_MISC_LEN];		/* temporary buffer */

	/*
	**	First form the buffer to take the MD5 of
	*/
	my_bcopy((void *) host_key_public_modulus.num, (void *) tmp_buf,
		host_key_public_modulus.data_bytes);
	my_bcopy((void *) server_key_public_modulus.num,
		(void *) (tmp_buf + host_key_public_modulus.data_bytes),
		server_key_public_modulus.data_bytes);
	my_bcopy((void *) cookie, 
		(void *) (tmp_buf + host_key_public_modulus.data_bytes
			+ server_key_public_modulus.data_bytes),
		SSH_COOKIE_SIZE);

	/*
	**	Take an MD5 of the buffer
	*/
	MD5_Init(&md5_context);
	MD5_Update(&md5_context, tmp_buf, 
		host_key_public_modulus.data_bytes
		+ server_key_public_modulus.data_bytes
		+ SSH_COOKIE_SIZE);
	MD5_Final(session_id, &md5_context);
}

/*********************************************************************
**
**     Function:                ssh_session_key_gen
**
**     Purpose:                 generate the (random) session key
**
**     Entry (pre) conditions:  memory for session_key valid
**				random number generator seeded
**
**     Parameters:              ssh_info	info on connection
**				*session_key	session key
**
**     Side effects:            session_key filled with random values
**
**     Author/Date:             Tadayoshi Kohno, 27 November 1997
**     Modified:		Tadayoshi Kohno, 14 March 1998
**					moved seeding to ssh_random.c
**
**     Notes:
**	xxx this need to be made much better.
**	random number generator seed in ssh_presetup.c
**
*********************************************************************/

void ssh_session_key_gen
(
	struct ssh_struct * ssh_info,	/* info on connection */
	uint8_t * session_key		/* session key */
)
{
/*
**	random number seeding moved to ssh_random.c (14 March 1998)
*/ 
	RAND_bytes(session_key, SSH_SESSION_KEY_SIZE);
	/*
	** uncomment this if want to test with a known session key
	** my_bzero(session_key, SSH_SESSION_KEY_SIZE);
	*/
}


/*********************************************************************
**
**     Function:                ssh_session_key_encrypt
**
**     Purpose:                 encrypt the session key to MP_Int form
**
**     Entry (pre) conditions:  parameters all initialized
**
**     Parameters:              ssh_info			connection info
**				encoded_session_key		output MP_Int
**
**				session_key			session key
**				session_id			session id
**
**				server_key_public_exponent	server e
**				server_key_public_modulus	server n
**
**				host_key_public_exponent	host e
**				host_key_public_modulus		host n
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**				ssh_errno set to SSH_ERRNO_MALLOC
**
**     Side effects:            encoded session key contains MP_Int
**				representation of the session key
**				encrypted with the two public keys
**
**     Author/Date:             Tadayoshi Kohno, 27 November 1997
**
**     Notes:
**	We first encrypt on the smallest of the keys.
**
**	On setting nbits of encoded_session_key, I could easily
**	subtract the 0 leading bits of the most significant byte.
**	I don't need to, however, because draft-ylonen-ssh-protocol-00.txt
**	says it's okay to over-state the number of bits (although I'm
**	sure it's best to not over-state the number of bytes)
**
*********************************************************************/

static int ssh_session_key_encrypt
(
	struct ssh_struct * ssh_info,		/* connection information */

	MP_Int * encoded_session_key,		/* encoded session key */

	const uint8_t * session_key,		/* session key */
	const uint8_t * session_id,		/* session id */
	MP_Int server_key_public_exponent,	/* server key exponent */
	MP_Int server_key_public_modulus,	/* server key modulus */
	MP_Int host_key_public_exponent,	/* host key exponent */
	MP_Int host_key_public_modulus		/* host key modulus */
)
{
	int msg_size;		/* length of encoded message */
	RSA * pub_key;		/* RSA public key */

	uint8_t xor_ses_key[SSH_SESSION_KEY_SIZE+1];/* session key w/ xor */
	uint8_t tmp_ses_key[CRYPT_BIG_SIZE];	/* transition key */
	uint8_t enc_ses_key[CRYPT_BIG_SIZE];	/* encrypted session key */

	int index;				/* index into session key */

	pub_key = RSA_new();

	/*
	**	xor the first SSH_SESSION_ID_SIZE bytes of key with session id.
	*/
	for (index = 0; index < SSH_SESSION_KEY_SIZE; index++)
	{
		xor_ses_key[index] = session_key[index];

		if (index < SSH_SESSION_ID_SIZE)
		{	
			xor_ses_key[index]
				= xor_ses_key[index] ^ session_id[index];
		}
	}

	/*
	**	Encrypt the session key using the public keys.  xxx these
	**	could be moved to their own function
	*/
	pub_key->n = BN_new();
	pub_key->e = BN_new();

	if (server_key_public_modulus.nbits < host_key_public_modulus.nbits)
	{
		/*
		**	First encrypt on the smallest key
		*/
		make_rsakey(server_key_public_modulus,
			server_key_public_exponent,
			pub_key);
		assert(RSA_size(pub_key) <= CRYPT_BIG_SIZE);
		msg_size = RSA_public_encrypt(SSH_SESSION_KEY_SIZE,
			xor_ses_key, tmp_ses_key, pub_key, RSA_PKCS1_PADDING);

		/*
		**	Now encrypt on the larger key
		*/
		make_rsakey(host_key_public_modulus, host_key_public_exponent,
			pub_key);
		assert(RSA_size(pub_key) <= CRYPT_BIG_SIZE);
		msg_size = RSA_public_encrypt(msg_size, tmp_ses_key,
			enc_ses_key, pub_key, RSA_PKCS1_PADDING);

	} else
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"encrypting on host key first, server key second",
			"ssh_connect_client_init");
		/*
		**	First encrypt on the smallest key
		*/
		make_rsakey(host_key_public_modulus, host_key_public_exponent,
			pub_key);
		assert(RSA_size(pub_key) <= CRYPT_BIG_SIZE);
		msg_size = RSA_public_encrypt(SSH_SESSION_KEY_SIZE,
			xor_ses_key, tmp_ses_key, pub_key, RSA_PKCS1_PADDING);

		ssh_debug_int_new(&(ssh_info->debug_info),
			"First pass Encrypt went %d bytes", msg_size,
			"ssh_connect_client_init");

		/*
		**	Now encrypt on the larger key
		*/
		make_rsakey(server_key_public_modulus,
			server_key_public_exponent, pub_key);
		assert(RSA_size(pub_key) <= CRYPT_BIG_SIZE);
		msg_size = RSA_public_encrypt(msg_size, tmp_ses_key,
			enc_ses_key, pub_key, RSA_PKCS1_PADDING);

		ssh_debug_int_new(&(ssh_info->debug_info),
			"Second pass Encrypt went %d bytes", msg_size,
			"ssh_connect_client_init");
	}

	/*
	**	Finally move the encrypted string into an MP_Int
	*/
	if ((encoded_session_key->num
		= (uint8_t *) malloc(sizeof(uint8_t) * (msg_size + 10)))
		== NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error mallocing memory for session key",
			"ssh_connect_client_init");

		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}

	my_bcopy((void *) enc_ses_key, (void *) encoded_session_key->num,
		msg_size);
	encoded_session_key->data_bytes = msg_size;
	encoded_session_key->nbits = msg_size * 8;

	return(S_GOOD);
}

