/*********************************************************************
**
**     File name:               ssh_cmsg.c
**				
**                              Copyright 1997, 1998 Tadayoshi Kohno.
**				All rights reserved.  See the LICENSE file.
**
**     Purpose:                 encode and decode client messages
**
**     Author/Date:             Tadayoshi Kohno, 22 November 1997
**
**     References:              draft-ylonen-ssh-protocol-00.txt
**
**     Notes:
**	The functions in this file are used to pack the "data" portion
**	of an SSH packet.  The names of the functions are very
**	self-explanitory (perhaps too much so :-).
**
**	Note that this "packed" "data" is not the full packet.  It still
**	needs the length, padding, packet type, crc. (and probably
**	compression and encryption).
**
**	Some of these functions are *way* overkill.  I have them here
**	for uniformity.  We'll see what happens.  I am hoping that this
**	method will scale nicely for v2.0.
**
**	In order to help efficiency (although it is only a constant factor),
**	I don't do too much error checking (NULL pointers, ...)
**
**	Bounds checking xxx
**
**     Functions:
**	ssh_cmsg_session_key_encode		encode SSH_CMSG_SESSION_KEY
**	ssh_cmsg_user_encode			encode SSH_CMSG_USER
**	ssh_cmsg_auth_password_encode		encode SSH_CMSG_AUTH_PASSWORD
**	ssh_cmsg_request_pty_encode		encode SSH_CMSG_REQUEST_PTY
**	ssh_cmsg_exec_shell_encode		encode SSH_CMSG_EXEC_SHELL
**	ssh_cmsg_stdin_data_encode		encode SSH_CMSG_STDIN_DATA
**	ssh_cmsg_exit_confirmation_encode	encd SSH_CMSG_EXIT_CONFIRMATION
**	ssh_cmsg_auth_rsa_encode		encode SSH_CMSG_AUTH_RSA
**	ssh_cmsg_auth_rsa_response_encode	encd SSH_CMSG_AUTH_RSA_RESPONSE
**	ssh_cmsg_window_size_encode		encode SSH_CMSG_WINDOW_SIZE
**	ssh_cmsg_request_compression_encode	encd SSH_CMSG_REQ._COMPRESSION
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_cmsg.c,v 3.6 1998/05/09 17:39:23 kohno Exp $";
#endif

#include "ssh.h"
#include "ssh_cmsg.h"

#include "ssh_util.h"

/*********************************************************************
**
**     Function:                ssh_cmsg_session_key_encode
**
**     Purpose:                 encode the SSH_CMSG_SESSION_KEY
**				data (data field of packet)
**
**     Entry (pre) conditions:  parameters available and correct
**
**     Parameters:              data		output data for packet
**				*data_len	length of resulting data
**
**				cipher_type	chosen cipher type
**				cookie		cookie to prove we listened
**				session_key	encrypted session key
**				protocol_flags	protocol flags
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		not used
**
**     Side effects:            data packed for a SSH_CMSG_SESSION_KEY
**				message (see the internet draft on SSHv1.5)
**
**				*data_len contains the length of the
**				packed data.
**
**     Author/Date:             Tadayoshi Kohno, 25 November 1997
**
**     Notes:
**
*********************************************************************/

int ssh_cmsg_session_key_encode
(
	uint8_t * data,		/* packed "data" for packet */
	uint32_t * data_len,	/* length of "data" */

	uint8_t cipher_type,	/* chosen cipher type */
	const uint8_t * cookie,	/* cookie to help avoid ip-spoofing */
	MP_Int session_key,	/* encrypted session key */
	uint32_t protocol_flags	/* protocol flags */

	/* xxx should really pass length of data (mp_int_to_stream) xxx */
)
{
	int offset;		/* offset for mp_int */

	my_bcopy((void *) &cipher_type, (void *) data, sizeof(cipher_type));
	my_bcopy((void *) cookie, (void *) (data + sizeof(cipher_type)),
		SSH_COOKIE_SIZE);

	offset = mp_int_to_stream(data + sizeof(cipher_type) + SSH_COOKIE_SIZE,
		session_key,
		SSH_MAX_PACKET - sizeof(cipher_type) - SSH_COOKIE_SIZE);

	protocol_flags = htonl(protocol_flags);
	my_bcopy((void *) &protocol_flags,
		(void *) (data + sizeof(cipher_type) + SSH_COOKIE_SIZE
		+ offset), sizeof(protocol_flags));

	*data_len = sizeof(cipher_type) + SSH_COOKIE_SIZE + offset
		+ sizeof(protocol_flags);

	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_cmsg_user_encode
**
**     Purpose:                 encode message of above type
**
**     Entry (pre) conditions:  memory addresses valid
**
**     Parameters:              data		"data" for packet
**				*data_len	length of "data"
**
**				user		user to login as
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		not implemented yet
**
**     Side effects:            data contains data for packet of this type
**				*data_len is length of "data"
**
**     Author/Date:             Tadayoshi Kohno, 23 November 1997
**
**     Notes:
**
*********************************************************************/

int ssh_cmsg_user_encode
(
	uint8_t * data,		/* data for packet */
	uint32_t * data_len,	/* length of data */

	const uint8_t * user	/* user to login as */
)
{
	*data_len = string_to_stream(data, user);
	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_cmsg_auth_password
**
**     Purpose:                 encode a message of above type
**
**     Entry (pre) conditions:  memory addresses valid
**
**     Parameters:              data		data for packet
**				*data_len	length of "data"
**
**				passwd		user password
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		not used
**
**     Side effects:            data contains data for packet
**				*data_len is length of "data"
**
**     Author/Date:             Tadayoshi Kohno, 23 November 1997
**
**     Notes:
**	I don't like even storing the password in memory in plaintext.
**	Oh well.
**
*********************************************************************/

int ssh_cmsg_auth_password_encode
(
	uint8_t * data,		/* data of packet */
	uint32_t * data_len,	/* length of data */

	const uint8_t * passwd	/* password to send */
)
{
	*data_len = string_to_stream(data, passwd);
	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_cmsg_request_pty_encode
**
**     Purpose:                 encode a request for a pseudo-terminal
**
**     Entry (pre) conditions:  memory for data, data_len valid
**				tty_mode computed
**				height, width appropriate for terminal
**
**     Parameters:              *data		data for packet
**				*data_len	length of data
**
**				terminal_type	environment var (vt100, .)
**				height		rows for terminal
**				width		columns for terminal
**
**				x_pixels	width, 0 for non graphics
**				y_pixles	heighth, 0 for non graphics
**
**				tty_mode	tty mode in binary form
**				tty_mode_len	bytes for tty_mode
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		not used
**
**     Side effects:            data for packet wrapped and stored in
**				*data
**
**     Author/Date:             Tadayoshi Kohno, 27 November 1997
**
**     Notes:
**	The tty_modes need to be computed beforehand.  I hope to provide
**	a tool to help with that (eventually).
**
*********************************************************************/

int ssh_cmsg_request_pty_encode
(
	uint8_t * data,		/* data of packet */
	uint32_t * data_len,	/* length of data */

	const char * terminal_type,	/* terminal environment variable */
	uint32_t height,		/* rows for terminal */
	uint32_t width,			/* columns for terminal */
	uint32_t x_pixels,		/* graphics, width in pixels */
	uint32_t y_pixels,		/* graphics, height in pixels */
	const uint8_t * tty_mode,	/* tty mode encoded in binary */
	int tty_mode_len		/* length of tty_mode */
)
{
	int offset = 0;		/* offset into data */

	height = htonl(height);
	width = htonl(width);
	x_pixels = htonl(x_pixels);
	y_pixels = htonl(y_pixels);

	offset = string_to_stream(data, (uint8_t *) terminal_type);

	my_bcopy((void *) &height, (void *) (data + offset), sizeof(height));
	offset += sizeof(height);

	my_bcopy((void *) &width, (void *) (data + offset), sizeof(width));
	offset += sizeof(height);

	my_bcopy((void *) &x_pixels, (void *) (data + offset),
		sizeof(x_pixels));
	offset += sizeof(x_pixels);

	my_bcopy((void *) &y_pixels, (void *) (data + offset),
		sizeof(y_pixels));
	offset += sizeof(y_pixels);

	my_bcopy((void *) tty_mode, (void *) (data + offset),
		(int) tty_mode_len);
	offset += (int) tty_mode_len;

	*data_len = offset;

	return(S_GOOD);
}

/*********************************************************************
**
**     Function                 ssh_cmsg_exec_shell_encode
**
**     Purpose:                 wrap data for requesting a shell
**
**     Entry (pre) conditions:  pointers valid
**
**     Parameters:              *data		data for packet
**				*data_len	length of *data
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		not used
**
**     Side effects:            *data_len, *data set for message type
**
**     Author/Date:             Tadayoshi Kohno, 27 November 1997
**
**     Notes:
**	This is a funny little function that just sets *data_len to 0.
**	This is because we don't attach any data to a message of this
**	type.
**
*********************************************************************/

int ssh_cmsg_exec_shell_encode
(
	uint8_t * data,		/* data for packet */
	uint32_t * data_len	/* length of data in packet */
)
{
	*data_len = 0;

	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_cmsg_stdin_data_encode
**
**     Purpose:                 encode a STDIN message for the server
**
**     Entry (pre) conditions:  len is valid length for str
**				pointers valid
**
**     Parameters:              *data		data for packet
**				*data_len	length of *data
**
**				str		string that's STDIN data
**				len		length of "str"
**
**				data_size	available size of *data
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		*data to small
**
**     Side effects:            *data, *data_len set appropriately
**				for this packet
**
**     Author/Date:             Tadayoshi Kohno, 28 November 1997
**     Modified:		Tadayoshi Kohno, 14 March 1998
**					set errno for data being too
**					small
**
**     Notes:
**	The reason we have "len" as a parameter (rather than just
**	using string_to_stream() [which uses strlen()]) is that
**	the data *may* contain NULLs or not be NULL terminated.  I
**	haven't encountered such a situation yet, but there may be someday.
**
**	only data_size bytes will be packed into data
**
*********************************************************************/

int ssh_cmsg_stdin_data_encode
(
	uint8_t * data,		/* data in resulting packet */
	uint32_t * data_len,	/* length of *data */

	const uint8_t * str,	/* string for STDIN on server */
	int len,		/* length of input string */
	int data_size		/* size of *data */
)
{
	long length_of_data;	/* length of the data */

	length_of_data = string_to_stream_n(data, str, len, data_size);

	if (length_of_data <= 0)
	{
		ssh_errno_set(SSH_ERRNO_PACKET_BIG);
		*data_len = 0;
		return(S_BAD);
	}

	*data_len = (uint32_t) length_of_data;

	return(S_GOOD);

	/*
	**	return(*data_len > 0 ? S_GOOD : S_BAD);
	*/
}

/*********************************************************************
**
**     Function:                ssh_cmsg_exit_confirmation_encode
**
**     Purpose:                 confirm that we know we're done
**
**     Entry (pre) conditions:  pointers valid
**
**     Parameters:              *data		data for packet
**				*data_len	length of *data
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		not used
**
**     Side effects:            *data_len, *data set for message type
**
**     Author/Date:             Tadayoshi Kohno, 8 December 1997
**
**     Notes:
**	This is a funny little function that just sets *data_len to 0.
**	This is because we don't attach any data to a message of this
**	type.
**
*********************************************************************/

int ssh_cmsg_exit_confirmation_encode
(
	uint8_t * data,		/* data for packet */
	uint32_t * data_len	/* length of data in packet */
)
{
	*data_len = 0;

	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_cmsg_auth_rsa_encode
**
**     Purpose:                 encode an SSH_CMSG_AUTH_RSA message
**
**     Entry (pre) conditions:  pointers valid, public modulus valid
**
**     Parameters:              *data		data for packet
**				*data_len	length of *data
**
**				public_key_modulus	our pub key
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		not used
**
**     Side effects:            *data formed as per SSHv1.5 specs
**
**     Author/Date:             Tadayoshi Kohno, 9 December 1997
**
**     Notes:
**
*********************************************************************/

int ssh_cmsg_auth_rsa_encode
(
	uint8_t * data,		/* data for packet */
	uint32_t * data_len,	/* length of data in packet */

	MP_Int public_key_modulus	/* user's public key modulus */

	/* xxx pass in size of data xxx */
)
{
	*data_len = mp_int_to_stream(data, public_key_modulus,
		SSH_MAX_PACKET);
	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_cmsg_auth_rsa_response_encode
**
**     Purpose:                 encode an SSH_CMSG_AUTH_RSA_RESPONSE
**
**     Entry (pre) conditions:  pointers valid, md5_digest is computed
**				as per SSHv1.5 protocol
**
**     Parameters:              *data		data for packet
**				*data_len	length of data in packet
**
**				md5_digest	md5 digest to send
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		not used
**
**     Side effects:            *data packet formed as per SSHv1.5 def.
**
**     Author/Date:             Tadayoshi Kohno, 9 December 1997
**
**     Notes:
**
*********************************************************************/

int ssh_cmsg_auth_rsa_response_encode
(
	uint8_t * data,			/* data for packet */
	uint32_t * data_len,		/* length of data */

	const uint8_t * md5_digest	/* md5 of challenge, session id */
)
{
	my_bcopy(md5_digest, data, MD5_DIGEST_LEN);
	*data_len = MD5_DIGEST_LEN;

	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_cmsg_window_size_encode
**
**     Purpose:                 compose a resize message for the server
**
**     Entry (pre) conditions:  connection established.
**
**     Parameters:              *data		packet of data
**				*data_len	length of *data
**
**				height		height in rows
**				width		width in columns
**				x_pixels	width, 0 for non-graphics
**				y_pixels	height, 0 for non-graphics
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD
**
**     Side effects:            *data packed for CMSG_WINDOW_SIZE message
**
**     Author/Date:             Tadayoshi Kohno, 16 December 1997
**
**     Notes:
**	Use this to send a resize message to the server.
**
*********************************************************************/

int ssh_cmsg_window_size_encode
(
	uint8_t * data,		/* packed data */
	uint32_t * data_len,	/* length of packet data */

	uint32_t height,	/* height in rows */
	uint32_t width,		/* width in columns */
	uint32_t x_pixels,	/* width in pixels */
	uint32_t y_pixels	/* height in pixels */
)
{
	int offset = 0;		/* offset into data */

	height = htonl(height);
	width = htonl(width);
	x_pixels = htonl(x_pixels);
	y_pixels = htonl(y_pixels);

	my_bcopy((void *) &height, (void *) (data + offset), sizeof(height));
	offset += sizeof(height);

	my_bcopy((void *) &width, (void *) (data + offset), sizeof(width));
	offset += sizeof(height);

	my_bcopy((void *) &x_pixels, (void *) (data + offset),
		sizeof(x_pixels));
	offset += sizeof(x_pixels);

	my_bcopy((void *) &y_pixels, (void *) (data + offset),
		sizeof(y_pixels));
	offset += sizeof(y_pixels);

	*data_len = offset;
	return(S_GOOD);
}

/******************************************************************************
**
**     File/Function name:	ssh_cmsg_request_compression_encode
**
**     Purpose:			encode a request for compression
**
**     Preconditions:		level is OK (1-9 -- see ssh_compress.h)
**
**     Parameters:		*data		packed data
**				*data_len	length of packed data
**
**				level		desired compression level
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD
**
**     Side effects:		*data packed for SSH_CMSG_REQUEST_COMPRESSION
**
**     Author/Date:		Tadayoshi Kohno, 4 April 1998
**
**     References:
**
**     Notes:
**
******************************************************************************/

int ssh_cmsg_request_compression_encode
(
	uint8_t * data,		/* packed data */
	uint32_t * data_len,	/* length of packet data */

	uint32_t level		/* desired compression level */
)
{
	level = htonl(level);
	my_bcopy((void *) &level, (void *) (data), sizeof(level));

	*data_len = sizeof(level);

	return(S_GOOD);
}

