Unit RRC2;

(****************************************************************************
*                                                                            *
*                      An Turbo-Pascal Implementation of                     *
*                                   RRC.2                                    *
*                                                                            *
*    Specification done by Peter Gutmann.                                    *
*                                                                            *
*    Implementation done on 12-21.02.96 by Daniel Vogelheim                  *
*                                                                            *
*    This document is placed into the public domain.                         *
*    I'd appreciate if my name was mentioned in derivative works.            *
*                                                                            *
 ****************************************************************************)


(* This implementation is based solely on the RRC.2 specification by
   Peter Gutmann and my own C implementation on Linux. These documents
   might bring additional insight in the code presented here.
*)


(* These are the "user"-functions. Procedures Encrypt and Decrypt encrypt
   and decrypt the variable Data (length Len) with key Key. Function Test
   tests implementation conformance with the specification.

   Users may require access to the "internal" functions as well. The
   Interface may be extended accordingly.
*)

(* Since the only big-endian Turbo-Pascal version I'm aware of was done
   for the now defunct Atari ST series, I did not bother with an endian
   neutral version. I am not sure how the various promising 32-bit Borland
   compatible Pascals deal with Byte and Word types. They may turn out to
   be incompatible.
*)


(****************************************************************************)

Interface

Procedure Encrypt(Key: String; Var Data; Len: Word);
Procedure Decrypt(Key: String; Var Data; Len: Word);

Function Test: Boolean;



(****************************************************************************)

Implementation


Type tSBox = Record
                Case Integer Of
                   0 : (b : Array[0..255] Of Byte);
                   1 : (w : Array[0..127] Of Word);
            End;

     tS = Record
                Case Integer Of
                   0 : (b : Array[0..127] Of Byte);
                   1 : (w : Array[0..63] Of Word);
            End;


(* Consts: copied from spec *)

Const Beale : Array[0..255] Of Word = (
  71, 194,  38,1701,  89,  76,  11,  83,1629,  48,  94,  63, 132,  16, 111,  95,
  84, 341, 975,  14,  40,  64,  27,  81, 139, 213,  63,  90,1120,   8,  15,   3,
 126,2018,  40,  74, 758, 485, 604, 230, 436, 664, 582, 150, 251, 284, 308, 231,
 124, 211, 486, 225, 401, 370,  11, 101, 305, 139, 189,  17,  33,  88, 208, 193,
 145,   1,  94,  73, 416, 918, 263,  28, 500, 538, 356, 117, 136, 219,  27, 176,
 130,  10, 460,  25, 485,  18, 436,  65,  84, 200, 283, 118, 320, 138,  36, 416,
 280,  15,  71, 224, 961,  44,  16, 401,  39,  88,  61, 304,  12,  21,  24, 283,
 134,  92,  63, 246, 486, 682,   7, 219, 184, 360, 780,  18,  64, 463, 474, 131,
 160,  79,  73, 440,  95,  18,  64, 581,  34,  69, 128, 367, 460,  17,  81,  12,
 103, 820,  62, 110,  97, 103, 862,  70,  60,1317, 471, 540, 208, 121, 890, 346,
  36, 150,  59, 568, 614,  13, 120,  63, 219, 812,2160,1780,  99,  35,  18,  21,
 136, 872,  15,  28, 170,  88,   4,  30,  44, 112,  18, 147, 436, 195, 320,  37,
 122, 113,   6, 140,   8, 120, 305,  42,  58, 461,  44, 106, 301,  13, 408, 680,
  93,  86, 116, 530,  82, 568,   9, 102,  38, 416,  89,  71, 216, 728, 965, 818,
   2,  38, 121, 195,  14, 326, 148, 234,  18,  55, 131, 234, 361, 824,   5,  81,
 623,  48, 961,  19,  26,  33,  10,1101, 365,  92,  88, 181, 275, 346, 201, 206
 );

Const Pad : Array[0..255] Of Byte = (
 158, 186, 223,  97,  64, 145, 190, 190, 117, 217, 163,  70, 206, 176, 183, 194,
 146,  43, 248, 141,   3,  54,  72, 223, 233, 153,  91, 210,  36, 131, 244, 161,
 105, 120, 113, 191, 113,  86,  19, 245, 213, 221,  43,  27, 242, 157,  73, 213,
 193,  92, 166,  10,  23, 197, 112, 110, 193,  30, 156,  51, 125,  51, 158,  67,
 197, 215,  59, 218, 110, 246, 181,   0, 135,  76, 164,  97,  47,  87, 234, 108,
 144, 127,   6,   6, 222, 172,  80, 144,  22, 245, 207,  70, 227, 182, 146, 134,
 119, 176,  73,  58, 135,  69,  23, 198,   0, 170,  32, 171, 176, 129,  91,  24,
 126,  77, 248,   0, 118,  69,  57,  60, 190, 171, 217,  61, 136, 169, 196,  84,
 168, 167, 163, 102, 223,  64, 174, 178, 166, 239, 242, 195, 249,  92,  59,  38,
 241,  46, 236,  31,  59, 114,  23,  50, 119, 186,   7,  66, 212,  97, 222, 182,
 230, 118, 122,  86, 105,  92, 179, 243, 255, 189, 223, 164, 194, 215,  98,  44,
  17,  20,  53, 153, 137, 224, 176, 100, 208, 114,  36, 200, 145, 150, 215,  20,
  87,  44, 252,  20, 235, 242, 163, 132,  63,  18,   5, 122,  74,  97,  34,  97,
 142,  86, 146, 221, 179, 166, 161,  74,  69, 182,  88, 120, 128,  58,  76, 155,
  15,  30,  77, 216, 165, 117, 107,  90, 169, 127, 143, 181, 208, 137, 200, 127,
 170, 195,  26,  84, 255, 132, 150,  58, 103, 250, 120, 221, 237,  37,   8,  99
);



(****************************************************************************)

(* Necessary rotation Operators; should be inlined asm for decent
   performance.
*)

Function LRot (w:Word; b:Byte) : Word;
Begin
LRot:=(w ShL b) Or (w ShR (16-b));
End; (* LRot *)



Function RRot(w:Word; b:Byte) : Word;
Begin
RRot:=(w ShR b) Or (w ShL (16-b));
End; (* RRot *)


(****************************************************************************)

(* Here come the obvious functions from the spec... *)

(* A lot less type-casting required than in C. Good! *)

Procedure Initialise_SBox(Var SBox: tSBox);

Var i : Word;

Begin
For i := 0 to 255
Do SBox.b[i] := (Beale[i] mod 256) XOr Pad[i];
End; (* Initialise_SBox *)



Procedure Expand_Key(Key: String; Var SBox: tSBox; Var S: tS);

Var j : Word;

Begin
For j := 0 to Length(Key)-1 Do S.b[j] := Byte(Key[j+1]);
For j := Length(Key) to 127 Do S.b[j] := sBox.b[(S.b[j - Length(Key)] + S.b[j - 1]) mod 256];
S.b[0] := sBox.b[S.b[0]];
End; (* Expand_Key *)


(* For decent performance the w? passwd as an array. Some pointer fiddling
   could then be use succesive values from array. I happen not to like
   pointer fiddling, however...
*)

Procedure _Encrypt(Var S: tS; Var w0, w1, w2, w3: Word);

Var i,j : Word;

Begin
For i := 0 To 15
Do Begin
   j := i * 4;

   w0 := LROT(w0 + (w1 And Not w3) + (w2 And w3) + S.w[j + 0], 1);
   w1 := LROT(w1 + (w2 And Not w0) + (w3 And w0) + S.w[j + 1], 2);
   w2 := LROT(w2 + (w3 And Not w1) + (w0 And w1) + S.w[j + 2], 3);
   w3 := LROT(w3 + (w0 And Not w2) + (w1 And w2) + S.w[j + 3], 5);

   If ((i=4) Or (i=10))
   Then Begin
        w0 := w0 + S.w[w3 And 63];
        w1 := w1 + S.w[w0 And 63];
        w2 := w2 + S.w[w1 And 63];
        w3 := w3 + S.w[w2 And 63];
        End; (* If *)

   End; (* For *)
End; (* _Encrypt *)



Procedure _Decrypt(Var S: tS; Var w0, w1, w2, w3: Word);

Var i,j : Word;

Begin
For i := 15 DownTo 0
Do Begin
    j := i * 4;

    w3 := RROT(w3, 5) - (w0 And Not w2) - (w1 And w2) - S.w[j + 3];
    w2 := RROT(w2, 3) - (w3 And Not w1) - (w0 And w1) - S.w[j + 2];
    w1 := RROT(w1, 2) - (w2 And Not w0) - (w3 And w0) - S.w[j + 1];
    w0 := RROT(w0, 1) - (w1 And Not w3) - (w2 And w3) - S.w[j + 0];

    If ((i=11) Or (i=5))
    Then Begin
         w3 := w3 - S.w[w2 And 63];
         w2 := w2 - S.w[w1 And 63];
         w1 := w1 - S.w[w0 And 63];
         w0 := w0 - S.w[w3 And 63];
         End; (* If *)

    End; (* For *)
End; (* _Decrypt *)


(****************************************************************************)

(* These are the "user" procedures. They use the cipher repeatedly
   on an untyped variable. They take the key in form of a string, since
   Pascal-Strings can deal with $00.

   Cunning type casting with the absolute arr to let the compiler do
   the address calculations :-)
*)


Procedure Encrypt(key: String; Var data; Len: Word);

Var SBox : tSBox;
    S    : tS;
    arr  : Array[0..1000,0..3] Of Word Absolute data;
    i    : Word;

Begin
If (Len Mod 8)<>0
Then Begin
     WriteLn('Error in RRC2.Encrypt: Length not multiple of Blocksize!');
     Halt(1);
     End;

Initialise_SBox(SBox);
Expand_Key(key,SBox,S);

For i:=0 To (Len div 8)-1 Do _Encrypt(S,arr[i,0],arr[i,1],arr[i,2],arr[i,3]);

End; (* Encrypt *)



Procedure Decrypt(key: String; Var data; Len: Word);

Var SBox : tSBox;
    S    : tS;
    arr  : Array[0..1000,0..3] Of Word Absolute data;
    i    : Word;

Begin
If (Len Mod 8)<>0
Then Begin
     WriteLn('Error in RRC2.Decrypt: Length not multiple of Blocksize!');
     Halt(1);
     End;

Initialise_SBox(SBox);
Expand_Key(key,SBox,S);

For i:=0 To (Len div 8)-1 Do _Decrypt(S,arr[i,0],arr[i,1],arr[i,2],arr[i,3]);

End; (* Decrypt *)


(****************************************************************************)

(* This Function tests spec conformance and correctness of implementation.
   This should be an assert, if Pascal knew one...
*)

Function Test: Boolean;

         Function _Test(Var k,p,c): Boolean;

         Type arr = Array[0..7] Of Byte;

         Var Key        : String Absolute k;
             Plain      : arr Absolute p;
             Cipher     : arr Absolute c;
             arr1, arr2 : arr;
             i          : Word;
             b          : Boolean;

         Begin
         arr1:=Plain;
         arr2:=Cipher;

         Encrypt(Key, arr1, 8);
         Decrypt(Key, arr2, 8);

         b:=True;
         For i:=0 To 7
         Do b:=b And (arr1[i]=Cipher[i]) And (arr2[i]=Plain[i]);

         _Test:=b;
         End; (* _Test *)



Const
     Key1 : Array[0..16] Of Byte = ( 16,
            $00, $00, $00, $00, $00, $00, $00, $00,
            $00, $00, $00, $00, $00, $00, $00, $00
            );
     Plain1 : Array[0..7] Of Byte = (
            $00, $00, $00, $00, $00, $00, $00, $00
            );
     Cipher1 : Array[0..7] Of Byte = (
            $1C, $19, $8A, $83, $8D, $F0, $28, $B7
            );

     Key2 : Array[0..16] Of Byte = ( 16,
            $00, $00, $00, $00, $00, $00, $00, $00,
            $00, $00, $00, $00, $00, $00, $00, $01
            );
     Plain2 : Array[0..7] Of Byte = (
            $00, $00, $00, $00, $00, $00, $00, $00
            );
     Cipher2 : Array[0..7] Of Byte = (
            $21, $82, $9C, $78, $A9, $F9, $C0, $74
            );

     Key3 : Array[0..16] Of Byte = ( 16,
            $00, $00, $00, $00, $00, $00, $00, $00,
            $00, $00, $00, $00, $00, $00, $00, $00
            );
     Plain3 : Array[0..7] Of Byte = (
            $FF, $FF, $FF, $FF, $FF, $FF, $FF, $FF
            );
     Cipher3 : Array[0..7] Of Byte = (
            $13, $DB, $35, $17, $D3, $21, $86, $9E
            );

     Key4 : Array[0..16] Of Byte = ( 16,
            $00, $01, $02, $03, $04, $05, $06, $07,
            $08, $09, $0A, $0B, $0C, $0D, $0E, $0F
            );
     Plain4 : Array[0..7] Of Byte = (
            $00, $00, $00, $00, $00, $00, $00, $00
            );
     Cipher4 : Array[0..7] Of Byte = (
            $50, $DC, $01, $62, $BD, $75, $7F, $31
            );


Var w1,w2 : Word;

Begin
Test := (LROT(1,3)=8) And
        (RROT($5c5c,4)=$c5c5) And
        _Test(Key1, Plain1, Cipher1) And
        _Test(Key2, Plain2, Cipher2) And
        _Test(Key3, Plain3, Cipher3) And
        _Test(Key4, Plain4, Cipher4);
End; (* Test *)


(****************************************************************************)

(* No global vars, no initialization. *)

Begin
End. (* RRC2 *)

(* FINI rrc2.pas *)
