/* SCCSID: fwk/cssm/cssm.c, dss_cdsa_fwk, fwk_rel3, dss_980413 1.67 4/13/98 11:18:55 */
/* ***************************************************************** *
 * Copyright 1998 International Business Machines Corporation. All   *
 * Rights Reserved.                                                  *
 *                                                                   *
 * Please read this carefully.  Your use of this reference           *
 * implementation of certain of the IETF public-key infrastructure   *
 * specifications ("Software") indicates your acceptance of the      *
 * following.  If you do not agree to the following, do not install  *
 * or use any of the Software.                                       *
 *                                                                   *
 * Permission to use, reproduce, distribute and create derivative    *
 * works from the Software ("Software Derivative Works"), and to     *
 * distribute such Software Derivative Works is hereby granted to    *
 * you by International Business Machines Corporation ("IBM").  This *
 * permission includes a license under the patents of IBM that are   *
 * necessarily infringed by your use of the Software as provided by  *
 * IBM.                                                              *
 *                                                                   *
 * IBM licenses the Software to you on an "AS IS" basis, without     *
 * warranty of any kind.  IBM HEREBY EXPRESSLY DISCLAIMS ALL         *
 * WARRANTIES OR CONDITIONS, EITHER EXPRESS OR IMPLIED, INCLUDING,   *
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OR CONDITIONS OF       *
 * MERCHANTABILITY, NON INFRINGEMENT AND FITNESS FOR A PARTICULAR    *
 * PURPOSE.  You are solely responsible for determining the          *
 * appropriateness of using this Software and assume all risks       *
 * associated with the use of this Software, including but not       *
 * limited to the risks of program errors, damage to or loss of      *
 * data, programs or equipment, and unavailability or interruption   *
 * of operations.                                                    *
 *                                                                   *
 * IBM WILL NOT BE LIABLE FOR ANY DIRECT DAMAGES OR FOR ANY SPECIAL, *
 * INCIDENTAL, OR  INDIRECT DAMAGES OR FOR ANY ECONOMIC              *
 * CONSEQUENTIAL DAMAGES (INCLUDING LOST PROFITS OR SAVINGS), EVEN   *
 * IF IBM HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.  IBM  *
 * will not be liable for the loss of, or damage to, your records or *
 * data, or any damages claimed by you based on a third party claim. *
 *                                                                   *
 * IBM wishes to obtain your feedback to assist in improving the     *
 * Software.  You grant IBM a world-wide, royalty-free right to use, *
 * copy, distribute, sublicense and prepare derivative works based   *
 * upon any feedback, including materials, error corrections,        *
 * Software Derivatives, enhancements, suggestions and the like that *
 * you provide to IBM relating to the Software (this does not        *
 * include products for which you charge a royalty and distribute to *
 * IBM under other terms and conditions).                            *
 *                                                                   *
 * You agree to distribute the Software and any Software Derivatives *
 * under a license agreement that: 1) is sufficient to notify all    *
 * licensees of the Software and Software Derivatives that IBM       *
 * assumes no liability for any claim that may arise regarding the   *
 * Software or Software Derivatives, and 2) that disclaims all       *
 * warranties, both express and implied, from IBM regarding the      *
 * Software and Software Derivatives.  (If you include this          *
 * Agreement with any distribution of the Software or Software       *
 * Derivatives you will have met this requirement.)  You agree that  *
 * you will not delete any copyright notices in the Software.        *
 *                                                                   *
 * This Agreement is the exclusive statement of your rights in the   *
 * Software as provided by IBM.   Except for the rights granted to   *
 * you in the second paragraph above, You are not granted any other  *
 * patent rights, including but not limited to the right to make     *
 * combinations of the Software with products that infringe IBM      *
 * patents. You agree to comply with all applicable laws and         *
 * regulations, including all export and import laws and regulation. *
 * This Agreement is governed by the laws of the State of New York.  *
 * This Agreement supersedes all other communications,               *
 * understandings or agreements we may have had prior to this        *
 * Agreement.                                                        *
 * ***************************************************************** */

/*-----------------------------------------------------------------------
 *      File:   CSSM.C
 *
 */
/*
 * (C) COPYRIGHT International Business Machines Corp. 1996, 1997
 * All Rights Reserved
 * Licensed Materials - Property of IBM
 *
 * Copyright (c) 1995, 1996, 1997 Intel Corporation. All rights reserved.
 *
 * US Government Users Restricted Rights - Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 */
/*
 * WARNING: EXPORT RESTRICTED.
 * This software listing contains cryptographic methods and technology.
 * It is export restricted by the Office of Defense Trade Controls, United
 * States Department of State and cannot be downloaded or otherwise
 * exported or re-exported (i) into (or to a national or resident of) Cuba,
 * Iraq, Libya, Yugoslavia, North Korea, Iran, Syria or any other country
 * to which the US has embargoed goods; or (ii) to anyone on the US
 * Treasury Department's list of Specially Designated Nationals or the US
 * Commerce Department's Table of Denial Orders. By downloading or using
 * this product, you are agreeing to the foregoing and you are representing
 * and warranting that you are not located in, under the control of, or a
 * national or resident of any such country or on any such list.
 */

#include "cssmdefs.h"
#include "cssm.h"
#include "cssmport.h"
#include "internal.h"
#include "time.h"
#include "cssmport.h"

#include "context.h"

#include "vendor.h"

#ifdef VERIFY_EXPIRATION
/* Set these if you want an expiration date */
#define YEAR yy
#define MONTH mm
#define DAY dd
#endif

#pragma data_seg("Shared")

/* GUID for CSSM */
#ifndef BOOTSTRAP
CSSM_GUID cssm_GUID = { 0x53ba1c39, 0xf3c1, 0x51cf, { 0x59, 0x72, 0x14, 0xac, 0x0, 0xb1, 0x89, 0x1d } };
#else
CSSM_GUID cssm_GUID = { 0x3a529ec0, 0x1b03, 0x11d1, { 0x90, 0xe9, 0x0, 0x4, 0xac, 0x60, 0x49, 0x9c } };
#endif

#pragma data_seg()

HANDLE hCssmAddinMutex			= NULL;		// addinInfo synchronization
HANDLE hCssmRegistryMutex		= NULL;		// registry synchronization

/* apps supplied memory functions to CSSM */
CSSM_FREE		app_free				= NULL;
CSSM_MALLOC		app_malloc				= NULL;
CSSM_REALLOC	app_realloc				= NULL;
CSSM_CALLOC		app_calloc				= NULL;
void 			*memRef;

#ifdef VERIFY_EXPIRATION
/*---------------------------------------------------------------
 *
 *Name: cssm_DaysExpire
 *
 *Description:
 *   This function calculates the number of days to the
 *   expiration date.
 *
 *Parameters:
 *   NOne
 *
 *Returns:
 *   Number of days to expiration date.  A negative value when
 *   current date is after expiration date.
 *
 *-------------------------------------------------------------*/
sint32 cssm_DaysExpire (void)
{
    struct tm *t;
    time_t ltime, xtime;
    double diff;

    /* Get the current time */

    time (&ltime);

    /* Create time_t of expiration date */
    t = localtime (&ltime);
    t->tm_mon = MONTH;
    t->tm_year = YEAR;
    t->tm_mday = DAY;
    xtime = mktime (t);

    /* Find the difference between times */
    diff = difftime (xtime, ltime);

    /* Return difference in days */
    return((sint32)(diff/(60*60*24)));
}
#endif /* VERIFY_EXPIRATION */

#ifdef VERIFY_EXPIRATION
/*---------------------------------------------------------------
 *
 *Name: cssm_Expire
 *
 *Description:
 *   This functions determines if code has expired.  When it does,
 *   it puts up a dialog box and return an error.  You also get a
 *   MessageBox 3 weeks, 2 weeks and everyone for the week before
 *   expiration.
 *
 *Parameters:
 *   None
 *
 *Returns:
 *   CSSM_FAIL - if pass expiration or clock set before 10/1/96
 *   CSSM_OK - current date before expiration date.
 *
 *Note:
 *   This module is not portable because of IO.
 *-------------------------------------------------------------*/
CSSM_RETURN cssm_Expire (void)
{
    sint32 days;

    days = cssm_DaysExpire ();

    /* Decide what to do */
    if (days > 0) {
        char st[80];

        if (days > 21) {
            if (days < 72)
                return(CSSM_FAIL);
            else {
                MessageBox (0, "Please check your internal clock.\nCSSM not loading.", "CSSM", 0);
                return(CSSM_OK);
            }
        } else if (days == 21) {
            sprintf (st, "%d days before beta program expires.", days);
            MessageBox (0, st, "CSSM", 0);
        } else if (days == 14) {
            sprintf (st, "%d days before beta program expires.", days);
            MessageBox (0, st, "CSSM", 0);
        } else if (days < 8) {
            sprintf (st, "%d days before beta program expires.", days);
            MessageBox (0, st, "CSSM", 0);
        }
        return(CSSM_FAIL);
    }

    MessageBox (0, "The beta program is over.", "CSSM", 0);
    return(CSSM_OK);
}
#endif /* VERIFY_EXPIRATION */

/*---------------------------------------------------------------
 *
 *Name: cssm_SpiMalloc
 *
 *Description:
 *   All add-in mallocs come through here to get memory from an app
 *
 *Parameters:
 *   handle - describe the application that supplied memory fxs
 *   size - the number of bytes to malloc
 *
 *Returns:
 *   NULL - error in allocating memory
 *   not NULL - pointer to memory
 *
 *-------------------------------------------------------------*/
void * cssm_SpiMalloc (CSSM_HANDLE handle, uint32 size)
{
	void	*rc;
    CSSM_API_MEMORY_FUNCS_PTR mem_ptr = cssm_GetMemory (handle);

    if (mem_ptr == NULL)
        return(NULL);

    rc = mem_ptr->malloc_func (size, mem_ptr->AllocRef);
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: cssm_SpiFree
 *
 *Description:
 *   All add-in frees come through here to get memory from an app
 *
 *Parameters:
 *   handle - describe the application that supplied memory fxs
 *   mem - pointer to memory for freeing
 *
 *Returns:
 *   None
 *
 *-------------------------------------------------------------*/
void cssm_SpiFree (CSSM_HANDLE handle, void *mem)
{
    CSSM_API_MEMORY_FUNCS_PTR mem_ptr = cssm_GetMemory (handle);

    if (mem_ptr == NULL)
        return;

   mem_ptr->free_func (mem, mem_ptr->AllocRef);
	return;
}

/*---------------------------------------------------------------
 *
 *Name: cssm_SpiRealloc
 *
 *Description:
 *   All add-in reallocs come through here to get memory from an app
 *
 *Parameters:
 *   handle - describe the application that supplied memory fxs
 *   mem - pointer to previously allocated memory
 *   size - the number of bytes to malloc
 *
 *Returns:
 *   NULL - error in allocating memory
 *   non NULL - pointer to memory
 *
 *-------------------------------------------------------------*/
void * cssm_SpiRealloc (CSSM_HANDLE handle, void *mem, uint32 size)
{
    CSSM_API_MEMORY_FUNCS_PTR	mem_ptr = cssm_GetMemory (handle);
	void						*rc;

    if (mem_ptr == NULL)
        return(NULL);

    rc = mem_ptr->realloc_func (mem, size, mem_ptr->AllocRef);
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: cssm_SpiCalloc
 *
 *Description:
 *   All add-in callocs come through here to get memory from an app
 *
 *Parameters:
 *   handle - describe the application that supplied memory fxs
 *   num - value to initialize memory
 *   size - the number of bytes to malloc
 *
 *Returns:
 *   NULL - error in allocating memory
 *   non NULL - pointer to memory
 *
 *-------------------------------------------------------------*/
void * cssm_SpiCalloc (CSSM_HANDLE handle, uint32 num, uint32 size)
{
    CSSM_API_MEMORY_FUNCS_PTR	mem_ptr = cssm_GetMemory (handle);
	void						*rc;

    if (mem_ptr == NULL)
        return(NULL);

    rc = mem_ptr->calloc_func (num, size, mem_ptr->AllocRef);
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: DllMain
 *
 *Description:
 * This is the main entry point of DLL required by Windows
 *
 *
 *Parameters:
 * hInstance(iuput) - instance handle of DLL.
 * dwReason (input) - reason for windows calling this function.
 * lpReserved (input) - data associated with this callback.
 *
 *Returns:
 * TRUE - call is successful
 * FALSE - abort loading of DLL
 *
 *-------------------------------------------------------------*/
BOOL WINAPI DllMain (HANDLE hInstance, DWORD dwReason, LPVOID lpReserved)
{

     switch (dwReason) {
        case DLL_PROCESS_ATTACH: {

			if(cssm_port_init() != CSSM_OK) {
		         return(FALSE);
			}

#if VERIFY_EXPIRATION
            /* Make sure the SW is still good */
            if (cssm_Expire () == CSSM_OK)
                return(FALSE);
#endif

            /* Initialize memory structure to hand to add-ins */
            /* Obsolete after migration to common memory function tables */
            InternalMemoryFuncs.malloc_func = cssm_SpiMalloc;
            InternalMemoryFuncs.free_func = cssm_SpiFree;
            InternalMemoryFuncs.realloc_func = cssm_SpiRealloc;
            InternalMemoryFuncs.calloc_func = cssm_SpiCalloc;

            /* allocate mem for error in that thread */
            CSSM_InitError ();

            /* clear the error */
            CSSM_ClearError ();

            /* DLL is attaching to the address space of the current process. */
#ifdef _DEBUG
            OutputDebugString("CSSM: Process attaching\n");
#endif
            break;
        }

        case DLL_THREAD_ATTACH:
            /* A new thread is being created in the current process */

            /* clear the error */
            CSSM_ClearError ();

#ifdef _DEBUG
            OutputDebugString("CSSM: Thread attaching\n");
#endif
        break;

        case DLL_THREAD_DETACH:
            /* A thread is exiting cleanly */
#ifdef _DEBUG
        OutputDebugString("CSSM: Thread detaching\n");
#endif
        break;

        case DLL_PROCESS_DETACH:
            /* The calling process is detaching the DLL from its address space */

            /* free mem for error in that process */
            cssm_CleanUp ();

            CSSM_DestroyError ();

            //Mutexes are closed
            if(hCssmAddinMutex != NULL) {
                cssm_CloseMutex(hCssmAddinMutex);
            }

            if(hCssmRegistryMutex != NULL) {
                cssm_CloseMutex(hCssmRegistryMutex);
            }

		cssm_port_terminate();

        break;
    }

    return(TRUE);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_GetInfo
 *
 *Description:
 *   Return information that describes the CSSM
 *
 *Parameters:
 *   None
 *
 *Returns:
 *   NULL - error in getting the structure
 *   non NULL - structure that describes CSSM
 *
 *-------------------------------------------------------------*/
CSSM_CSSMINFO_PTR CSSMAPI CSSM_GetInfo (void)
{
	CSSM_CSSMINFO_PTR	rc;

    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(NULL);

    rc = cssm_RegReadCSSMInfo();
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_FreeInfo
 *
 *Description:
 *   Function to remove memory of CSSM_INFO structure
 *
 *Parameters:
 *   CssmInfo - pointer to CSSM_INFO structure returned from
 *              a CSSM_GetInfo call.
 *
 *Returns:
 *   CSSM_OK - structure has been freed.
 *   CSSM_FAIL - unable to free structure.
 *
 *Note:
 *-------------------------------------------------------------*/
CSSM_RETURN CSSMAPI CSSM_FreeInfo (CSSM_CSSMINFO_PTR CssmInfo)
{
    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(CSSM_FAIL);

    /* Check validity of pointers passed in */
    if (cssm_IsBadReadPtr(CssmInfo, sizeof (CSSM_CSSMINFO))) {
        CSSM_SetError (&cssm_GUID, CSSM_INVALID_POINTER);
        return(CSSM_FAIL);
    }

    app_free (CssmInfo, memRef);
    return(CSSM_OK);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_Init
 *
 *Description:
 *   Exported function for application to initialize with the
 *   CSSM.
 *
 *Parameters:
 *  Version - version of CSSM
 *  MemoryFunctions - Memory functions from the application
 *  Reserved - TBD
 *
 *Returns:
 *  CSSM_FAIL - CSSM unable to initialize
 *  CSSM_OK - Initialization complete
 *
 *Note:
 *-------------------------------------------------------------*/
CSSM_RETURN CSSMAPI CSSM_Init (const CSSM_VERSION_PTR Version,
                               const CSSM_API_MEMORY_FUNCS_PTR MemoryFunctions,
                               const void *Reserved)
{

    /* clear the error */
    CSSM_ClearError ();

    /* Make sure that memory functions are filled */
    if (cssm_CheckAPIMemFuncs (MemoryFunctions) == CSSM_FAIL)
        return(CSSM_FAIL);

    /* initialize memory functions */

	if ((cssm_CheckInit()== CSSM_OK)) {
	/* app_malloc has been initialized before, check and make sure
	 * memory functions are the same
	 */
		if( (app_malloc		!= MemoryFunctions->malloc_func)  ||
			(app_calloc		!= MemoryFunctions->calloc_func)  ||
			(app_realloc	!= MemoryFunctions->realloc_func) ||
			(app_free		!= MemoryFunctions->free_func)  )
		{
			CSSM_SetError( &cssm_GUID, CSSM_MEM_FUNCS_NOT_MATCHING);
			return(CSSM_FAIL);
		}
	} else {
		app_malloc = MemoryFunctions->malloc_func;
		app_realloc = MemoryFunctions->realloc_func;
		app_free = MemoryFunctions->free_func;
		app_calloc = MemoryFunctions->calloc_func;
		memRef = MemoryFunctions->AllocRef;
	}
    /* Check for version information */
    if (Version->Major != CSSM_MAJOR || Version->Minor != CSSM_MINOR) {
        CSSM_SetError (&cssm_GUID, CSSM_INCOMPATIBLE_VERSION);
        return(CSSM_FAIL);
    }

    if(cssm_port_init() != CSSM_OK)
         return(CSSM_FAIL);

    /* Mutexes are created */
    if(hCssmAddinMutex == NULL) {
        hCssmAddinMutex = cssm_CreateMutex("INTELCSSMADDINMUTEX");
        if(hCssmAddinMutex == NULL) {
            CSSM_SetError(&cssm_GUID, CSSM_CANT_CREATE_MUTEX);
            return(CSSM_FAIL);
        }
    }

	if ( cssm_InitContextTable() != CSSM_OK) {
		// cssm error is set
		return(CSSM_FAIL);;
	}

    if(hCssmRegistryMutex == NULL) {
        hCssmRegistryMutex = cssm_CreateMutex("INTELCSSMREGISTRYMUTEX");
        if(hCssmRegistryMutex == NULL) {
            CSSM_SetError(&cssm_GUID, CSSM_CANT_CREATE_MUTEX);
            return(CSSM_FAIL);
        }
    }

    return(CSSM_OK);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_VerifyComponents
 *
 *Description:
 *   Call into the test and check library
 *
 *Parameters:
 *   None
 *
 *Returns:
 *   CSSM_FAIL - Test and check failed
 *   CSSM_OK - Test and check OK
 *
 *Note:
 *-------------------------------------------------------------*/
extern cssm_INTERNALMODULELIST_PTR ModuleListHead;

CSSM_RETURN CSSMAPI CSSM_VerifyComponents (void)
{
	CSSM_RETURN ret = CSSM_OK;

    /* clear the error */
    CSSM_ClearError ();

	return(ret);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_Free
 *
 *Description:
 *   Frees memory allocated by addin module
 *
 *Parameters:
 *   MemPtr - memory to be freed
 *   AddInHandle - handle to addin that allocated the memory
 *
 *Returns:
 *   CSSM_FAIL - unable to free memory
 *   CSSM_OK - memory is freed
 *
 *-------------------------------------------------------------*/
void CSSMAPI CSSM_Free (void *MemPtr, CSSM_HANDLE AddInHandle)
{
    CSSM_API_MEMORY_FUNCS_PTR mem_ptr = cssm_GetMemory (AddInHandle);

    if (mem_ptr == NULL)
        return;

    mem_ptr->free_func (MemPtr, mem_ptr->AllocRef);
	return;
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_GetAPIMemoryFunctions
 *
 *Description:
 *   Retreives the memory functions associated with the addin
 *
 *Parameters:
 *   AddInHandle - handle to addin
 *
 *Returns:
 *   NULL - unable to get memory functions
 *   not NULL - pointer to structure that holds memory functions
 *
 *-------------------------------------------------------------*/
CSSM_API_MEMORY_FUNCS_PTR CSSMAPI CSSM_GetAPIMemoryFunctions (
                                                   CSSM_HANDLE AddInHandle)
{
	CSSM_API_MEMORY_FUNCS_PTR	rc;

    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(0);

    rc = cssm_GetMemory (AddInHandle);
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_ModuleAttach
 *
 *Description:
 *   Attach and load the addin module
 *
 *Parameters:
 *   GUID - GUID to addin module
 *   Version - version requested of addin module
 *   MemoryFuncs - memory functions for addin module
 *   SubServiceID - ID of service requested
 *   SubServiceFlags - flags of service requested
 *   Application - ID passed back in notification callback
 *   Notification - callback used by addin to notify application
 *                  on addin events
 *   Reserved - TBD
 *
 *Returns:
 *   0 - unable to attach to addin
 *   not 0 - handle to addin
 *
 *-------------------------------------------------------------*/
CSSM_MODULE_HANDLE CSSMAPI CSSM_ModuleAttach (
                                    const CSSM_GUID_PTR GUID,
                                    const CSSM_VERSION_PTR pVersion,
                                    const CSSM_API_MEMORY_FUNCS_PTR MemoryFuncs,
                                    uint32 SubServiceID,
                                    uint32 SubServiceFlags,
                                    uint32 Application,
                                    const CSSM_NOTIFY_CALLBACK Notification,
                                    const void * Reserved)
{
	CSSM_MODULE_HANDLE	rc;

    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(0);

    /* Make sure that memory functions are filled */
    if (cssm_CheckAPIMemFuncs (MemoryFuncs) == CSSM_FAIL)
        return(0);

    rc = cssm_ModuleAttach (GUID, pVersion, MemoryFuncs, SubServiceID,
                              SubServiceFlags, Application, Notification,
                              CSSM_EVENT_ATTACH, CSSM_EVENT_DETACH);
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_ModuleDetach
 *
 *Description:
 *   Detach addin module from the infrastructure
 *
 *Parameters:
 *   ModuleHandle - handle to addin
 *
 *Returns:
 *   CSSM_FAIL - unable to detach module
 *   CSSM_OK - module is detached
 *
 *-------------------------------------------------------------*/
CSSM_RETURN CSSMAPI CSSM_ModuleDetach (CSSM_MODULE_HANDLE ModuleHandle)
{
	CSSM_RETURN	rc = CSSM_FAIL;

    /* clear the error */

    CSSM_ClearError ();

	// lock the addin queue
    if (cssm_LockMutex (hCssmAddinMutex, CSSM_INFINITE_WAIT) != CSSM_MUTEX_LOCKED) {
        return(CSSM_FAIL);
    }

    rc = cssm_AddInDetach (ModuleHandle, CSSM_EVENT_DETACH);

	// clean the stage
	if (cssm_UnlockMutex (hCssmAddinMutex) != CSSM_MUTEX_UNLOCKED) {
		return(CSSM_FAIL);
	}

	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_ListModules
 *
 *Description:
 *   List the registered addin modules
 *
 *Parameters:
 *   UsageMask - mask to describe the usage of addin
 *   MatchAll - flag to indicate if addin has to match all
 *              the conditions express in UsageMask
 *
 *Returns:
 *   NULL - unable to locate modules that match mask or an error
 *   not NULL - list of modules that match mask
 *
 *-------------------------------------------------------------*/
CSSM_LIST_PTR CSSMAPI CSSM_ListModules (CSSM_SERVICE_MASK UsageMask,
                                        CSSM_BOOL MatchAll)
{
	CSSM_LIST_PTR	rc;

    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(NULL);

    rc = cssm_RegReadModuleList (UsageMask, MatchAll);
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_GetGUIDUsage
 *
 *Description:
 *   Return the usage of module given the GUID
 *
 *Parameters:
 *   ModuleGUID - GUID of the addin module
 *
 *Returns:
 *   0 - error
 *   not 0 - Mask that desribes addin usage
 *
 *-------------------------------------------------------------*/
CSSM_SERVICE_MASK CSSMAPI CSSM_GetGUIDUsage (const CSSM_GUID_PTR ModuleGUID)
{
	CSSM_SERVICE_MASK	rc;

    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(0);

	/* check for bad GUID */
	if (ModuleGUID == NULL ||		
		cssm_IsBadReadPtr(ModuleGUID, sizeof(CSSM_GUID)) == CSSM_TRUE) {
		CSSM_SetError(&cssm_GUID,CSSM_INVALID_GUID);
        return(0);
	}

    rc = cssm_RegReadUsage (ModuleGUID);
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_GetHandleUsage
 *
 *Description:
 *   Return the usage of module given the handle to addin
 *
 *Parameters:
 *   ModuleHandle - handle to the addin module
 *
 *Returns:
 *   0 - error
 *   not 0 - Mask that desribes addin usage
 *
 *-------------------------------------------------------------*/
CSSM_SERVICE_MASK CSSMAPI CSSM_GetHandleUsage (CSSM_HANDLE ModuleHandle)
{
	CSSM_SERVICE_MASK	rc;

    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(0);

    rc = cssm_GetUsageMask (ModuleHandle);
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_SetModuleInfo
 *
 *Description:
 *   Update the registry with new module info
 *
 *Parameters:
 *   ModuleGUID - GUID of the addin module
 *   ModuleInfo - information on the addin module
 *
 *Returns:
 *   CSSM_FAIL - error in updating the registry
 *   CSSM_OK - updated registry with new module info
 *
 *-------------------------------------------------------------*/
CSSM_RETURN CSSMAPI CSSM_SetModuleInfo (const CSSM_GUID_PTR ModuleGUID,
                                        const CSSM_MODULE_INFO_PTR ModuleInfo)
{
    char		regKey[MAX_REG];
    char		ID[MAX_REG];
	CSSM_RETURN	rc;

    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(CSSM_FAIL);

    /* Generate sub-key name */
    strcpy (regKey, CSSMSECTION);
    cssm_GUIDToStr (ModuleGUID, ID);
    strcat (regKey, ID);

    rc = cssm_RegWriteModuleInfo (regKey, ModuleInfo);
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_GetModuleInfo
 *
 *Description:
 *   Get from the registry module info of the addin
 *
 *Parameters:
 *   ModuleGUID - GUID of the addin module
 *   UsageMask - the type of service
 *   SubserviceID - internal ID of service
 *   InfoLevel - data to return in module info
 *
 *Returns:
 *   NULL - error in retreiving info from the registry
 *   not NULL - pointer to module info
 *
 *-------------------------------------------------------------*/
CSSM_MODULE_INFO_PTR CSSMAPI CSSM_GetModuleInfo (const CSSM_GUID_PTR ModuleGUID,
                                                 CSSM_SERVICE_MASK UsageMask,
                                                 uint32 SubserviceID,
                                                 CSSM_INFO_LEVEL InfoLevel)
{
	CSSM_MODULE_INFO_PTR	rc;

    /* clear the error */

    CSSM_ClearError ();

	/* check for bad GUID */
	if (ModuleGUID == NULL ||		
		cssm_IsBadReadPtr(ModuleGUID, sizeof(CSSM_GUID)) == CSSM_TRUE) {
		CSSM_SetError(&cssm_GUID,CSSM_INVALID_GUID);
        return(NULL);
	}

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(NULL);

    rc = cssm_RegReadModuleInfo (ModuleGUID, UsageMask, SubserviceID, InfoLevel);
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_FreeModuleInfo
 *
 *Description:
 *   Free module info of the addin
 *
 *Parameters:
 *   ModuleInfo - data to be freed
 *
 *Returns:
 *   CSSM_FAIL - error in freeing the info
 *   CSSM_OK -  info is freed
 *
 *-------------------------------------------------------------*/
CSSM_RETURN CSSMAPI CSSM_FreeModuleInfo (CSSM_MODULE_INFO_PTR ModuleInfo)
{
	CSSM_RETURN	rc;

    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(CSSM_FAIL);

    rc = cssm_FreeModule (ModuleInfo);
	return(rc);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_GetCSSMRegistryPath
 *
 *Description:
 *   Get the registry path of the CSSM
 *
 *Parameters:
 *   void
 *
 *Returns:
 *   NULL - error in getting the info
 *   not NULL - pointer to registry path information
 *
 *-------------------------------------------------------------*/
CSSM_DATA_PTR CSSMAPI CSSM_GetCSSMRegistryPath (void)
{
    CSSM_DATA_PTR Result;

    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(NULL);

    if ((Result = (CSSM_DATA_PTR)app_malloc (sizeof (CSSM_DATA), memRef)) == NULL) {
        CSSM_SetError (&cssm_GUID, CSSM_MEMORY_ERROR);
        return(NULL);
    }

    if ((Result->Data = app_malloc (strlen (CSSMSECTION), memRef)) == NULL) {
        app_free (Result, memRef);
        CSSM_SetError (&cssm_GUID, CSSM_MEMORY_ERROR);
        return(NULL);
    }

    strcpy ((char *) Result->Data, CSSMSECTION);
    Result->Length = strlen (CSSMSECTION);
    return(Result);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_GetModuleLocation
 *
 *Description:
 *   Get the path of the addin
 *
 *Parameters:
 *   GUID - guid of the addin
 *
 *Returns:
 *   NULL - error in getting the info
 *   not NULL - pointer to path information
 *
 *-------------------------------------------------------------*/
CSSM_DATA_PTR CSSMAPI CSSM_GetModuleLocation (const CSSM_GUID_PTR GUID)
{
    char path[MAX_REG];
    CSSM_DATA_PTR Result;

    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(NULL);

    if (cssm_RegGetLocation (GUID, path) != CSSM_OK)
        return(NULL);

    if ((Result = (CSSM_DATA_PTR)app_malloc (sizeof (CSSM_DATA), memRef)) == NULL) {
        CSSM_SetError (&cssm_GUID, CSSM_MEMORY_ERROR);
        return(NULL);
    }

    if ((Result->Data = app_malloc (strlen (path) + 1, memRef)) == NULL) {
        app_free (Result, memRef);
        CSSM_SetError (&cssm_GUID, CSSM_MEMORY_ERROR);
        return(NULL);
    }

    strcpy ((char *) Result->Data, path);
    Result->Length = strlen (path);
    Result->Data[strlen (path)] = 0;

    return(Result);
}

/*---------------------------------------------------------------
 *
 *Name: CSSM_GetModuleGUIDFromHandle
 *
 *Description:
 *   Get the GUID associated with the addin handle
 *
 *Parameters:
 *   ModuleHandle - handle of the addin
 *
 *Returns:
 *   NULL - error in getting the info
 *   not NULL - pointer to guid associated with addin
 *
 *-------------------------------------------------------------*/
CSSM_GUID_PTR CSSMAPI CSSM_GetModuleGUIDFromHandle (CSSM_HANDLE ModuleHandle)
{
	CSSM_GUID_PTR	rc;
    /* clear the error */

    CSSM_ClearError ();

    /* make sure that app as done a CSSM_Init */
    if (cssm_CheckInit () == CSSM_FAIL)
        return(NULL);

    rc = cssm_GetGUIDFromHandle (ModuleHandle);
	return(rc);
}
