/* ***************************************************************** *
 * Copyright 1998 International Business Machines Corporation. All   *
 * Rights Reserved.                                                  *
 *                                                                   *
 * Please read this carefully.  Your use of this reference           *
 * implementation of certain of the IETF public-key infrastructure   *
 * specifications ("Software") indicates your acceptance of the      *
 * following.  If you do not agree to the following, do not install  *
 * or use any of the Software.                                       *
 *                                                                   *
 * Permission to use, reproduce, distribute and create derivative    *
 * works from the Software ("Software Derivative Works"), and to     *
 * distribute such Software Derivative Works is hereby granted to    *
 * you by International Business Machines Corporation ("IBM").  This *
 * permission includes a license under the patents of IBM that are   *
 * necessarily infringed by your use of the Software as provided by  *
 * IBM.                                                              *
 *                                                                   *
 * IBM licenses the Software to you on an "AS IS" basis, without     *
 * warranty of any kind.  IBM HEREBY EXPRESSLY DISCLAIMS ALL         *
 * WARRANTIES OR CONDITIONS, EITHER EXPRESS OR IMPLIED, INCLUDING,   *
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OR CONDITIONS OF       *
 * MERCHANTABILITY, NON INFRINGEMENT AND FITNESS FOR A PARTICULAR    *
 * PURPOSE.  You are solely responsible for determining the          *
 * appropriateness of using this Software and assume all risks       *
 * associated with the use of this Software, including but not       *
 * limited to the risks of program errors, damage to or loss of      *
 * data, programs or equipment, and unavailability or interruption   *
 * of operations.                                                    *
 *                                                                   *
 * IBM WILL NOT BE LIABLE FOR ANY DIRECT DAMAGES OR FOR ANY SPECIAL, *
 * INCIDENTAL, OR  INDIRECT DAMAGES OR FOR ANY ECONOMIC              *
 * CONSEQUENTIAL DAMAGES (INCLUDING LOST PROFITS OR SAVINGS), EVEN   *
 * IF IBM HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.  IBM  *
 * will not be liable for the loss of, or damage to, your records or *
 * data, or any damages claimed by you based on a third party claim. *
 *                                                                   *
 * IBM wishes to obtain your feedback to assist in improving the     *
 * Software.  You grant IBM a world-wide, royalty-free right to use, *
 * copy, distribute, sublicense and prepare derivative works based   *
 * upon any feedback, including materials, error corrections,        *
 * Software Derivatives, enhancements, suggestions and the like that *
 * you provide to IBM relating to the Software (this does not        *
 * include products for which you charge a royalty and distribute to *
 * IBM under other terms and conditions).                            *
 *                                                                   *
 * You agree to distribute the Software and any Software Derivatives *
 * under a license agreement that: 1) is sufficient to notify all    *
 * licensees of the Software and Software Derivatives that IBM       *
 * assumes no liability for any claim that may arise regarding the   *
 * Software or Software Derivatives, and 2) that disclaims all       *
 * warranties, both express and implied, from IBM regarding the      *
 * Software and Software Derivatives.  (If you include this          *
 * Agreement with any distribution of the Software or Software       *
 * Derivatives you will have met this requirement.)  You agree that  *
 * you will not delete any copyright notices in the Software.        *
 *                                                                   *
 * This Agreement is the exclusive statement of your rights in the   *
 * Software as provided by IBM.   Except for the rights granted to   *
 * you in the second paragraph above, You are not granted any other  *
 * patent rights, including but not limited to the right to make     *
 * combinations of the Software with products that infringe IBM      *
 * patents. You agree to comply with all applicable laws and         *
 * regulations, including all export and import laws and regulation. *
 * This Agreement is governed by the laws of the State of New York.  *
 * This Agreement supersedes all other communications,               *
 * understandings or agreements we may have had prior to this        *
 * Agreement.                                                        *
 * ***************************************************************** */

#include "exception.hpp"
#include "cssmerr.h"
#include "misc.hpp"
#include "deskey.hpp"
#include "descipher.hpp"

/**********************************************************
 *
 *	DESCipherEngine class
 *
 **********************************************************/
DESCipherEngine::DESCipherEngine(uint32 mode) 
:SymBlockCipherEngine(CSSM_ALGID_DES, mode, DES_BLOCK_SIZE)
{
}

void DESCipherEngine::storeExpandedKey(Key *key) {
	KeyMaterial	*keyMat;
	uchar		*keyBlob;

	if (key == NULL)
		throw InputException(CSSM_CSP_INVALID_KEY_POINTER);

	keyMat = key->exportKeyMat();
	//check if the blob is DES
	if (keyMat->getAlgId() != CSSM_ALGID_DES) {
		delete keyMat;
		throw InputException(CSSM_CSP_KEY_ALGID_NOTMATCH);
	}

	keyBlob = ((DESKey *)keyMat)->getBlob();
	delete keyMat;
	if (keyBlob == NULL)
		throw InputException(CSSM_CSP_INVALID_KEY);

	DESKeyExpand(keyBlob, expandedKey);
	delete keyBlob;
}

void DESCipherEngine::encryptInit(Key *key, void *params) {
	SymBlockCipherEngine::encryptInit(key, params);
	storeExpandedKey(key);
}

void DESCipherEngine::encryptBlock(CssmData &clearBuff, CssmData &cipherBuff, uint32 *bytesEncrypted) {
	int		rc;
	uint32	cMode = getCylinkMode(mode);

	rc = DESEncrypt(IV.getData(), (uchar*)expandedKey, cMode, clearBuff.getData(), cipherBuff.getData(), clearBuff.getLength());	
	if (rc != SUCCESS) {
		*bytesEncrypted = 0;
		cipherBuff.setLength(0);
		throw CryptoLibException(CSSM_CSP_ENC_FAILED);
	}
	else {
		*bytesEncrypted = clearBuff.getLength();
	}
}

void DESCipherEngine::decryptInit(Key *key, void *params) {
	SymBlockCipherEngine::decryptInit(key, params);
	storeExpandedKey(key);
}

void DESCipherEngine::decryptBlock(CssmData &cipherBuff, CssmData &clearBuff, uint32 *bytesDecrypted) {
	int		rc;
	uchar*	inOutBuff;
	uint32	cMode;

	if (cipherBuff.getLength() == 0) {
		*bytesDecrypted = 0;
		clearBuff.setLength(0);
		return;
	}

	inOutBuff = new uchar[cipherBuff.getLength()];

	memcpy(inOutBuff, cipherBuff.getData(), cipherBuff.getLength());

	cMode = getCylinkMode(mode);

	rc = DESDecrypt(IV.getData(), (uchar*)expandedKey, cMode, inOutBuff, cipherBuff.getLength());
	if (rc != SUCCESS) {
		*bytesDecrypted = 0;
		clearBuff.setLength(0);
		delete inOutBuff;
		throw CryptoLibException(CSSM_CSP_ENC_FAILED);
	}
	else {
		*bytesDecrypted = cipherBuff.getLength();
		clearBuff.setData(cipherBuff.getLength(), inOutBuff);
	}

	delete inOutBuff;
}

void DESCipherEngine::cleanUp() {
	SymBlockCipherEngine::cleanUp();
	memset(expandedKey, '\0', EXPANDED_KEY_LENGTH);
}
