/*
Copyright (C) 1992,1993,1994 Trusted Information Systems, Inc.

Export of this software from the United States of America or
Canada requires a specific license from the United States
Government.  This version of this software is not suitable for
export.

WITHIN THAT CONSTRAINT, the full text of the license agreement
that specifies the conditions under which this software may be
used is published in the file license.txt in the same directory
as that containing the TIS/PEM source.

Trusted Information Systems makes no representation about the
suitability of this software for any purpose.  It is provided
"as is" without express or implied warranty.
*/

/*
 * get_key - return an indication of whether or not a user's private key
 * is accessible, and the key if requested and possible
 *
 * if invoked as get_key(user, NULLBB, NULLBB), it functions like a
 * login check.  Note that it will return OK if could have given you the
 * private key, i.e., whether or not you included space for it.
 *
 * PARAMETERS
 *
 *   user	- (input)  a user record
 *   pw		- (input)  password/authentication data according to access
 *   key	- (output) ASN.1 encoded private key
 *
 * RETURN VALUES
 *
 *   OK		- private key is accessible
 *   NOTOK	- private key is inaccessible or does not exist
 *   EINVAL	- NULL or incorrect pw was entered
 *
 * At all times, if key is non-NULL it will be set correctly.
 * 
 * If pw is NULL and required, get_key will check the user's login
 * record for the information.  If found, it will be used, otherwise
 * EINVAL will be returned.
 */


#include "config.h"
#include "general.h"

#include "bbuf.h"
#include "crypto.h"
#include "key.h"
#include "new.h"
#include "util.h"

static int do_file();

get_key(user, pw, key)
char        **user;		/* user record */
struct bbuf  *pw;		/* password data according to access */
struct bbuf **key;		/* ASN.1 encoded private key */
{
    char **av_key, **av_access, **av_pkey;

    if (key != (struct bbuf **)0)
	FREE_BBUF(*key);

    if (user == NULLVP)
	return NOTOK;

    if ((av_access = tag_user(user, "private-key-access")) == NULLVP)
	return NOTOK;

    if ((av_pkey = tag_user(user, "public-key")) == NULLVP)
	return NOTOK;

    av_key = tag_user(user, "private-key");

    if (!strcasecmp(*(av_access+1), "file")) {
	if (av_key == NULLVP || *(av_key+1) == NULLCP || **(av_key+1) == NULLC)
	    return NOTOK;
	return do_file(*(av_pkey+1), *(av_key+1), pw, key);
    }
    else
	return NOTOK;
}


/* split a bbuf into two bbufs that was previously joined by bbufjoin */

static bbufsplit(bb, bb1, bb2)
struct bbuf *bb, **bb1, **bb2;
{
    short length;

    if (bb == NULLBB || bb1 == (struct bbuf **)0 || bb2 == (struct bbuf **)0)
	return NOTOK;

    FREE_BBUF(*bb1);
    FREE_BBUF(*bb2);

    *bb1 = alloc_bbuf();
    *bb2 = alloc_bbuf();

    BCOPY(bb->data, (char *)&length, sizeof(length));
    BSWAP(length);
    (*bb1)->length = length;
    (*bb1)->data = alloc_uchar(length);
    BCOPY(bb->data + sizeof(length), (*bb1)->data, length);

    BCOPY(bb->data + sizeof(length) + (*bb1)->length,
	  (char *)&length, sizeof(length));
    BSWAP(length);
    (*bb2)->length = length;
    (*bb2)->data = alloc_uchar(length);
    BCOPY(bb->data + 2 * sizeof(length) + (*bb1)->length,
	  (*bb2)->data, length);

    return OK;
}


/* private key file is formatted as follows
 *
 * CLEARTEXT
 *
 * 	1 byte status - always 0x00
 * 	bbuf (len, data) which is md5 hash of key
 * 	bbuf (len, data) which is ASN.1 encoded key
 *
 * ENCRYPTED
 *
 * 	1 bytes status - always 0x01
 * 	bbuf (len, data) which is encrypted join of hash bbuf and key bbuf
 */

static do_file(pkey, file, pw, key)
char         *pkey;
char         *file;
struct bbuf  *pw;
struct bbuf **key;
{
    int            fd = -1;
    unsigned char  status;
    struct bbuf   *hash = NULLBB,
                  *thash = NULLBB,
                  *tkey = NULLBB;
    int            ret = NOTOK;

    if (filchk(file, -1, 0) != OK)
	return NOTOK;

    if ((fd = open(file, O_RDONLY)) < 0)
	return NOTOK;

    if (read(fd, (char *)&status, 1) != 1)
	goto cleanup;

    if (status & 0x01) {
	struct bbuf  *ekey = NULLBB,
                     *join = NULLBB,
                     *ejoin = NULLBB;
	struct key   *dkey = NULL_KEY;

	if (pw == NULLBB || pw -> length <= 0) { /* must check login */
	    char        *pwfile = NULLCP;
	    struct bbuf *t_pw = NULLBB,
                        *t_pkey = NULLBB,
                        *t_disp = NULLBB;
	    int          rfd = -1;

	    pwfile = pw_file();

	    if (filchk(pwfile, -1, 0) != OK)
		goto lose;
	    if ((rfd = open(pwfile, O_RDONLY)) < 0)
		goto lose;

	    while (read_pwrec(rfd, &t_pw, &t_pkey, &t_disp) == OK)
		if (!strcmp(pkey, (char *)(t_pkey->data)))
		    break;

	    if (t_pw == NULLBB) {
	    lose:
		FREE(pwfile);
		if (rfd != -1)
		    (void) close(rfd);
		FREE_BBUF(t_pw);
		FREE_BBUF(t_pkey);

		ret = EINVAL;
		goto cleanup;
	    }

	    pw = t_pw;
	    FREE(pwfile);
	    (void) close(rfd);
	    FREE_BBUF(t_pkey);
	}
	if (read_bbuf(fd, &ejoin) != OK)
	    goto cleanup;
	if (str2deskey((char *)pw->data, &dkey) != OK)
	    goto cleanup;
	if (encode_key(dkey, &ekey) != OK)
	    goto cleanup;
	if (set_key(ekey) != OK)
	    goto cleanup;
	if (decipher(ejoin, &join) != OK) {
	    ret = EINVAL;	/* must have given bad password */
	    goto cleanup;
	}
	if (bbufsplit(join, &hash, &tkey) != OK)
	    goto cleanup;

	FREE_BBUF(ejoin);
	FREE_BBUF(ekey);
	FREE_BBUF(join);

	FREE_KEY(dkey);
    }
    else {
	if (read_bbuf(fd, &hash) != OK)
	    goto cleanup;
	if (read_bbuf(fd, &tkey) != OK)
	    goto cleanup;
    }

    /* check hash of private key */

    if (gen_md5(NULL_KEY, tkey, &thash) != OK)
	goto cleanup;
    if (bbufcmp(hash, thash)) {
	if (status & 0x01)	/* must have given bad password */
	    ret = EINVAL;
	goto cleanup;
    }

    /* return the key if requested */

    if (key != (struct bbuf **)0) {
	*key = tkey;
	tkey = NULLBB;
    }

    ret = OK;

 cleanup:

    if (fd != -1)
	(void) close(fd);

    FREE_BBUF(hash);
    FREE_BBUF(thash);
    FREE_BBUF(tkey);

    return ret;
}
