//
// LiDIA - a library for computational number theory
//   Copyright (c) 1994, 1995 by the LiDIA Group
//
// File        : dlp_appl.c
// Author      : Damian Weber (DW)
// Last change : TP, Aug 8 1995, initial version
//

#include <LiDIA/dlp.h>
#include <LiDIA/timer.h>
#include <stdlib.h>
#include <strstream.h>

#define PROGNAME "dlp_appl"

void help()
{
   cerr << "Solve the Discrete Logarithm Problem a^x=b mod p, p prime" << "\n";
   cerr << "\n" << "usage: dlp_appl a b p" << "\n" << "\n";
   exit(0);
}


char *ord_suffix(const bigint &k)
     {
     static char suffix[3];

     bigint ord = k % 10;
     int i;
     ord.intify(i);

     switch(i % 10)
           {
           case 1 : if (k % 100 != bigint(11)) strcpy(suffix,"st");
                           else        strcpy(suffix,"th");
                       break;
           case 2 : if (k % 100 != bigint(12)) strcpy(suffix,"nd");
                           else        strcpy(suffix,"th");
                       break;
           case 3 : if (k % 100 != bigint(13)) strcpy(suffix,"rd");
                           else        strcpy(suffix,"th");
                       break;
           default: if (k.is_gt_zero()) strcpy(suffix,"th");
                       else  suffix[0]=0;
                    break;
           }
      return suffix;
      }

int  flag_set(int argc, char **argv, char *s) 
     {
     int j;

     for (j=1;j<=argc-1;j++)
	 if (!strcmp(argv[j],s))
	    return j;
     return 0;
     }

int  main(int argc, char **argv)
{
  bigint a,b,p,q,tq;
  bigint g;
  bigint up,ua,xa,da,xc,x,mc;
  bigint y1,y2;
  long td;
  int j,k,np;            
  int no_gen;
  timer tr,t1,t2;
  
  char buffer[1024];
  ostrstream os(buffer,1024);
  rational_factorization f;
  
  if (flag_set(argc,argv,"-h") || argc!=4)
    help();
  
  if (!string_to_bigint(argv[1],a)) help();
  if (!string_to_bigint(argv[2],b)) help();
  if (!string_to_bigint(argv[3],p)) help();
  
  tr.start_timer();
  
  if (!is_prime(p, 6)) 
    {
      cerr << "dlp_appl: " << p << " isn't a prime number." << "\n";
      exit(1);
    }
  
  cout << "The DLP is " << a << "^x=" << b << " mod " << p << "\n";
  cout << "factoring p-1 ..." << "\n" ;
  
  f.assign(p-(bigint)1);
  f.factor();
  
  cout << p-(bigint)1 << " = " ;
  
  np=f.no_of_comp();
  for (j=0;j<=np-1;j++)
    {
      if (j) cout << " * ";
      cout << f.base(j);
      if (f.exponent(j)>1) 
	cout << "^" << f.exponent(j);
    }
  
  cout << "\n" << "\n";
  
  no_gen=0;
  
  for (j=0;j<=np-1;j++)
    {
      for (k=1;k<=f.exponent(j);k++)
	{
	  power(q,f.base(j),k);
	  power_mod(y1,a,(p-(bigint)1)/q,p);
	  if (y1.is_one())
	    {
	      no_gen++;
	      power_mod(y2,b,(p-(bigint)1)/q,p);
	      if (!y2.is_one())
		{
		  cout << "The subgroup of " 
		    << q << ord_suffix(q) << " powers" << "\n";
		  cout << "mod " << p << "\n" 
		    << "contains " << a
		      << " but not " << b << "." << "\n";
		  cout << "So there doesn't exist a solution." 
		    << "\n";
		  exit(0);
		}
	    }
	}
    }
  
  mc=p-(bigint) 1;
  
  if (no_gen) 
    {
      g=search_generator(p,(bigint)2,f);
      cout << "using generator " << g << "\n";
    }
  else   g=a;
  
  if (g!=a)
    cout << "solving " << g << "^y = " << b << "\n";
  
  t1.start_timer();
  xc=discrete_log(g,b,p,f);
  t1.stop_timer();
  if (g!=a)
    {
      cout << "\n" << "y=" << xc << " mod " << mc;
      td=t1.real_time()/100;
      cout << "  (" << td/60 << ":" << td%60 << " min)" << ")" << "\n";
      cout << "\n";
    }
  
  if (g==a)
    {
      xa=1;
    }
  else
    {
      cout << "solving " << g << "^y = " << a << "\n";
      t2.start_timer();
      xa=discrete_log(g,a,p,f);
      t2.stop_timer();
      cout << "\n" << "y=" << xa << " mod " << mc;
      td=t2.real_time()/100;
      cout << "  (" << td/60 << ":" << td%60 << " min)" << ")" << "\n";
    }
  
  if (!xa.is_one())
    {
      da=gcd(xa,p-(bigint) 1);
      xgcd(ua,up,xa/da,(p-(bigint) 1)/da);
      
      x=(ua*xc/da) % ((p-(bigint) 1)/da);
      if (x.is_lt_zero()) x=x+p-(bigint)1;
    }
  else x=xc;
  
  cout << "\n";          
  cout << "Solution:" << "\n";          
  cout << "x = " << x << " mod " << mc << "\n";
  cout << "-------------------------------------" << "\n";
  
  tr.stop_timer();
  
  td=tr.real_time()/100;
  cout << "      CPU time : " << td/60 << " min " << td%60 << " sec.\n";
  cout << "=====================================" << "\n";
}



