

//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : math_vector.h 
// Author      : Frank Lehmann (FL), Markus Maurer (MM),
//               Thomas Papanikolaou (TP)
// Last change : FL/MM, Feb 15 1995, initial version
//               FL/MM, May 10 1995, clean-up
//		 FL/MM, May 15 1995, changed type of size variables to base_vector<T>::size_type
//		 FL/MM, Jul  6 1995, replaced base_vector<T>::size_type by lidia_size_t
//               FL/MM, Jul 24 1995, added 'const' to T* parameters of constructors


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// *
// *    File        :  math_vector.h
// *
// *    Description :  -) definition of template class 'math_vector<T>'
// *                      to allow use of vectors with arithmetical
// *                      functions
// *
// *                   -) virtual base class is 'base_vector<T>'
// *
// *		       -) to use this class the type T must support the
// *			  operators + , - , * , /, - (unary), != 
// *
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *




#ifndef LIDIA_MATH_VECTOR_H
#define LIDIA_MATH_VECTOR_H

#include <LiDIA/lidia.h>
#include <LiDIA/base_vector.h>


template <class T> class math_vector :  virtual public base_vector<T>

 {

   /* * * * * * * * * * * * * class description * * * * * * * * * * *

   public :

   // *****  constructor functions  *****

   math_vector () ;
   math_vector ( char md ) ;

   math_vector ( lidia_size_t all ) ;
   math_vector ( lidia_size_t all,  char md ) ;

   math_vector ( lidia_size_t all,  lidia_size_t len ) ;
   math_vector ( lidia_size_t all,  lidia_size_t len,  char md ) ;

   math_vector ( const base_vector < T > & v ) ;
   math_vector ( const base_vector < T > & v,  char md ) ;

   math_vector ( const T *v,  lidia_size_t len ) ;
   math_vector ( const T *v,  lidia_size_t len,  char md ) ;


   // ***** destructor *****

   LiDIA_VEC_VIRTUAL ~math_vector() ;


   // ***** assignment *****

   math_vector<T> & operator= ( const math_vector<T> & v ) ;


   // ***** operations on math_vectors *****

   friend math_vector <T>  operator + ( const math_vector <T> & v, const math_vector <T> & w ) ;
   friend math_vector <T>  operator - ( const math_vector <T> & v, const math_vector <T> & w ) ;
   friend math_vector <T>  operator * ( const math_vector <T> & v, const math_vector <T> & w ) ;
   friend math_vector <T>  operator / ( const math_vector <T> & v, const math_vector <T> & w ) ;
   friend math_vector <T>  operator - ( const math_vector <T> & v )  ;

   friend int add      ( math_vector <T> & r, const math_vector <T> & v, const math_vector <T> & w ) ;
   friend int subtract ( math_vector <T> & r, const math_vector <T> & v, const math_vector <T> & w ) ;
   friend int multiply ( math_vector <T> & r, const math_vector <T> & v, const math_vector <T> & w ) ;
   friend int divide   ( math_vector <T> & r, const math_vector <T> & v, const math_vector <T> & w ) ;
   friend int negate   ( math_vector <T> & r, const math_vector <T> & v ) ;


   // ***** operations on math_vector and type T element  *****

   friend math_vector<T> operator + ( const math_vector <T> & v, const T & t ) ; 
   friend math_vector<T> operator + ( const T & t , const math_vector <T> & v ) ;

   friend math_vector<T> operator - ( const math_vector <T> & v, const T & t ) ;
   friend math_vector<T> operator - ( const T & t , const math_vector <T> & v ) ;

   friend math_vector<T> operator * ( const math_vector <T> & v, const T & t ) ;
   friend math_vector<T> operator * ( const T & t , const math_vector <T> & v ) ;

   friend math_vector<T> operator / ( const math_vector <T> & v, const T & t ) ;
   friend math_vector<T> operator / ( const T & t , const math_vector <T> & v ) ;

   friend int add ( math_vector <T> & r, const math_vector <T> & v, const T & t ) ;
   friend int add ( math_vector <T> & r, const T & t , const math_vector <T> & v ) ;

   friend int subtract ( math_vector <T> & r, const math_vector <T> & v, const T & t ) ;
   friend int subtract ( math_vector <T> & r, const T & t , const math_vector <T> & v ) ;

   friend int multiply ( math_vector <T> & r, const math_vector <T> & v, const T & t ) ;
   friend int multiply ( math_vector <T> & r, const T & t , const math_vector <T> & v ) ;

   friend int divide   ( math_vector <T> & r, const math_vector <T> & v, const T & t ) ;
   friend int divide   ( math_vector <T> & r, const T & t , const math_vector <T> & v ) ;


   // ***** miscellaneous *****

   friend T inner_product  ( const math_vector <T> & v, const math_vector <T> & w ) ;
   T sum_of_squares ( ) ;


   // ***** comparisons *****

   bool operator == ( const math_vector <T> & v )  const ;
   bool operator != ( const math_vector <T> & v )  const ;

   */




   /************************************************************************************/
   /*                    declaration and definition of functions                       */
   /************************************************************************************/


   public :

   // *****  constructor functions  *****

   math_vector () : base_vector< T > ( FIXED )
    {
       debug_handler ( "math_vector", "constructor ()" ) ;
    }

   math_vector ( char md ) : base_vector< T > ( md )
    {
       debug_handler ( "math_vector", "constructor ( char )" ) ;
    }


   math_vector ( lidia_size_t all ) : base_vector< T > ( all, FIXED )
    {
       debug_handler ( "math_vector", "constructor ( lidia_size_t )" ) ;
    }

   math_vector ( lidia_size_t all, char md ) : base_vector< T > ( all, md )
    {
       debug_handler ( "math_vector", "constructor ( lidia_size_t , char )" ) ;
    }


   math_vector ( lidia_size_t all, lidia_size_t len ) : base_vector< T > ( all, len, FIXED )
    {
       debug_handler ( "math_vector", "constructor ( lidia_size_t, lidia_size_t )" ) ;
    }

   math_vector ( lidia_size_t all, lidia_size_t len, char md ) : base_vector< T > ( all, len, md )
    {
       debug_handler ( "math_vector", "constructor ( lidia_size_t, lidia_size_t, char )" ) ;
    }


   math_vector ( const base_vector < T > & v ) : base_vector< T > ( v, FIXED )
    {
       debug_handler ( "math_vector", "constructor ( base_vector < T > & )" ) ;
    }

   math_vector ( const base_vector < T > & v, char md ) : base_vector< T > ( v, md )
    {
       debug_handler ( "math_vector", "constructor ( base_vector < T > &, char  )" ) ;
    }


   math_vector ( const T *v, lidia_size_t len ) : base_vector< T > ( v, len, FIXED )
    {
       debug_handler ( "math_vector", "constructor ( const T *, lidia_size_t )" ) ;
    }

   math_vector ( const T *v, lidia_size_t len, char md ) : base_vector< T > ( v, len, md )
    {
       debug_handler ( "math_vector", "constructor ( const T *, lidia_size_t, char )" ) ;
    }


  // ***** destructor *****

  LiDIA_VEC_VIRTUAL ~math_vector()
  {
    debug_handler ( "math_vector", "destructor()" ) ;

    //  memory deallocation is done by
    //  base_vector - destructor
  }


  // ***** assignment *****

  math_vector<T> &  operator= ( const math_vector<T> & v )
  {
     debug_handler ( "math_vector", "operator=" ) ;

     if ( &v != this )

        base_vector<T>::operator= ( v ) ;

     return ( *this ) ;
   }



  // *****  addition of two math_vectors  *****

  friend math_vector<T> operator + ( const math_vector <T> & v, const math_vector <T> & w )
  {
    debug_handler ( "math_vector" , "operator+ (math_vector<T>&,math_vector<T>&)" ) ;

    lidia_size_t i ;

    math_vector <T> sum ( v.length ) ;

    if ( v.length == w.length )
     {
        for ( i = 0 ; i < v.length ; i++ )
         {
            sum.data[i] = v.data[i] + w.data[i] ;
         }

        sum.length = v.length ;
     }
    else
     {
        lidia_error_handler ( "math_vector", "operator+(math_vector<T> &)::different sizes" ) ;
     }

    return sum ;
  }



  // *****  subtraction of two math_vectors  *****

  friend math_vector<T> operator - ( const math_vector <T> & v, const math_vector <T> & w )
  {
    debug_handler ( "math_vector", "operator-(math_vector<T> &,math_vector<T>&)" ) ;

    lidia_size_t i ;

    math_vector <T> dif ( v.length ) ;

    if ( v.length == w.length )
     {
       for ( i = 0 ; i < v.length ; i++ )
        {
           dif.data[i] = v.data[i] - w.data[i] ;
        }

       dif.length = v.length ;
     }
    else
     {
        lidia_error_handler ( "math_vector", "operator-(math_vector<T> &)::different sizes" ) ;
     }

    return dif ;
  }



  // ***** multiplication of two math_vectors *****

  friend math_vector<T> operator * ( const math_vector <T> & v, const math_vector <T> & w )
  {
    debug_handler ( "math_vector", "operator*(math_vector<T> &,math_vector<T>&)" ) ;

    lidia_size_t i ;

    math_vector <T> prod ( v.length ) ;

    if ( v.length == w.length )
     {
       for ( i = 0 ; i < v.length ; i++ )
        {
           prod.data[i] = v.data[i] * w.data[i] ;
        }

       prod.length = v.length ;
     }
    else
     {
        lidia_error_handler ( "math_vector", "operator*(math_vector<T> &)::different sizes" ) ;
     }

    return prod;
  }



  // *****  division of two math_vectors  *****

  friend math_vector<T> operator / ( const math_vector <T> & v, const math_vector <T> & w )
  {
    debug_handler ( "math_vector", "operator/(math_vector<T> &,math_vector<T>&)" ) ;

    lidia_size_t i ;

    math_vector <T> quot ( v.length ) ;

    if ( v.length == w.length )
     {
        for ( i = 0 ; i < v.length ; i++ )
         {
            quot.data[i] = v.data[i] / w.data[i] ;
         }

        quot.length = v.length ;
     }
    else
     {
        lidia_error_handler ( "math_vector", "operator/(math_vector<T> &)::different sizes" ) ;
     }

    return quot;
  }



  // *****  negation of a math_vector (unary -)  *****

  friend math_vector<T> operator - ( const math_vector <T> & v )
  {
    debug_handler ( "math_vector", "operator-(unary)" ) ;

    lidia_size_t i ;

    math_vector <T> neg ( v.length ) ;

    for ( i = 0 ; i < v.length ; i++ )
     {
        neg.data[i] = -v.data[i] ;
     }

    neg.length = v.length ;

    return neg ;
  }




  // ***** ***** procedural functions ***** *****



  // **** addition of two math_vectors

  friend int add ( math_vector <T> & r, const math_vector <T> & v, const math_vector <T> & w )
  {
    debug_handler ( "math_vector" , "add (math_vector <T>&, math_vector <T>&, math_vector<T>&)" ) ;

    lidia_size_t i ;
    lidia_size_t l = w.length ;

    if ( l == v.length )
     {
        r.set_capacity ( l ) ;

        for ( i = 0 ; i < l ; i++ )
         {
            r.data[i] = v.data[i] + w.data[i] ;
         }

        r.length = l ;
     }
    else
     {
        lidia_error_handler ( "math_vector", "add(math_vector <T>&, math_vector <T>&, math_vector<T>&)::different sizes" ) ;
     }

    return (0) ;
  }



  // *****  subtraction of two math_vectors  *****

  friend int subtract ( math_vector <T> & r, const math_vector <T> & v, const math_vector <T> & w )
  {
    debug_handler ( "math_vector", "subtract(math_vector <T>&, math_vector <T>&, math_vector<T>&)" ) ;

    lidia_size_t i ;
    lidia_size_t l = w.length ;

    if ( l == v.length )
     {
       r.set_capacity ( l ) ;

       for ( i = 0 ; i < l ; i++ )
        {
           r.data[i] = v.data[i] - w.data[i] ;
        }

       r.length = l ;
     }
    else
     {
        lidia_error_handler ( "math_vector", "subtract(math_vector <T>&, math_vector <T>&, math_vector<T>&)::different sizes" ) ;
     }

    return (0) ;
  }



  // ***** multiplication of two math_vectors *****

  friend int multiply ( math_vector <T> & r, const math_vector <T> & v, const math_vector <T> & w )
  {
    debug_handler ( "math_vector", "multiply(math_vector <T>&, math_vector <T>&, math_vector<T>&)" ) ;

    lidia_size_t i ;
    lidia_size_t l = w.length ;

    if ( l == v.length )
     {
       r.set_capacity ( l ) ;

       for ( i = 0 ; i < l ; i++ )
        {
           r.data[i] = v.data[i] * w.data[i] ;
        }

       r.length = l ;
     }
    else
     {
        lidia_error_handler ( "math_vector", "multiply(math_vector <T>&, math_vector <T>&, math_vector<T>&)::different sizes" ) ;
     }

    return (0) ;
  }



  // *****  division of two math_vectors  *****

  friend int divide ( math_vector <T> & r, const math_vector <T> & v, const math_vector <T> & w )
  {
    debug_handler ( "math_vector", "divide(math_vector <T>&, math_vector <T>&, math_vector<T>&)" ) ;

    lidia_size_t i ;
    lidia_size_t l = w.length ;

    if ( l == v.length )
     {
        r.set_capacity ( l ) ;

        for ( i = 0 ; i < l ; i++ )
         {
            r.data[i] = v.data[i] / w.data[i] ;
         }

        r.length = l ;
     }
    else
     {
        lidia_error_handler ( "math_vector", "divide(math_vector <T>&, math_vector <T>&, math_vector<T>&)::different sizes" ) ;
     }

    return (0) ;
  }



  // *****  negation of a math_vector  *****

  friend int negate ( math_vector<T> & r, const math_vector <T> & v )
  {
    debug_handler ( "math_vector", "negate(math_vector<T>&, math_vector<T>&)" ) ;

    lidia_size_t i ;
    lidia_size_t l = v.length ;

    r.set_capacity ( l ) ;

    for ( i = 0 ; i < l ; i++ )
     {
        r.data[i] = -v.data[i] ;
     }

    r.length = l ;

    return (0) ;
  }





  // *****  addition of math_vector and type T element  *****

  friend math_vector<T> operator + ( const math_vector <T> & v, const T & t )
  {
    debug_handler ( "math_vector", "operator+T" ) ;

    lidia_size_t i ;

    math_vector <T> sum ( v.length ) ;

    for (i = 0 ; i < v.length ; i++ )
     {
       sum.data[i] = v.data[i] + t;
     }

    sum.length = v.length ;

    return sum ;
  }

  friend math_vector<T> operator + ( const T & t , const math_vector <T> & v )
  {
    debug_handler ( "math_vector", "operator T+" ) ;

    lidia_size_t i ;

    math_vector <T> sum ( v.length ) ;

    for (i = 0 ; i < v.length ; i++ )
     {
       sum.data[i] = t + v.data[i] ;
     }

    sum.length = v.length ;

    return sum ;
  }



  // *****  subtraction of math_vector and type T element  *****

  friend math_vector<T> operator - ( const math_vector <T> & v, const T & t )
  {
    debug_handler ( "math_vector", "operator-T" ) ;

    lidia_size_t i ;

    math_vector <T> dif ( v.length ) ;

    for ( i = 0 ; i < v.length ; i++ )
     {
        dif.data[i] = v.data[i] - t ;
     }

    dif.length = v.length ;

    return dif ;
  }

  friend math_vector<T> operator - ( const T & t , const math_vector <T> & v )
  {
    debug_handler ( "math_vector", "operator T-" ) ;

    lidia_size_t i ;

    math_vector <T> dif ( v.length ) ;

    for ( i = 0 ; i < v.length ; i++ )
     {
        dif.data[i] = t - v.data[i] ;
     }

    dif.length = v.length ;

    return dif ;
  }



  // ***** multiplication of math_vector and type T element *****

  friend math_vector<T> operator * ( const math_vector <T> & v, const T & t )
  {
    debug_handler ("math_vector", " operator*T" ) ;

    lidia_size_t i ;

    math_vector <T> prod ( v.length ) ;

    for ( i = 0 ; i < v.length ; i++ )
     {
        prod.data[i] = v.data[i] * t;
     }

    prod.length = v.length ;

    return prod ;
  }

  friend math_vector<T> operator * ( const T & t , const math_vector <T> & v )
  {
    debug_handler ("math_vector", " operator T*" ) ;

    lidia_size_t i ;

    math_vector <T> prod ( v.length ) ;

    for ( i = 0 ; i < v.length ; i++ )
     {
        prod.data[i] = t * v.data[i] ;
     }

    prod.length = v.length ;

    return prod ;
  }


  // ***** division of math_vector and type T element *****

  friend math_vector<T> operator / ( const math_vector <T> & v, const T & t )
  {
    debug_handler ( "math_vector", "operator/T" ) ;

    lidia_size_t i ;

    math_vector <T> quot ( v.length ) ;

    for ( i = 0 ; i < v.length ; i++ )
     {
        quot.data[i] = v.data[i] / t ;
     }

    quot.length = v.length ;

    return quot ;
  }

  friend math_vector<T> operator / ( const T & t , const math_vector <T> & v )
  {
    debug_handler ( "math_vector", "operator T/" ) ;

    lidia_size_t i ;

    math_vector <T> quot ( v.length ) ;

    for ( i = 0 ; i < v.length ; i++ )
     {
        quot.data[i] = t / v.data[i] ;
     }

    quot.length = v.length ;

    return quot ;
  }


  // ***** ***** procedural functions ***** *****


  // *****  addition of math_vector and type T element  *****

  friend int add ( math_vector <T> & r, const math_vector <T> & v, const T & t )
  {
    debug_handler ( "math_vector", "add+T" ) ;

    lidia_size_t i ;
    lidia_size_t l = v.length ;

    r.set_capacity ( l ) ;

    for (i = 0 ; i < l ; i++ )
     {
       r.data[i] = v.data[i] + t;
     }

    r.length = l ;

    return (0) ;
  }

  friend int add ( math_vector <T> & r, const T & t , const math_vector <T> & v )
  {
    debug_handler ( "math_vector", "add T+" ) ;

    lidia_size_t i ;
    lidia_size_t l = v.length ;

    r.set_capacity ( l ) ;

    for (i = 0 ; i < l ; i++ )
     {
       r.data[i] = t + v.data[i] ;
     }

    r.length = l ;

    return (0) ;
  }


  // *****  subtraction of math_vector and type T element  *****

  friend int subtract ( math_vector <T> & r, const math_vector <T> & v, const T & t )
  {
    debug_handler ( "math_vector", "subtract -T" ) ;

    lidia_size_t i ;
    lidia_size_t l = v.length ;

    r.set_capacity ( l ) ;

    for ( i = 0 ; i < l ; i++ )
     {
        r.data[i] = v.data[i] - t ;
     }

    r.length = l ;

    return (0) ;
  }

  friend int subtract ( math_vector <T> & r, const T & t , const math_vector <T> & v )
  {
    debug_handler ( "math_vector", "subtract T-" ) ;

    lidia_size_t i ;
    lidia_size_t l = v.length ;

    r.set_capacity ( l ) ;

    for (i = 0 ; i < l ; i++ )
     {
       r.data[i] = t - v.data[i] ;
     }

    r.length = l ;

    return (0) ;
  }



  // ***** multiplication of math_vector and type T element *****

  friend int multiply ( math_vector <T> & r, const math_vector <T> & v, const T & t )
  {
    debug_handler ("math_vector", "multiply*T" ) ;

    lidia_size_t i ;
    lidia_size_t l = v.length ;

    r.set_capacity ( l ) ;

    for ( i = 0 ; i < l ; i++ )
     {
        r.data[i] = v.data[i] * t;
     }

    r.length = l ;

    return (0) ;
  }

  friend int multiply ( math_vector <T> & r, const T & t , const math_vector <T> & v )
  {
    debug_handler ("math_vector", "multiply T*" ) ;

    lidia_size_t i ;
    lidia_size_t l = v.length ;

    r.set_capacity ( l ) ;

    for ( i = 0 ; i < l ; i++ )
     {
        r.data[i] = t * v.data[i] ;
     }

    r.length = l ;

    return (0) ;
  }


  // ***** division of math_vector and type T element *****

  friend int divide ( math_vector <T> & r, const math_vector <T> & v, const T & t )
  {
    debug_handler ( "math_vector", "divide/T" ) ;

    lidia_size_t i ;
    lidia_size_t l = v.length ;

    r.set_capacity ( l ) ;

    for ( i = 0 ; i < l ; i++ )
     {
        r.data[i] = v.data[i] / t ;
     }

    r.length = l ;

    return (0) ;
  }

  friend int divide ( math_vector <T> & r, const T & t , const math_vector <T> & v )
  {
    debug_handler ( "math_vector", "divide T/" ) ;

    lidia_size_t i ;
    lidia_size_t l = v.length ;

    r.set_capacity ( l ) ;

    for ( i = 0 ; i < l ; i++ )
     {
        r.data[i] = t / v.data[i] ;
     }

    r.length = l ;

    return (0) ;
  }



  // *****  inner product of two math_vectors  *****

  friend T  inner_product ( const math_vector <T> & v, const math_vector <T> & w )
  {
    debug_handler ( "math_vector", "inner_product(math_vector&, math_vector&)") ;

    T         res ;
    lidia_size_t i   ;

    if ( v.length == w.length )
     {
       if ( v.length > 0 )
        {
           res = v.data[0] * w.data[0] ;
        }

       for ( i = 1 ; i < v.length ; i++ )
        {
           res = res + v.data[i] * w.data[i] ;
        }
     }
    else
     {
        lidia_error_handler ( "math_vector", "inner_product(math_vector&,math_vector&)::different sizes" ) ;
     }

    return res ;
  }



  // *****  sum of squares of two math_vectors  *****

  T  sum_of_squares ( )
  {
    debug_handler ( "math_vector", "sum_of_squares()" ) ;

    T   res ;
    lidia_size_t i   ;

    if ( length > 0 )
     {
       res = data[0] * data[0];
     }
    for ( i = 1 ; i < length ; i++ )
     {
       res = res + data[i] * data[i] ;
     }

    return res ;
  }



  // *****  testing for equality  *****

  bool operator == ( const math_vector <T> & v ) const
  {
    debug_handler ( "math_vector", "operator==" ) ;

    lidia_size_t i ;

    if ( length == v.length )
     {
       for (i = 0; i < length; i++)
        {
          if ( data[i] != v.data[i] )

            return false ;
        }

       return true ;
     }
    else
     {
       return false ;
     }
  }



  // *****  testing for inequality  *****

  int operator != ( const math_vector <T> &v ) const
  {
    debug_handler ( "math_vector", "operator!=" ) ;

    lidia_size_t i ;

    if ( length == v.length )
     {
       for ( i = 0 ; i < length ; i++ )
        {
           if (data[i] != v.data[i])

             return true ;
        }

       return false ;
     }
    else
     {
       return true ;
     }
  }


 } ; // end of class math_vector<T>


#endif


