/* routines that interface with the kernel's IPsec mechanism
 * Copyright (C) 1997 Angelos D. Keromytis.
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: kernel.c,v 1.66 2000/01/07 05:07:36 dhr Exp $
 */

#include <stddef.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <wait.h>

#ifdef KLIPS
/* A little experimentation indicates that the order of the next six
 * includes is critical.  I don't know why, or what the rules are.  Ugh!
 */
# include <sys/types.h>
# include <linux/types.h>
# include <sys/socket.h>
# include <netinet/in.h>
# include <arpa/inet.h>
# include <netdb.h>

# include <unistd.h>
# include <fcntl.h>

# include <freeswan.h>

# include <linux/autoconf.h>    /* CONFIG_IPSEC_PFKEYv2 */

# ifdef CONFIG_IPSEC_PFKEYv2
#   include <signal.h>
#   include <pfkeyv2.h>
#   include <pfkey.h>
u_int32_t pfkey_seq = 0;
# endif /* CONFIG_IPSEC_PFKEYv2 */

# include <radij.h>
# include <ipsec_encap.h>
# include <ipsec_netlink.h>
# include <ipsec_xform.h>
# include <ipsec_ipe4.h>
# include <ipsec_ah.h>
# include <ipsec_esp.h>

#else /* !KLIPS */

/* this collection is included in the other case, but in an order
 * that makes it hard to make the #includes common.
 */
# include <sys/socket.h>
# include <netinet/in.h>
# include <arpa/inet.h>

# include <freeswan.h>

#endif /* !KLIPS */

#include "constants.h"
#include "defs.h"
#include "rnd.h"
#include "id.h"
#include "connections.h"	/* needs id.h */
#include "state.h"
#include "kernel.h"
#include "log.h"
#include "server.h"


bool no_klips = FALSE;	/* don't actually use KLIPS */

#ifdef CONFIG_IPSEC_PFKEYv2
void
pfkey_handle(void)
{
    char buffer[4096];
    struct sadb_ext *extensions[SADB_EXT_MAX + 1];
    ssize_t len = read(pfkeyfd, buffer, sizeof(buffer));

    DBG(DBG_CONTROL,
	DBG_log(BLANK_FORMAT);
	DBG_log("pfkey_handle: read %d octets of pfkey message", len));

    if (len == -1)
    {
	log_errno((e, "read() failed in pfkey_handle()"));
#if 0	/* this may be a dumb idea */
	close(pfkeyfd);
	pfkeyfd = NULL_FD;
#endif
	return;
    }

    /* Parse with default extension parsers */
    if (pfkey_msg_parse((struct sadb_msg *)buffer, NULL, extensions, EXT_BITS_OUT))
	log("pfkey_handle: insane test message received from below, \"don't panic\".");
    else
	log("pfkey_handle: sane message parsed but not processed (yet) from below.");

    /* Need to add pluto's own extension
     * processors and message processing here
     */
}
#endif /* CONFIG_IPSEC_PFKEYv2 */

/* Generate Unique SPI numbers.
 *
 * The specs say that the number must not be less than 0x100.
 * XXX This should be replaced by a call to the kernel when
 * XXX we get an API.
 * The returned SPI is in network byte order.
 * We use a random number as the initial SPI so that there is
 * a good chance that different Pluto instances will choose
 * different SPIs.  This is good for two reasons.
 * - the keying material for the initiator and responder only
 *   differs if the SPIs differ.
 * - if Pluto is restarted, it would otherwise recycle the SPI
 *   numbers and confuse everything.  When the kernel generates
 *   SPIs, this will no longer matter.
 */
ipsec_spi_t
get_ipsec_spi(void)
{
    static ipsec_spi_t spi = 0;	/* host order, so not returned directly! */

    spi++;
    while (spi < 0x100)
	get_rnd_bytes((u_char *)&spi, sizeof(spi));

    DBG(DBG_CONTROL,
	{
	    ipsec_spi_t spi_net = htonl(spi);

	    DBG_dump("generate SPI:", (u_char *)&spi_net, sizeof(spi_net));
	});

    return htonl(spi);
}

/* invoke the updown script to do the routing and firewall commands required
 *
 * The user-specified updown script is run.  Parameters are fed to it in
 * the form of environment variables.  All such environment variables
 * have names starting with "PLUTO_".
 *
 * The operation to be performed is specified by PLUTO_VERB.  This
 * verb has a suffix "-host" if the client on this end is just the
 * host; otherwise the suffix is "-client".
 *
 * "prepare-host" and "prepare_client" are used to delete a route
 * that may exist (due to forces outside of Pluto).  It is used to
 * prepare for pluto creating a route.
 *
 * "route-host" and "route-client" are used to install a route.
 * Since routing is based only on destination, the PLUTO_MY_CLIENT_*
 * values are probably of no use (using them may signify a bug).
 *
 * "unroute-host" and "unroute-client" are used to delete a route.
 * Since routing is based only on destination, the PLUTO_MY_CLIENT_*
 * values are probably of no use (using them may signify a bug).
 *
 * "up-host" and "up-client" are run when an eroute is added (not replaced).
 * They are useful for adjusting a firewall: usually for adding a rule
 * to let processed packets flow between clients.  Note that only
 * one eroute may exist for a pair of client subnets but inbound
 * IPsec SAs may persist without an eroute.
 *
 * "down-host" and "down-client" are run when an eroute is deleted.
 * They are useful for adjusting a firewall.
 */

#ifndef DEFAULT_UPDOWN
# define DEFAULT_UPDOWN	"ipsec _updown"
#endif

static bool
do_command(struct connection *c, const char *verb)
{
    char cmd[1024];

    const char
	*verb_suffix = c->this.client_net.s_addr == c->this.host_addr.s_addr
	     && c->this.client_net.s_addr == mask32.sin_addr.s_addr
	    ? "-host" : "-client";
    FILE *f;

    /* form the command string */
    {
	char
	    me_str[SOCKADDR_STRING_SIZE],
	    myclientnet_str[SOCKADDR_STRING_SIZE],
	    myclientmask_str[SOCKADDR_STRING_SIZE],
	    peer_str[SOCKADDR_STRING_SIZE],
	    nexthop_str[SOCKADDR_STRING_SIZE],
	    peerclientnet_str[SOCKADDR_STRING_SIZE],
	    peerclientmask_str[SOCKADDR_STRING_SIZE];

	if (-1 == snprintf(cmd, sizeof(cmd),
	    "2>&1 "	/* capture stderr along with stdout */
	    "PLUTO_VERSION='1.0' "	/* change VERSION when interface space changes */
	    "PLUTO_VERB='%s%s' "
	    "PLUTO_CONNECTION='%s' "
	    "PLUTO_NEXT_HOP='%s' "
	    "PLUTO_INTERFACE='%s' "
	    "PLUTO_ME='%s' "
	    "PLUTO_MY_CLIENT_NET='%s' "
	    "PLUTO_MY_CLIENT_MASK='%s' "
	    "PLUTO_PEER='%s' "
	    "PLUTO_PEER_CLIENT_NET='%s' "
	    "PLUTO_PEER_CLIENT_MASK='%s' "
	    "%s"	/* actual script */
	    , verb, verb_suffix
	    , c->name
	    , strcpy(nexthop_str, inet_ntoa(c->this.host_nexthop))
	    , c->interface->vname
	    , strcpy(me_str, inet_ntoa(c->this.host_addr))
	    , strcpy(myclientnet_str, inet_ntoa(c->this.client_net))
	    , strcpy(myclientmask_str, inet_ntoa(c->this.client_mask))
	    , strcpy(peer_str, inet_ntoa(c->that.host_addr))
	    , strcpy(peerclientnet_str, inet_ntoa(c->that.client_net))
	    , strcpy(peerclientmask_str, inet_ntoa(c->that.client_mask))
	    , c->this.updown == NULL? DEFAULT_UPDOWN : c->this.updown))
	{
	    log("%s%s command too long!", verb, verb_suffix);
	    return FALSE;
	}
    }

    DBG(DBG_CONTROL, DBG_log("executing %s%s: %s"
	, verb, verb_suffix, cmd));

#ifdef KLIPS
    if (!no_klips)
    {
	/* invoke the script, catching stderr and stdout */
	f = popen(cmd, "r");
	if (f == NULL)
	{
	    log("unable to popen %s%s command", verb, verb_suffix);
	    return FALSE;
	}

	/* log any output */
	for (;;)
	{
	    char resp[1024];

	    if (fgets(resp, sizeof(resp), f) == NULL)
	    {
		if (ferror(f))
		{
		    log_errno((e, "fgets failed on output of %s%s command"
			, verb, verb_suffix));
		    return FALSE;
		}
		else
		{
		    passert(feof(f));
		    break;
		}
	    }
	    else
	    {
		char *e = resp + strlen(resp);

		if (e > resp && resp[-1] == '\n')
		    *e-- = '\0';	/* trim trailing '\n' */
		log("%s%s output: %s", verb, verb_suffix, resp);
	    }
	}

	/* report on and react to return code */
	{
	    int r = pclose(f);

	    if (r == -1)
	    {
		log_errno((e, "pclose failed for %s%s command"
		    , verb, verb_suffix));
		return FALSE;
	    }
	    else if (WIFEXITED(r))
	    {
		if (WEXITSTATUS(r) != 0)
		{
		    log("%s%s command exited with status %d"
			, verb, verb_suffix, WEXITSTATUS(r));
		    return FALSE;
		}
	    }
	    else if (WIFSIGNALED(r))
	    {
		log("%s%s command exited with signal %d"
		    , verb, verb_suffix, WTERMSIG(r));
		return FALSE;
	    }
	    else
	    {
		log("%s%s command exited with unknown status %d"
		    , verb, verb_suffix, r);
		return FALSE;
	    }
	}
    }
#endif /* KLIPS */
    return TRUE;
}

bool
route_connection(struct connection *c)
{
    if (c->routed)
    {
	/* already done */
    }
    else if (c->this.host_port != IKE_UDP_PORT
    && inside_client(c->that.host_addr, c->that))
    {
	log("cannot install route: peer is within its client");
    }
    else
    {
	struct connection *d = route_owner(c, FALSE);

	passert(c->eroute_owner == SOS_NOBODY);
	if (d == NULL)
	{
	    /* nobody's got it: we can try to install our route */
	    (void) do_command(c, "prepare");	/* just in case; ignore failure */
	    if (do_command(c, "route"))
		c->routed = TRUE;
	}
	else if (d->this.host_nexthop.s_addr == c->this.host_nexthop.s_addr
	&& d->interface == c->interface)
	{
	    /* The other connection has the same nexthop and interface,
	     * so we're done.
	     */
	    c->routed = TRUE;
	}
	else if (d->eroute_owner == SOS_NOBODY)
	{
	    /* The other connection has the route, and it conflicts
	     * (the nexthop or interface differs), but that connection
	     * isn't using the route.  We'll steal it!  There might be
	     * other connections using the same route, but none of
	     * them is using it either (otherwise route_owner()
	     * would have returned one that did).
	     *
	     * A feature of LINUX allows us to install the new route
	     * before deleting the old if the nexthops differ.
	     * This reduces the "window of vulnerability" when packets
	     * might flow in the clear.
	     *
	     * c->routed must be set last so that route_owner()
	     * doesn't find it.
	     */

	    bool preadd = d->this.host_nexthop.s_addr != c->this.host_nexthop.s_addr;

	    if (!preadd || do_command(c, "route"))
	    {
		/* one unroute for all */
		if (do_command(d, "unroute"))
		{
		    do {
			passert(d->eroute_owner == SOS_NOBODY);
			d->routed = FALSE;
			d = route_owner(c, FALSE);
		    } while (d != NULL);

		    if (preadd || do_command(c, "route"))
			c->routed = TRUE;
		}
	    }
	}
	else
	{
	    log("cannot install route: connection \"%s\" already has it"
		, d->name);
	}
    }
    return c->routed;
}

void
unroute_connection(struct connection *c)
{
    /* only unroute if no other connection shares it */
    if (c->routed)
    {
	c->routed = FALSE;
	if (route_owner(c, FALSE) == NULL && !do_command(c, "unroute"))
	    c->routed = TRUE;	/* undo on failure */
    }
}

#ifdef KLIPS
/* Setup an IPsec route entry. Code taken from addrt.c.
 * op is the KLIPS operator: EMT_DELEROUTE, EMT_SETEROUTE, or EMT_RPLACEROUTE
 */
static bool
do_eroute(int fd, struct state *st, u_int8_t op, const char *opname)
{
    struct connection *c = st->st_connection;
    struct encap_msghdr emh;

    memset(&emh, '\0', sizeof(emh));	/* clear unused parts */

    emh.em_magic = EM_MAGIC;
    emh.em_msglen = sizeof(emh);
    emh.em_version = 0;
    emh.em_type = op;
    emh.em_eaddr.sen_len = emh.em_emask.sen_len
			 = sizeof(struct sockaddr_encap);
    emh.em_eaddr.sen_family = emh.em_emask.sen_family = 26;
    emh.em_eaddr.sen_type = SENT_IP4;
    emh.em_emask.sen_type = 255;

    emh.em_eaddr.sen_ip_src.s_addr = c->this.client_net.s_addr;
    emh.em_emask.sen_ip_src.s_addr = c->this.client_mask.s_addr;

    emh.em_eaddr.sen_ip_dst.s_addr = c->that.client_net.s_addr;
    emh.em_emask.sen_ip_dst.s_addr = c->that.client_mask.s_addr;

    /* Fill in peer's address (the other security gateway) and
     * figure out the SPI to reference.
     * these are not needed for deleting an eroute
     * At this time, we only need these for setting the eroute.
     */
    if (op != EMT_DELEROUTE)
    {
	emh.em_erdst.s_addr = c->that.host_addr.s_addr;

	/* multiple SPIs will be grouped; we must refer to the
	 * last in the sequence: ah esp encap.
	 */
	if (st->st_ah.present)
	{
	    emh.em_erspi = st->st_ah.attrs.spi;
	    emh.em_erproto = IPPROTO_AH;
	}

	if (st->st_esp.present)
	{
	    emh.em_erspi = st->st_esp.attrs.spi;
	    emh.em_erproto = IPPROTO_ESP;
	}

	if (st->st_ah.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL
	|| st->st_esp.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL)
	{
	    emh.em_erspi = st->st_tunnel_spi;
	    emh.em_erproto = IPPROTO_IPIP;
	}
    }

    DBG(DBG_CONTROL,
	{
	    char mybuf[SOCKADDR_STRING_SIZE*2];
	    char peerbuf[SOCKADDR_STRING_SIZE*2];

	    subnettoa(c->this.client_net, c->this.client_mask, 0, mybuf, sizeof(mybuf));
	    subnettoa(c->that.client_net, c->that.client_mask, 0, peerbuf, sizeof(peerbuf));
	    DBG_log("%s eroute %s to %s", opname, mybuf, peerbuf);
	});

    DBG_cond_dump(DBG_KLIPS, "eroute => KLIPS:\n", &emh, sizeof(emh));

    if (!no_klips)
    {
	if (write(fd, &emh, sizeof(emh)) != sizeof(emh))
	{
	    log_errno((e, "write to %s eroute", opname));
	    return FALSE;
	}
    }
    return TRUE;
}

#ifdef CONFIG_IPSEC_PFKEYv2
static void
set_text_said(char text_said[SATOA_BUF], ip_address dst, ipsec_spi_t spi, int proto)
{
    struct sa_id said;

    said.dst = dst;
    said.spi = spi;
    said.proto = proto;
    satoa(said, 0, text_said, SATOA_BUF);
}

static bool
pfkey_build(int error
, const char *description
, const char *text_said
, struct sadb_ext *extensions[SADB_EXT_MAX + 1])
{
    if (error == 0)
    {
	return TRUE;
    }
    else
    {
	log("building of %s %s failed, code %d"
	    , description, text_said, error);
	pfkey_extensions_free(extensions);
	return FALSE;
    }
}

static bool
finish_pfkey_msg(struct sadb_ext *extensions[SADB_EXT_MAX + 1]
, const char *description
, const char *text_said)
{
    struct sadb_msg *pfkey_msg;
    int error = pfkey_msg_build(&pfkey_msg, extensions, EXT_BITS_IN);

    if (error != 0)
    {
	log("building of pfkey_msg %s %s failed, code %d"
	    , description, text_said, error);
	pfkey_extensions_free(extensions);
	pfkey_msg_free(&pfkey_msg);
	return FALSE;
    }

    {
	size_t len = pfkey_msg->sadb_msg_len * IPSEC_PFKEYv2_ALIGN;
	ssize_t r;

	DBG_cond_dump(DBG_KLIPS, description, (void *) pfkey_msg, len);

	r = write(pfkeyfd, pfkey_msg, len);

	pfkey_extensions_free(extensions);
	pfkey_msg_free(&pfkey_msg);

	if (r != (ssize_t)len)
	{
	    log_errno((e, "pfkey write() of %s %s failed"
		, description, text_said));
	    return FALSE;
	}

	log("building of pfkey_msg %s %s succeeded", description, text_said);
    }
    return TRUE;
}
#endif

static bool
del_spi(int fd, ipsec_spi_t spi, int proto, struct in_addr dest)
{
#ifdef CONFIG_IPSEC_PFKEYv2
    struct sadb_ext *extensions[SADB_EXT_MAX + 1];
    struct sockaddr_in pfkey_address_s_ska;
    struct sockaddr_in pfkey_address_d_ska;
    char text_said[SATOA_BUF];

    pfkey_extensions_init(extensions);
    set_text_said(text_said, dest, spi, proto);

    mksin(pfkey_address_s_ska, INADDR_ANY, 0);
    mksin(pfkey_address_d_ska, dest.s_addr, 0);

    return pfkey_build(pfkey_msg_hdr_build(&extensions[0], SADB_DELETE
	, proto == SA_ESP ? SADB_SATYPE_ESP
	    : proto == SA_AH ? SADB_SATYPE_AH
	    : proto == SA_IPIP ? SADB_X_SATYPE_IPIP
	    : SADB_SATYPE_UNSPEC
	, 0, ++pfkey_seq, getpid())
	, "pfkey_msg_hdr delete SA", text_said, extensions)

    && pfkey_build(pfkey_sa_build(&extensions[SADB_EXT_SA]
	, SADB_EXT_SA
	, spi	/* in host order */
	, 0, SADB_SASTATE_MATURE, 0, 0, 0)
	, "pfkey_sa delete SA", text_said, extensions)

    && pfkey_build(pfkey_address_build(&extensions[SADB_EXT_ADDRESS_SRC]
	, SADB_EXT_ADDRESS_SRC, 0, 0, (struct sockaddr*)&pfkey_address_s_ska)
	, "pfkey_addr_s delete SA", text_said, extensions)

    && pfkey_build(pfkey_address_build(&extensions[SADB_EXT_ADDRESS_DST]
	, SADB_EXT_ADDRESS_DST, 0, 0, (struct sockaddr*)&pfkey_address_d_ska)
	, "pfkey_addr_d delete SA", text_said, extensions)

    && finish_pfkey_msg(extensions, "Delete SA", text_said);

#else /* !CONFIG_IPSEC_PFKEYv2 */
    struct encap_msghdr delmsg;

    memset(&delmsg, '\0', sizeof(delmsg));
    delmsg.em_magic = EM_MAGIC;
    delmsg.em_version = 0;
    delmsg.em_msglen = EMT_SETSPI_FLEN;
    delmsg.em_type = EMT_DELSPI;
    delmsg.em_alg = XF_DEL;
    delmsg.em_dst = dest;
    delmsg.em_proto = proto;
    delmsg.em_spi = spi;
    DBG_cond_dump(DBG_KLIPS, "delete SPI => KLIPS:\n",
	(void *) &delmsg, EMT_SETSPI_FLEN);
    if (!no_klips)
    {
	if (write(fd, (caddr_t)&delmsg, EMT_SETSPI_FLEN) != EMT_SETSPI_FLEN)
	{
	    char text_said[SATOA_BUF];

	    log_errno((e, "write() of delete SPI %s failed"
		, (satoa(delmsg.em_said, 0, text_said, sizeof(text_said)), text_said)));
	    return FALSE;
	}
    }
    return TRUE;
#endif /* !CONFIG_IPSEC_PFKEYv2 */
}

/* Setup a pair of SAs. Code taken from setsa.c and spigrp.c, in
 * ipsec-0.5.
 */
static bool
setup_half_ipsec_sa(int fd, struct state *st, bool inbound)
{
    /* Build an inbound or outbound SA */

    struct connection *c = st->st_connection;
    struct in_addr
	src = inbound? c->that.host_addr : c->this.host_addr,
	dst = inbound? c->this.host_addr : c->that.host_addr;

    /* SPIs, saved for spigrouping or undoing, if necessary */
    struct sa_id
	said[EM_MAXRELSPIS],
	*said_next = said;

#ifdef CONFIG_IPSEC_PFKEYv2
    struct sadb_ext *extensions[SADB_EXT_MAX + 1];
    struct sockaddr_in any_sin
	, src_sin
	, dst_sin;
    char text_said[SATOA_BUF];

    mksin(any_sin, INADDR_ANY, 0);
    mksin(src_sin, src.s_addr, 0);
    mksin(dst_sin, dst.s_addr, 0);
#else /* !CONFIG_IPSEC_PFKEYv2 */
    /* encap_msghdr is from "ipsec_netlink.h"
     * It preceded each netlink message.
     * We create a generic header to save work.
     * User must still fill in:
     *	.em_alg
     *	.em_msglen
     *	.em_type
     *	.em_spi
     *	.em_proto
     */
    struct encap_msghdr h;

    memset(&h, '\0', sizeof(h));
    h.em_magic = EM_MAGIC;
    h.em_version = 0;
    h.em_if = 0;	/* ??? what is this? */

    if (inbound)
	h.em_flags |= EMT_INBOUND;
    h.em_dst = dst;
#endif /* !CONFIG_IPSEC_PFKEYv2 */

    /* set up AH SA, if any */

    if (st->st_ah.present)
    {
	ipsec_spi_t ah_spi = inbound? st->st_ah.our_spi : st->st_ah.attrs.spi;
	u_char *ah_dst_keymat = inbound? st->st_ah.our_keymat : st->st_ah.peer_keymat;

#ifdef CONFIG_IPSEC_PFKEYv2
	unsigned char authalg
	    , encryptalg;
#else /* !CONFIG_IPSEC_PFKEYv2 */
	int alg;
	size_t ah_alen;
	/* ??? mysterious code stolen from setsa.c:f_hmac()
	 * ??? why is struct ahhmacmd5_edata OK for SHA too?
	 */

	/* ahhmacmd5_edata is from "ipsec_ah.h"
	 * ahhmacsha1_edata has identical fields except that the ame_key
	 * field has a different length.
	 */
	union {
	    struct encap_msghdr h;
	    struct {
		char dummy[offsetof(struct encap_msghdr, em_dat)];
		struct ahhmacmd5_edata d;	/* works for both */
	    } u;
	} ah;

	memset(&ah, '\0', sizeof(ah));

	ah.h = h;
	ah.h.em_type = EMT_SETSPI;

	ah.h.em_spi = ah_spi;
	ah.h.em_proto = IPPROTO_AH;
#endif /* !CONFIG_IPSEC_PFKEYv2 */

	switch (st->st_ah.attrs.auth)
	{
	case AUTH_ALGORITHM_HMAC_MD5:
#ifdef CONFIG_IPSEC_PFKEYv2
	    authalg = SADB_AALG_MD5HMAC;
#else /* !CONFIG_IPSEC_PFKEYv2 */
	    alg = XF_AHHMACMD5;
	    ah_alen = MD5_DIGEST_SIZE;
#endif /* !CONFIG_IPSEC_PFKEYv2 */
	    break;

	case AUTH_ALGORITHM_HMAC_SHA1:
#ifdef CONFIG_IPSEC_PFKEYv2
	    authalg = SADB_AALG_SHA1HMAC;
#else /* !CONFIG_IPSEC_PFKEYv2 */
	    alg = XF_AHHMACSHA1;
	    ah_alen = SHA1_DIGEST_SIZE;
#endif /* !CONFIG_IPSEC_PFKEYv2 */
	    break;

	case AUTH_ALGORITHM_KPDK:
	case AUTH_ALGORITHM_DES_MAC:
	default:
	    log("%s not implemented yet"
		, enum_show(&auth_alg_names, st->st_ah.attrs.auth));
	    goto fail;
	}

#ifdef CONFIG_IPSEC_PFKEYv2
	pfkey_extensions_init(extensions);
	set_text_said(text_said, dst, ah_spi, IPPROTO_AH);

	if (!(pfkey_build(pfkey_msg_hdr_build(&extensions[0], SADB_ADD
		, SADB_SATYPE_AH, 0, ++pfkey_seq, getpid())
	    , "pfkey_msg_hdr Add AH SA", text_said, extensions)

	&& pfkey_build(pfkey_sa_build(&extensions[SADB_EXT_SA]
		, SADB_EXT_SA
		, ah_spi	/* in network order */
		, 32, SADB_SASTATE_MATURE, authalg, encryptalg, 0)
	    , "pfkey_sa Add AH SA", text_said, extensions)

	&& pfkey_build(pfkey_address_build(&extensions[SADB_EXT_ADDRESS_SRC]
		, SADB_EXT_ADDRESS_SRC, 0, 0
		, (struct sockaddr *)&any_sin)
	    , "pfkey_addr_s Add AH SA", text_said, extensions)

	&& pfkey_build(pfkey_address_build(&extensions[SADB_EXT_ADDRESS_DST]
		, SADB_EXT_ADDRESS_DST, 0, 0
		, (struct sockaddr *)&dst_sin)
	    , "pfkey_addr_d Add AH SA", text_said, extensions)

	&& pfkey_build(pfkey_key_build(&extensions[SADB_EXT_KEY_AUTH]
		, SADB_EXT_KEY_AUTH, st->st_ah.keymat_len * IPSEC_PFKEYv2_ALIGN
		, ah_dst_keymat)
	    , "pfkey_key_a Add AH SA", text_said, extensions)

	&& finish_pfkey_msg(extensions, "Add AH SA", text_said)))

	    goto fail;

	said_next->dst = dst;
	said_next->spi = ah_spi;
	said_next->proto = IPPROTO_AH;
	said_next++;

#else /* !CONFIG_IPSEC_PFKEYv2 */
	/* ??? this length calculation is repulsive */
	ah.h.em_msglen =
	    EMT_SETSPI_FLEN + 3 * sizeof(u_short) + 2 * sizeof(u_char) + st->st_ah.keymat_len;
	ah.h.em_alg = alg;
	ah.u.d.ame_alen = ah_alen;
	ah.u.d.ame_klen = st->st_ah.keymat_len;
	ah.u.d.ame_replayp = TRUE;
	ah.u.d.ame_ooowin = 32;    /* ??? pure guess */
	memcpy(ah.u.d.ame_key, ah_dst_keymat, st->st_ah.keymat_len);

	DBG_cond_dump(DBG_KLIPS, "setup AH transform => KLIPS:\n", &ah, ah.h.em_msglen);
	if (!no_klips)
	{
	    if (write(fd, &ah, ah.h.em_msglen) != ah.h.em_msglen)
	    {
		char text_said[SATOA_BUF];

		log_errno((e, "write() of AH SPI %s failed"
		    , (satoa(ah.h.em_said, 0, text_said, sizeof(text_said)), text_said)));
		goto fail;
	    }
	}
	*said_next++ = ah.h.em_said;
#endif /* !CONFIG_IPSEC_PFKEYv2 */
    }

    /* set up ESP SA, if any */

    if (st->st_esp.present)
    {
	ipsec_spi_t esp_spi = inbound? st->st_esp.our_spi : st->st_esp.attrs.spi;
	u_char *esp_dst_keymat = inbound? st->st_esp.our_keymat : st->st_esp.peer_keymat;

	struct esp_info {
	    u_int8_t transid;	/* negotiated ESP transform */
	    u_int16_t auth;	/* negotiated AUTH */

	    size_t ivlen;	/* length of IV [no longer used] */
	    size_t enckeylen;	/* keylength for ESP transform */
	    size_t authkeylen;	/* keylength for AUTH */
#ifdef CONFIG_IPSEC_PFKEYv2
	    u_int8_t encryptalg;
	    u_int8_t authalg;
#else
	    int alg;	/* KLIPS encoding for ESP x AUTH */
#endif
	};

	const struct esp_info *ei;

#ifdef CONFIG_IPSEC_PFKEYv2
	static const struct esp_info esp_info[] = {
	    { ESP_NULL, AUTH_ALGORITHM_HMAC_MD5,
		0, 0, AHMD596_KLEN,
	        SADB_EALG_NULL, SADB_AALG_MD5HMAC },
	    { ESP_NULL, AUTH_ALGORITHM_HMAC_SHA1,
		0, 0, AHSHA196_KLEN,
	        SADB_EALG_NULL, SADB_AALG_SHA1HMAC },

	    { ESP_DES, AUTH_ALGORITHM_NONE,
		EMT_ESPDES_IV_SZ, EMT_ESPDES_KEY_SZ, 0,
	        SADB_EALG_DESCBC, SADB_AALG_NONE },
	    { ESP_DES, AUTH_ALGORITHM_HMAC_MD5,
		EMT_ESPDES_IV_SZ, EMT_ESPDES_KEY_SZ, AHMD596_KLEN,
	        SADB_EALG_DESCBC, SADB_AALG_MD5HMAC },
	    { ESP_DES, AUTH_ALGORITHM_HMAC_SHA1,
		EMT_ESPDES_IV_SZ, EMT_ESPDES_KEY_SZ,
	        AHSHA196_KLEN, SADB_EALG_DESCBC, SADB_AALG_SHA1HMAC },

	    { ESP_3DES, AUTH_ALGORITHM_NONE,
		EMT_ESPDES_IV_SZ, EMT_ESP3DES_KEY_SZ, 0,
	        SADB_EALG_3DESCBC, SADB_AALG_NONE },
	    { ESP_3DES, AUTH_ALGORITHM_HMAC_MD5,
		EMT_ESPDES_IV_SZ, EMT_ESP3DES_KEY_SZ, AHMD596_KLEN,
	        SADB_EALG_3DESCBC, SADB_AALG_MD5HMAC },
	    { ESP_3DES, AUTH_ALGORITHM_HMAC_SHA1,
		EMT_ESPDES_IV_SZ, EMT_ESP3DES_KEY_SZ, AHSHA196_KLEN,
	        SADB_EALG_3DESCBC, SADB_AALG_SHA1HMAC },
	};

#else /* !CONFIG_IPSEC_PFKEYv2 */
	static const struct esp_info esp_info[] = {
	    { ESP_NULL, AUTH_ALGORITHM_HMAC_MD5,
		0, 0, AHMD596_KLEN,
		XF_ESPNULLMD596 },
	    { ESP_NULL, AUTH_ALGORITHM_HMAC_SHA1,
		0, 0, AHSHA196_KLEN,
		XF_ESPNULLSHA196 },

	    { ESP_DES, AUTH_ALGORITHM_NONE,
		EMT_ESPDES_IV_SZ, EMT_ESPDES_KEY_SZ, 0,
		XF_ESPDES },
	    { ESP_DES, AUTH_ALGORITHM_HMAC_MD5,
		EMT_ESPDES_IV_SZ, EMT_ESPDES_KEY_SZ, AHMD596_KLEN,
		XF_ESPDESMD596 },
	    { ESP_DES, AUTH_ALGORITHM_HMAC_SHA1,
		EMT_ESPDES_IV_SZ, EMT_ESPDES_KEY_SZ, AHSHA196_KLEN,
		XF_ESPDESSHA196 },

	    { ESP_3DES, AUTH_ALGORITHM_NONE,
		EMT_ESPDES_IV_SZ, EMT_ESP3DES_KEY_SZ, 0,
		XF_ESP3DES },
	    { ESP_3DES, AUTH_ALGORITHM_HMAC_MD5,
		EMT_ESPDES_IV_SZ, EMT_ESP3DES_KEY_SZ, AHMD596_KLEN,
		XF_ESP3DESMD596 },
	    { ESP_3DES, AUTH_ALGORITHM_HMAC_SHA1,
		EMT_ESPDES_IV_SZ, EMT_ESP3DES_KEY_SZ, AHSHA196_KLEN,
		XF_ESP3DESSHA196 },
	};

	/* espblkrply_edata is from "ipsec_esp.h"
	 */
	union {
	    struct encap_msghdr h;
	    struct {
		char dummy[offsetof(struct encap_msghdr, em_dat)];
		struct espblkrply_edata d;	/* works for all */
	    } u;
	} esp;

	memset(&esp, '\0', sizeof(esp));

	esp.h = h;

	esp.h.em_type = EMT_SETSPI;

	esp.h.em_spi = esp_spi;
	esp.h.em_proto = IPPROTO_ESP;

#endif /* !CONFIG_IPSEC_PFKEYv2 */

	for (ei = esp_info; ; ei++)
	{
	    if (ei == &esp_info[elemsof(esp_info)])
	    {
		/* note: enum_show may use a static buffer, so two
		 * calls in one printf would be a mistake.
		 * enum_name does the same job, without a static buffer,
		 * assuming the name will be found.
		 */
		log("ESP transform %s / auth %s not implemented yet",
		    enum_name(&esp_transformid_names, st->st_esp.attrs.transid),
		    enum_name(&auth_alg_names, st->st_esp.attrs.auth));
		goto fail;
	    }

	    if (st->st_esp.attrs.transid == ei->transid
	    && st->st_esp.attrs.auth == ei->auth)
		break;
	}

	/* divide up keying material */
	passert(st->st_esp.keymat_len == ei->enckeylen + ei->authkeylen);

#ifdef CONFIG_IPSEC_PFKEYv2
	pfkey_extensions_init(extensions);

	set_text_said(text_said, dst, esp_spi, IPPROTO_ESP);

	if (!(pfkey_build(pfkey_msg_hdr_build(&extensions[0], SADB_ADD
		, SADB_SATYPE_ESP, 0, ++pfkey_seq, getpid())
	    , "pfkey_msg_hdr Add ESP SA", text_said, extensions)

	&& pfkey_build(pfkey_sa_build(&extensions[SADB_EXT_SA]
		, SADB_EXT_SA
		, esp_spi	/* in network order */
		, 32, SADB_SASTATE_MATURE, ei->authalg, ei->encryptalg, 0)
	    , "pfkey_sa Add ESP SA", text_said, extensions)

	&& pfkey_build(pfkey_address_build(&extensions[SADB_EXT_ADDRESS_SRC]
		, SADB_EXT_ADDRESS_SRC, 0, 0
		, (struct sockaddr*)&any_sin)
	    , "pfkey_addr_s Add ESP SA", text_said, extensions)

	&& pfkey_build(pfkey_address_build(&extensions[SADB_EXT_ADDRESS_DST]
		, SADB_EXT_ADDRESS_DST, 0, 0
		, (struct sockaddr*)&dst_sin)
	    , "pfkey_addr_d Add ESP SA", text_said, extensions)

	&& pfkey_build(pfkey_key_build(&extensions[SADB_EXT_KEY_AUTH]
		, SADB_EXT_KEY_AUTH, ei->authkeylen * IPSEC_PFKEYv2_ALIGN
		, esp_dst_keymat + ei->enckeylen)
	    , "pfkey_key_a Add ESP SA", text_said, extensions)

	&& pfkey_build(pfkey_key_build(&extensions[SADB_EXT_KEY_ENCRYPT]
		, SADB_EXT_KEY_ENCRYPT, ei->enckeylen * IPSEC_PFKEYv2_ALIGN
		, esp_dst_keymat)
	    , "pfkey_key_a Add ESP SA", text_said, extensions)

	&& finish_pfkey_msg(extensions, "Add ESP SA", text_said)))

	    goto fail;

	said_next->dst = dst;
	said_next->spi = esp_spi;
	said_next->proto = IPPROTO_ESP;
	said_next++;

#else /* !CONFIG_IPSEC_PFKEYv2 */

	esp.h.em_alg = ei->alg;

	passert(ei->enckeylen <= sizeof(esp.u.d.eme_key));
	passert(ei->authkeylen <= sizeof(esp.u.d.ame_key));

	memcpy(esp.u.d.eme_key, esp_dst_keymat, ei->enckeylen);
	esp.u.d.eme_klen = ei->enckeylen;

	memcpy(esp.u.d.ame_key
	    , (inbound? st->st_esp.our_keymat : st->st_esp.peer_keymat)
		+ ei->enckeylen
	    , ei->authkeylen);
	esp.u.d.ame_klen = ei->authkeylen;

	/* window: always full size */
	esp.u.d.eme_ooowin = 32;

	esp.h.em_msglen = EMT_SETSPI_FLEN + sizeof(struct espblkrply_edata);

	DBG_cond_dump(DBG_KLIPS, "setup ESP transform => KLIPS:\n", &esp, esp.h.em_msglen);
	if (!no_klips)
	{
	    if (write(fd, &esp, esp.h.em_msglen) != esp.h.em_msglen)
	    {
		char text_said[SATOA_BUF];

		log_errno((e, "write() of ESP SPI %s failed"
		    , (satoa(esp.h.em_said, 0, text_said, sizeof(text_said)), text_said)));
		goto fail;
	    }
	}
	*said_next++ = esp.h.em_said;
#endif /* !CONFIG_IPSEC_PFKEYv2 */
    }

    /* set up IP in IP pseudo SA for tunnelling */

    if (!inbound
    && (st->st_ah.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL
     || st->st_esp.attrs.encapsulation == ENCAPSULATION_MODE_TUNNEL))
    {
	/* XXX hack alert -- we SHOULD NOT HAVE TO HAVE A DIFFERENT SPI
	 * XXX FOR IP-in-IP ENCAPSULATION!
	 */

#ifndef CONFIG_IPSEC_PFKEYv2
	/* ipe4_xdata is from "ipsec_ipe4.h" */
	union {
	    struct encap_msghdr h;
	    struct {
		char dummy[offsetof(struct encap_msghdr, em_dat)];
		struct ipe4_xdata d;
	    } u;
	} m;	/* encapsulating message */
#endif

	/* Allocate an SPI for the outgoing tunnel.
	 * Since our peer will never see this,
	 * and it comes from its own number space,
	 * it is purely a local implementation wart.
	 */
	{
	    static ipsec_spi_t last_tunnel_spi = 0x100;

	    st->st_tunnel_spi = htonl(++last_tunnel_spi);
	}

#ifdef CONFIG_IPSEC_PFKEYv2
	pfkey_extensions_init(extensions);
	set_text_said(text_said
	    , c->that.host_addr, st->st_tunnel_spi, IPPROTO_IPIP);

	if (!(pfkey_build(pfkey_msg_hdr_build(&extensions[0]
		, SADB_ADD, SADB_X_SATYPE_IPIP, 0, ++pfkey_seq, getpid())
	    , "pfkey_msg_hdr Add outgoing IPIP SA", text_said, extensions)

	&& pfkey_build(pfkey_sa_build(&extensions[SADB_EXT_SA]
		, SADB_EXT_SA
		, st->st_tunnel_spi		/* in network order */
		, 0, SADB_SASTATE_MATURE
		, 0	/* authalg */
		, 0	/* encryptalg */
		, 0)
	    , "pfkey_sa Add outgoing IPIP SA", text_said, extensions)

	&& pfkey_build(pfkey_address_build(&extensions[SADB_EXT_ADDRESS_SRC]
		, SADB_EXT_ADDRESS_SRC, 0, 0
		, (struct sockaddr*)&src_sin)
	    , "pfkey_addr_s Add outgoing IPIP SA", text_said, extensions)

	&& pfkey_build(pfkey_address_build(&extensions[SADB_EXT_ADDRESS_DST]
		, SADB_EXT_ADDRESS_DST, 0, 0
		, (struct sockaddr*)&dst_sin)
	    , "pfkey_addr_d Add outgoing IPIP SA", text_said, extensions)

	&& finish_pfkey_msg(extensions, "Add outgoing IPIP SA", text_said)))

	    goto fail;

	said_next->dst = dst;
	said_next->spi = st->st_tunnel_spi;
	said_next->proto = IPPROTO_IPIP;
	said_next++;

#else /* !CONFIG_IPSEC_PFKEYv2 */

	memset(&m, '\0', sizeof(m));
	m.h = h;

	m.h.em_type = EMT_SETSPI;
	m.h.em_spi = st->st_tunnel_spi;
	m.h.em_proto = IPPROTO_IPIP;

	m.h.em_msglen = EMT_SETSPI_FLEN + EMT_IPE4_ULEN;
	m.h.em_alg = XF_IP4;

	m.u.d.i4_src = src;
	m.u.d.i4_dst = dst;

	DBG_cond_dump(DBG_KLIPS, "setup encapsulation => KLIPS:\n", &m, m.h.em_msglen);

	if (!no_klips)
	{
	    if (write(fd, &m, m.h.em_msglen) != m.h.em_msglen)
	    {
		char text_said[SATOA_BUF];

		log_errno((e, "write() of encapsulation %s failed"
		    , (satoa(m.h.em_said, 0, text_said, sizeof(text_said)), text_said)));
		goto fail;
	    }
	}
	*said_next++ = m.h.em_said;
#endif /* !CONFIG_IPSEC_PFKEYv2 */
    }

    if (!inbound && said_next > &said[1])
    {
	/* Group the multiple outbound SPIs
	 * All the info lives in the header!
	 * EM_MAXRELSPIS is max number of groupings.
	 */
	struct encap_msghdr m;
	int i;

	memset(&m, '\0', sizeof(m));

	m.em_magic = EM_MAGIC;
	m.em_version = 0;
	m.em_type = EMT_GRPSPIS;
	m.em_msglen = EMT_GRPSPIS_FLEN;

	/* reverse order of creation in above code:
	 * encap, esp, ah.
	 */
	for (i = 0; i != said_next - said; i++)
	{
	    m.em_rel[i].emr_said = said_next[ - 1 - i];
	    m.em_msglen += sizeof(m.em_rel[0]);

	    DBG(DBG_KLIPS,
		char text_said[SATOA_BUF];

		DBG_log("grouping %s"
		    , (satoa(m.em_rel[i].emr_said, 0, text_said, sizeof(text_said)), text_said)));
	}

	DBG_cond_dump(DBG_KLIPS, "setup grouping => KLIPS:\n", &m, m.em_msglen);

	if (!no_klips)
	{
	    if (write(fd, &m, m.em_msglen) != m.em_msglen)
	    {
		log_errno((e, "write() of grouping failed"));
		goto fail;
	    }
	}
	/* could update said, but it will not be used */
    }

    return TRUE;

fail:
    {
	/* undo the done SPIs */
	while (said_next-- != said)
	    (void) del_spi(fd, said_next->spi, said_next->proto
		, said_next->dst);
	return FALSE;
    }
}

static bool
setup_ipsec_sa(int fd, struct state *st)
{
    return setup_half_ipsec_sa(fd, st, TRUE)
	&& setup_half_ipsec_sa(fd, st, FALSE);
}

/* teardown_ipsec_sa is a canibalized version of setup_ipsec_sa */

static bool
teardown_ipsec_sa(int fd, struct state *st)
{
    /* We need to tear down both a Sending SA and a Receiving SA.
     * On the receiving side, only AH and ESP are possible (no encapsulation).
     * On the sending side, first of AH and ESP is fine:
     * since only one of a group needs to be deleted, either there
     * was no group, or the AH or ESP will be in it.
     */
    struct connection *c = st->st_connection;
    ipsec_spi_t *an_out_spi = NULL;
    int an_out_proto;

    /* delete receiving SPIs */

    if (st->st_ah.present)
    {
	(void) del_spi(fd, st->st_ah.our_spi, IPPROTO_AH, c->this.host_addr);
	an_out_spi = &st->st_ah.attrs.spi;
	an_out_proto = IPPROTO_AH;
    }

    if (st->st_esp.present)
    {
	(void) del_spi(fd, st->st_esp.our_spi, IPPROTO_ESP, c->this.host_addr);
	an_out_spi = &st->st_esp.attrs.spi;
	an_out_proto = IPPROTO_ESP;
    }

    /* delete some outgoing SPI (grouping gets rest) */

    passert(an_out_spi != NULL);
    (void) del_spi(fd, *an_out_spi, an_out_proto, c->that.host_addr);
    return TRUE;
}

#endif /* KLIPS */

bool
install_ipsec_sa(struct state *st)
{
#ifdef KLIPS
    int fd = NULL_FD;
    bool res;
    struct connection
	*c = st->st_connection,
	*ero = route_owner(c, TRUE);	/* who, if anyone, owns our eroute? */

    if (ero != NULL && ero != c && ero->eroute_owner != SOS_NOBODY)
    {
	log("cannot install eroute -- it is in use for \"%s\"", ero->name);
	return FALSE;	/* another connection already using the eroute */
    }

    if (!route_connection(c))
	return FALSE;

    if (!no_klips)
    {
	fd = open("/dev/ipsec", O_WRONLY);
	if (fd < 0)
	{
	    log_errno((e, "open() of /dev/ipsec failed in install_ipsec_sa()"));
	    return FALSE;
	}
    }

    /* no other connection has this eroute, but
     * this connection might have it for another SA.
     * This is expected when rekeying.
     * If another SA has it, we replace the eroute.
     */
    res = setup_ipsec_sa(fd, st)
	&& (c->eroute_owner == SOS_NOBODY
	    ? do_eroute(fd, st, EMT_SETEROUTE, "set")
	      && do_command(st->st_connection, "up")
	    : do_eroute(fd, st, EMT_RPLACEROUTE, "replace"));

    if (res)
	c->eroute_owner = st->st_serialno;

    if (fd != NULL_FD)
	close(fd);

    return res;
#else /* !KLIPS */
    DBG(DBG_CONTROL, DBG_log("install_ipsec_sa()"));
    return route_connection(st->st_connection);
#endif /* !KLIPS */
}

bool
delete_ipsec_sa(struct state *st)
{
#ifdef KLIPS
    struct connection *c = st->st_connection;
    bool own_eroute = c->eroute_owner == st->st_serialno;
    int fd = NULL_FD;
    bool res = TRUE;

    if (!no_klips)
    {
	fd = open("/dev/ipsec", O_WRONLY);
	if (fd < 0)
	{
	    log_errno((e, "open() of /dev/ipsec failed in delete_ipsec_sa()"));
	    return FALSE;
	}
    }

    if (own_eroute)
    {
	res = do_command(st->st_connection, "down")
	    && do_eroute(fd, st, EMT_DELEROUTE, "delete");
	if (res)
	    c->eroute_owner = SOS_NOBODY;
    }
    if (res)
	res = teardown_ipsec_sa(fd, st);

    if (fd != NULL_FD)
	close(fd);	/* rain or shine */

    return res;
#else /* !KLIPS */
    DBG(DBG_CONTROL, DBG_log("if I knew how, I'd do_eroute() and teardown_ipsec_sa()"));
    return TRUE;
#endif /* !KLIPS */
}
