/* information about connections between hosts and clients
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: connections.c,v 1.50 2000/01/06 22:15:33 dhr Exp $
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "id.h"
#include "connections.h"	/* needs id.h */
#include "packet.h"
#include "demux.h"	/* needs packet.h */
#include "state.h"
#include "ipsec_doi.h"	/* needs demux.h and state.h */
#include "server.h"
#include "kernel.h"	/* for no_klips */
#include "log.h"
#include "preshared.h"
#include "whack.h"

static struct connection *connections = NULL;

/* check to see that almost all particulars of peers match */
bool
same_peers(const struct connection *c, const struct connection *d
, bool wildOK, const struct id *his_id)
{
    return c->this.host_addr.s_addr == d->this.host_addr.s_addr
	&& (c->that.host_addr.s_addr == d->that.host_addr.s_addr
#ifdef ROAD_WARRIOR_FUDGE
	       || (wildOK && HasWildcardIP(*d))
#endif /* ROAD_WARRIOR_FUDGE */
	   )
	&& c->this.host_port == d->this.host_port
	&& c->that.host_port == d->that.host_port
	&& id_same(&c->this.id, &d->this.id)
	&& id_same(his_id == NULL? &c->that.id : his_id, &d->that.id);
}

/* find a connection by name.
 * move the winner (if any) to the front.
 * If none is found, and wft is not NULL_FD, a diagnostic is logged to whack.
 */
struct connection *
con_by_name(const char *nm, int wfd)
{
    struct connection *p, *prev;

    prev = NULL;
    for (p = connections; ; p = p->next)
    {
	if (p == NULL)
	{
	    if (wfd != NULL_FD)
		whack_log(wfd, RC_UNKNOWN_NAME, "no connection with that name");
	    break;
	}
	if (strcmp(p->name, nm) == 0)
	{
	    if (prev != NULL)
	    {
		struct connection *nxt = p->next;

		p->next = connections;
		connections = p;
		prev->next = nxt;
	    }
	    break;
	}
	prev = p;
    }
    return p;
}

static void
unorient_connection(struct connection *c)
{
#ifdef ROAD_WARRIOR_FUDGE
    if (c->rw_state == rwcs_instance)
    {
	/* This does everything we need.
	 * Note that we will be called by delete_connection,
	 * but rw_state will be rwcs_going_away.
	 */
	delete_connection(c);
    }
#endif /* ROAD_WARRIOR_FUDGE */
    {
	delete_states_by_connection(c);
	unroute_connection(c);
	c->interface = NULL;
    }
}

/* Delete a connection */

void
delete_connection(struct connection *c)
{
    struct connection *old_cur_connection = cur_connection;

    cur_connection = c;
#ifdef ROAD_WARRIOR_FUDGE
    /* Must be careful to avoid circularity:
     * we mark c as going away so it won't get deleted recursively.
     */
    passert(c->rw_state != rwcs_going_away);
    if (c->rw_state == rwcs_instance)
    {
	char that_host[ADDRTOA_BUF];

	addrtoa(c->that.host_addr, 0, that_host, sizeof(that_host));
	log("deleting connection \"%s\" with Road Warrior %s"
	    , c->name, that_host);
	c->rw_state = rwcs_going_away;
    }
    else
#endif /* ROAD_WARRIOR_FUDGE */
    {
	log("deleting connection");
    }
    unorient_connection(c);	/* won't delete c */

    /* find and delete c from connections list */
    {
	struct connection **p;

	for (p = &connections; *p != c; p = &(*p)->next)
	    passert(*p != NULL);    /* we must not come up empty-handed */
	*p = c->next;
    }
    cur_connection = old_cur_connection == c? NULL : old_cur_connection;
    pfreeany(c->name);
    free_id_content(&c->this.id);
    pfreeany(c->this.updown);
    free_id_content(&c->that.id);
    pfreeany(c->that.updown);
    pfree(c);
}

void
delete_every_connection(void)
{
    while (connections != NULL)
	delete_connection(connections);
}

void
release_interface(struct iface *i)
{
    struct connection *c, *n;

    passert(i != NULL);
    for (c = connections; c != NULL; c = n)
    {
	n = c->next;
	if (c->interface == i)
	    unorient_connection(c);	/* may delete c */
    }
}

static void
default_end(struct end *e, struct in_addr dflt_nexthop)
{
    /* default ID to IP (but only if not NO_IP -- WildCard) */
    if (e->id.kind == ID_NONE && e->host_addr.s_addr != NO_IP)
    {
	e->id.kind = ID_IPV4_ADDR;
	e->id.ipv4 = e->host_addr;
    }

    /* default nexthop to other side */
    if (e->host_nexthop.s_addr == NO_IP)
	e->host_nexthop = dflt_nexthop;

    /* default client to subnet containing only self */
    if (!e->has_client)
    {
	e->client_net = e->host_addr;
	e->client_mask = mask32.sin_addr;
    }
}

/* format the topology of an end, leaving out defaults
 * Note: if that==NULL, skip nexthop
 */
size_t
format_end(char *buf, size_t buf_len, struct end *this, struct end *that, bool is_left)
{
    char client[SUBNETTOA_BUF];
    const char *client_sep = "";
    char host[ADDRTOA_BUF];
    char host_port[10];
    char host_id[30];
    char hop[ADDRTOA_BUF];
    const char *hop_sep = "";

    /* [client===] */
    client[0] = '\0';
    if (this->has_client)
    {
	subnettoa(this->client_net, this->client_mask, 0
	    , client, sizeof(client));
	client_sep = "===";
    }

    /* host */
    addrtoa(this->host_addr, 0, host, sizeof(host));
    host_port[0] = '\0';
    if (this->host_port != IKE_UDP_PORT)
	snprintf(host_port, sizeof(host_port), ":%u"
	    , this->host_port);

    /* id, if different from host */
    host_id[0] = '\0';
    if (!(this->id.kind == ID_NONE
    || (this->id.kind == ID_IPV4_ADDR && this->id.ipv4.s_addr == this->host_addr.s_addr)))
    {
	int len = idtoa(&this->id, host_id+1, sizeof(host_id)-2);

	host_id[0] = '[';
	strcpy(&host_id[len < 0? sizeof(host_id)-2 : 1 + len], "]");
    }

    /* [---hop] */
    hop[0] = '\0';
    hop_sep = "";
    if (that != NULL && this->host_nexthop.s_addr != that->host_addr.s_addr)
    {
	addrtoa(this->host_nexthop, 0, hop, sizeof(hop));
	hop_sep = "---";
    }

    if (is_left)
	snprintf(buf, buf_len, "%s%s%s%s%s%s%s"
	    , client, client_sep
	    , host, host_port, host_id
	    , hop_sep, hop);
    else
	snprintf(buf, buf_len, "%s%s%s%s%s%s%s"
	    , hop, hop_sep
	    , host, host_port, host_id
	    , client_sep, client);
    return strlen(buf);
}

static void
unshare_connection_strings(struct connection *c)
{
    c->name = clone_str(c->name, "connection name");

    clone_id_content(&c->this.id);
    c->this.updown = clone_str(c->this.updown, "updown");
    clone_id_content(&c->that.id);
    c->that.updown = clone_str(c->that.updown, "updown");
}

static void
extract_end(struct end *dst, const struct whack_end *src, const char *which, int whack_fd)
{
    /* decode id, if any */
    if (src->id == NULL)
    {
	dst->id.kind = ID_NONE;
    }
    else
    {
	const char *ugh = atoid(src->id, &dst->id);

	if (ugh != NULL)
	{
	    log("bad %s --id: %s (ignored)", which, ugh);
	    whack_log(whack_fd, RC_BADID
		, "bad %s --id: %s (ignored)", which, ugh);
	    dst->id = empty_id;	/* ignore bad one */
	}
    }

    /* the rest is simple copying of corresponding fields */
    dst->host_addr = src->host_addr;
    dst->host_nexthop = src->host_nexthop;
    dst->client_net = src->client_net;
    dst->client_mask = src->client_mask;

    dst->has_client = src->has_client;
    dst->updown = src->updown;
    dst->host_port = src->host_port;
}

void
add_connection(const struct whack_message *wm, int whack_fd)
{
    if (con_by_name(wm->name, NULL_FD) != NULL)
    {
	log("attempt to redefine connection \"%s\"", wm->name);
	whack_log(whack_fd, RC_DUPNAME, "attempt to redefine connection \"%s\"", wm->name);
    }
    else if (wm->left.host_addr.s_addr == NO_IP
    && wm->right.host_addr.s_addr == NO_IP)
    {
	log("connection must specify host IP address for our side");
	whack_log(whack_fd, RC_ORIENT, "connection must specify host IP address for our side");
    }
    else
    {
	struct connection *c = alloc_thing(struct connection, "struct connection");

	c->name = wm->name;

	c->policy = wm->policy;

	c->sa_ike_life_seconds = wm->sa_ike_life_seconds;
	c->sa_ipsec_life_seconds = wm->sa_ipsec_life_seconds;
	c->sa_rekey_margin = wm->sa_rekey_margin;
	c->sa_rekey_fuzz = wm->sa_rekey_fuzz;
	c->sa_keying_tries = wm->sa_keying_tries;

	extract_end(&c->this, &wm->left, "left", whack_fd);
	extract_end(&c->that, &wm->right, "right", whack_fd);

	default_end(&c->this, c->that.host_addr);
	default_end(&c->that, c->this.host_addr);

	/* force any wildcard host IP address to that end */
	if (wm->left.host_addr.s_addr == NO_IP)
	{
	    struct end t = c->this;

	    c->this = c->that;
	    c->that = t;
	}

	/* set internal fields */
	c->next = connections;
	connections = c;
	c->interface = NULL;
	c->routed = FALSE;
	c->newest_isakmp_sa = SOS_NOBODY;
	c->newest_ipsec_sa = SOS_NOBODY;
	c->eroute_owner = SOS_NOBODY;

#ifdef ROAD_WARRIOR_FUDGE
	c->rw_state = rwcs_permanent;
	c->parent = NULL;
#endif

	unshare_connection_strings(c);

	/* log all about this connection */
	log("added connection description \"%s\"", c->name);
	DBG(DBG_CONTROL,
	    char lhs[100];
	    char rhs[100];

	    (void) format_end(lhs, sizeof(lhs), &c->this, &c->that, TRUE);
	    (void) format_end(rhs, sizeof(rhs), &c->that, &c->this, FALSE);

	    DBG_log("%s...%s", lhs, rhs);

	    DBG_log("ike_life: %lus; ipsec_life: %lus; rekey_margin: %lus;"
		" rekey_fuzz: %lu%%; keyingtries: %lu; policy: %s"
		, (unsigned long) c->sa_ike_life_seconds
		, (unsigned long) c->sa_ipsec_life_seconds
		, (unsigned long) c->sa_rekey_margin
		, (unsigned long) c->sa_rekey_fuzz
		, (unsigned long) c->sa_keying_tries
		, bitnamesof(sa_policy_bit_names, c->policy));
	);
    }
}

#ifdef ROAD_WARRIOR_FUDGE
struct connection *
rw_connection(struct connection *c, struct in_addr him)
{
    struct connection *d = clone_thing(*c, "temporary connection");

    unshare_connection_strings(c);

    DBG(DBG_CONTROL, DBG_log("instantiating \"%s\" for %s"
	, c->name, inet_ntoa(him)));
    d->rw_state = rwcs_instance;
    d->parent = c;

    passert(oriented(*d));
    passert(HasWildcardIP(*d));
    d->that.host_addr = him;
    default_end(&d->that, d->this.host_addr);

    /* We cannot guess what our next_hop should be, but if it was
     * explicitly specified as 0.0.0.0, we set it to be him.
     * (whack will not allow nexthop to be elided in RW case.)
     */
    default_end(&d->this, d->that.host_addr);

    /* set internal fields */
    d->next = connections;
    connections = d;
    d->routed = FALSE;
    d->newest_isakmp_sa = SOS_NOBODY;
    d->newest_ipsec_sa = SOS_NOBODY;
    d->eroute_owner = SOS_NOBODY;

    return d;
}
#endif /* ROAD_WARRIOR_FUDGE */

bool
orient(struct connection *c, bool must)
{
    if (!oriented(*c))
    {
	struct iface *p;

	/* Note: this loop does not stop when it finds a match:
	 * it continues checking to catch any ambiguity.
	 */
	for (p = interfaces; p != NULL; p = p->next)
	{
	    for (;;)
	    {
		/* check if this interface matches this end */
		if (c->this.host_addr.s_addr == p->addr.s_addr
		&& (!no_klips || c->this.host_port == pluto_port))
		{
		    if (oriented(*c))
		    {
			if (c->interface == p)
			    log("both sides of \"%s\" are our interfaces!"
				, c->name);
			else
			    log("two interfaces match \"%s\"", c->name);
			c->interface = NULL;	/* withdraw orientation */
			return FALSE;
		    }
		    c->interface = p;
		}

		/* done with this interface if it doesn't match that end */
		if (!(c->that.host_addr.s_addr == p->addr.s_addr
		&& (!no_klips || c->that.host_port == pluto_port)))
		    break;

		/* swap ends and try again */
		{
		    struct end t = c->this;

		    c->this = c->that;
		    c->that = t;
		}
	    }
	}
	if (!oriented(*c))
	{
	    if (must)
		log("cannot find interface for connection \"%s\"", c->name);
	    return FALSE;
	}
    }
    return TRUE;
}

void
initiate_connection(const char *name, int whackfd)
{
    struct connection *c = con_by_name(name, whackfd);

    if (c != NULL)
    {
	cur_connection = c;
	if (!orient(c, TRUE))
	{
	    log("could not orient connection");
	    whack_log(whackfd, RC_ORIENT, "could not orient connection");
	}
#ifdef ROAD_WARRIOR_FUDGE
	else if (HasWildcardIP(*c))
	{
	    log("cannot initiate connection without peer IP");
	    whack_log(whackfd
		, RC_NOPEERIP, "cannot initiate connection without peer IP");
	}
#endif /* ROAD_WARRIOR_FUDGE */
	else
	{
	    /* We will only request an IPsec SA if policy isn't empty
	     * (ignoring Main Mode items).
	     * This is a fudge, but not yet important.
	     */
	    ipsecdoi_initiate(whackfd == NULL_FD? NULL_FD : dup(whackfd)
		, c
		, (c->policy>>POLICY_IPSEC_SHIFT) != 0
		, c->policy
		, 1);
	}
	cur_connection = NULL;
    }
}

void
terminate_connection(const char *nm, int whackfd)
{
#ifdef ROAD_WARRIOR_FUDGE
    /* Loop because more than one may match in ROAD_WARRIOR_FUDGE.
     * But at least one is required (enforced by con_by_name).
     */
    struct connection *c, *n;

    for (c = con_by_name(nm, whackfd); c != NULL; c = n)
    {
	n = c->next;	/* grab this before c might disappear */
	if (strcmp(c->name, nm) == 0)
	{
	    cur_connection = c;
	    log("terminating SAs using connection");
	    delete_states_by_connection(c);
	    cur_connection = NULL;
	}
    }
#else /* !ROAD_WARRIOR_FUDGE */
    struct connection *c = con_by_name(nm, whackfd);

    if (c != NULL)
    {
	cur_connection = c;
	log("terminating SAs using connection");
	delete_states_by_connection(c);
	cur_connection = NULL;
    }
#endif /* !ROAD_WARRIOR_FUDGE */
}

/* check nexthop safety
 * Our nexthop must not be within a routed client subnet, and vice versa.
 * Note: we don't think this is true.  We think that KLIPS will
 * not process a packet output by an eroute.
 */
#ifdef NEVER
bool
check_nexthop(const struct connection *c)
{
    struct connection *d;

    if (inside_client(c->this.host_nexthop, c->that))
    {
	log("cannot perform routing for connection \"%s\""
	    " because nexthop is within peer's client network",
	    c->name);
	return FALSE;
    }

    for (d = connections; d != NULL; d = d->next)
    {
	if (d->routed)
	{
	    if (inside_client(c->this.host_nexthop, d->that))
	    {
		log("cannot do routing for connection \"%s\"
		    " because nexthop is contained in"
		    " existing routing for connection \"%s\"",
		    c->name, d->name);
		return FALSE;
	    }
	    if (inside_client(d->this.host_nexthop, c->that))
	    {
		log("cannot do routing for connection \"%s\"
		    " because it contains nexthop of"
		    " existing routing for connection \"%s\"",
		    c->name, d->name);
		return FALSE;
	    }
	}
    }
    return TRUE;
}
#endif /* NEVER */

/* Find a connection that owns the route to a connection's peer's client.
 * Preference is given to one that has an eroute too.
 * If eroute is true, our client must match too (as for eroutes).
 */
struct connection *
route_owner(struct connection *c, bool eroute)
{
    struct connection
	*d,
	*best = NULL;

    passert(oriented(*c));

    if (c->routed && c->eroute_owner != SOS_NOBODY)
	return c;

    for (d = connections; d != NULL; d = d->next)
    {
	if (d->routed)
	{
	    passert(oriented(*d));
	    if (same_client(c->that, d->that)
	    && (!eroute || same_client(c->this, d->this)))
	    {
		if (d->eroute_owner != SOS_NOBODY)
		    return d;	/* definite winner */

		best = d;	/* winner if no better comes along */
	    }
	}
    }
    return best;
}

struct connection *
find_host_connection(struct in_addr me, struct in_addr him)
{
    struct connection *d;

    for (d = connections; d != NULL; d = d->next)
    {
	if (d->this.host_addr.s_addr == me.s_addr
	|| d->that.host_addr.s_addr == me.s_addr)
	{
	    if (!oriented(*d))
		(void)orient(d, FALSE);

	    /* check this.host last -- it is almost certainly us */
	    if (d->that.host_addr.s_addr == him.s_addr
	    && d->this.host_addr.s_addr == me.s_addr
	    && (!no_klips || d->this.host_port == pluto_port))
		return d;
	}
    }
    return NULL;
}

/* given an up-until-now satisfactory connection, find the best
 * connection now that we've got the Id Payload from the peer.
 * We can only accept connections with compatible authentication.
 */
struct connection *
refine_host_connection(const struct state *st, const struct id *peer_id
, bool initiator)
{
    struct connection *c = st->st_connection;
    u_int16_t auth = st->st_oakley.auth;
    struct connection *best = NULL;
    struct connection *d;
    lset_t auth_policy;
    const chunk_t *psk;
    const struct RSA_public_key *his_RSA_pub;
    const struct RSA_private_key *my_RSA_pri;

    if (id_same(&c->that.id, peer_id))
	return c;

    switch (auth)
    {
    case OAKLEY_PRESHARED_KEY:
	auth_policy = POLICY_PSK;
	psk = get_preshared_secret(c);
	if (psk == NULL)
	    return NULL;	/* cannot determine PSK! */
	break;
    case OAKLEY_RSA_SIG:
	auth_policy = POLICY_RSASIG;
	my_RSA_pri = get_RSA_private_key(c);
	if (my_RSA_pri == NULL)
	    return NULL;	/* cannot determine my RSA private key! */
	if (initiator)
	{
	    his_RSA_pub = get_RSA_public_key(&c->that.id);
	    if (his_RSA_pub == NULL)
		return NULL;	/* cannot determine his RSA public key! */
	}
	break;
    default:
	passert(FALSE);
    }
    /* The current connection won't do: search for one that will.
     * Made more complicated by the fact that the connection might
     * not be oriented and it might have 0.0.0.0 for peer.
     */
    for (d = connections; d != NULL; d = d->next)
    {
	if (c->this.host_addr.s_addr == d->this.host_addr.s_addr
	|| c->this.host_addr.s_addr == d->that.host_addr.s_addr)
	{
	    /* one end is us -- try to orient & investigate further */
	    if (!oriented(*d))
		(void)orient(d, FALSE);

	    if (same_peers(c, d, TRUE, peer_id))
	    {
		/* check whether authentication would work the same */

		if (!((d->policy & auth_policy) != 0
		|| (d->policy & POLICY_ISAKMP_MASK) == 0))
		    continue;	/* our auth isn't OK for this connection */

		switch (auth)
		{
		case OAKLEY_PRESHARED_KEY:
		    if (psk != get_preshared_secret(d))
			continue;	/* different secret */
		    break;
		case OAKLEY_RSA_SIG:
		    if (my_RSA_pri != get_RSA_private_key(d))
			continue;	/* different private key */
		    if (initiator
		    && his_RSA_pub != get_RSA_public_key(&d->that.id))
			continue;	/* different public key */
		    break;
		default:
		    passert(FALSE);
		}

		/* passed all the tests */

#ifdef ROAD_WARRIOR_FUDGE
		if (HasWildcardIP(*c))
		{
		    best = d;	/* RW match -- best so far */
		}
		else
#endif /* ROAD_WARRIOR_FUDGE */
		{
		    return d;	/* exact match -- we have a winner */
		}
	    }
	}
    }
    return best;
}

/* given a connection suitable for ISAKMP (i.e. the hosts match)
 * find a one suitable for IPSEC (i.e. with matching clients).
 */
struct connection *
find_client_connection(
    struct connection *c,
    struct in_addr our_net, struct in_addr our_mask,
    struct in_addr peer_net, struct in_addr peer_mask)
{
    struct connection
	*d,
	*unrouted = NULL;

    /* give priority to current connection
     * but even greater priority to a routed connection
     */

    if (c->this.client_net.s_addr == our_net.s_addr
    && c->this.client_mask.s_addr == our_mask.s_addr
    && c->that.client_net.s_addr == peer_net.s_addr
    && c->that.client_mask.s_addr == peer_mask.s_addr)
    {
	passert(oriented(*c));
	if (c->routed)
	    return c;
	unrouted = c;
    }

    for (d = connections; d != NULL; d = d->next)
    {
	if (d->this.host_addr.s_addr == c->this.host_addr.s_addr
	|| d->that.host_addr.s_addr == c->this.host_addr.s_addr)
	{
	    if (!oriented(*d))
		(void)orient(d, FALSE);

	    if (same_peers(c, d, FALSE, NULL)
	    && our_net.s_addr ==  d->this.client_net.s_addr
	    && our_mask.s_addr ==  d->this.client_mask.s_addr
	    && peer_net.s_addr ==  d->that.client_net.s_addr
	    && peer_mask.s_addr ==  d->that.client_mask.s_addr)
	    {
		if (d->routed)
		    return d;
		if (unrouted == NULL)
		    unrouted = d;
	    }
	}
    }
    return unrouted;
}

void
show_connections_status(int wfd)
{
    struct connection *c;

    for (c = connections; c != NULL; c = c->next)
    {
	const char *routed = c->routed? "; routed" : "";
	const char *ifn = oriented(*c)? c->interface->rname : "";

	/* show topology */
	{
	    char lhs[100];
	    size_t ll = format_end(lhs, sizeof(lhs), &c->this, &c->that, TRUE);
	    char rhs[100];
	    size_t rl = format_end(rhs, sizeof(rhs), &c->that, &c->this, FALSE);

	    if (ll + rl < 70)
	    {
		/* display on one line */
		whack_log(wfd, RC_COMMENT, "\"%s\": %s...%s", c->name, lhs, rhs);
	    }
	    else
	    {
		/* split over two lines */
		whack_log(wfd, RC_COMMENT, "\"%s\": %s...", c->name, lhs);
		whack_log(wfd, RC_COMMENT, "\"%s\": ...%s", c->name, rhs);
	    }
	}

	whack_log(wfd, RC_COMMENT
	    , "\"%s\":  ike_life: %lus; ipsec_life: %lus; rekey_margin: %lus;"
	    " rekey_fuzz: %lu%%; keyingtries: %lu"
	    , c->name
	    , (unsigned long) c->sa_ike_life_seconds
	    , (unsigned long) c->sa_ipsec_life_seconds
	    , (unsigned long) c->sa_rekey_margin
	    , (unsigned long) c->sa_rekey_fuzz
	    , (unsigned long) c->sa_keying_tries);


	whack_log(wfd, RC_COMMENT
	    , "\"%s\":  policy: %s; interface: %s%s"
	    , c->name
	    , bitnamesof(sa_policy_bit_names, c->policy)
	    , ifn
	    , routed);

	whack_log(wfd, RC_COMMENT
	    , "\"%s\":  newest ISAKMP SA: #%ld; newest IPsec SA: #%ld; eroute owner: #%lu"
	    , c->name
	    , c->newest_isakmp_sa
	    , c->newest_ipsec_sa
	    , c->eroute_owner);
    }
}
